#include <linux/sched.h>
#include <linux/errno.h>
#include <linux/signal.h>
#include <linux/stat.h>
#include <linux/proc_fs.h>
#include <asm/uaccess.h>

/*
 * Called when the time is modified below with settimeofday in order
 * to notify all of the tasks that requested notification.
 */
void do_notify_timechange(void)
{
	struct task_struct *p;

	read_lock(&tasklist_lock);
	for_each_process(p) {
		if (p->settime_sig)
			group_send_sig_info(p->settime_sig, SEND_SIG_PRIV, p);
	}
	read_unlock(&tasklist_lock);
}

/*
 * Set a signal to be delivered to the calling process whenever
 * time of day is changed. A signal value of zero indicates that
 * no signal should be sent.  On success, the call returns the 
 * old value of the signal.  A return value of -1 indicates an error.
 */
int do_notify_settime(int sig)
{
	int ret = -EINVAL;

	if ((sig > 0) && (sig <= _NSIG)) {
		ret = current->settime_sig;
		current->settime_sig = sig;
	}
	return ret;
}

#ifdef CONFIG_PROC_FS
#define NOTIFY_SETTIME_PROCFS_NAME "notify_settime_signal"

static int show_notify_settime_signal(char *page, char **start, off_t offset,
				      int count, int *eof, void *data)
{
	return snprintf(page, count, "%d\n", current->settime_sig);
}

static int store_notify_settime_signal(struct file *file, const char __user * buffer,
				       unsigned long count, void *data)
{
	char str[16];
	int sig = 0;

	if (count > sizeof(str)-1) return -EINVAL;

	memset(str, 0, sizeof(str));
	if (copy_from_user(str, buffer, count)) return -EFAULT;

	sscanf(str, "%d", &sig);
	return do_notify_settime(sig);
}

static int __init notify_settime_init(void)
{
	struct proc_dir_entry *e;
	int ret = 0;

	e = create_proc_entry(NOTIFY_SETTIME_PROCFS_NAME, S_IFREG | S_IRUGO | S_IWUSR, NULL);
	if (e) {
		e->read_proc = show_notify_settime_signal;
		e->write_proc = store_notify_settime_signal;
		e->data = NULL;
	} else {
		ret = -ENOMEM;
	}
	return ret;
}
__initcall(notify_settime_init);
#endif
