#include <linux/errno.h>
#include <linux/sched.h>
#include <linux/prctl.h>
#include <asm/uaccess.h>

void handle_already_monitoring(struct signotifier *node, 
       struct task_state_notify_info *args,
       struct task_state_notify_info *oldargs)
{
	/* Store the old values */
	oldargs->sig = node->sig;
	oldargs->events = node->events;
 
	/* We know that args->sig is 0 or a valid signal. */
	if (args->sig > 0) {
		/* Update the new values */
		node->sig = args->sig;
		node->events = args->events;
	} else if (!args->sig) {
		/* args->sig of 0 means to deregister */
		unlink_status_notifier(node);
	}
}
 
void setup_new_node(struct task_struct *p,
	struct signotifier *node,
	struct task_state_notify_info *args)
{
	node->notify_tsk = current;
	node->sig = args->sig;
	node->events = args->events;
 
	/* Add this node to the list of notification requests
	 * for the specified process.
	 */
	list_add_tail(&node->notify_list, &p->notify);
 
	/* Also add this node to the list of monitor requests
	 * for the current process.
	 */
	list_add_tail(&node->monitor_list, &current->monitor);
}
 
 
/* Returns 0 if arguments are valid, 1 if they are not. */
int invalid_args(struct task_state_notify_info *args)
{
	int ret = 1;

	if (args->pid <= 0) goto out;
 
	/* Sig of -1 implies query, sig of 0 implies deregistration.
	 * Otherwise sig must be positive and within range.
	 */
	if ((args->sig < -1) || (args->sig > _NSIG)) goto out;
 
	/* If positive sig, must have valid events. */
	if (args->sig > 0) {
		if (!args->events || (args->events >= (1 << (NSIGCHLD+1)))) goto out;
	}
 
	ret = 0;
out:
	return ret;
}

/* If the config is defined, then processes can call this routine
 * to request notification when the specified task's state changes.
 * On the death (or other state change) of the specified process, 
 * we will send them the specified signal if the event is listed
 * in their event bitfield.
 *
 * A sig of 0 means that we want to deregister.
 *
 * The sig/events fields are value/result.  On success we update them
 * to reflect what they were before the call.
 *
 * Returns error code on error, on success we return 0.
 */
int do_notify_task_state(unsigned long arg)
{
	int err;
	struct task_struct *p;
	struct signotifier *node, *tmp;
	struct task_state_notify_info args, oldargs;

	if (copy_from_user(&args, (struct task_state_notify_info __user *)arg, 
			sizeof(args)))
		return -EFAULT;
	oldargs.pid = args.pid;
 
	/* Validate the arguments passed in. */
	err = -EINVAL;
	if (invalid_args(&args))
		goto out;
 
	/* We must hold a write lock on tasklist_lock to add the notification
	 * later on, and we need some lock on tasklist_lock for
	 * find_task_by_pid(), so may as well take the write lock now.
	 * Must use write_lock_irq().
	 */
	write_lock_irq(&tasklist_lock);
 
	err = -ESRCH;
	p = find_task_by_pid(args.pid);
	if (!p) goto unlock_out;
 
	/* Now we know pid exists, unlikely to fail. */
	err = 0;
 
	/* Check if we're already monitoring the specified pid. If so, update
	 * the monitoring parameters and return the old ones.
	 */
	list_for_each_entry(tmp, &p->notify, notify_list) {
		if (tmp->notify_tsk == current) {
			handle_already_monitoring(tmp, &args, &oldargs);
			goto unlock_out;
		}
	}
 
	/* If we get here, we're not currently monitoring the process.  */
	oldargs.sig = 0;
	oldargs.events = 0;
 
	/* If we wanted to set up a new monitor, do it now. If we didn't
	 * manage to allocate memory for the new node, then we return
	 * an appropriate error.
	 */
	if (args.sig > 0) {
		node = kmalloc(sizeof(*node), GFP_KERNEL);
		if (node)
			setup_new_node(p, node, &args);
		else
			err = -ENOMEM;
	}
 
unlock_out:
	write_unlock_irq(&tasklist_lock);
 
	/* Copy the old values back to caller. */
	if (copy_to_user((struct task_state_notify_info __user *)arg,
			&oldargs, sizeof(oldargs)))
		err = -EFAULT;
	
out:
	return err;
}
