/*****************************************************************
 *                                                               *
 * Copyright 2000-2002 LSI Logic. All rights reserved.           *
 *                                                               *
 * This file is confidential and a trade secret of LSI Logic.    *
 * The receipt of or possession of this file does not convey any *
 * rights to reproduce or disclose its contents or to            *
 * manufacture, use, or sell anything it may describe, in whole, *
 * or in part, without the specific written consent of LSI Logic.*
 *                                                               *
 *****************************************************************/
/*
 * System Include Files
 */
#include <stdio.h>
#include <curses.h>      //getchar
#include <string.h>
#include <stdlib.h>     //exit
#include <ctype.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <linux/compiler.h>
#include <linux/kernel.h>
#include <unistd.h>
#include <errno.h>
#include <getopt.h>

/* Typedefs
 */
#ifndef uchar
typedef unsigned char uchar;
#endif

#ifndef u8
typedef unsigned char u8;
#endif

#ifndef u16
typedef unsigned short u16;
#endif

#ifndef u32
typedef unsigned int u32;
#endif

#ifndef u64
typedef unsigned long long u64;
#endif

#ifndef min_t
#define min_t(type,x,y) \
        ({ type __x = (x); type __y = (y); __x < __y ? __x: __y; })
#endif

#ifndef max_t
#define max_t(type,x,y) \
        ({ type __x = (x); type __y = (y); __x > __y ? __x: __y; })
#endif

/*
 * Local Include Files
 */
#include "../mptbase.h"	/* WHAT string and VERSION information */
#include "../mptctl.h"
#include "../lsi/mpi_type.h"
#include "../lsi/mpi.h"
#include "../lsi/mpi_ioc.h"
#include "../lsi/mpi_cnfg.h"
#include "../lsi/mpi_init.h"
#include "../lsi/mpi_raid.h"

typedef struct mpt_ioctl_command mpiIoctlBlk_t;

/*
 * The following strucure is used to link the user selection
 * with a specific function.  Selection of the cmdString results
 * in execution fo the function cmdFunction.  The structures
 * are populated in the dot-c modules.
 */

typedef struct _CmdStrings
{
	char *cmdString;        // Pointer to a String
	void (*cmdFunction)();  // Pointer to a CommandFunction - invoked
				// if string is selected.
} CmdStrings;

typedef struct _ErrStrings
{
	int  value;
	char *errString;

} ErrStrings;

#define IMAGE_VERSION_SIZE 48

#define IMAGE_FLAGS_IS_BIOS 	0x0000000001 /* 0 - FW; 1 - Option Rom */
#define IMAGE_FLAGS_IS_FC 	0x0000000002 /* 0 - SCSI or SAS; 1 - FC */
#define IMAGE_FLAGS_LAST_IMG	0x0000000004 /* 1 - last Option ROM image */
#define IMAGE_FLAGS_MULTI_IMG	0x0000000008 /* 1 - Image contains multiple Option Rom images */
#define IMAGE_FLAGS_CONCAT_IMG	0x0000000010 /* 1 - Option Rom Image to be concatenated */

typedef struct _ImageInfo
{
	char *pBuf;
	char version[IMAGE_VERSION_SIZE];
	int  fd;
	uint size;
	int  flags;
	ushort pci_id;
	ushort adap_pci_id;
} ImageInfo_t;


/*
 * Defines
 */
#define REPLY_SIZE		128
#define DATA_DIR_NONE		0
#define DATA_DIR_IN		1
#define DATA_DIR_OUT		2

#define PROC_DISPLAY_PER_LINE	40	
#define PROC_READ_LINE		256	

#define MAX_FILE_NAME_LENGTH	48
#define DATAIN_LENGTH		24
#define DATAIN_LENGTH_SMALL	 8

#define BYTES_PER_LINE		16

/* To turn on the application debug, set the following */
/* #define DEBUG_APP_SMALL */
/* #define DEBUG_APP */
/* #define DEBUG_APP_BIG */

#ifdef DEBUG_APP_BIG
#ifndef DEBUG_APP
#define DEBUG_APP
#endif
#endif

#ifdef DEBUG_APP
#ifndef DEBUG_APP_SMALL
#define DEBUG_APP_SMALL
#endif
#endif


#define FDU_IS_NOT_RAID		0
#define FDU_IS_IM		1
#define FDU_IS_IS		2

#define FDU_TRUE		1
#define FDU_FALSE		0

#define FDU_EBUF_SIZE		72
#define FDU_MAX_BYTES		0x10000
#define FDU_MIN_BYTES		0x1000
#define FDU_SKIP_CHECKSUM	0x80

/*
 * LOG_ERROR defines and associated strings
 */
#define FDU_GOOD			0x00
#define FDU_NO_MEMORY			0x01
#define FDU_IOCTL_FAILED		0x02
#define FDU_FILE_WRITE_FAILED		0x03
#define FDU_FILE_READ_FAILED		0x04
#define FDU_FSTAT_FAILED		0x05
#define FDU_UNKNOWN_FILE_OPTION		0x06
#define FDU_CREATE_FAILED		0x07
#define FDU_OPEN_FAILED			0x08
#define FDU_INVALID_ADAPTER		0x09
#define FDU_INVALID_USAGE		0x0A
#define FDU_UNSUPPORTED			0x0B
#define FDU_NO_OPTION_ROM		0x0C
#define RESERVEDD			0x0D
#define RESERVEDE			0x0E
#define RESERVEDF			0x0F
#define FDU_FACTS_FAILED		0x10
#define FDU_FWUPLOAD_FAILED		0x11
#define FDU_CHECKSUM_FAILED		0x12
#define FDU_WRONG_FW_TYPE		0x13
#define FDU_WRONG_CHIP_REVISION		0x14
#define FDU_WRONG_FW_PRODUCT		0x15
#define FDU_FWDOWNLOAD_FAILED		0x16
#define FDU_BIOSDOWNLOAD_FAILED		0x17
#define FDU_WRONG_BIOS_TYPE		0x18
#define FDU_WRONG_BIOS_PRODUCT		0x19
#define FDU_ERASE_FLASH_FAILED		0x1A
#define FDU_DL_UL_MISCOMPARE		0x1B
#define FDU_NO_VERIFY			0x1C
#define FDU_FWDOWNLOADBOOT_FAILED	0x1D

#define FDU_FAILED_PROC_FS_READ		0x20

char *error_info[] =
{
	"Successful",
	"Memory Allocation Failed (malloc)",
	"IOCTL Failed",
	"Write of Uploaded F/W image Failed",
	"Read of New F/W image Failed",
	"FSTAT of New F/W image Failed",
	"Unknown file permissions",
	"File create (open) for Write Failed",
	"File open for Read Failed",
	"Invalid IO Controller",
	"Invalid Application Usage",
	"Unsupported Feature",
	"Option ROM Size 0 or non-existent",
	"ReservedMsg-d",
	"ReservedMsg-e",
	"ReservedMsg-f",
	"IOCFacts Failed",
	"F/W Upload Failed",
	"Checksum on New F/W image Failed",
	"New F/W has Wrong Type",
	"New F/W for Different Chip Revision",
	"New F/W has Wrong Product ID",
	"F/W Download Failed",
	"Option ROM Download Failed",
	"New Option Rom image has Wrong Type",
	"New Option Rom image is Wrong Product ID",
	"Erase Flash Failed",
	"Miscompare when verifying downloaded image",
	"FW Upload unsupported. No verify performed.",
	"rsvd-1d",
	"rsvd-1e",
	"rsvd-1f",
	"Failed to Read /proc File System"
};


/*
 * Global Variables
 */
static char *ToolTitle    = "LSI Logic Linux MPT Firmware Download Utility";
static char *IdString     = WHAT_MAGIC_STRING "MptFDU-" MPT_LINUX_VERSION_COMMON "(GCA)";
static char *LsiCopyright = "Copyright 2002 LSI Logic";
static char 		*specialFile = MPT_MISCDEV_PATHNAME;

static int		g_iocnum = -1;
static int		g_fd = -1;
static int		g_status = FDU_INVALID_ADAPTER;
static int		g_menuMode = 1;	/* 1 if menu operation */
static int		g_silent = 0;	/* 1, if silent operation */
static int		g_doReset = 0;	/* 1, if perform reset (cmdline) */
static int		g_force = 0;	/* 1, if ignore Product Mismatches (cmdline) */
static int		g_utilMpiVersion = 0; /* Utility F/W version */
static int		g_multi = 0; 	/* Multiple image flag */
static int		g_lastDownLoad = FDU_GOOD;
static char 		g_upload[MAX_FILE_NAME_LENGTH];
static char 		g_download[MAX_FILE_NAME_LENGTH];
static char 		g_downloadboot[MAX_FILE_NAME_LENGTH];
static char 		g_uploadBios[MAX_FILE_NAME_LENGTH];
static char 		g_downloadBios[MAX_FILE_NAME_LENGTH];
static char 		g_downloadBiosConcat[MAX_FILE_NAME_LENGTH];
static char		g_error_buf[FDU_EBUF_SIZE];
static char		*g_pErrBuf = &g_error_buf[0];
FILE 			*g_filePtr;
static IOCFactsReply_t  g_facts;
static int		fdu_max_write = FDU_MAX_BYTES;

extern int 		errno;
/* extern int optind, opterr, optopt */
/* extern char *optarg */
/* extern char *sys_errlist[]; */

/*
 * MACRO's
 */
#define FDU_LOG_ERROR(error, msg) { \
	g_status = error; \
	if (g_silent) { \
		fprintf(g_filePtr, "0x%x ", g_status); \
	} else { \
		fprintf(g_filePtr, "Error (0x%x): %s", g_status, error_info[error]); \
		if (strlen(msg) > 0) \
			fprintf(g_filePtr, ": %s", msg); \
		fprintf(g_filePtr, "\n"); \
	} \
}

#define FDU_LOG_INFO(time, info) { \
	if (!g_silent) { \
		info; \
		sleep(time); \
	} \
}

#if defined(DEBUG_APP)
#define FDU_LOG_DEBUG(pause, info) { \
	info; \
	if (pause > 0) \
		getchar(); \
}
#else
#define FDU_LOG_DEBUG(pause, info) ;
#endif

#define clr_scr() printf("\033[2J\033[0;0H")

/*
 * Menu Function Prototypes
 */
void SelectAdapter (void);
void FactsIoctl (void);
#if defined(DEBUG_APP_SMALL)
void Erase (void);
void UploadFirmwareCustom (void);
#endif
void UploadFirmware (void);
void UploadBIOS (void);
void DownloadFirmware (void);
void DownloadBootFirmware (void);
void DownloadBIOS (void);
void DiagnosticReset(void);
void ShowHelp(void);
void ShowAbout(void);
void DoExit(void);
void NullFunction (void);

void MainMenuCommands(void);

CmdStrings mainMenuList[] =
{
	{"Specify Action:  ",NullFunction},
	{"Select Adapter ", SelectAdapter},
#if defined(DEBUG_APP_SMALL)
	{"Issue MPI_FUNCTION_IOC_FACTS ", FactsIoctl},
	{"Upload x bytes z type", UploadFirmwareCustom},
	{"Erase Image", Erase},
#endif
	{"Upload the F/W Image from Flash ", UploadFirmware},
	{"Download New F/W Image ", DownloadFirmware},
	{"DownloadBoot New F/W Image ", DownloadBootFirmware},
	{"Upload the Option ROM Image from Flash ", UploadBIOS},
	{"Download New Option ROM Image ", DownloadBIOS},
	{"Issue Diagnostic Reset", DiagnosticReset},
	{"Help and Usage", ShowHelp},
	{"Quit", DoExit},
	{"NULL", NullFunction}
};

/*
 * Worker and Utility functions - these handle the
 * allocation and freeing of memory, user input
 * and the verification etc needed for uploads and downloads of images.
 */
void printDisableWarning(char *imgString);
void printQuiesceMsg(void);
void ShowStatus(void);
void FirmWareDownloadIoctl (char imageType);
void FirmWareDownloadBootIoctl (char imageType);
void FirmWareUploadIoctl (char imageType, uint imageSize, uint *calcImSz, u8 * pCmpImage, u32 readOffset);
void MenuHandler(CmdStrings *menuPtr);
uchar CheckNumericalInput(char *dataString);
int OpenDevice (void);
#if defined(DEBUG_APP)
void ShowBuf (char *titleP, void *dataBufP, int count);
#endif
void eraseFlash (char imageType, char printWarning);
int IssueMptCommand (mpiIoctlBlk_t *mpiBlkPtr, int errIndex);
void freeAllocMem (mpiIoctlBlk_t *mpiBlkPtr);
mpiIoctlBlk_t *allocIoctlBlk (uint numBytes);
int  allocReplyFrame (mpiIoctlBlk_t *mpiBlkPtr);
int  allocDataFrame (mpiIoctlBlk_t *mpiBlkPtr, int dir);
void calcChecksum (ImageInfo_t *image, int bufSize);
int  getFileDesc (int perms, int flags);
int verifyFwPID(ImageInfo_t *image);
int verifyBiosPID(ImageInfo_t *image);
static char *getMagicWhatString(ImageInfo_t *image);
static char *getOldFwVersion(char *resultsBuf, char *isFc);
static void updateBiosImage(ImageInfo_t *image);
static char *whatSearch(const char *buf, int *size_out, int size_in);

/*****************************************************************/
/*                                                               */
/*                                                               */
/*  MAIN                                                         */
/*                                                               */
/*                                                               */
/*****************************************************************/
int main (int argc, char *argv[])
{
	FILE *pfile;
	char *name = "mptctl";
	char dataIn[PROC_READ_LINE];
	int c;

	/* Initialize (force a buffer flush)
	 */
	setvbuf(stdout,NULL,_IONBF,0);

	g_upload[0] = g_download[0] = g_downloadboot[0] = '\0';
	g_uploadBios[0] = g_downloadBios[0] = g_downloadBiosConcat[0] = '\0';
	g_filePtr = stderr;
	g_utilMpiVersion = (MPI_VERSION <<  16);
#ifdef MPI_HEADER_VERSION
	g_utilMpiVersion |= MPI_HEADER_VERSION;
#endif

	ShowAbout();

	/* Open /proc/modules to determine if
	 * driver loaded.
	 */
	snprintf(dataIn, PROC_READ_LINE, "/proc/modules");
	if ((pfile = fopen(dataIn, "r")) == NULL ) {
		printf("Open of /proc/modules failed!\n");
		DoExit();
	}

	c = 0;
	while (1) {
		if (fgets(dataIn, PROC_READ_LINE, pfile) == NULL)
			break;

		if (strstr(dataIn, name) != NULL) {
			c = 1;
			break;
		}
	}
	fclose(pfile);

	if (c == 0) {
		memset(g_pErrBuf, 0, FDU_EBUF_SIZE);
		sprintf(g_pErrBuf, "Driver mptctl not loaded. Executing insmod mptctl\n");
		system ("insmod mptctl");
		FDU_LOG_INFO(5, fprintf(stdout,"%s\n", g_pErrBuf));
	} else {
		/* mptctl driver is loaded. Continue processing.
		 */
		;
	}


	/* Parse the command line options
	 */
	while (1) {
		int option_index = 0;
		static struct option long_options[] =
		{
			{"adapter", 1, 0, 'a'},
			{"bios", 1, 0, 'b'},
			{"download", 1, 0, 'd'},
			{"downloadboot", 1, 0, 'e'},
			{"force", 0, 0, 'f'},
			{"help", 0, 0, 'h'},
			{"logfile", 1, 0, 'l'},
			{"upload", 1, 0, 'u'},
			{"reset", 0, 0, 'r'},
			{"silent", 0, 0, 's'},
			{"multi", 0, 0, 'm'},
			{"concat", 0, 0, 'c'},
			{"add", 1, 0, 0},	/* FIX  - Dell Options ?*/
			{"append", 0, 0, 0},	/* FIX */
			{0, 0, 0, 0,},
		};

		c = getopt_long (argc, argv, "a:b:c:d:fhl:mu:rs", long_options, &option_index);
		if (c == -1)
			break;

		switch (c) {
		case 0:
			printf("option %s", long_options[option_index].name);
			if (optarg)
				printf(" with arg %s", optarg);
			printf("\n");	/* FIX */
			break;

		case 'a':
			FDU_LOG_INFO(2, fprintf (stdout, "Adapter %s selected.\n", optarg));
			if (CheckNumericalInput(optarg) == FDU_FALSE) {
				/* Invalid usage. Display help.
				 */
				ShowHelp();
				FDU_LOG_ERROR(FDU_INVALID_USAGE, "");
				DoExit();
			} else {
				g_iocnum = atoi(optarg);
				if (g_iocnum < 0 || g_iocnum > MPT_MAX_ADAPTERS) {
					ShowHelp();
					memset(g_pErrBuf, 0, FDU_EBUF_SIZE);
					sprintf(g_pErrBuf, "0 <= iocnum < %d, entered value %d.",
						MPT_MAX_ADAPTERS, g_iocnum);
					FDU_LOG_ERROR(FDU_INVALID_ADAPTER, g_pErrBuf);
					DoExit();
				} else {
					if (OpenDevice()) {
						memset(g_pErrBuf, 0, FDU_EBUF_SIZE);
						sprintf(g_pErrBuf, "Failed to open controller #%d.", g_iocnum);
						FDU_LOG_ERROR(FDU_INVALID_ADAPTER, g_pErrBuf);
						DoExit();
					}
				}
			}
			g_menuMode = 0;
			break;

		case 'b':
			FDU_LOG_INFO(1, fprintf (stdout, "Option ROM download source: %s\n", optarg));
			strncpy(g_downloadBios, optarg, MAX_FILE_NAME_LENGTH);
			break;

		case 'c':
			FDU_LOG_INFO(1, fprintf (stdout, "Secondary Option ROM image source: %s\n", optarg));
			strncpy(g_downloadBiosConcat, optarg, MAX_FILE_NAME_LENGTH);
			break;


		case 'd':
			FDU_LOG_INFO(1, fprintf (stdout, "F/W download source: %s\n", optarg));
			strncpy(g_download, optarg, MAX_FILE_NAME_LENGTH);
			break;

		case 'e':
			FDU_LOG_INFO(1, fprintf (stdout, "F/W downloadboot source: %s\n", optarg));
			strncpy(g_downloadboot, optarg, MAX_FILE_NAME_LENGTH);
			g_force = 1;
			break;

		case 'f':
#ifdef FDU_ALLOW_FORCE
			FDU_LOG_INFO(1, fprintf (stdout, "Product-Mismatch Override Enabled.\n"));
			g_force = 1;
#endif
			break;

		case '?':
		case 'h':
			g_menuMode = 0;
			ShowHelp();
			break;

		case 'l':
			if ((g_filePtr = fopen(optarg, "w")) == NULL)
				g_filePtr = stderr;
			break;

		case 'm':
			/* User supplied BIOS image contains multiple sub images
			 */
			FDU_LOG_INFO(1, fprintf (stdout, "Multi-Image Enabled.\n"));
			g_multi = 1;
			break;

		case 'r':
			FDU_LOG_INFO(1, fprintf (stdout, "Hard Reset Enabled.\n"));
			g_doReset = 1;
			break;

		case 's':
			g_silent = 1;
			break;

		case 'u':
			FDU_LOG_INFO(1, fprintf (stdout, "F/W Upload destination: %s\n", optarg));
			strncpy(g_upload, optarg, MAX_FILE_NAME_LENGTH);
			break;

		default:
			printf ("?? getopt returned character code 0%o ??\n", c);
		}
	}

	if (optind < argc) {
		FDU_LOG_INFO(1, fprintf (stdout, "Unknown Options: "));
		while (optind < argc) {
			FDU_LOG_INFO(1, fprintf (stdout, "%s ", argv[optind++]));
		}

		FDU_LOG_INFO(0, fprintf (stdout, "\n"));
	}

	/* If menu mode, enter the handler and let the user select options.
	 */
	if (g_menuMode) {
		MenuHandler(mainMenuList);
	} else {
		/*
		 * 1. Issue FW Upload if enabled
		 * 2. Issue FW Replace or FW Download
		 * 3. Issue Diage Reset if enabled
		 * 4. Exit
		 */
		if ((g_status == FDU_GOOD) && (strlen(g_upload) > 0)) {
			UploadFirmware();
			ShowStatus();
		}

		if ((g_status == FDU_GOOD) && (strlen(g_download) > 0)) {
			DownloadFirmware();
			ShowStatus();
		}

		if ((g_status == FDU_GOOD) && (strlen(g_downloadboot) > 0)) {
			DownloadBootFirmware();
			ShowStatus();
		}
		if ((g_status == FDU_GOOD) && (g_doReset)) {
			DiagnosticReset();
			ShowStatus();
		}

		if ((g_status == FDU_GOOD) && (strlen(g_uploadBios) > 0)) {
			UploadBIOS();
			ShowStatus();
		}

		if ((g_status == FDU_GOOD) && (strlen(g_downloadBios) > 0)) {
			DownloadBIOS();
			ShowStatus();
		}

		DoExit();
	}

	/* Done. Program Complete.
	 */
	return 0;
}


/*****************************************************************
 *  MenuHandler
 *
 *  Inputs:   menuPtr - pointer to a menu of type CmdStrings
 *            the last entry must be NULL
 *  Outputs:   None.
 *  Returns:  None.
 *  Globals:  set g_status.
 *
 *  Purpose:
 *      Will display the supplied menu and check for the validity
 *      of the user input agains the menu choices.  On a successful
 *      input, will invoke the specified menu function.
 *****************************************************************/
void MenuHandler(CmdStrings *menuPtr)
{
	char dataIn[DATAIN_LENGTH];
	int input;
	char counter;
	uchar notDone = FDU_TRUE;
	uchar menuNotDone;

	while (notDone) {
		clr_scr();

		/* Show the tool name, version number and build date
		 */
		ShowAbout();
		printQuiesceMsg();

		/* Show last command status
		 */
		ShowStatus();

		g_status = FDU_GOOD;
		input = 0;
		counter = 0;
		menuNotDone = FDU_TRUE;
		printf("\n");
		while (menuNotDone) {
			if (strcmp(menuPtr[(int)counter].cmdString,"NULL") != 0) {
				if (counter == 0) {
					printf("\n%s", menuPtr[(int)counter].cmdString);
				} else {
					printf("\n%d - %s", counter, menuPtr[(int)counter].cmdString);
				}
				counter++;
			} else {
				/* Done. Decrement counter.
				 * Valid values = [1, counter]
				 */
				menuNotDone = FDU_FALSE;
				counter--;
			}
		}
		printf("\nPlease enter your selection:  ");
		fgets (dataIn, DATAIN_LENGTH, stdin);
	
		/* Verify that a numeric value was entered
		 */
		if (CheckNumericalInput(dataIn) == FDU_FALSE)
			continue;
	
		/* Input string consists only of digits, convert to a number.
		 */
		sscanf(dataIn, "%d", &input);
		
		/* Check input to make sure that entered value is legal
		 */
		if ( (input < 1)||(input > counter) ) {
			printf("\nInvalid entry, please try again.\n");
		} else {
			/* Execute specified funcion
			 */
			menuPtr[input].cmdFunction();
		}
	}

	return;
}

/*****************************************************************
 *  OpenDevice
 *
 *  Inputs:   inputDevice - IOC number
 *  Outputs:  None.
 *  Returns:  g_status
 *  Globals:  g_status to FDU_GOOD (success)
 *                   or non-zero if fail.
 *  Purpose:
 *      Given the ioc number, verify acceptable and open
 *	the specialFile. The iocnum (global) is set.
 *****************************************************************/
int OpenDevice (void)
{
	unsigned int major = 10;
	unsigned int minor = 220;
	dev_t device = (dev_t) ((major << 8) | minor);

	/* Valid ioc number, open the special file.
	 */
	if ((g_fd = open(specialFile, O_RDWR)) < 0) {
		switch(errno)
		{
		case ENOENT:
		case ENXIO:
		case ENODEV:
			/* See man 2 mknod:
			 * S_IFCHR - a character special file.
			 * Permissions: 0644: rwx user; r og
			 *
			 * makedev(10, 200) creates dev_t type
			 * of variable. system macro in
			 * (sys/sysmacros.h) Depending on
			 * platform, dev_t is either an
			 * unsigned int or an unsigned short.
			 */
		
			mknod(specialFile, 0644 | S_IFCHR, device);

			/* See man 2 chmod or
			 * linux/include/stat.h for defines
			 * Set perms. to 0644: rwx user; r og
			 */
			chmod(specialFile, S_IRUSR | S_IWUSR | S_IRGRP | S_IROTH);
			if ((g_fd = open(specialFile, O_RDWR)) >= 0) {
				memset(g_pErrBuf, 0, FDU_EBUF_SIZE);
				sprintf(g_pErrBuf, "Created and opened %s.", specialFile);
				FDU_LOG_INFO(1, fprintf(stdout,"%s\n", g_pErrBuf));
				g_status = FDU_GOOD;
				break;
			}

			/* Fall through to the default error case when open fails.
			 */
		default:
			memset(g_pErrBuf, 0, FDU_EBUF_SIZE);
			sprintf(g_pErrBuf, "Can't open %s.", specialFile);
			FDU_LOG_INFO(1, fprintf(stdout,"%s\n", g_pErrBuf));
			FDU_LOG_ERROR(FDU_INVALID_ADAPTER, g_pErrBuf);
			break;
		}
	} else
		g_status = FDU_GOOD;

	FDU_LOG_DEBUG(0, fprintf(stdout, "iocnum = %d, file descriptor %d\n",
							g_iocnum, g_fd));
	return g_status;
}

/*****************************************************************
 *  CheckNumericalInput
 *
 *  Inputs:   string pointer with user data
 *  Outputs:  None.
 *  Returns:  FDU_TRUE if isDecimal, FDU_FALSE else
 *  Globals:  None.
 *
 *  Purpose:  To ensure that the entered data is numerical
 *****************************************************************/
uchar CheckNumericalInput(char *dataString)
{
	uint ii;
	uchar isDecimal = FDU_TRUE;
	char *newLine = NULL;
	
	/* Replace new line character if it exists.
	 */
	if ( (newLine = strchr(dataString, '\n')) != NULL )
		*newLine = 0;
	
	for (ii=0; ii < strlen(dataString); ii++)
	{
		/* If not a digit, continue
		 */
		if (isdigit(dataString[ii]) == 0)
		{
			isDecimal = FDU_FALSE;
			break;
		}
	}
	
	return isDecimal;
}

#if defined(DEBUG_APP)
/*****************************************************************
 *  ShowBuf
 *
 *  Inputs:   titleP - pointer to string for buffer title
 *            dataBufP - pointer to data buffer
 *            count - data buffer size
 *  Outputs:  None.
 *  Returns:  None.
 *  Globals:  None.
 *
 *  Purpose:  Dump the buffer contents w/ header and ascii conversion.
 *****************************************************************/
void ShowBuf (char *titleP, void *dataBufP, int count)
{
	int done;
	int lineDone = 0;
	int i;
	unsigned char *bufP = (unsigned char *) dataBufP;
	unsigned char textBuf [BYTES_PER_LINE + 1];
	unsigned char nextByte;
	unsigned char sepChar;

	if (g_silent)
		return;

	printf ("%s\n", titleP);

	if (count > 96)
		count = 96;
	
	done = 0;
	while (done < count)
	{
		if (lineDone == 0)
			printf(" %.4X: ", done);

		if (lineDone == (BYTES_PER_LINE/2))
			sepChar = '-';
		else
			sepChar = ' ';

		nextByte = bufP[done];

		printf ("%c%.2X", sepChar, nextByte);
		if ( (nextByte >= 0x20) && (nextByte <= 0x7F))
			textBuf[lineDone] = nextByte;
		else
			textBuf[lineDone] = '.';

		lineDone++;
		done++;

		if ( (lineDone == BYTES_PER_LINE) || (done == count))
		{
			for (i=lineDone; i<BYTES_PER_LINE; i++)
				printf("   ");

			textBuf[lineDone]='\0';
			printf("  %s\n", textBuf);
			lineDone =0;
		}
	}
	return;
}
#endif /* defined(DEBUG_APP) */

/*****************************************************************
 *
 * Primary Menu Functions
 *
 *****************************************************************/
/*****************************************************************
 *  NullFunction
 *
 *  Inputs:   None.
 *  Outputs:  None.
 *  Returns:  None.
 *  Globals:  None.
 *
 *  Purpose:  Key for end of menu listing.
 *****************************************************************/
void NullFunction (void)
{
	return;
}

/*****************************************************************
 *  MainMenuCommands
 *
 *  Inputs:   None.
 *  Outputs:  None.
 *  Returns:  None.
 *  Globals:  None.
 *
 *  Purpose:  MenuDisplay callback for the main menu commands.
 *****************************************************************/
void MainMenuCommands(void)
{
	MenuHandler(mainMenuList);
	return;
}


/*****************************************************************
 *  DoExit
 *
 *  Inputs:   None.
 *  Outputs:  None.
 *  Returns:  None.
 *  Globals:  g_fd - adapter file descriptor
 *            g_filePtr - file pointer for log file
 *
 *  Purpose:  If a file pointer is open, close the file pointer
 *            and exit the program.  Remove the special file
 *            "scsinode" from the working directory.
 *****************************************************************/
void DoExit(void)
{
	if (g_fd != -1)
		close(g_fd);

	if (g_filePtr != stderr)
		fclose(g_filePtr);

	exit(0);
}

/*****************************************************************
 *  SelectAdapter
 *
 *  Inputs:   None.
 *  Outputs:  None.
 *  Returns:  None.
 *  Globals:  sets g_status, g_fd, g_iocnum
 *
 *  Purpose:  A generic routine used as a place holder.
 *            Parses /proc to get the adapter information.
 *****************************************************************/
void SelectAdapter (void)
{
	FILE *pfile;
	char *ptr;
	char iipath[PROC_DISPLAY_PER_LINE];
	char buf[MPT_MAX_ADAPTERS][PROC_DISPLAY_PER_LINE];
	char dataIn[PROC_READ_LINE];
	int  maxAdapt, ii;

	snprintf(iipath, PROC_DISPLAY_PER_LINE, "/proc/mpt/summary");

	if ((pfile = fopen(iipath, "r")) == NULL) {
		FDU_LOG_ERROR(FDU_FAILED_PROC_FS_READ, "");
		sprintf(g_pErrBuf, "Adapter selection: %s\n",
			error_info[g_status]);

		FDU_LOG_INFO(1, fprintf(stdout, "%s", g_pErrBuf));
		return;
	}

	for (ii = 0; ii < MPT_MAX_ADAPTERS; ii++) {
		memset(buf[ii], 0, PROC_DISPLAY_PER_LINE);
		if (fgets(dataIn, PROC_READ_LINE, pfile) == NULL)
			break;
		else if (strstr(dataIn,"ioc") == NULL)
			break;
		else {
			strncpy(buf[ii], dataIn, PROC_DISPLAY_PER_LINE);
			if ((ptr = strstr(buf[ii],"Port")) != NULL) {
				ptr--;
				ptr--;
				*ptr = '\0';
			} else if ((ptr = strstr(buf[ii],"Exp")) != NULL) {
				ptr--;
				ptr--;
				*ptr = '\0';
			} else
				buf[ii][PROC_DISPLAY_PER_LINE-1] = '\0';
		}
	}
	maxAdapt = ii;

	/* Display the data and get the user selection
	 */
	clr_scr();
	while (1) {
		printf("Adapters Found: \n");
		for (ii = 0; ii < maxAdapt; ii++)
			printf("%d - %s\n", ii, buf[ii]);

		printf("Please enter your selection:  ");
		fgets (dataIn, DATAIN_LENGTH, stdin);

		/* Verify that a numeric value was entered
		 */
		if (CheckNumericalInput(dataIn) == FDU_FALSE)
			continue;
	
		/* Input string consists only of digits, convert to a number.
		 */
		sscanf(dataIn, "%d", &g_iocnum);
		
		/* Check input to make sure that entered value is legal
		 */
		if ( (g_iocnum < 0)||(g_iocnum >= maxAdapt) ) {
			printf("\nInvalid entry, please try again.\n");
		} else {
			/* Close any existing file descriptors
			 */
			if (g_fd != -1)
				close(g_fd);
			/* Open this device.
			 */
			if (OpenDevice() != FDU_GOOD) {
				FDU_LOG_INFO(2, fprintf (stdout, "Exiting\n"));
				DoExit();
			}

			g_status = FDU_GOOD;
			break;
		}
	}

	sprintf(g_pErrBuf, "Adapter selection %s.\n",
			g_status == FDU_GOOD ? "successful" : "failed");

	fclose(pfile);
	return;
}

/*****************************************************************
 *  ShowHelp
 *
 *  Inputs:   None.
 *  Outputs:  None.
 *  Returns:  None.
 *  Globals:  None.
 *
 *  Purpose:  Displays the user help information
 *****************************************************************/
void ShowHelp(void)
{
	printf("\n\n\t Firmware Download Utility (FDU)\n");
	printf("The utility may be used to replace the F/W image\n");
	printf("stored in NVS or host-memory (no NVS available).\n\n");

	fprintf(stdout,	"If the F/W or Option ROM download fails, download the latest\n"
			"F/W and/or Option ROM images and flash utilities from the\n"
			"LSI Logic website (www.lsilogic.com).\n"
			"   ia64 arch: efimptfl (Place on a FAT formatted diskette)\n"
			"    x86 arch: flsh1030 (1030 adapters); fcutil (9xx adapters)\n"
			"              (Place on a bootable diskette).\n\n");

	fprintf(stdout, "Use the Flash utility downloaded from the website to write a\n"
			"a new F/W or Option ROM image onto the adapter's flash part.\n\n");

	printf("Press Enter key for usage information ...\n");
	getchar();

	clr_scr();
	printf("Menu Driven Operation:\n\n");
	printf("mptfdu [-l logfile] [-s]\n");
	printf("where:\n\t -l logfile (create a log and save in logfile)\n");
	printf("\t -s (turn on silent mode)\n\n");
	
	printf("Suggested Sequence of Operations:\n");
	printf("Step 1: Select the IO Controller whose F/W is to be upgraded.\n");
	printf("Step 2: (Optional) Issue a FW Upload request to save the \n");
	printf("\tcurrently executing F/W image.\n");
	printf("Step 3: Issue a FW Download request to download the new F/W image.\n");
	printf("\tIf the FW image is cached in memory, a FWDownload\n");
	printf("\trequest will be rejected, so issue a ReplaceFW Request.\n");
	printf("Step 4: To begin immediate execution of the new F/W image, \n");
	printf("\tissue a diagnostic reset.\n\n");

	printf("Press Enter key to continue ...\n");
	getchar();

	clr_scr();
	printf("Command-line Driven Operation:\n\n");
	printf("mptfdu [-l logfile] [-u uFWname] [-d dFWname] [-a iocnum] [-s] [-r] [-h]\n"
	        "                      [-b dORname] [-m] [-c cORname]\n");
	printf("where:\n");
	printf("  -a iocnum  (0 <= iocnum < %d)\n", MPT_MAX_ADAPTERS);
	printf("  -b dORname (new Option ROM image in dORname)\n");
	printf("  -c cORname (concate Option ROM image in cORname)\n");
	printf("  -d dFWname (new F/W image in dFWname)\n");
	printf("  -h         (show help)\n");
	printf("  -l logfile (create a log and save in logfile)\n");
	printf("  -m         (if set, dORname contains multiple images OR\n");
	printf("             dORname and cORname are to be contatenated.\n");
	printf("  -r         (reset the adapter if download successful)\n");
	printf("  -s         (turn on silent mode)\n");
	printf("  -u uFWname (save F/W image in uFWname)\n");

	if (g_menuMode) {
		printf("Press Enter key to continue ...\n");
		getchar();
	}
	return;
}

/*****************************************************************
 *  ShowAbout
 *
 *  Inputs:   None.
 *  Outputs:  None.
 *  Returns:  None.
 *  Globals:  None.
 *
 *  Purpose:  Displays the tool ID string
 *****************************************************************/
void ShowAbout(void)
{
	printf("\n%s\n",ToolTitle);
	printf("%s\n",IdString+4);
	printf("%s\n",LsiCopyright);
	printf("MPI Version: %2x.%02x.%02x.%02x\n\n",
			(g_utilMpiVersion >> 24) & 0xFF,
			(g_utilMpiVersion >> 16) & 0xFF,
			(g_utilMpiVersion >> 8) & 0xFF,
			g_utilMpiVersion & 0xFF);
	return;
}


/*****************************************************************
 *  ShowStatus
 *
 *  Inputs:   None.
 *  Outputs:  None.
 *  Returns:  None.
 *  Globals:  None.
 *
 *  Purpose:  Displays the current contents of global Error Buf.
 *****************************************************************/
void ShowStatus (void) {
	if (g_pErrBuf && (strlen(g_pErrBuf) > 0))
		fprintf(stdout, "STATUS: %s", g_pErrBuf);

	g_error_buf[0] = 0;
}

/*****************************************************************
 *  DiagnosticReset
 *
 *  Inputs:   None.
 *  Outputs:  None.
 *  Returns:  None.
 *  Globals:  set g_status
 *
 *  Purpose:  Reset the adapter hardware and F/W. Risky!
 *****************************************************************/
void DiagnosticReset(void)
{
	struct mpt_ioctl_diag_reset  thisBuf;
	int sz = sizeof(struct mpt_ioctl_diag_reset);

	if (g_fd < 0) {
		sprintf(g_pErrBuf, "No Adapter Selected!\n");
		FDU_LOG_ERROR(FDU_INVALID_ADAPTER, g_pErrBuf);
		return;
	}
	
	if (g_lastDownLoad != FDU_GOOD) {
		sprintf(g_pErrBuf, "Diagnostic Reset: "
			" Last F/W download failed. Reset aborted.\n");
		return;
	}

	printf("Issuing Diagnostic Reset. Please be patient.\n");
	memset (&thisBuf, 0, sz);
	
	/* Complete the header.
	 */
	thisBuf.hdr.iocnum = g_iocnum;
	
	FDU_LOG_DEBUG(0, ShowBuf("Requesting Diagnostic Reset:", &thisBuf, sz));
	
	if (ioctl(g_fd, (unsigned long) MPTHARDRESET, (char *) &thisBuf) != 0) {
		sprintf(g_pErrBuf, "Diagnostic Reset failed.\n");
		FDU_LOG_ERROR(FDU_IOCTL_FAILED, "");
	} else {
		sprintf(g_pErrBuf, "Diagnostic Reset successful.\n");
	}

	return;
}

/*****************************************************************
 *  FactsIoctl
 *
 *  Inputs:   None.
 *  Outputs:  None.
 *  Returns:  None.
 *  Globals:  Sets g_status and g_facts.
 *
 *  Purpose:  Get the IOC Facts data for internal use. Memory
 *            freed in function call IssueMptCommand.
 *****************************************************************/
void FactsIoctl (void)
{
	mpiIoctlBlk_t	*mpiBlkPtr;
	IOCFacts_t 	*pReq;
	IOCFactsReply_t *pReply;
	int		 rc;
	uint numBytes = sizeof (IOCFacts_t) + sizeof (SGESimple64_t);
	static int	 didWarning = 0;

	if (g_fd < 0) {
		sprintf(g_pErrBuf, "No Adapter Selected!\n");
		FDU_LOG_ERROR(FDU_INVALID_ADAPTER, g_pErrBuf);
		return;
	}
	
	if ((mpiBlkPtr = allocIoctlBlk(numBytes)) == NULL)
		goto done_and_free;
	
	/* Populate the Facts MPI Message
	 */
	pReq = (IOCFacts_t *) mpiBlkPtr->MF;
	mpiBlkPtr->dataInSize = 0;
	mpiBlkPtr->dataOutSize = 0;
	mpiBlkPtr->dataSgeOffset = sizeof (IOCFacts_t)/ 4;
	
	pReq->Reserved[0]  = 0;
	pReq->Reserved[1]  = 0;
	pReq->ChainOffset  = 0;
	pReq->Function     = MPI_FUNCTION_IOC_FACTS;
	pReq->Reserved1[0] = 0;
	pReq->Reserved1[1] = 0;
	pReq->Reserved1[2] = 0;
	pReq->MsgFlags     = 0;
	pReq->MsgContext   = -1;

	/* Issue the MPT Command
	 */
	rc = IssueMptCommand(mpiBlkPtr, FDU_FACTS_FAILED);
	if (rc != 0) {
		/* Ioctl Failed. Failure logged by IssueMptCommand.
		 * Clear the facts structure.
		 */
		memset(&g_facts, 0, sizeof(IOCFactsReply_t));

	} else if (g_status == FDU_GOOD) {
		/* Ioctl Completed. Process Reply Information
		 */
		pReply = (IOCFactsReply_t *) mpiBlkPtr->replyFrameBufPtr;

		if ((pReply) && (pReply->IOCStatus & MPI_IOCSTATUS_MASK) == MPI_IOCSTATUS_SUCCESS) {
			int fwMpiVersion;
			memcpy(&g_facts, mpiBlkPtr->replyFrameBufPtr, sizeof(IOCFactsReply_t));

			/* If the Utility is an older MPI version than the F/W,
			 * upgrade the utility to get the latest functionality.
			 */
			fwMpiVersion = (g_facts.MsgVersion << 16);
#ifdef MPI_HEADER_VERSION
			fwMpiVersion |= g_facts.HeaderVersion;
#else
			fwMpiVersion |= g_facts.Reserved;
#endif
			if ((fwMpiVersion > g_utilMpiVersion) && !didWarning) {
				didWarning++;
				FDU_LOG_INFO(0, fprintf(stdout,
					"\nINFO: Utility MPI Version %2x.%02x.%02x.%02x\n"
					"is older than Current F/W MPI Version %2x.%02x.%02x.%02x\n"
					"Some F/W functionality may have changed.\n"
					"Please download a newer Utility.\n\n",
					(g_utilMpiVersion >> 24) & 0xFF,
					(g_utilMpiVersion >> 16) & 0xFF,
					(g_utilMpiVersion >> 8) & 0xFF,
					g_utilMpiVersion & 0xFF,
					(fwMpiVersion >> 24) & 0xFF,
					(fwMpiVersion >> 16) & 0xFF,
					(fwMpiVersion >> 8) & 0xFF,
					fwMpiVersion & 0xFF));
			}

		} else {
			memset(g_pErrBuf, 0, FDU_EBUF_SIZE);
			sprintf( g_pErrBuf, "IOCStatus = 0x%04x",
				pReply->IOCStatus & MPI_IOCSTATUS_MASK);

			FDU_LOG_INFO(3, fprintf(stdout, "ERROR: IOCFacts %s\n", g_pErrBuf));
			FDU_LOG_ERROR(FDU_FACTS_FAILED, g_pErrBuf);
			memset(&g_facts, 0, sizeof(IOCFactsReply_t));
		}
	}

done_and_free:
	sprintf(g_pErrBuf, "Facts: %s.\n", error_info[g_status]);
	freeAllocMem (mpiBlkPtr);
	
	return;
}

/*****************************************************************
 *  UploadFirmware
 *
 *  Inputs:   None.
 *  Outputs:  None.
 *  Returns:  None.
 *  Globals:  Set g_stauts, g_facts.
 *
 *  Purpose: Top level function to upload a F/W image to user space
 *           file.
 *****************************************************************/
void UploadFirmware (void)
{
	clr_scr();
	FactsIoctl();
	if (g_status != FDU_GOOD)
		return;

	if (g_facts.MsgLength < 0x0D) {
		sprintf(g_pErrBuf,
			"F/W upload: Current F/W size not available!\n");
		FDU_LOG_INFO(3, fprintf(stdout, "%s", g_pErrBuf));
		return;
	}

	if ( g_facts.FWImageSize == 0) {
		sprintf(g_pErrBuf, "F/W upload: Current F/W has size 0!\n");
		FDU_LOG_INFO(3, fprintf(stdout, "%s", g_pErrBuf));
		return;
	} else
		FirmWareUploadIoctl(MPI_FW_UPLOAD_ITYPE_FW_FLASH, g_facts.FWImageSize, 0, NULL, 0);
}

/*****************************************************************
 *  UploadBIOS
 *
 *  Inputs:   None.
 *  Outputs:  None.
 *  Returns:  None.
 *  Globals:  Set g_stauts.
 *
 *  Purpose: Read the first 64 bytes to get Option ROM size
 *           and then read the Option ROM image.
 *****************************************************************/
void UploadBIOS (void)
{
	uint imageSize = 64;
	uint newSize = 0;

	clr_scr();
	if (g_fd < 0){
		sprintf(g_pErrBuf, "No Adapter Selected!\n");
		FDU_LOG_ERROR(FDU_INVALID_ADAPTER, g_pErrBuf);
		return;
	}
	FirmWareUploadIoctl (MPI_FW_UPLOAD_ITYPE_BIOS_FLASH, imageSize, &newSize, NULL, 0);

	if ((newSize == 0) && (g_status == FDU_GOOD)) {
		FDU_LOG_ERROR(FDU_NO_OPTION_ROM, "");
		sprintf(g_pErrBuf, "Option ROM upload: %s.\n",
			error_info[g_status]);
	} else if ((newSize > 0) && (g_status == FDU_GOOD))
		FirmWareUploadIoctl (MPI_FW_UPLOAD_ITYPE_BIOS_FLASH, newSize, 0, NULL, 0);

	return;
}

#if defined(DEBUG_APP_SMALL)
/*****************************************************************
 *  Erase
 *
 *  Inputs:   None.
 *  Outputs:  None.
 *  Returns:  None.
 *  Globals:  Set g_stauts.
 *
 *  Purpose: Call eraseFlash to erase the first 36 bytes of data
 *****************************************************************/
void Erase (void)
{
	int  oldstatus = g_status;
	uint input;
	char dataIn[PROC_READ_LINE];
	char type;

	while (1) {
		printf("\nPlease enter 0 - FW; 1 - BIOS:  ");
		fgets (dataIn, DATAIN_LENGTH, stdin);

		/* Verify that a numeric value was entered
		 */
		if (CheckNumericalInput(dataIn) != FDU_FALSE) {
			sscanf(dataIn, "%d", &input);
			if (input == 0) {
				type = MPI_FW_UPLOAD_ITYPE_FW_FLASH;
				break;
			} else if (input == 1) {
				type = MPI_FW_UPLOAD_ITYPE_BIOS_FLASH;
				break;
			}
		}
	}
	
	eraseFlash(type, 1);
	g_status = oldstatus;
}
#endif

#if defined(DEBUG_APP_SMALL)
/*****************************************************************
 *  UploadFirmwareCustom
 *
 *  Inputs:   None.
 *  Outputs:  None.
 *  Returns:  None.
 *  Globals:  Set g_stauts.
 *
 *  Purpose: Top level function to upload a BIOS image to user space
 *           file.
 *****************************************************************/
void UploadFirmwareCustom (void)
{
	uint imageSize = 64;
	uint input;
	u32  offset = 0;
	char dataIn[PROC_READ_LINE];
	char type;

	clr_scr();
	if (g_fd < 0) {
		sprintf(g_pErrBuf, "No Adapter Selected!\n");
		FDU_LOG_ERROR(FDU_INVALID_ADAPTER, g_pErrBuf);
		return;
	}
	memset(g_pErrBuf, 0, FDU_EBUF_SIZE);

	while (1) {
		printf("\nPlease enter the read size in bytes:  ");
		fgets (dataIn, DATAIN_LENGTH, stdin);

		/* Verify that a numeric value was entered
		 */
		if (CheckNumericalInput(dataIn) != FDU_FALSE)
			break;
	}
	sscanf(dataIn, "%d", &imageSize);
	
	while (1) {
		printf("\nPlease enter 0 - FW; 1 - BIOS:  ");
		fgets (dataIn, DATAIN_LENGTH, stdin);

		/* Verify that a numeric value was entered
		 */
		if (CheckNumericalInput(dataIn) != FDU_FALSE) {
			sscanf(dataIn, "%d", &input);
			if (input == 0) {
				type = MPI_FW_UPLOAD_ITYPE_FW_FLASH;
				break;
			} else if (input == 1) {
				type = MPI_FW_UPLOAD_ITYPE_BIOS_FLASH;
				break;
			}
		}
	}
	

	if ((imageSize > 0) && (g_status == FDU_GOOD))
		FirmWareUploadIoctl (type | FDU_SKIP_CHECKSUM, imageSize, 0, NULL, offset);

	return;
}

#endif /*  defined(DEBUG_APP_SMALL) */

/*****************************************************************
 *  FirmWareUploadIoctl
 *
 *  Inputs:   imageType (F/W or Option ROM)
 *            imageSize (bytes)
 *            calcImSz (bytes), if not 0, do not write to file
 *		but return image size here.
 *            pCmpImage, if not NULL do a compare instead
 *		of a write to file.		
 *            imgOffset (bytes), typically 0, first byte to read from
 *		The Databuffer should contain the first byte of data,
 *		do not shift by imgOffset.
 *	
 *  Outputs:  None.
 *  Returns:  None.
 *  Globals:  Set g_stauts.
 *
 *  Purpose:  Construct and issue the F/W upload MPI message.
 *****************************************************************/
void FirmWareUploadIoctl (char imageType, uint imageSize, uint *calcImSz, u8 *pCmpImage, u32 imgOffset)
{
	mpiIoctlBlk_t	*mpiBlkPtr=NULL;
	FWUpload_t	*pReq;
	FWUploadReply_t	*pReply;
	FWUploadTCSGE_t	*ptcsge;
	ImageInfo_t	 image;
	u8		 *pImage;
	char		 imageStr[DATAIN_LENGTH];
	int		 jj;
	int		 bytesLeft, readOffset, readSize;
	int		 rc;
	uint		 numBytes=sizeof(FWUpload_t);
	char		 doCompare=0;
	char		 noCkSum = 0;

	/* Clear Image structures
	 */
	memset(&image, 0, sizeof(ImageInfo_t));
	image.fd = -1;
	image.size = imageSize;

	noCkSum = imageType & FDU_SKIP_CHECKSUM ? 1 : 0;	

	imageType &= 0x0f;
	if (imageType == MPI_FW_UPLOAD_ITYPE_FW_FLASH)
		strncpy(imageStr, "F/W", DATAIN_LENGTH);
	else if (imageType == MPI_FW_UPLOAD_ITYPE_BIOS_FLASH) {
		strncpy(imageStr, "Option ROM", DATAIN_LENGTH);
		image.flags |= IMAGE_FLAGS_IS_BIOS;
	}

	if (pCmpImage)
		doCompare = 1;
	else {
		/* Open a file only to write result
		 */
		if ( !((image.flags & IMAGE_FLAGS_IS_BIOS) && (calcImSz != NULL)) ) {
			if ((image.fd = getFileDesc(O_WRONLY, image.flags)) < 0) {
				goto done_and_free;
			}
		}
	}

	/* Alloc memory - single block for image + data
	 */
	numBytes += image.size;
	if ((mpiBlkPtr = allocIoctlBlk(numBytes)) == NULL)
		goto done_and_free;

	/* Populate the Facts MPI Message
	 */
	pReq = (FWUpload_t *) mpiBlkPtr->MF;
	mpiBlkPtr->timeout = 20;
	mpiBlkPtr->dataOutSize = 0;
	mpiBlkPtr->dataSgeOffset = (sizeof(FWUpload_t) - sizeof(SGE_MPI_UNION)
							+ sizeof(FWUploadTCSGE_t)) / 4;

	image.pBuf = (char *)(((char *)pReq) + sizeof(FWUpload_t));
	bytesLeft = imageSize;
	readOffset = 0;
	rc = 0;
	if (pCmpImage == NULL) {
		printf("Uploading %s image. Timeout is %d "
			"seconds per iteration.\n",
			imageStr, mpiBlkPtr->timeout);
	}
	while (bytesLeft) {
		readSize = (bytesLeft > fdu_max_write) ? fdu_max_write : bytesLeft;
		pReq->ImageType    = imageType;
		pReq->Reserved     = 0;
		pReq->ChainOffset  = 0;
		pReq->Function     = MPI_FUNCTION_FW_UPLOAD;
		pReq->Reserved1[0] = 0;
		pReq->Reserved1[1] = 0;
		pReq->Reserved1[2] = 0;
		pReq->MsgFlags     = 0;
		pReq->MsgContext   = -1;
		ptcsge = (FWUploadTCSGE_t *) &pReq->SGL;
		ptcsge->Reserved = 0;
		ptcsge->ContextSize = 0;
		ptcsge->DetailsLength = 12;
		ptcsge->Flags = MPI_SGE_FLAGS_TRANSACTION_ELEMENT;
		ptcsge->Reserved1 = 0;
		ptcsge->ImageOffset = imgOffset + readOffset;
		ptcsge->ImageSize = readSize;

		/* Set up the information that the driver uses
		 * for SGE construction
		 */
		mpiBlkPtr->dataInSize = readSize;
		mpiBlkPtr->dataInBufPtr = ((char *) image.pBuf + readOffset) ;

		/* Issue the Command
		 */
		printf("Uploading %d bytes. %d bytes remain.\n",
				readSize, bytesLeft - readSize);

		rc = IssueMptCommand(mpiBlkPtr, FDU_FWUPLOAD_FAILED);
		if (rc != 0) {
			/* IssueMptCommand logged an error
			 * Failure Case C
			 */
			if (rc == -ENOMEM) {
				if (fdu_max_write > FDU_MIN_BYTES) {
					fdu_max_write /= 2;

					FDU_LOG_INFO(3, fprintf(stdout,
					"No Memory. Retrying with block size"
					" 0x%x.\n", fdu_max_write));
					continue;
				}
			}
		} else if (g_status == FDU_GOOD) {
			/* Parse the Reply Frame.
			 * Error if expected != actual image size
			 */
			pReply = (FWUploadReply_t *) mpiBlkPtr->replyFrameBufPtr;
			if ((pReply) && (pReply->MsgLength > 0)){
				pReply->IOCStatus &= MPI_IOCSTATUS_MASK;
				switch (pReply->IOCStatus) {
				case MPI_IOCSTATUS_SUCCESS:
					/* readSize may, or may not, = imageSize
					 * According to the spec, the FWUpload
					 * Reply should always contain
					 * the actual image size and not the
					 * number of bytes requested.
					 * Uploads of partial images may occur, in this
					 * case readSize == imageSize but readSize is
					 * not the REAL image size....skip this test.
					 * Do not compare
					 * pReply->ActualImageSize to readSize
					 */
#if 0
					if ((readSize == imageSize) && (readSize != pReply->ActualImageSize))  {
						FDU_LOG_ERROR(FDU_FWUPLOAD_FAILED,"Image Size Mismatch");
					}
#endif
					break;

				case MPI_IOCSTATUS_INVALID_FIELD:
					FDU_LOG_INFO(3, fprintf(stdout,
						"INFO: Cannot upload %s from"
						" flash - no flash present for "
						"this chip\n", imageStr));
					sprintf(g_pErrBuf,"IOCStatus = 0x%04x",
							pReply->IOCStatus);
					FDU_LOG_ERROR(FDU_FWUPLOAD_FAILED,
							g_pErrBuf);
					break;

				case MPI_IOCSTATUS_INVALID_FUNCTION:
					sprintf(g_pErrBuf,"IOCStatus = 0x%04x",
							pReply->IOCStatus);
					FDU_LOG_INFO(3, fprintf(stdout,
						"INFO: F/W does not support"
						" a %s Upload\n",imageStr));
					if (doCompare) {
						FDU_LOG_INFO(3, fprintf(stdout,
							"Unable to verify FW download.\n"));
						FDU_LOG_ERROR(FDU_NO_VERIFY, g_pErrBuf);
					} else {
						FDU_LOG_ERROR(FDU_FWUPLOAD_FAILED, g_pErrBuf);
					}
					break;

				default:
					/* Failure Case D
					 */
					sprintf(g_pErrBuf,"IOCStatus = 0x%04x",
							pReply->IOCStatus);
					FDU_LOG_INFO(3, fprintf(stdout,
							"ERROR: %s Upload %s",
							imageStr, g_pErrBuf));

					FDU_LOG_ERROR(FDU_FWUPLOAD_FAILED,
							g_pErrBuf);
				}
			} else {
				FDU_LOG_ERROR(FDU_FWUPLOAD_FAILED, "Reply Frame Length is 0!");
			}
		}


		if (g_status != FDU_GOOD)
			break;

		/* Update the read size and read offset for the next loop.
		 */
		bytesLeft -= readSize;
		readOffset += readSize;
	}

	if (doCompare) {
		/* Compare this image to the supplied image
		 */
		if (g_status == FDU_GOOD) {
			pImage = image.pBuf;
			for (jj = 0;  jj < imageSize; jj++) {
				if (pCmpImage[jj] != pImage[jj]) {
					/* Failure Case E
					 */
					FDU_LOG_INFO(5, fprintf(stdout,
					"Miscompare: offset %d, upload: 0x%x, dl: 0x%x\n",
					jj, pImage[jj], pCmpImage[jj]));

					FDU_LOG_ERROR(FDU_DL_UL_MISCOMPARE,"");
					break;
				}
			}
		} else if (g_status == FDU_NO_VERIFY) {
			g_status = FDU_GOOD;
		}

	} else if ((g_status == FDU_GOOD) && (image.flags & IMAGE_FLAGS_IS_BIOS) && (calcImSz != NULL)) {
		/* Try to determine the size of the image to upload.
		 */
		uchar *pPci = image.pBuf + ((image.pBuf[0x19] << 8) | image.pBuf[0x18]);

		if (strncmp(pPci,"PCIR", 4) == 0) {
			*calcImSz = (pPci[0x11] << 8 | pPci[0x10] ) * 512;
			FDU_LOG_DEBUG(1, fprintf(stdout, "Calculated BIOS Image size 0x%x (%d) "
							 " bytes <enter>\n", *calcImSz, *calcImSz));
		} else {
			*calcImSz = 0;
		}

	} else if (g_status == FDU_GOOD) {

		if (!noCkSum) {
			/* Verify the checksum prior to write
			 */
			calcChecksum (&image, imageSize);
		}

		/* Write data to destination File
		 */
		if (g_status == FDU_GOOD) {
			pImage = image.pBuf;
			if ((jj = write(image.fd,  pImage, imageSize)) < 0){
				FDU_LOG_INFO(3, fprintf(stdout,
					"ERROR: %s write to \"%s\" FAILED: %s\n",
					imageStr, g_upload, strerror(errno)));
				FDU_LOG_ERROR(FDU_FILE_WRITE_FAILED, strerror(errno));
			}
			FDU_LOG_DEBUG(0, fprintf(stdout, "INFO: %s wrote %d bytes\n",
						imageStr, imageSize));
		}
	}

done_and_free:
	sprintf(g_pErrBuf, "%s upload: %s.\n",
		imageStr, error_info[g_status]);

	/* Free Memory
	 */
	if (image.fd >= 0)
		close(image.fd);

	/* NULL dataInBufPtr, since memory allocated as one block
	 */
	if (mpiBlkPtr)
		mpiBlkPtr->dataInBufPtr = NULL;

	return;
}

/*****************************************************************
 *  DownloadFirmware
 *
 *  Inputs:   None.
 *  Outputs:  None.
 *  Returns:  None.
 *  Globals:  None.
 *
 *  Purpose:  Call the download worker function.
 *****************************************************************/
void DownloadFirmware (void)
{
	FirmWareDownloadIoctl (MPI_FW_DOWNLOAD_ITYPE_FW);
}

/*****************************************************************
 *  DownloadBootFirmware
 *
 *  Inputs:   None.
 *  Outputs:  None.
 *  Returns:  None.
 *  Globals:  None.
 *
 *  Purpose:  Call the download worker function.
 *****************************************************************/
void DownloadBootFirmware (void)
{
	FirmWareDownloadBootIoctl (MPI_FW_DOWNLOAD_ITYPE_BOOTLOADER);
}

/*****************************************************************
 *  DownloadBIOS
 *
 *  Inputs:   None.
 *  Outputs:  None.
 *  Returns:  None.
 *  Globals:  None.
 *
 *  Purpose:  Call the download worker function.
 *****************************************************************/
void DownloadBIOS (void)
{
	FirmWareDownloadIoctl (MPI_FW_DOWNLOAD_ITYPE_BIOS);
	return;
}

/*****************************************************************
 *  FirWareDownloadIoctl
 *
 *  Inputs:   imageType (F/W or BIOS)
 *            imageSize (bytes)
 *  Outputs:  None.
 *  Returns:  None.
 *  Globals:  Set g_stauts.
 *
 *  Purpose:  Construct and issue the F/W upload MPI message.
 *
 *  Remarks:
 *  (1) There are different algorithms and processes for
 *      validating a F/W and BIOS image. When possible, a single function
 *      is called, otherwise, we switch based on type.
 *  (2) Only 1030-type of chips may be implemented without onboard flash.
 *      If the chip is of type FC, FWReplace is not legal.
 *****************************************************************/
void FirmWareDownloadIoctl (char imageType)
{
	mpiIoctlBlk_t		*mpiBlkPtr = NULL;
	FWDownload_t 		*pReq;
	FWDownloadReply_t	*pReply;
	FWDownloadTCSGE_t	*ptcsge;
	mpt_ioctl_replace_fw_t	*pIoctlReq = NULL;
	ImageInfo_t		image[2];
	char 			*oldverp = NULL;
	char 			*newverp = NULL;
	char			*pDataBuf = NULL;
	char			*pImage = NULL;
	char 			oldver[IMAGE_VERSION_SIZE];
	char		 	imageStr[DATAIN_LENGTH];
	struct stat		fwstat;
	int			maxImage = 2;
	int			pid;
	int			jj, kk;
	int			errDefine;
	uint 			numBytes;
	uint			bytesLeft, writeSize, writeOffset;
	int			rc;
	char			doCachedFw = 0;
	char			allowReplace = 0;
	char			isFc = 0;
	char 			vtype = 0;
	
	clr_scr();
	if (g_fd < 0) {
		sprintf(g_pErrBuf, "No Adapter Selected!\n");
		FDU_LOG_ERROR(FDU_INVALID_ADAPTER, g_pErrBuf);
		return;
	}

	/* Clear Image structures
	 */
	for (jj = 0; jj < maxImage; jj++) {
		memset(&image[jj], 0, sizeof(ImageInfo_t));
		image[jj].fd = -1;
	}

	strncpy(imageStr, "", DATAIN_LENGTH);

	/* Issue the Facts IOCTL - current FW information
	 */
	FactsIoctl ();
	if (g_status != FDU_GOOD)
		goto done_and_free;

	if (imageType == MPI_FW_DOWNLOAD_ITYPE_FW) {
		vtype = MPI_FW_UPLOAD_ITYPE_FW_FLASH;
		maxImage = 1;
		allowReplace = 1;

		strncpy(imageStr, "F/W", DATAIN_LENGTH);
		errDefine = FDU_FWDOWNLOAD_FAILED;
	} else {
		strncpy(imageStr, "Option ROM", DATAIN_LENGTH);
		errDefine = FDU_BIOSDOWNLOAD_FAILED;
		if (g_facts.MsgVersion <= 0x0100) {
			FDU_LOG_ERROR(FDU_UNSUPPORTED,
				"Option Rom download requires"
				" MPI Version > 1.00");
			goto done_and_free;
		}
		vtype = MPI_FW_UPLOAD_ITYPE_BIOS_FLASH;

		/* Two cases - (1) a multi image is supplied
		 * (2) multiple images are supplied and must be
		 *     concatenated.
		 */
		for (jj = 0; jj < maxImage; jj++)
			image[jj].flags |= IMAGE_FLAGS_IS_BIOS;

		if (g_multi == 0) {
			maxImage = 1;
			image[0].flags |= IMAGE_FLAGS_LAST_IMG;
		} else if (g_downloadBiosConcat[0] == 0){
			maxImage = 1;
			image[0].flags |= IMAGE_FLAGS_MULTI_IMG;
		} else {
			image[1].flags |= IMAGE_FLAGS_LAST_IMG;
			image[1].flags |= IMAGE_FLAGS_CONCAT_IMG;
		}
	}

	/* Remark: For multi-image case, can set an int
	 * numImages = 1 for FW and single BIOS image case
	 * make imagefd, imageSize, arrays.. index from
	 * ii = 0; ii < numImages.  Or create a structure
	 * of image information...
	 */

	for (jj = 0; jj < maxImage; jj++) {
		/* Get the image filename and open the file.
		 */
		if ((image[jj].fd = getFileDesc(O_RDONLY, image[jj].flags)) < 0)
			goto done_and_free;

		if (fstat(image[jj].fd, &fwstat) < 0) {
			FDU_LOG_INFO(3, fprintf(stdout, "ERROR: fstat(%d) FAILED: %s \n",
						image[jj].fd, strerror(errno)));
			FDU_LOG_ERROR(FDU_FSTAT_FAILED, strerror(errno));
			goto done_and_free;
		}
		image[jj].size = (int) fwstat.st_size;
		FDU_LOG_DEBUG(0, fprintf(stdout, "%s Image #%d Size %d bytes\n",
				      imageStr, jj, image[jj].size));
	}

	
	/* Set numBytes to be the size of the request AND the data areas.
	 * Data area is sizeof (FWdownload_t) bytes from start of MF.
	 *  *** Must set dataOutBufPtr to NULL prior to freeing memory,
	 *  *** because there has not been a separate alloc just for data
	 *  *** failure to clear, will result in the memory being freed twice!
	 */
	numBytes = sizeof (FWDownload_t);
	if (numBytes < sizeof(mpt_ioctl_replace_fw_t))
		numBytes = sizeof(mpt_ioctl_replace_fw_t);

	for (jj = 0; jj < maxImage; jj++)
		numBytes += image[jj].size;

	/* FW Download MPI Message */
	if ((mpiBlkPtr = allocIoctlBlk(numBytes)) == NULL)
		goto done_and_free;

	/* Populate the MPI Message
	 * The host driver will append at most 1 SGE
	 * dataOutSize and dataOutBufPtr are set later.
	 */
	pReq = (FWDownload_t *) mpiBlkPtr->MF;
	mpiBlkPtr->timeout = 30;
	mpiBlkPtr->dataInSize = 0;
	mpiBlkPtr->dataSgeOffset = (sizeof(FWDownload_t)
			- sizeof(SGE_MPI_UNION) + sizeof(FWDownloadTCSGE_t))/4;
	pDataBuf =  (char *) (((char *) pReq) + sizeof (FWDownload_t));
	
	pImage = pDataBuf;
	kk = 0;
	for (jj = 0; jj < maxImage; jj++) {
		/* Set the image pointer. kk = 0 on first iteration
		 * and kk = image[0].size on second iteration
		 */
		image[jj].pBuf = pImage + kk;

		/* Read image number jj into memory.
		 */
		if ((kk = read(image[jj].fd,  image[jj].pBuf, image[jj].size)) < 0){
			FDU_LOG_INFO(3, fprintf(stdout, "ERROR: %s read from \"%s\" FAILED: %s\n",
							 imageStr, g_download, strerror(errno)));
			FDU_LOG_ERROR(FDU_FILE_READ_FAILED, strerror(errno));
		} else {
			FDU_LOG_DEBUG(0, fprintf(stdout, "%s read %d bytes\n", imageStr, image[jj].size));

			/* Compute and verify checksum, sets g_status if fails
			 */
			calcChecksum (&image[jj], kk);
		}

		/* If there has been a failure, terminate the download procedure.
		 */
		if (g_status != FDU_GOOD)
			goto done_and_free;
	}

#ifdef DEBUG_APP
	for (jj =0; jj < maxImage; jj++) {
		printf("flags 0x%x, size %d, fd %d, buff @ %p, version %s\n",
			image[jj].flags, image[jj].size, image[jj].fd,
			image[jj].pBuf, image[jj].version);
		ShowBuf("D1 pbuf:", image[jj].pBuf, 48);
	}
#endif
	FDU_LOG_DEBUG(1, fprintf(stdout, "Finished read of %d image(s) <enter>\n", maxImage));

	if (imageType == MPI_FW_DOWNLOAD_ITYPE_FW) {
		/* Now that the new image is loaded, perform the verification process:
		 * (1) get the version of the current F/W
		 *     including a RAID identified (none, IM, IS)
		 * (2) clear the allowReplace if FC
		 * (3) Get the version of the new F/W image
		 * (4) Verify the Product ID
		 */
		oldverp = getOldFwVersion(oldver, &isFc);
		if (isFc && allowReplace)
			allowReplace = 0;

		newverp = getMagicWhatString(&image[0]);
		verifyFwPID(&image[0]);
	} else {
		/* Now that the new image is loaded, perform the verification process:
		 * (1) Get the version of the current BIOS
		 *     including a RAID identified (none, IM, IS)
		 *     Issue mpt_ioctl_iocinfo to get BIOS version
		 * (2) Get the version of the new BIOS image
		 * (3) Verify the Product
		 * (4) Update the Image as necessary
		 */
		struct mpt_ioctl_iocinfo info;

		memset(&info, 0, sizeof(struct mpt_ioctl_iocinfo));

		info.hdr.iocnum = g_iocnum;
		info.hdr.port = 0;
		info.hdr.maxDataSize = sizeof(struct mpt_ioctl_iocinfo);
		if (ioctl(g_fd, (unsigned long) MPTIOCINFO, (char *) &info) != 0) {
			/* Continue but log error...
			 */
		 	FDU_LOG_ERROR(FDU_IOCTL_FAILED,
					"Unable to get PCI Information.");
		 	FDU_LOG_INFO(3, fprintf(stdout,
					"Unknown PCI ID - Unable to verify!!!"));

			g_status = FDU_GOOD;
			strncpy(oldver, "Unknown", IMAGE_VERSION_SIZE);
			image[0].adap_pci_id = -1;
			image[1].adap_pci_id = -1;
			oldverp = &oldver[0];
		} else {
			if (info.BIOSVersion == 0) {
				strncpy(oldver, "Unknown", IMAGE_VERSION_SIZE);
			} else {
				sprintf(oldver, "MPTBIOS-%02d.%02d.%02d.%02d",
						info.BIOSVersion >> 24,
						(info.BIOSVersion >> 16) & 0x00FF,
						(info.BIOSVersion >> 8) & 0x00FF,
						info.BIOSVersion  & 0x00FF);
			}
			oldverp = &oldver[0];
			image[0].adap_pci_id = info.pciId;
			image[1].adap_pci_id = info.pciId;
		}

		for (jj=0; jj < maxImage; jj++) {
			newverp = getMagicWhatString(&image[jj]);
			if (g_status == FDU_GOOD)
				verifyBiosPID(&image[jj]);
			if (g_status == FDU_GOOD)
				updateBiosImage(&image[jj]);
		}
		newverp = image[0].version;
	}

	if (oldverp == NULL)
		oldverp = "unknown";

	FDU_LOG_DEBUG(0, ShowBuf("Modified:", image[0].pBuf, 128));


	/* If there has been a failure, terminate the download procedure.
	 */
	if (g_status != FDU_GOOD)
		goto done_and_free;

	/* Print Command Information
	 */
	if (imageType == MPI_FW_DOWNLOAD_ITYPE_FW) {
		FDU_LOG_INFO(0, fprintf(stdout,
		"    Current %s Version = \"%s\"\n", imageStr, oldverp));
	} else {
		FDU_LOG_INFO(0, fprintf(stdout,
		"     Booted %s Version = \"%s\"\n", imageStr, oldverp));
	}

	FDU_LOG_INFO(0, fprintf(stdout,
		"        New %s Version = \"%s\"\n", imageStr, newverp));

	bytesLeft = 0;
	for (jj = 0; jj < maxImage; jj++) {
		bytesLeft += image[jj].size;
	}
	writeOffset = 0;

	FDU_LOG_DEBUG(0, fprintf(stdout, "Total Num Bytes %d (0x%x) \n", bytesLeft, bytesLeft));
	printf("Downloading %s image. Timeout is %d seconds per iteration.\n",
		imageStr, mpiBlkPtr->timeout);
	while (bytesLeft){
		/* Set the writeSize
		 */
		writeSize = (bytesLeft > fdu_max_write) ? fdu_max_write : bytesLeft;

		pReq->ImageType  = imageType;
		pReq->Reserved  = 0;
		pReq->ChainOffset  = 0;
		pReq->Function     = MPI_FUNCTION_FW_DOWNLOAD;
		pReq->Reserved1[0] = 0;
		pReq->Reserved1[1] = 0;
		pReq->Reserved1[2] = 0;
		pReq->MsgFlags     = 0;
		pReq->MsgContext   = -1;

		ptcsge = (FWDownloadTCSGE_t *) &pReq->SGL;
		ptcsge->Reserved = 0;
		ptcsge->ContextSize = 0;
		ptcsge->DetailsLength = 12;
		ptcsge->Flags = MPI_SGE_FLAGS_TRANSACTION_ELEMENT;
		ptcsge->ImageOffset = writeOffset;
		ptcsge->ImageSize = writeSize;

		/* Set up the information that the driver uses
		 * for SGE construction
		 */
		mpiBlkPtr->dataOutSize = writeSize;
		mpiBlkPtr->dataOutBufPtr = ((char *) pDataBuf + writeOffset);

		if (g_facts.MsgVersion <= 0x0100) {
			u8 *pTmp = (u8 *) mpiBlkPtr->dataOutBufPtr;
			u32 cksum = 0;

			for (kk=0; kk < writeSize; kk++)
				cksum += ((u32)(pTmp[kk])) << (8 * (3 & kk));

			cksum = (0 - cksum);
			ptcsge->Reserved_0100_Checksum = cksum;

			FDU_LOG_DEBUG(1, fprintf(stdout, "cksum 0x%x <enter>\n", cksum));

		} else {
			ptcsge->Reserved_0100_Checksum = 0;
		}

		FDU_LOG_DEBUG(1, fprintf(stdout, "Issuing cmd: offset 0x%x, size 0x%x, left 0x%x"
				" imageSize 0x%x <enter>\n",
				writeOffset, writeSize, bytesLeft-writeSize,
				image[0].size + image[1].size));

		printf("Downloading %d bytes (%d bytes remain) ...\n",
				writeSize, bytesLeft - writeSize);

		/* Issue the Command
		 */
		rc = IssueMptCommand(mpiBlkPtr, errDefine);
		if (rc != 0) {
			printf("\n");
			/* Ioctl Failed. Special processing here?
			 * REMARK: Failure Case A
			 */
			if (rc == -ENOMEM) {
				if (fdu_max_write > FDU_MIN_BYTES) {
					fdu_max_write /= 2;

					FDU_LOG_INFO(3, fprintf(stdout,
					"No Memory. Retrying with block size"
					" 0x%x.\n", fdu_max_write));
					continue;
				}
			}
		} else if (g_status == FDU_GOOD) {
			/* Parse the Reply Frame.
			 */
			pReply = (FWDownloadReply_t *) mpiBlkPtr->replyFrameBufPtr;

			if ((pReply) && (pReply->MsgLength > 0)) {
				pReply->IOCStatus &= MPI_IOCSTATUS_MASK;

				if (pReply->IOCStatus == MPI_IOCSTATUS_SUCCESS) {
					;
				} else if (allowReplace && (pReply->IOCStatus == MPI_IOCSTATUS_INVALID_FIELD)) {
					/* FW Download failed, FWReplace if legal,
					 */
					doCachedFw = 1;
					printf("\n");
				} else {
					/* REMARK: Failure Case B
					 * set g_status, invalidate image
					 */
					printf("\n");
					memset(g_pErrBuf, 0, FDU_EBUF_SIZE);
					sprintf(g_pErrBuf, "IOCStatus = 0x%04x", pReply->IOCStatus);

					FDU_LOG_INFO(3, fprintf(stdout, "ERROR: %s Download %s\n",
									imageStr, g_pErrBuf));
					FDU_LOG_ERROR(errDefine, g_pErrBuf);
				}
			} else {
				printf("\n");
				FDU_LOG_ERROR(errDefine, "Reply Frame Length is 0!");
			}
		}

		/* If the cached FW flag is set or if the previous F/W download failed,
		 * terminate the F/W download procedure.
		 */
		if ((g_status != FDU_GOOD) || doCachedFw)	
			break;

		/* Update the write size and write offset for the next loop.
		 */
		bytesLeft -= writeSize;
		writeOffset += writeSize;
	}

	/* The following loop can only be executed if a F/W to
	 * flash request failed due to no flash part.
	 */
	if (doCachedFw) {
		/* FW aready read, set the pointer to the
		 * mpt_ioctl_replace_fw location.
		 */
		pImage = pDataBuf;
		pIoctlReq = (mpt_ioctl_replace_fw_t *) (((char *) pImage) - sizeof (mpt_ioctl_replace_fw_t *));

		if ( pImage != (char *)pIoctlReq->newImage ) {
			fprintf (stderr, "Hmm didn't calculate thing right\n");
			fprintf (stderr, "pImage %p, pIoctlReq %p, newImage %p,"
					" sizeof(mpt_ioctl_replace_fw_t) %d \n",
					pImage, pIoctlReq, pIoctlReq->newImage,
					(int) sizeof (mpt_ioctl_replace_fw_t));
		}
		/*
		 * Populate the structure.
		 */
		pIoctlReq->hdr.iocnum = g_iocnum;
		pIoctlReq->hdr.port = 0;
		pIoctlReq->hdr.maxDataSize = 0;
		pIoctlReq->newImageSize = image[0].size; /* FW has a single image */

		FDU_LOG_DEBUG(0, ShowBuf("MPTFWREPLACE Start:", pIoctlReq, sizeof(mpt_ioctl_replace_fw_t)));
		if (ioctl(g_fd, (unsigned long) MPTFWREPLACE, (char *) pIoctlReq) != 0) {
					FDU_LOG_INFO(3, fprintf(stdout, "ERROR: FW Replace failed.\n"));
					FDU_LOG_ERROR(FDU_IOCTL_FAILED, "FW Replace Failed.");
		} else {
			FDU_LOG_DEBUG(0, ShowBuf("MPTFWREPLACE Data:", pIoctlReq, sizeof(mpt_ioctl_replace_fw_t)));
			g_status = FDU_GOOD;
		}
	}
	/* Set the status of the last download.
	 * Do not allow a diag reset if DL failed.
	 */
	g_lastDownLoad = g_status;


	/* Perform the verification process
	 */
	if ((g_status == FDU_GOOD) && (doCachedFw == 0)) {
		/* Verify: Upload will perform compare
		 * if possible. Will Log error if compare fails.
		 */
		bytesLeft = 0;
		for (jj = 0; jj < maxImage; jj++) {
			bytesLeft += image[jj].size;
		}

		pid = g_facts.ProductID & MPI_FW_HEADER_PID_TYPE_MASK;
		if (pid == MPI_FW_HEADER_PID_TYPE_SAS) {
			printf("Begin verification....Issuing Diag Reset!\n");
			sleep(10);

			DiagnosticReset();
			ShowStatus();
		}

		printf("Begin verification....please be patient!\n");
		sleep(10);

		FDU_LOG_INFO(1, fprintf(stdout, "Verifying %d bytes...\n",
				bytesLeft));

		FirmWareUploadIoctl(vtype, bytesLeft, 0,(u8 *)pDataBuf, 0);
	}

	/* If the FW download failed, write a few bytes to the
	 * beginning of the flash part.
	 * Print error message w/ user instructions.
	 */
	if (g_status != FDU_GOOD) {
		if (imageType == MPI_FW_UPLOAD_ITYPE_FW_FLASH) {
			eraseFlash(MPI_FW_DOWNLOAD_ITYPE_BIOS, 0);
		}
		eraseFlash(imageType, 1);
	}

done_and_free:
	sprintf(g_pErrBuf, "%s download: %s.\n",
			imageStr, error_info[g_status]);

	/* Due to data memory being included in mpiBlk allocation,
	 * NULL this pointer before calling freeAlloc. See comments above.
	 */
	for (jj = 0; jj < maxImage; jj++) {
		if (image[jj].fd >= 0)
			close(image[jj].fd);
	}

	if (mpiBlkPtr)
		mpiBlkPtr->dataOutBufPtr =  NULL;

	freeAllocMem(mpiBlkPtr);
	return;
}


void FirmWareDownloadBootIoctl (char imageType)
{
	struct mpt_fw_xfer fwdnldboot;
	char			*pImage = NULL;
	ImageInfo_t		image[2];
	char		 	imageStr[DATAIN_LENGTH];
	struct stat		fwstat;
	int			maxImage = 2;
	int			jj, kk;
	uint 			numBytes=0;

	/* Clear Image structures
	 */
	for (jj = 0; jj < maxImage; jj++) {
		memset(&image[jj], 0, sizeof(ImageInfo_t));
		image[jj].fd = -1;
	}
	maxImage = 1;
	for (jj = 0; jj < maxImage; jj++) {
		/* Get the image filename and open the file.
		 */
		if ((image[jj].fd = getFileDesc(O_RDONLY, image[jj].flags)) < 0)
			goto done;

		if (fstat(image[jj].fd, &fwstat) < 0) {
			FDU_LOG_INFO(3, fprintf(stdout, "ERROR: fstat(%d) FAILED: %s \n",
						image[jj].fd, strerror(errno)));
			FDU_LOG_ERROR(FDU_FSTAT_FAILED, strerror(errno));
			goto done;
		}
		image[jj].size = (int) fwstat.st_size;
		FDU_LOG_DEBUG(0, fprintf(stdout, "%s Image #%d Size %d bytes\n",
				      imageStr, jj, image[jj].size));
	}

	for (jj = 0; jj < maxImage; jj++)
		numBytes += image[jj].size;

	if ((pImage = (char *) malloc (numBytes)) == NULL) {
		goto done;
	}

	kk = 0;
	for (jj = 0; jj < maxImage; jj++) {
		/* Set the image pointer. kk = 0 on first iteration
		 * and kk = image[0].size on second iteration
		 */
		image[jj].pBuf = pImage + kk;

		/* Read image number jj into memory.
		 */
		if ((kk = read(image[jj].fd,  image[jj].pBuf, image[jj].size)) < 0){
			FDU_LOG_INFO(3, fprintf(stdout, "ERROR: %s read from \"%s\" FAILED: %s\n",
							 imageStr, g_download, strerror(errno)));
			FDU_LOG_ERROR(FDU_FILE_READ_FAILED, strerror(errno));
		} else {
			FDU_LOG_DEBUG(0, fprintf(stdout, "%s read %d bytes\n", imageStr, image[jj].size));

			/* Compute and verify checksum, sets g_status if fails
			 */
			calcChecksum (&image[jj], kk);
		}

		/* If there has been a failure, terminate the download procedure.
		 */
		if (g_status != FDU_GOOD)
			goto done_and_free;
	}
	fwdnldboot.iocnum = g_iocnum;
	fwdnldboot.fwlen = numBytes;
	fwdnldboot.bufp = (void *)pImage;
	if (ioctl(g_fd, (unsigned long) MPTFWDOWNLOADBOOT, (char *) &fwdnldboot) != 0) {
		FDU_LOG_INFO(3, fprintf(stdout, "ERROR: FW DownloadBoot failed.\n"));
		FDU_LOG_ERROR(FDU_IOCTL_FAILED, "FW DownloadBoot Failed.");
	} else {
		g_status = FDU_GOOD;
	}

done_and_free:
	free (pImage);
done:
	sprintf(g_pErrBuf, "%s download: %s.\n",
			imageStr, error_info[g_status]);

	for (jj = 0; jj < maxImage; jj++) {
		if (image[jj].fd >= 0)
			close(image[jj].fd);
	}
}

/*****************************************************************
 *  eraseFlash
 *
 *  Inputs:   imageType (MPI_FW_DOWNLOAD_ITYPE_FW or MPI_FW_DOWNLOAD_ITYPE_BIOS);
 *  Outputs:  None.
 *  Returns:  None.
 *  Globals:  Sets g_status to FDU_IOCTL_FAILED - if ioctl failed
 *                             FDU_GOOD - if ioctl successful
 *
 *  Purpose:  Write 36 bytes of 0xFF to first or last section of the
 *            flash part. Only called if FW Download Failed.
 *
 *****************************************************************/
void eraseFlash (char imageType, char printWarning)
{
	mpiIoctlBlk_t		*mpiBlkPtr = NULL;
	FWDownload_t 		*pReq;
	FWDownloadReply_t	*pReply;
	FWDownloadTCSGE_t	*ptcsge;
	int			ii;
	int			dataSize = 36;
	uint 			numBytes;
	int			oldstatus = g_status;

	FDU_LOG_INFO(1, fprintf(stdout,
			"Attempting to erase %s due to"
			" download failure.\n",
			imageType == MPI_FW_UPLOAD_ITYPE_FW_FLASH ?
				"F/W" : "Option ROM"));

	numBytes = dataSize + sizeof (FWDownload_t);
	if ((mpiBlkPtr = allocIoctlBlk(numBytes)) == NULL) {
		sprintf(g_pErrBuf, "Failed - no memory allocation.");
		FDU_LOG_INFO(3, fprintf(stdout, "ERROR: Erase Flash %s\n",g_pErrBuf));
		FDU_LOG_ERROR(FDU_ERASE_FLASH_FAILED, g_pErrBuf);
		return;
	}

	/* Populate the MPI Message
	 * The host driver will append at most 1 SGE
	 */
	pReq = (FWDownload_t *) mpiBlkPtr->MF;
	mpiBlkPtr->timeout = 30;
	mpiBlkPtr->dataInSize = 0;

	mpiBlkPtr->dataOutSize = dataSize;
	mpiBlkPtr->dataOutBufPtr = ((char *) mpiBlkPtr->MF + sizeof(FWDownload_t));
	for (ii = 0; ii < dataSize; ii++) {
		mpiBlkPtr->dataOutBufPtr[ii] = 0xFF;
	}

	mpiBlkPtr->dataSgeOffset = (sizeof(FWDownload_t)
			- sizeof(SGE_MPI_UNION) + sizeof(FWDownloadTCSGE_t))/4;

	pReq->ImageType  = imageType;
	pReq->Reserved  = 0;
	pReq->ChainOffset  = 0;
	pReq->Function     = MPI_FUNCTION_FW_DOWNLOAD;
	pReq->Reserved1[0] = 0;
	pReq->Reserved1[1] = 0;
	pReq->Reserved1[2] = 0;
	pReq->MsgFlags     = 0;
	pReq->MsgContext   = -1;

	ptcsge = (FWDownloadTCSGE_t *) &pReq->SGL;
	ptcsge->Reserved = 0;
	ptcsge->ContextSize = 0;
	ptcsge->DetailsLength = 12;
	ptcsge->Flags = MPI_SGE_FLAGS_TRANSACTION_ELEMENT;

	if (g_facts.MsgVersion <= 0x0100) {
		u8 *pTmp = (u8 *) mpiBlkPtr->dataOutBufPtr;
		u32 cksum = 0;

		for (ii=0; ii < dataSize; ii++)
			cksum += ((u32)(pTmp[ii])) << (8 * (3 & ii));

		cksum = (0 - cksum);
		ptcsge->Reserved_0100_Checksum = cksum;

		FDU_LOG_DEBUG(1, fprintf(stdout,
					"Erase: cksum 0x%x <enter>\n", cksum));
	} else  {
		ptcsge->Reserved_0100_Checksum = 0;
	}
	ptcsge->ImageOffset = 0;
	ptcsge->ImageSize = dataSize;


	IssueMptCommand(mpiBlkPtr, FDU_ERASE_FLASH_FAILED);

	if (g_status == FDU_GOOD) {
		pReply = (FWDownloadReply_t *) mpiBlkPtr->replyFrameBufPtr;

		if ((pReply) && (pReply->MsgLength > 0)) {
			if ((pReply->IOCStatus & MPI_IOCSTATUS_MASK) != MPI_IOCSTATUS_SUCCESS) {
				memset(g_pErrBuf, 0, FDU_EBUF_SIZE);
				sprintf(g_pErrBuf, "IOCStatus = 0x%04x",
					pReply->IOCStatus & MPI_IOCSTATUS_MASK);
				FDU_LOG_INFO(3, fprintf(stdout, "ERROR: Erase Flash %s\n",g_pErrBuf));
				FDU_LOG_ERROR(FDU_ERASE_FLASH_FAILED, g_pErrBuf);
			}
		}
	}

	if (printWarning) {
		if (imageType == MPI_FW_UPLOAD_ITYPE_FW_FLASH)
			printDisableWarning("F/W");
		else
			printDisableWarning("Option ROM");
	}

	g_status = oldstatus;
	return;
}

/*****************************************************************
 *  IssueMptIoctl
 *
 *  Inputs:   None.
 *  Outputs:  None.
 *  Returns:  None.
 *  Globals:  Sets g_status to FDU_IOCTL_FAILED - if ioctl failed
 *                             FDU_GOOD - if ioctl successful
 *
 *  Purpose:  Generic command to issue the MPT command using the
 *            MPTCOMMAND command.
 *
 *****************************************************************/
int IssueMptCommand (mpiIoctlBlk_t *mpiBlkPtr, int errIndex)
{
	int CmdBlkSize;
	int rc;

	CmdBlkSize = sizeof(mpiIoctlBlk_t) + ((mpiBlkPtr->dataSgeOffset)*4) + 8;
	
#if defined(DEBUG_APP)
	printf("reply %p  max (%d) \n", mpiBlkPtr->replyFrameBufPtr, mpiBlkPtr->maxReplyBytes);
	printf("data in %p size (%d) \n", mpiBlkPtr->dataInBufPtr, mpiBlkPtr->dataInSize);
	printf("data out %p size (%d) \n", mpiBlkPtr->dataOutBufPtr, mpiBlkPtr->dataOutSize);
	printf("sense %p size (%d) \n", mpiBlkPtr->senseDataPtr, mpiBlkPtr->maxSenseBytes);
	printf("Timeout %d, \n", mpiBlkPtr->timeout);
	printf ("CmdBlkSize %d  mpiBlkPtr %p\n", CmdBlkSize, mpiBlkPtr);
	printf (" dataSgeOffset*4 0x%x, size of SGESimple64 0x%x\n",
				mpiBlkPtr->dataSgeOffset*4, (uint) sizeof(SGESimple64_t));
	ShowBuf ("Command Block Before: ", mpiBlkPtr, CmdBlkSize);
	if (g_menuMode) {
		printf ("Press enter to continue..");
		getchar();
	}
#endif

	/* Set the IOC number prior to issuing this command.
	 */
	mpiBlkPtr->hdr.iocnum = g_iocnum;
	mpiBlkPtr->hdr.port = 0;
	
	rc = ioctl (g_fd, (unsigned long) MPTCOMMAND, (char *) mpiBlkPtr);
	if (rc != 0) {
		/* Possible Returns:
		 *     -EBUSY: adapter reset in progress
		 *     -EFAULT: copy in failed, request frame too big, data unavailable
		 *     -ENODEV: invalid adapter
		 *     -ETIME: timer expires
		 *     -ENOMEM: no memory, malloc failed
		 *     -EAGAIN: try again - driver resource issue
		 *     -ENODATA: copy out failed
		 */
		rc = -errno;
		memset(g_pErrBuf, 0, FDU_EBUF_SIZE);
		sprintf(g_pErrBuf, "IOCTL Failed (%d): %s", errno, strerror(errno));
		FDU_LOG_ERROR(errIndex, g_pErrBuf);
	} else {
		g_status = FDU_GOOD;

#if defined (DEBUG_APP)
		if ((mpiBlkPtr->replyFrameBufPtr) && (mpiBlkPtr->replyFrameBufPtr[2] > 0) ) {
			ShowBuf ("Reply Frame : ", mpiBlkPtr->replyFrameBufPtr,
						mpiBlkPtr->replyFrameBufPtr[2] * 4);
		}

		if (mpiBlkPtr->dataInBufPtr)
			ShowBuf ("Data In: ", mpiBlkPtr->dataInBufPtr, mpiBlkPtr->dataInSize);

		if ((mpiBlkPtr->dataInBufPtr) && (mpiBlkPtr->dataInSize > 64))
			ShowBuf ("Data In: ",
				(char *) mpiBlkPtr->dataInBufPtr + (mpiBlkPtr->dataInSize - 64), 64);

		if (mpiBlkPtr->dataOutBufPtr)
			ShowBuf ("Data Out: ", mpiBlkPtr->dataOutBufPtr, mpiBlkPtr->dataOutSize);

		if ((mpiBlkPtr->dataOutBufPtr) && (mpiBlkPtr->dataOutSize > 64))
			ShowBuf ("Data Out: ",
				(char *) mpiBlkPtr->dataOutBufPtr + (mpiBlkPtr->dataOutSize - 64), 64);

#endif
	}

#if defined(DEBUG_APP)
	if (g_menuMode) {
		printf ("Press enter to continue ..");
		getchar();
	}
#endif
	return rc;
}

/*****************************************************************
 *
 *
 *      Utility Functions
 *
 *
 *****************************************************************/
/*****************************************************************
 *  calcChecksum
 *
 *  Inputs:   pBuf - pointer to data buffer
 *            bufSize - buffer size (bytes)
 *            fname - that contains the data
 *  Outputs:  None.
 *  Returns:  None.
 *  Globals:  Set g_status if failed.
 *
 *  Purpose:  Compute the checksum on a FW or BIOS image. From S. Sharon.
 *****************************************************************/
void calcChecksum (ImageInfo_t *image, int bufSize) {
	char *  pBuf = image->pBuf;
	int	ii;
	u32	checksum = 0;
	u8	checku8 = 0;

	FDU_LOG_DEBUG(0, fprintf (stdout, "Calling calcChecksum. flags 0x%x, size %d\n", image->flags, image->size));

	if (image->flags & IMAGE_FLAGS_IS_BIOS) {
		/* Do a 8-bit checksum calculation
		 * Check each image individually.
		 */
		if (image->flags & IMAGE_FLAGS_MULTI_IMG) {
			uchar *pPci = pBuf + ((pBuf[0x19] << 8) | pBuf[0x18]);
			uint size = (pPci[0x11] << 8 | pPci[0x10] ) * 512;
			u8 checku8 = 0, chk2 = 0;

			FDU_LOG_DEBUG(0, fprintf (stdout, " pbuf @ %p  \n", image->pBuf));
			FDU_LOG_DEBUG(0, fprintf (stdout, " pPci @ %p \n", pPci));
			FDU_LOG_DEBUG(0, ShowBuf("CK EROM:", image->pBuf, 48));
			FDU_LOG_DEBUG(0, ShowBuf("CK PCI Struct:", pPci, 32));

			/* User said this was a multiple image - but this
			 * is false! Reset the flags.
			 */
			if (size == image->size) {
				image->flags &= ~IMAGE_FLAGS_MULTI_IMG;
				image->flags |= IMAGE_FLAGS_LAST_IMG;
			}

			for (ii=0; ii < size; ii++)
				checku8 += pBuf[ii];

			for (ii=size; ii < bufSize; ii++)
				chk2 += pBuf[ii];

			if (checku8) {
				checksum = checku8;
			} else if (chk2) {
				checksum = chk2;
			}
		} else {
			for (ii=0; ii < bufSize; ii++)
				checku8 += pBuf[ii];
		}

		checksum = checku8;
	} else {
		/* Do a 32-bit checksum calculation
		 */
		for (ii=0; ii < bufSize; ii++)
			checksum += ((u32)((u8)pBuf[ii])) << (8 * (3 & ii));
	}

	if (checksum != 0) {
		if (image->flags & IMAGE_FLAGS_IS_BIOS) {
			FDU_LOG_INFO(3, fprintf(stdout, "ERROR: %s Option Rom Image Checksum INVALID\n",
				image->flags & IMAGE_FLAGS_CONCAT_IMG ? "Second" : "First"));
		} else {
			FDU_LOG_INFO(3, fprintf(stdout, "ERROR: FW Image Checksum INVALID\n"));
		}

		FDU_LOG_ERROR(FDU_CHECKSUM_FAILED, "");
	}

	return ;
}

/*****************************************************************
 *  getFileDesc
 *
 *  Inputs:   perms - Read or Write permissions
 *            isBios - set, if this is a BIOS image
 *  Outputs:  None.
 *  Returns:  None.
 *  Globals:  Set g_status if failed.
 *
 *  Purpose:  Utility to acquire the data-in and data-out file names
 *            from the user or command line arguments. Sets the global
 *            g_upload(Bios) or g_download(Bios).
 *****************************************************************/
int getFileDesc (int perms, int flags) {
	char	*pTmp;
	char 	*pFileName = NULL;
	int 	fwfd = -1;
	int	error = 0;

	if (perms == O_WRONLY) {
		if (flags & IMAGE_FLAGS_IS_BIOS)
			pFileName = g_uploadBios;
		else
			pFileName = g_upload;

		if (g_menuMode) {
			/* Read in the file name from user
			 */
			printf("\nEnter the name of the file to save the %s Image:  ",
				(flags & IMAGE_FLAGS_IS_BIOS) ? "Option ROM" : "F/W");
			fgets(pFileName, MAX_FILE_NAME_LENGTH, stdin);
			pTmp = strchr(pFileName,'\n');
			if (pTmp)
				strcpy(pTmp, "");

			pTmp = NULL;
		} else {
			/* g_upload or g_uploadBios set by command line parser
			 */
			;
		}

		if (pFileName) {
			//if ((fwfd = creat(pFileName, O_CREAT| O_TRUNC|O_RDWR|O_BINARY, 0644)) < 0) {
			//if ((fwfd = creat(pFileName, O_WRONLY)) < 0) {
			if ((fwfd = creat(pFileName, 0644)) < 0) {
				FDU_LOG_INFO(3, fprintf(stdout, "ERROR: open(\"%s\") FAILED: %s\n",
					pFileName, strerror(errno)));

				error = FDU_CREATE_FAILED;
			}
		}


	} else if (perms == O_RDONLY) {
		FDU_LOG_DEBUG(0, fprintf (stdout, "flags: 0x%x \n", flags));
		if (flags & IMAGE_FLAGS_IS_BIOS) {
			if (flags & IMAGE_FLAGS_CONCAT_IMG)
				pFileName = g_downloadBiosConcat;
			else
				pFileName = g_downloadBios;

		} else
			pFileName = g_download;

		if (g_menuMode) {
			/* Read in the file name from user
			 */
			printf("\nEnter the name of the file containing the %s Image:  ",
				(flags & IMAGE_FLAGS_IS_BIOS) ? "Option ROM" : "F/W");
			fgets(pFileName, MAX_FILE_NAME_LENGTH, stdin);

			pTmp = strchr(pFileName,'\n');
			if (pTmp)
				strcpy(pTmp, "");

			pTmp = NULL;
		} else {
			/* g_download or g_downloadBios set by command line parser
			 */
			;
		}

		if (pFileName) {
			if ((fwfd = open(pFileName, O_RDONLY)) < 0) {
				FDU_LOG_INFO(3, fprintf(stdout, "ERROR: open(\"%s\") FAILED: %s\n",
					 pFileName, strerror(errno)));
				error = FDU_OPEN_FAILED;
			}
		}
	} else
		error = FDU_UNKNOWN_FILE_OPTION;

	if (fwfd < 0) {
		FDU_LOG_ERROR(error, strerror(errno));
	}

	return fwfd;
}

/*****************************************************************
 *  freeAllocMem
 *
 *  Inputs:   None.
 *  Outputs:  None.
 *  Returns:  None.
 *  Globals:  Uses global mpiBlkPto to free allocated memory.
 *
 *  Purpose:  Utility to free all the allocated memory.
 *            If datain(Out)BufPtr != NULL, memory is assumed to be
 *            allocated separately from mpiBlk allocation.
 *****************************************************************/
void freeAllocMem (mpiIoctlBlk_t *mpiBlkPtr)
{
	if (mpiBlkPtr == NULL)
		return;

	if (mpiBlkPtr->replyFrameBufPtr)
		free (mpiBlkPtr->replyFrameBufPtr);
	
	if (mpiBlkPtr->dataOutBufPtr)
		free (mpiBlkPtr->dataOutBufPtr);
	
	if (mpiBlkPtr->dataInBufPtr)
		free (mpiBlkPtr->dataInBufPtr);
	
	if (mpiBlkPtr->senseDataPtr)
		free (mpiBlkPtr->senseDataPtr);
	
	free (mpiBlkPtr);

	mpiBlkPtr = NULL;

	return;
}

/*****************************************************************
 *  allocIocltBlk
 *
 *  Inputs:   numBytes - number of bytes of memory needed.
 *  Outputs:  None.
 *  Returns:  NULL if mpiBlk or reply frame alloc fails.
 *            mpiBlkPtr if successful.
 *  Globals:  Uses global mpiBlkPto to free allocated memory.
 *            Sets g_status.
 *
 *  Purpose:  Utility to allocate an mpiIoctlBlk_t structure that
 *            is large enough to handle the entire MF + ioctl overhead.
 *****************************************************************/
mpiIoctlBlk_t *allocIoctlBlk (uint numBytes)
{
	mpiIoctlBlk_t *thisBlkPtr = NULL;
	uint blksize  = sizeof(mpiIoctlBlk_t) + numBytes;
	
	if ( (thisBlkPtr = (mpiIoctlBlk_t *) malloc (blksize)) == NULL) {
		FDU_LOG_ERROR(FDU_NO_MEMORY, strerror(errno));
		return (NULL);
	}
	
	memset (thisBlkPtr, 0, blksize);
	
	if (allocReplyFrame (thisBlkPtr)) {
		freeAllocMem(thisBlkPtr);
		return (NULL);
	}
	else
		return (thisBlkPtr);
}

/*****************************************************************
 *  allocReplyFrame
 *
 *  Inputs:  Pointer to the mpiIoctlBlk_t structure
 *  Outputs:
 *  Returns: 0: success   1: fail
 *  Globals:  set g_status.
 *
 *  Purpose: Utility to allocate a reply frame (MPI SIZE)
 *           and save the address in the incomping mpiBlkPtr
 *****************************************************************/
int allocReplyFrame (mpiIoctlBlk_t *mpiBlkPtr)
{
	if (( mpiBlkPtr->replyFrameBufPtr = (char *) malloc (REPLY_SIZE)) == NULL) {
		FDU_LOG_ERROR(FDU_NO_MEMORY, strerror(errno));
		return (1);
	}
	
	memset (mpiBlkPtr->replyFrameBufPtr, 0, REPLY_SIZE);
	mpiBlkPtr->maxReplyBytes = REPLY_SIZE;
	return (0);
}

/*****************************************************************
 *  allocDataFrame
 *
 *  Inputs: DATA_DIR_IN (1) or DATA_DIR_OUT (2) or DATA_DIR_NONE (0)
 *  Outputs:
 *  Returns: 0: success   1: fail
 *  Globals:  set g_status.
 *
 *  Purpose:  Utility to allocate a data frame (mpiBlkPtr->DataInSize)
 *            or mpiBlkPtr->DataOutSize and save the address in the
 *            incomping mpiBlkPtr
 *****************************************************************/
int allocDataFrame (mpiIoctlBlk_t *mpiBlkPtr, int dir)
{
	if (dir == DATA_DIR_OUT) {
		if ((mpiBlkPtr->dataOutBufPtr = (char *) malloc (
							mpiBlkPtr->dataOutSize)) == NULL) {
			FDU_LOG_ERROR(FDU_NO_MEMORY, strerror(errno));
			return (1);
		}
	
		memset (mpiBlkPtr->dataOutBufPtr, 0, mpiBlkPtr->dataOutSize);
	
	} else if (dir == DATA_DIR_IN) {
		if ((mpiBlkPtr->dataInBufPtr = (char *) malloc (
							mpiBlkPtr->dataInSize)) == NULL) {
			FDU_LOG_ERROR(FDU_NO_MEMORY, strerror(errno));
			return (1);
		}
	
		memset (mpiBlkPtr->dataInBufPtr, 0, mpiBlkPtr->dataInSize);
	}
	return (0);
}

/*****************************************************************
 *  verifyFwPID
 *
 *  Inputs:  Pointer to the FW buffer
 *           FW size in bytes.	
 *  Outputs: None.
 *  Returns: 0: success   1: fail
 *  Globals:  set g_status.
 *
 *  Purpose:  Verify that the ProductID in the FW matches the board we're
 *            sending the image to. ProductID format:
 *            4 bits - type { FC, SCSI, SAS, Unknown }
 *            4 bits - Product { Initiator, Target, ...}
 *            8 bits - Family { 1030 A0, 1030 B0, ...}
 *****************************************************************/
int verifyFwPID(ImageInfo_t *image)
{
	char  userinp[DATAIN_LENGTH_SMALL];
	int   ii;
	u16   fwpid, iocpid;
	int   bigWarning = 0;

	/* ioc = currently on adapter
	 * fw = supplied image
	 */
	iocpid = g_facts.ProductID;
	if (iocpid == MPI_MANUFACTPAGE_DEVICEID_FC909) {
		u8 *fwptr = (u8 *) image->pBuf;

		for (ii = 0x017000; (ii+7) < image->size; ii++) {
			if ((((fwptr[ii+1] << 8)| fwptr[ii]) == MPI_MANUFACTPAGE_DEVICEID_FC909)
			    && (fwptr[ii+2] == 0x00) && (fwptr[ii+3] == 0x00)
			    && (fwptr[ii+4] == 0x00) && (fwptr[ii+5] == 0x00)
			    && (fwptr[ii+6] == 0xff) && (fwptr[ii+7] == 0x03)) {
				return (0);
			}
		}
		fwpid = 0xdead;
	} else {
		fwpid = (((u16)((u8)image->pBuf[35])) << 8) | ((u16)((u8)image->pBuf[34]));
	}

	/* If 909 controller and fwpid = 0xdead, then the image failed the
	 * required 909 FW image signature. Log error.
	 */
	if (fwpid == 0xdead) {
		sprintf(g_pErrBuf, "Invalid Image for FC909 Adapter!");
		FDU_LOG_INFO(3, fprintf(stdout, "ERROR: %s \n", g_pErrBuf));
		FDU_LOG_ERROR(FDU_WRONG_FW_TYPE, g_pErrBuf);
		return 0;
	}

	/* 909 images are a differnt format, if 909 in version
	 * string, print message
	 */
	if (strstr(image->version,"909") != NULL) {
		sprintf(g_pErrBuf, "Image for FC909 Adapter!");
		FDU_LOG_INFO(3, fprintf(stdout, "ERROR: %s \n", g_pErrBuf));
		FDU_LOG_ERROR(FDU_WRONG_FW_TYPE, g_pErrBuf);
		return 0;
	}

	/* Check the Type (FC vs SCSI), upper nibble of pid. MUST Match
	 */
	if ((fwpid & MPI_FW_HEADER_PID_TYPE_MASK) !=
					(iocpid & MPI_FW_HEADER_PID_TYPE_MASK)){
		uint fwtype = fwpid & MPI_FW_HEADER_PID_TYPE_MASK;
		uint ioctype = iocpid & MPI_FW_HEADER_PID_TYPE_MASK;

		sprintf(g_pErrBuf, "%s Image %s Adapter",
			fwtype==MPI_FW_HEADER_PID_TYPE_SCSI?"SCSI":(fwtype == MPI_FW_HEADER_PID_TYPE_FC?"FC":(fwtype == MPI_FW_HEADER_PID_TYPE_SAS?"SAS":"Unknown")),
			ioctype==MPI_FW_HEADER_PID_TYPE_SCSI?"SCSI":(ioctype == MPI_FW_HEADER_PID_TYPE_FC?"FC":(ioctype == MPI_FW_HEADER_PID_TYPE_SAS?"SAS":"Unknown")));

		FDU_LOG_INFO(3, fprintf(stdout, "ERROR: %s \n", g_pErrBuf));
		FDU_LOG_ERROR(FDU_WRONG_FW_TYPE, g_pErrBuf);
	}

	if (g_status == FDU_GOOD) {
		/* Check the Family, lower byte. MUST Match
		 */
		if ((fwpid & MPI_FW_HEADER_PID_FAMILY_MASK) !=
				(iocpid & MPI_FW_HEADER_PID_FAMILY_MASK)){
			uint fwfam = fwpid & MPI_FW_HEADER_PID_FAMILY_MASK;
			uint iocfam = iocpid & MPI_FW_HEADER_PID_FAMILY_MASK;

			sprintf(g_pErrBuf, "F/W %d Chip %d",
				fwfam & MPI_FW_HEADER_PID_FAMILY_MASK,
				iocfam & MPI_FW_HEADER_PID_FAMILY_MASK);

			FDU_LOG_INFO(3, fprintf(stdout,
				"ERROR: FW - Chip Revsion Mismatch: %s.\n",
				g_pErrBuf));
			FDU_LOG_ERROR(FDU_WRONG_CHIP_REVISION, g_pErrBuf);
		}
	}

	if (g_status == FDU_GOOD) {
		/* Check the Product
		 */
		if ((fwpid & MPI_FW_HEADER_PID_PROD_MASK) !=
						(iocpid & MPI_FW_HEADER_PID_PROD_MASK)){
			uint fwprod = fwpid & MPI_FW_HEADER_PID_PROD_MASK;
			uint iocprod = iocpid & MPI_FW_HEADER_PID_PROD_MASK;
			char *fwProdString;
			char *iocProdString;

			if ((fwpid & MPI_FW_HEADER_PID_TYPE_MASK) ==
						MPI_FW_HEADER_PID_TYPE_SCSI) {
				switch(fwprod) {
				case MPI_FW_HEADER_PID_PROD_INITIATOR_SCSI:
					fwProdString="Initiator";
					break;
				case MPI_FW_HEADER_PID_PROD_TARGET_INITIATOR_SCSI:
					fwProdString="Initiator-Target";
					break;
				case MPI_FW_HEADER_PID_PROD_TARGET_SCSI:
					fwProdString="Target";
					break;
				case MPI_FW_HEADER_PID_PROD_IM_SCSI:
					fwProdString="Integrated Mirroring";
					bigWarning++;
					break;
				case MPI_FW_HEADER_PID_PROD_IS_SCSI:
					fwProdString="Integrated Striping";
					bigWarning++;
					break;
				case MPI_FW_HEADER_PID_PROD_CTX_SCSI:
					fwProdString="Context";
					break;
				default:
					fwProdString="Unknown";
					break;
				}
	
				switch(iocprod) {
				case MPI_FW_HEADER_PID_PROD_INITIATOR_SCSI:
					iocProdString="Initiator";
					break;
				case MPI_FW_HEADER_PID_PROD_TARGET_INITIATOR_SCSI:
					iocProdString="Initiator-Target";
					break;
				case MPI_FW_HEADER_PID_PROD_TARGET_SCSI:
					iocProdString="Target";
					break;
				case MPI_FW_HEADER_PID_PROD_IM_SCSI:
					iocProdString="Integrated Mirroring";
					break;
				case MPI_FW_HEADER_PID_PROD_IS_SCSI:
					iocProdString="Integrated Striping";
					break;
				case MPI_FW_HEADER_PID_PROD_CTX_SCSI:
					iocProdString="Context";
					break;
				default:
					iocProdString="Unknown";
					break;
				}

			} else  {
				fwProdString="Unknown";
				iocProdString="Unknown";
			}
	
			if (g_menuMode && g_force) {
				fprintf(stdout,
					"\nProduct Mismatch Warning:\n"
					"       New F/W: %s\n"
					"   Current F/W: %s\n"
					"Altering the F/W product may leave "
					"your system unable to reboot!\n\n",
					fwProdString, iocProdString);
				if (bigWarning) {
					fprintf(stdout,
						"\nPrior to downloading %s FW,\n"
						"please save your current FW image to diskette (FAT) format\n"
						"and download flsh1030 from the LSI website. Should your\n"
						"adapter not support %s, a diagnostic \n"
						"reset will result in your adapter malfunctioning.\n"
						"The ONLY method of recovery is to boot to DOS and use \n"
						"flsh1030 to download a non-%s FW image.\n"
						"Do NOT download %s if this is your boot adapter!!!.\n\n",
						fwProdString, fwProdString, fwProdString, fwProdString);
				}


				fprintf(stdout,
					"Are you absolutely sure you want "
					"to update the firmware? (y/[n]) ");
				fgets(userinp, DATAIN_LENGTH_SMALL - 1, stdin);
				if (userinp[0] == 'y')
					;
				else {
					FDU_LOG_ERROR(FDU_WRONG_FW_PRODUCT, "");
				}

				fprintf(stdout, "\n");
			} else if (g_force) {
				fprintf(stdout,
					"\nOverriding Product Mismatch.\n"
					"       New F/W: %s\n"
					"   Current F/W: %s\n"
					"Altering the F/W product may leave "
					"your system unable to reboot!\n\n",
					fwProdString, iocProdString);
			} else {
				FDU_LOG_ERROR(FDU_WRONG_FW_PRODUCT, "");
			}
		}
	}
	return (0);
}

/*****************************************************************
 *  verifyBiosPID
 *
 *  Inputs:   imageBuf - pointer to image buffer
 *            imageSize - size in bytes of image
 *            imageName - image versioning string
 *  Outputs:  For EFI, set the version string based on the revision
 *            field, bytes 12 and 13.
 *  Returns:  0.
 *  Globals:  set g_status.
 *
 *  Purpose:  Verify the BIOS Product information is acceptable.
 *****************************************************************/
int verifyBiosPID(ImageInfo_t *image)
{
	char *pPci;
	int isRaid = FDU_IS_NOT_RAID;
	int newImage = FDU_IS_NOT_RAID;
	char currentFw[DATAIN_LENGTH];
	char newBios[DATAIN_LENGTH];
	char userinp[DATAIN_LENGTH_SMALL];
	int  isFC = 0;
	int  iocpid = g_facts.ProductID & MPI_FW_HEADER_PID_PROD_MASK;
	int  ioctype = g_facts.ProductID & MPI_FW_HEADER_PID_TYPE_MASK;
	uint class_code, doMatch;
	uchar ident;

	FDU_LOG_DEBUG(5, fprintf(stdout, "Calling verifyBios: flags 0x%x, size %d "
			" version %s <enter>\n",
			image->flags, image->size, image->version));

	pPci = image->pBuf + ((image->pBuf[0x19] << 8) | image->pBuf[0x18]);

	if (iocpid == MPI_FW_HEADER_PID_PROD_IM_SCSI) {
		isRaid = FDU_IS_IM;
		strncpy(currentFw, "Integrated Mirror", DATAIN_LENGTH);
	} else if (iocpid == MPI_FW_HEADER_PID_PROD_IS_SCSI) {
		isRaid = FDU_IS_IS;
		strncpy(currentFw, "Integrated Striping", DATAIN_LENGTH);
	} else if (iocpid == MPI_FW_HEADER_PID_PROD_INITIATOR_SCSI)
		strncpy(currentFw, "Initiator", DATAIN_LENGTH);
	else if (iocpid == MPI_FW_HEADER_PID_PROD_TARGET_INITIATOR_SCSI)
		strncpy(currentFw, "Target Initiator", DATAIN_LENGTH);
	else if (iocpid == MPI_FW_HEADER_PID_PROD_TARGET_SCSI)
		strncpy(currentFw, "Target", DATAIN_LENGTH);

	/* Verify string and type: 0 - BIOS, 1 - FCode, 3 - EFI
	 * FCode, match PCI Ids as well....
	 * Strings:
	 * MPI P SCSI:
	 * 	<Class Code: 0x010000>
	 * 	MPTBIOS		<Identifyier: 0> <Same all PCI IDs>
	 *	MPT SCSI FCode 	<Identifyier: 1> <Same all PCI IDs>
	 *	MPT-SCSI-EFI	<Identifyier: 3> <Same all PCI IDs>
	 * MPI FCP:
	 * 	<Class Code: 0x0C0400>
	 * 	FC MPTBIOS	<Identifyier: 0> <Same all PCI IDs>
	 *	MPT FC FCode	<Identifyier: 1> <PCI ID Specific>
	 *	FCode, 				 <PCI ID Specific>
	 *	<no version> 			 <PCI ID Specific>
	 *	MPT-FC-EFI	<Identifyier: 3> <Same all PCI IDs>
	 *
	 */

#ifdef DEBUG_APP
	ShowBuf("PCI HEADER:", pPci, 48);
	getchar();
#endif

	if (strncmp(pPci,"PCIR", 4) != 0) {
		sprintf(g_pErrBuf,"This is not a PCI image!!");
		FDU_LOG_ERROR(FDU_WRONG_BIOS_PRODUCT,g_pErrBuf);
		return 0;
	}

	image->pci_id = (pPci[0x07] << 8) | pPci[0x06];
	ident = pPci[0x14];
	class_code = pPci[0xF] << 16 | pPci[0xE] << 8 | pPci[0xD];
	isFC = 0;
	doMatch = 0;
	newImage = isRaid;


	if ((strstr(image->version, "MPT FC FCode") != NULL) && (ident == 0x01)) {
		isFC = 1;
		doMatch = 1;
	} else if ((strstr(image->version, "MPT SCSI FCode") != NULL) && (ident == 0x01)) {
		;
	} else if ((strstr(image->version, "FC MPTBIOS") != NULL) && (ident == 0x00)) {
		isFC = 1;
	} else if ((strstr(image->version, "MPTBIOS") != NULL) && (ident == 0x00)) {
		if (image->pci_id == MPI_MANUFACTPAGE_DEVICEID_FC909) {
			/* Old  FC909 BIOS images
			 */
			isFC = 1;
			doMatch = 1;
		} else if (strstr(image->version, "-IM") != NULL) {
			newImage = FDU_IS_IM;
			strncpy(newBios, "Integrated Mirror (IM)", DATAIN_LENGTH);
		} else if (strstr(image->version, "-IS") != NULL) {
			newImage = FDU_IS_IS;
			strncpy(newBios, "Integrated Striping (IS)", DATAIN_LENGTH);
		} else {
			newImage = FDU_IS_NOT_RAID;
			strncpy(newBios, "Non-IM and Non-IS", DATAIN_LENGTH);
		}
	} else if ((strstr(image->version, "FCode") != NULL) && (ident == 0x01)){
		/* Obsolete FC format 2 */
		isFC = 1;
		doMatch = 1;
	} else if ((strlen(image->version) == 0) && (ident == 0x01) && (class_code == 0x0C0400)){
		/* Obsolete FC format 1 */
		isFC = 1;
		doMatch = 1;
	} else if ((strlen(image->version) == 0) && (ident == 0x03)) {
		/* EFI Image Found  - no magic what string.
		 * Check PCI ID.If proper, construct version strings.
		 */
		uchar major, minor, test, dev;

		major = (pPci[0x13] & 0xE0) >> 5;
		minor = pPci[0x13] & 0x1F;
		test = (pPci[0x12] & 0xF0) >> 4;
		dev = pPci[0x12] & 0x0F;

		if (image->pci_id == MPI_MANUFACTPAGE_DEVICEID_FC929) {
			isFC = 1;
			sprintf(image->version, "MPT-FC-EFI %02d.%02d.%02d.%02d",
				major, minor,test,dev);
		} else if (image->pci_id == MPI_MANUFACTPAGE_DEVID_53C1030) {
			sprintf(image->version, "MPT-SCSI-EFI %02d.%02d.%02d.%02d",
				major, minor,test,dev);
		} else {
			sprintf(g_pErrBuf,"Unknown PCI EFI Image! (0x%x)", image->pci_id);
			FDU_LOG_ERROR(FDU_WRONG_BIOS_PRODUCT,g_pErrBuf);
		}
	} else {
		sprintf(g_pErrBuf,"Unknown PCI Image!");
		FDU_LOG_ERROR(FDU_WRONG_BIOS_PRODUCT,g_pErrBuf);
	}


	if ((g_status == FDU_GOOD) && (doMatch)) {
		if (image->pci_id != image->adap_pci_id) {
			sprintf(g_pErrBuf,"PCI IDs must match: IOC 0x%x, image 0x%x",
				image->adap_pci_id, image->pci_id);
			FDU_LOG_ERROR(FDU_WRONG_BIOS_PRODUCT,g_pErrBuf);
		}
	}

	if (g_status != FDU_GOOD)
		return 0;

	FDU_LOG_DEBUG(0, fprintf(stdout, "isFC = %d \n", isFC));
	if (isFC)
		image->flags |= IMAGE_FLAGS_IS_FC;

	/* Start verification - check type, then Product ID
	 */
	if (ioctype == MPI_FW_HEADER_PID_TYPE_SCSI) {
		if (isFC == 1) {
			FDU_LOG_ERROR(FDU_WRONG_BIOS_TYPE, "");
		}
	} else if (ioctype == MPI_FW_HEADER_PID_TYPE_FC) {
		if (isFC == 0) {
			FDU_LOG_ERROR(FDU_WRONG_BIOS_TYPE, "");
		}
	} else  {
		FDU_LOG_ERROR(FDU_WRONG_BIOS_TYPE, "");
	}

	if (g_status != FDU_GOOD)
		return 0;

	if (isRaid == newImage)
		;	/* We have a match, continue */
	else {
		if (g_menuMode && g_force) {
			fprintf(stdout,
				"\nProduct Mismatch Warning:\n"
				"New Option ROM: %s\n"
				"   Current F/W: %s\n"
				"Altering the Option ROM product may leave "
				"your system unable to reboot!\n\n",
				newBios, currentFw);
			fprintf(stdout, "Are you absolutely sure you want "
				"to update the Option ROM? (y/[n]) ");
			fgets(userinp, DATAIN_LENGTH_SMALL - 1, stdin);
			if (userinp[0] == 'y')
					;
			else {
				FDU_LOG_ERROR(FDU_WRONG_BIOS_PRODUCT, "");
			}

			fprintf(stdout, "\n");
		} else if (g_force) {
			fprintf(stdout,
				"\nOverriding Product Mismatch.\n"
				"New Option ROM: %s\n"
				"   Current F/W: %s\n"
				"Altering the Option ROM product may leave "
				"your system unable to reboot!\n\n",
				newBios, currentFw);
		} else {
			FDU_LOG_ERROR(FDU_WRONG_BIOS_PRODUCT, "");
		}
	}
	
	return 0;
}

/*****************************************************************
 *  getMagicWhatString
 *
 *  Inputs:   ImageInfo_t - pointer to results buffer
 *  Outputs:  None.
 *  Returns:  newver;
 *  Globals:  set g_status.
 *
 *  Purpose:  Scan the image for the magic what string new version.
 *****************************************************************/
static char *getMagicWhatString(ImageInfo_t *pImage)
{
	char *whatstr;
	int   imageSize = pImage->size;
	char *imageBuf = pImage->pBuf;
	char  newver[IMAGE_VERSION_SIZE];
	char  what2[IMAGE_VERSION_SIZE];
	int   whatsz;

	/* Scan file for new version (magic what string identifier)
	 */
	newver[0] = what2[0] = '\0';
	if ((whatstr = whatSearch(imageBuf, &whatsz, imageSize)) != NULL && whatsz) {
		strncpy(newver, whatstr, min_t(int,sizeof(newver)-1, whatsz));
		newver[min_t(int, sizeof(newver)-1, whatsz)] = '\0';
		if ((whatstr = whatSearch(whatstr, &whatsz, imageSize-(whatstr-imageBuf+whatsz))) != NULL && whatsz) {
			strncpy(what2, whatstr, min_t(int, sizeof(what2)-1, whatsz));
			what2[min_t(int, sizeof(what2)-1, whatsz)] = '\0';
		}
	}

	strncpy(pImage->version, newver, IMAGE_VERSION_SIZE);

	return pImage->version;
}

/*****************************************************************
 *  getOldFwVersion
 *
 *  Inputs:   resultsBuf - pointer to results buffer
 *  Outputs:  set isFc, is Fibre Channel, flag
 *  Returns:  FW version;
 *  Globals:  set g_status.
 *
 *  Purpose:  Using the IOCFactsReply_t data, extract the version
 *            of the F/W that is currently executing. Append
 *            -IM or -IS string if appropriate.
 *****************************************************************/
static char *getOldFwVersion(char *resultsBuf, char *isFc)
{
	char *str = NULL;
	unsigned int oldver_value = 0;
	int isDbg, len;
	int pid = 0;

	*isFc = 0;
	isDbg = 0;

	pid = g_facts.ProductID & MPI_FW_HEADER_PID_TYPE_MASK;
	if (pid == MPI_FW_HEADER_PID_TYPE_FC)
		*isFc = 1;
	else if ( (pid != MPI_FW_HEADER_PID_TYPE_SCSI) &&
	 	(pid != MPI_FW_HEADER_PID_TYPE_SAS) )
		return NULL;

	if (g_facts.MsgLength >= 0x0F) {
		oldver_value = g_facts.FWVersion.Word;

		if ((*isFc) && (oldver_value & 0x00008000)) {
			oldver_value &= ~0x00008000;
			isDbg++;
		}
	} else  {

		oldver_value = 0;

		oldver_value = g_facts.Reserved_0101_FWVersion & 0x7000;
		oldver_value = (oldver_value << 4);
		oldver_value |= g_facts.Reserved_0101_FWVersion & 0x0FFF;
		oldver_value = (oldver_value << 8);

		if  (g_facts.Reserved_0101_FWVersion & 0x8000)
			isDbg++;
	}

	/* Create display string
	 */
	len = sprintf(resultsBuf, "%s-%02d.%02d.%02d.%02d",
				*isFc ? "LSIFC9x9" : "MPTFW",
				oldver_value >> 24,
				(oldver_value >> 16) & 0x00FF,
				(oldver_value >> 8) & 0x00FF,
				oldver_value  & 0x00FF);

	pid = g_facts.ProductID & MPI_FW_HEADER_PID_PROD_MASK;
	if (pid == MPI_FW_HEADER_PID_PROD_INITIATOR_SCSI)
		strcat(resultsBuf, "-I");
	else if (pid == MPI_FW_HEADER_PID_PROD_TARGET_INITIATOR_SCSI)
		strcat(resultsBuf, "-TI");
	else if (pid == MPI_FW_HEADER_PID_PROD_TARGET_SCSI)
		strcat(resultsBuf, "-T");
	else if (pid == MPI_FW_HEADER_PID_PROD_IM_SCSI)
		strcat(resultsBuf, "-IM");
	else if (pid == MPI_FW_HEADER_PID_PROD_IS_SCSI)
		strcat(resultsBuf, "-IS");

	if (isDbg)
		strcat(resultsBuf, " (DBG)");

	str = resultsBuf;

	return str;
}
/*****************************************************************
 *  updateBiosImage
 *
 *  Inputs:   image - pointer to BIOS image
 *            imageSize - number of bytes
 *            pIocInfo - pointer to mpt_ioctl_iocinfo
 *  Outputs:  None.
 *  Returns:  None.
 *  Globals:  None.
 *
 *  Purpose:  Make sure that the class code is set,
 *            the lat bit indicator is set and
 *            that the device ID is correct.
 *****************************************************************/
static void updateBiosImage(ImageInfo_t *image)
{
	uchar *pPci;
	uchar *pEROM;
	uint offset = 0;
	int numImages = 1;
	int didChange = 0;
	int val;
	int jj;

	FDU_LOG_DEBUG(0, fprintf (stdout, "Calling updateBios: flags 0x%x, size %d\n", image->flags, image->size));

	/* Set pointer to PCI Expansion ROM image
	 */
	pEROM = image->pBuf;

	/* Set pointer to PCI Data Structure
	 */
	pPci = pEROM + ((pEROM[0x19] << 8) | pEROM[0x18]);

#ifdef DEBUG_APP_BIG
	printf( "pEROM @ %p, pbuf @ %p  \n", pEROM, image->pBuf);
	printf( "pPci @ %p \n", pPci);
	ShowBuf("U1 EROM:", pEROM, 48);
	ShowBuf("U1 PCI Struct:", pPci, 32);
#endif

	if (image->flags & IMAGE_FLAGS_MULTI_IMG) {
		numImages = 2;
		offset = (pPci[0x11] << 8 | pPci[0x10] ) * 512;
	}

	for (jj = 0; jj < numImages; jj++) {
		pEROM = image->pBuf + (jj * offset);
		pPci = pEROM+ ((pEROM[0x19] << 8) | pEROM[0x18]);

		/* Set the Device Identification */
		val = pPci[0x7] << 8 | pPci[0x6];

		if (image->adap_pci_id != 0xFFFF) {
			FDU_LOG_DEBUG(0, fprintf (stdout, "pid = %x, val = %x \n", image->adap_pci_id, val));

			if (val != (image->adap_pci_id)) {
				pPci[0x7] = (image->adap_pci_id & 0xFF00) >> 8;
				pPci[0x6] = image->adap_pci_id & 0xFF;
				didChange = 1;
			}
		}


		/* If Multi, make sure last image bit clear on first
		 *    image and set on second image.
		 * If flag is set, make sure last image bit is set.
		 * If flag is clear, make sure last image bit is clear.
		 */
		if (image->flags & IMAGE_FLAGS_MULTI_IMG) {
			if ((jj == 0) && (pPci[0x15] & 0x80)) {
				pPci[0x15] &= ~0x80;
				didChange = 1;
			} else if ((jj == 1) && (!(pPci[0x15] & 0x80))) {
				pPci[0x15] |= 0x80;
				didChange = 1;
			}
		} else if ((!(pPci[0x15] & 0x80)) && (image->flags & IMAGE_FLAGS_LAST_IMG)) {
				pPci[0x15] |= 0x80;
				didChange = 1;
		} else if ((pPci[0x15] & 0x80) && (!(image->flags & IMAGE_FLAGS_LAST_IMG))) {
				pPci[0x15] &= ~0x80;
				didChange = 1;
		}

		/* Set the class code */
		val = pPci[0xF] << 16 | pPci[0xE] << 8 | pPci[0xD];

		FDU_LOG_DEBUG(0, fprintf(stdout, "flags 0x%x , val = 0x%x \n", image->flags,  val));

		if (image->flags & IMAGE_FLAGS_IS_FC) {
			if (val != 0x0C0400) {
				pPci[0xF] = 0x0c;
				pPci[0xE] = 0x04;
				pPci[0xD] = 0x00;
				didChange = 1;
			}
		} else {
			if (val != 0x010000) {
				pPci[0xF] = 0x01;
				pPci[0xE] = pPci[0xD] = 0x00;
				didChange = 1;
			}
		}

#ifdef DEBUG_APP_BIG
	printf( "didChange %d\n", didChange);
	ShowBuf("U2 EROM:", pEROM, 48);
	ShowBuf("U2 PCI Struct:", pPci, 32);
#endif

		/* If modified anything, update the checksum
		 */
		if (didChange) {
			uint size, ii;
			uchar chksum = 0;

			pPci = pEROM+ ((pEROM[0x19] << 8) | pEROM[0x18]);
			size = (pPci[0x11] << 8 | pPci[0x10] ) * 512;

			FDU_LOG_DEBUG(0, fprintf(stdout, "Updating checksum...size 0x%x (%d)\n", size, size));
			FDU_LOG_DEBUG(0, fprintf(stdout, "Old: 0x%02x ", pEROM[size - 1]));

			for (ii = 0; ii < (size - 1); ii++) {
				chksum += pEROM[ii];
			}
			pEROM[ii] = (~chksum) + 1;

			FDU_LOG_DEBUG(0, fprintf(stdout, "New: 0x%02x\n", (u8) pEROM[ii]));
		}
	}

	return;
}


/*****************************************************************
 *  whatSearch
 *
 *  Inputs:   buf - data buffer pointer
 *            size_in - number of bytes
 *            size_out - number of bytes
 *  Outputs:  None.
 *  Returns:  Pointer to what string.
 *  Globals:  None.
 *
 *  Purpose:  Search for the magic what string.
 *****************************************************************/
static char *whatSearch(const char *buf, int *size_out, int size_in)
{
	register char c;
	register char *found = NULL;

	*size_out = 0;
	while (--size_in) {
		c = *buf++;
loop:
		if (size_in < 5)
			return NULL;
		if (c != '@')
			continue;
		if (--size_in && (c = *buf++) != '(')
			goto loop;
		if (--size_in && (c = *buf++) != '#')
			goto loop;
		if (--size_in && (c = *buf++) != ')')
			goto loop;
		found = (char*)buf;
		while (--size_in && (c = *buf++) && c != '"' && c != '>' && c != '\n')
			(*size_out)++;
//		fprintf(stderr, "\nDbG: offset=%d, size_out=%d\n", found-fwbuf, *size_out);
		return found;
	}
	return NULL;
}

/*****************************************************************
 *  printDisableWarning
 *
 *  Inputs:   None.
 *  Outputs:  None.
 *  Returns:  None.
 *  Globals:  None.
 *
 *  Purpose:  Display a warning message.
 *****************************************************************/
void printDisableWarning(char *imgString)
{
	clr_scr();
	fprintf(stdout, "\n *** WARNING %s Download Failed ***\n\n", imgString);

	if (strcmp(imgString,"F/W") == 0) {
		fprintf(stdout, "Adapter Flash part now contains invalid F/W\n"
				"and Option ROM images and will malfunction\n"
				"once reset.\n\n");
	} else {
		fprintf(stdout, "Adapter Flash part now contains invalid\n"
				"an Option ROM image and will malfunction\n"
				"once reset.\n\n");
	}


	fprintf(stdout, "Please download the latest F/W images, Option ROM images and\n"
			"flash utilities from the LSI Logic website (www.lsilogic.com).\n"
			"   ia64 arch: efimptfl (Place on a FAT formatted diskette)\n"
			"    x86 arch: flsh1030 (1030 adapters); fcutil (9xx adapters)\n"
			"              (Place on a bootable diskette).\n\n");

	if (g_status == FDU_GOOD) {
		/* erase was OK
		 */
		;
	} else {
		/* erase failed! This shouldn't be an issue, as the FW will have the
		 * flash erase the first sector if write at offset 0. Whether or not
		 * the data write succeeds is not relevant.
		 */
		fprintf(stdout, "Erase  of %s image failed!\n", imgString);
	}


	fprintf(stdout, "Do _not_ issue a diagnostic reset or perform a power\n"
			"cycle until you have successfully downloaded\n");
	if (strcmp(imgString,"F/W") == 0) {
		fprintf(stdout, "new F/W and Option ROM images! ");
	} else {
		fprintf(stdout, "a new Option ROM image! ");
	}
	fprintf(stdout, " Such an action may leave \n"
			"your system unusable and requiring a reboot.\n");

	fprintf(stdout, "If a re-attempt with this utility fails, reboot and\n"
			"use the Flash utility downloaded from the website\n"
			"to write new F/W and/or Option ROM images onto the \n"
			"adapter's flash part.\n");

	fprintf(stdout, "\n *** Adapter Requries a New %s Image For Proper Operation ***\n\n", imgString);

	if (g_menuMode) {
		fprintf(stdout, "Press enter to continue...");
		getchar();
	} else {
		sleep(20);
	}
}

/*****************************************************************
 *  printQiesceMsg
 *
 *  Inputs:   None.
 *  Outputs:  None.
 *  Returns:  None.
 *  Globals:  None.
 *
 *  Purpose:  Display a warning message.
 *****************************************************************/
void printQuiesceMsg(void)
{
	static int did=0 ;

	if (!did)
		fprintf(stdout, "Please close all applications!\n");

	did++;
	return;
}
