/*
 * LSIUtil -- configuration utility for MPT adapters (FC, SCSI, and SAS/SATA)
 *
 * Written by Stephen F. Shirron, October 11, 2002
 */


#define LSIUTIL_VERSION "Version 1.29, December 2, 2004"


char what[] = "@(#)LSI Logic MPT Configuration Utility, " LSIUTIL_VERSION;


#include <fcntl.h>
#if WIN32
#include <io.h>
#include "inc\getopt.h"
#define sleep(x) _sleep(x*1000)
#endif
#include <sys/stat.h>
#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#if WIN32
#include <windows.h>
#include "inc\devioctl.h"
typedef unsigned long * ULONG_PTR;
#define strcasecmp stricmp
#define strncasecmp strnicmp
#include "inc\ntddscsi.h"
#endif
#if __linux__ || __sparc__ || __irix__ || __alpha__
#include <unistd.h>
#include <strings.h>
#include <sys/ioctl.h>
#define FALSE 0
#define TRUE 1
typedef int HANDLE;
#define O_BINARY 0
#define min(x,y) ((int)(x) < (int)(y) ? (x) : (y))
#define max(x,y) ((int)(x) > (int)(y) ? (x) : (y))
#endif

#if WIN32
typedef unsigned __int64 uint64_t;
#pragma pack(1)
#include "inc\mpi_type.h"
#include "inc\mpi.h"
#include "inc\mpi_ioc.h"
#include "inc\mpi_cnfg.h"
#include "inc\mpi_init.h"
#include "inc\mpi_fc.h"
#include "inc\mpi_sas.h"
#include "inc\mpi_raid.h"
#pragma pack()
#include "inc\sym_dmi.h"
#define ISSUE_BUS_RESET 0x800000FF
typedef struct
{
	SRB_IO_CONTROL	 Sic;
	UCHAR			 Buf[8+128+1024];
} SRB_BUFFER;
#endif
#if __linux__
typedef unsigned long long uint64_t;
#pragma pack(1)
#include "lsi/mpi_type.h"
#include "lsi/mpi.h"
#include "lsi/mpi_ioc.h"
#include "lsi/mpi_cnfg.h"
#include "lsi/mpi_init.h"
#include "lsi/mpi_fc.h"
#include "lsi/mpi_sas.h"
#include "lsi/mpi_raid.h"
#pragma pack()
typedef U8 u8;
typedef U16 u16;
typedef U32 u32;
#include "mptctl.h"
#endif
#if __sparc__
#define MPI_POINTER *
typedef uint8_t U8;
typedef uint16_t U16;
typedef uint32_t U32;
typedef struct
{
	U32		 Low;
	U32		 High;
} U64;
#pragma pack(1)
#include "salimpi/inc/mpi.h"
#include "salimpi/inc/mpi_ioc.h"
#include "salimpi/inc/mpi_cnfg.h"
#include "salimpi/inc/mpi_init.h"
#include "salimpi/inc/mpi_fc.h"
#include "salimpi/inc/mpi_sas.h"
#include "salimpi/inc/mpi_raid.h"
#pragma pack()
#define TARGET_MPT
typedef uint8_t  UINT8;
typedef uint16_t UINT16;
typedef uint32_t UINT32;
typedef uint64_t UINT64;
#include "inc/dmi_ioctl.h"
#endif
#if __irix__
#define MPI_POINTER *
typedef uint8_t U8;
typedef uint16_t U16;
typedef uint32_t U32;
typedef struct
{
	U32		 Low;
	U32		 High;
} mpiU64;
#pragma pack(1)
#define U64 mpiU64
#include "mpi.h"
#include "mpi_ioc.h"
#include "mpi_cnfg.h"
#include "mpi_init.h"
#include "mpi_fc.h"
#include "mpi_sas.h"
#include "mpi_raid.h"
#pragma pack(0)
#undef U64
#define TARGET_MPT
typedef uint8_t  UINT8;
typedef uint16_t UINT16;
typedef uint32_t UINT32;
typedef uint64_t UINT64;
#include "dmi_ioctl.h"
#include <sys/scsi.h>
#endif
#if __alpha__
typedef unsigned __int64 uint64_t;
#define MPI_POINTER *
typedef unsigned char U8;
typedef unsigned short U16;
typedef unsigned int U32;
typedef unsigned long U64;
typedef struct
{
	U32		 Low;
	U32		 High;
} mpiU64;
#pragma pack(1)
#define U64 mpiU64
#include "mpi.h"
#include "mpi_ioc.h"
#include "mpi_cnfg.h"
#include "mpi_init.h"
#include "mpi_fc.h"
#include "mpi_sas.h"
#include "mpi_raid.h"
#pragma pack(0)
#undef U64
typedef U8 u8;
typedef U16 u16;
typedef U32 u32;
typedef U64 u64;
#include "mptctl.h"
#endif


#define swap16(x)            \
	((((U16)(x)>>8)&0xff) |  \
	 (((U16)(x)&0xff)<<8))
#define swap32(x)                 \
	((((U32)(x)>>24)&0xff) |      \
	((((U32)(x)>>16)&0xff)<<8) |  \
	((((U32)(x)>>8)&0xff)<<16) |  \
	 (((U32)(x)&0xff)<<24))
#if WIN32 || __alpha__
#define get16(x) (x)
#define get32(x) (x)
#define set16(x) (x)
#define set32(x) (x)
#endif
#if __linux__
#include <endian.h>
#include <linux/types.h>
#if __BYTE_ORDER == __BIG_ENDIAN
#include <linux/byteorder/big_endian.h>
#endif
#if __BYTE_ORDER == __LITTLE_ENDIAN
#include <linux/byteorder/little_endian.h>
#endif
#define get16(x) __le16_to_cpu(x)
#define get32(x) __le32_to_cpu(x)
#define set16(x) __cpu_to_le16(x)
#define set32(x) __cpu_to_le32(x)
#endif
#if __sparc__ || __irix__
#if i386
#define get16(x) (x)
#define get32(x) (x)
#define set16(x) (x)
#define set32(x) (x)
#else
#define get16(x) swap16(x)
#define get32(x) swap32(x)
#define set16(x) swap16(x)
#define set32(x) swap32(x)
#endif
#endif
#define get64(x) (((uint64_t)get32(((U32 *)&(x))[1])<<32) | get32(((U32 *)&(x))[0]))


#define NUM_PORTS 64

typedef struct
{
	char			 signature[4];
	unsigned short	 vendorId;
	unsigned short	 deviceId;
	unsigned char	 reserved1[2];
	unsigned short	 pcirLength;
	unsigned char	 pcirRevision;
	unsigned char	 classCode[3];
	unsigned short	 imageLength;
	unsigned short	 imageRevision;
	unsigned char	 type;
	unsigned char	 indicator;
	unsigned char	 reserved2[2];
} PCIR;


typedef struct
{
	int				 portNumber;
	char			 portName[16];
	HANDLE			 fileHandle;
	int				 ioctlValue;
	int				 iocNumber;
	int				 mptVersion;
	int				 fwVersion;
	int				 whoInit;
	U16				 deviceId;
	U8				 revisionId;
	U16				 productId;
	char			*chipName;
	int				 payOff;
	int				 portType;
	int				 maxPersistentIds;
	int				 maxBuses;
	int				 maxTargets;
	int				 maxLuns;
	int				 hostScsiId;
} MPT_PORT;


typedef struct
{
	SCSIIOReply_t	 reply;
	U8				 sense[18];
} SCSI_REPLY;


typedef struct
{
	int				 bus;
	int				 target;
	int				 lun;
	int				 mode;
	unsigned int	 size;
} DIAG_TARGET;


MPT_PORT			*mptPorts[NUM_PORTS];


#if __sparc__
MPT_PORT			*mappedPort;
int					 mappedTarget;
int					 mappedValue;
#endif


char				*fileNames[3];
int					 numFileNames;
int					 yesFlag;


int					 tagType;


/* user command line arguments */
typedef struct
{
	int				 portNums[NUM_PORTS];	/* port numbers specified */
	int				 numPortNums;			/* number of port numbers specified */
	int				 scan;					/* boolean */
	int				 info;					/* boolean */
	int				 dump;					/* boolean */
	char			 linkspeed;				/* desired link speed ('a', '1', '2', or '4') */
	char			 topology;				/* desired topology ('a', '1', or '2') */
	int				 reset;					/* boolean for chip reset */
	int				 linkReset;				/* boolean for link reset */
	int				 linkResetFlag;			/* boolean for link reset type */
	int				 coalescing;			/* boolean */
	int				 ic_depth;				/* desired interrupt coalescing depth */
	int				 ic_timeout;			/* desired interrupt coalescing timeout */
} CMNDLINE_ARGS;


unsigned char LoopIdToAlpa[126] =
{
	0xef, 0xe8, 0xe4, 0xe2, 0xe1, 0xe0, 0xdc, 0xda, 0xd9, 0xd6,  /* 0 to 9 */
	0xd5, 0xd4, 0xd3, 0xd2, 0xd1, 0xce, 0xcd, 0xcc, 0xcb, 0xca,  /* 10 to 19 */
	0xc9, 0xc7, 0xc6, 0xc5, 0xc3, 0xbc, 0xba, 0xb9, 0xb6, 0xb5,  /* 20 to 29 */
	0xb4, 0xb3, 0xb2, 0xb1, 0xae, 0xad, 0xac, 0xab, 0xaa, 0xa9,  /* 30 to 39 */
	0xa7, 0xa6, 0xa5, 0xa3, 0x9f, 0x9e, 0x9d, 0x9b, 0x98, 0x97,  /* 40 to 49 */
	0x90, 0x8f, 0x88, 0x84, 0x82, 0x81, 0x80, 0x7c, 0x7a, 0x79,  /* 50 to 59 */
	0x76, 0x75, 0x74, 0x73, 0x72, 0x71, 0x6e, 0x6d, 0x6c, 0x6b,  /* 60 to 69 */
	0x6a, 0x69, 0x67, 0x66, 0x65, 0x63, 0x5c, 0x5a, 0x59, 0x56,  /* 70 to 79 */
	0x55, 0x54, 0x53, 0x52, 0x51, 0x4e, 0x4d, 0x4c, 0x4b, 0x4a,  /* 80 to 89 */
	0x49, 0x47, 0x46, 0x45, 0x43, 0x3c, 0x3a, 0x39, 0x36, 0x35,  /* 90 to 99 */
	0x34, 0x33, 0x32, 0x31, 0x2e, 0x2d, 0x2c, 0x2b, 0x2a, 0x29,  /* 100 to 109 */
	0x27, 0x26, 0x25, 0x23, 0x1f, 0x1e, 0x1d, 0x1b, 0x18, 0x17,  /* 110 to 119 */
	0x10, 0x0f, 0x08, 0x04, 0x02, 0x01  /* 120 to 125 */
};


unsigned char AlpaToLoopId[256] =
{
	0x7e, 0x7d, 0x7c, 0xff, 0x7b, 0xff, 0xff, 0xff,  /* 0x00 to 0x07 */
	0x7a, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x79,  /* 0x08 to 0x0f */
	0x78, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x77,  /* 0x10 to 0x17 */
	0x76, 0xff, 0xff, 0x75, 0xff, 0x74, 0x73, 0x72,  /* 0x18 to 0x1f */
	0xff, 0xff, 0xff, 0x71, 0xff, 0x70, 0x6f, 0x6e,  /* 0x20 to 0x27 */
	0xff, 0x6d, 0x6c, 0x6b, 0x6a, 0x69, 0x68, 0xff,  /* 0x28 to 0x2f */
	0xff, 0x67, 0x66, 0x65, 0x64, 0x63, 0x62, 0xff,  /* 0x30 to 0x37 */
	0xff, 0x61, 0x60, 0xff, 0x5f, 0xff, 0xff, 0xff,  /* 0x38 to 0x3f */
	0xff, 0xff, 0xff, 0x5e, 0xff, 0x5d, 0x5c, 0x5b,  /* 0x40 to 0x47 */
	0xff, 0x5a, 0x59, 0x58, 0x57, 0x56, 0x55, 0xff,  /* 0x48 to 0x4f */
	0xff, 0x54, 0x53, 0x52, 0x51, 0x50, 0x4f, 0xff,  /* 0x50 to 0x57 */
	0xff, 0x4e, 0x4d, 0xff, 0x4c, 0xff, 0xff, 0xff,  /* 0x58 to 0x5f */
	0xff, 0xff, 0xff, 0x4b, 0xff, 0x4a, 0x49, 0x48,  /* 0x60 to 0x67 */
	0xff, 0x47, 0x46, 0x45, 0x44, 0x43, 0x42, 0xff,  /* 0x68 to 0x6f */
	0xff, 0x41, 0x40, 0x3f, 0x3e, 0x3d, 0x3c, 0xff,  /* 0x70 to 0x77 */
	0xff, 0x3b, 0x3a, 0xff, 0x39, 0xff, 0xff, 0xff,  /* 0x78 to 0x7f */
	0x38, 0x37, 0x36, 0xff, 0x35, 0xff, 0xff, 0xff,  /* 0x80 to 0x87 */
	0x34, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x33,  /* 0x88 to 0x8f */
	0x32, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x31,  /* 0x90 to 0x97 */
	0x30, 0xff, 0xff, 0x2f, 0xff, 0x2e, 0x2d, 0x2c,  /* 0x98 to 0x9f */
	0xff, 0xff, 0xff, 0x2b, 0xff, 0x2a, 0x29, 0x28,  /* 0xa0 to 0xa7 */
	0xff, 0x27, 0x26, 0x25, 0x24, 0x23, 0x22, 0xff,  /* 0xa8 to 0xaf */
	0xff, 0x21, 0x20, 0x1f, 0x1e, 0x1d, 0x1c, 0xff,  /* 0xb0 to 0xb7 */
	0xff, 0x1b, 0x1a, 0xff, 0x19, 0xff, 0xff, 0xff,  /* 0xb8 to 0xbf */
	0xff, 0xff, 0xff, 0x18, 0xff, 0x17, 0x16, 0x15,  /* 0xc0 to 0xc7 */
	0xff, 0x14, 0x13, 0x12, 0x11, 0x10, 0x0f, 0xff,  /* 0xc8 to 0xcf */
	0xff, 0x0e, 0x0d, 0x0c, 0x0b, 0x0a, 0x09, 0xff,  /* 0xd0 to 0xd7 */
	0xff, 0x08, 0x07, 0xff, 0x06, 0xff, 0xff, 0xff,  /* 0xd8 to 0xdf */
	0x05, 0x04, 0x03, 0xff, 0x02, 0xff, 0xff, 0xff,  /* 0xe0 to 0xe7 */
	0x01, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00,  /* 0xe8 to 0xef */
	0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,  /* 0xf0 to 0xf7 */
	0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff   /* 0xf8 to 0xff */
};


int findPorts(void);
int closePorts(int numPorts);
int getFileName(char *buf, int len, FILE *file, char *fileString, int fileType);
int getString(char *buf, int len, FILE *file);
int getYesNoAnswer(int defvalue);
int getNumberAnswer(int low, int high, int defvalue);
int getHexNumberAnswer(U32 *value);
int getHexDoubleNumberAnswer(U32 *value1, U32 *value2);
int readFile(char *name, unsigned char **outBuf, int *outLen);
int printWhatString(char *type, unsigned char *buf, int len);
int getCompatible(int deviceId, int type);
int checkCompatible(int deviceId1, int deviceId2, int type);
int doPort(MPT_PORT *port);
int doPortOption(MPT_PORT *port, int option);
int doIdentify(MPT_PORT *port);
int doFirmwareDownload(MPT_PORT *port);
int doFirmwareUpload(MPT_PORT *port);
int doBiosFcodeDownload(MPT_PORT *port);
int doBiosFcodeUpload(MPT_PORT *port, unsigned char **outBuf, int *outLen, int type);
int verifyBiosFcodeImage(MPT_PORT *port, unsigned char *buf, int len, int type);
int splitBiosImage(MPT_PORT *port, unsigned char **buf1, int *len1, unsigned char **buf2, int *len2);
int fixupBiosFcodeImage(MPT_PORT *port, unsigned char *buf, int len, int last);
int doSeepromDownload(MPT_PORT *port);
int doSeepromUpload(MPT_PORT *port);
int doScanForDevices(MPT_PORT *port, int flag);
int doConfigPage(MPT_PORT *port);
int doInterruptCoalescingValues(MPT_PORT *port, int timeout, int depth);
int doIocSettings(MPT_PORT *port);
int doScsiInitiatorSettings(MPT_PORT *port);
int doScsiTargetSettings(MPT_PORT *port);
int doFcLinkSpeedValue(MPT_PORT *port, int t);
int doFcTopologyValue(MPT_PORT *port, int t);
int doFcPortOffline(MPT_PORT *port);
int doFcPortOnline(MPT_PORT *port);
int doFcTopologyNLPort(MPT_PORT *port);
int doFcTopologyNPort(MPT_PORT *port);
int doFcPortSettings(MPT_PORT *port);
int doSasIoUnitSettings(MPT_PORT *port);
int doMultiPathing(MPT_PORT *port);
int doFcPersistentMappings(MPT_PORT *port, int command);
int doSasPersistentMappings(MPT_PORT *port, int command);
int doDisplayLoggedInDevices(MPT_PORT *port);
int doDisplayAttachedDevices(MPT_PORT *port);
int doShowPortAliases(MPT_PORT *port);
int doTestConfigPageActions(MPT_PORT *port);
int doDiagnostics(MPT_PORT *port, int command);
int doInquiryTest(MPT_PORT *port);
int doWriteBufferReadBufferCompareTest(MPT_PORT *port);
int doReadTest(MPT_PORT *port);
int doWriteReadCompareTest(MPT_PORT *port);
int doWriteTest(MPT_PORT *port);
int doReadCompareTest(MPT_PORT *port);
int doEchoTest(MPT_PORT *port);
int doReadLinkErrorStatusTest(MPT_PORT *port);
int doDisplayPortCounters(MPT_PORT *port);
int doClearPortCounters(MPT_PORT *port);
int doRaidActions(MPT_PORT *port, int command);
int doShowVolumes(MPT_PORT *port, IOCPage2_t *IOCPage2);
int doShowPhysDisks(MPT_PORT *port, IOCPage2_t *IOCPage2);
int doGetVolumeState(MPT_PORT *port, IOCPage2_t *IOCPage2);
int doModifyVolume(MPT_PORT *port, IOCPage2_t *IOCPage2, int action, char *string);
int doCreateVolume(MPT_PORT *port, IOCPage2_t *IOCPage2);
int doDeleteVolume(MPT_PORT *port, IOCPage2_t *IOCPage2);
int doVolumeSettings(MPT_PORT *port, IOCPage2_t *IOCPage2);
int doModifyPhysDisk(MPT_PORT *port, IOCPage2_t *IOCPage2, int action, char *string);
int doCreatePhysDisk(MPT_PORT *port, IOCPage2_t *IOCPage2);
int doPhysDiskSettings(MPT_PORT *port, IOCPage2_t *IOCPage2);
int doResetBus(MPT_PORT *port);
int doResetTarget(MPT_PORT *port);
int doGID_FT(MPT_PORT *port);
int doGA_NXT(MPT_PORT *port);
int doExLinkServiceSend(MPT_PORT *port);
int doResetFcLink(MPT_PORT *port, int flag);
int doResetSasPhy(MPT_PORT *port);
int doResetSasLink(MPT_PORT *port);
int doDumpPortState(MPT_PORT *port, int flag);
int doPortStateSummary(MPT_PORT *port);
int getPortInfo(MPT_PORT *port);
int showPortInfo(MPT_PORT *port);
int showPortInfoHeader(MPT_PORT *port);
int getDeviceInfo(MPT_PORT *port, int bus, int target, char *buf, int len);
int getDeviceInfoHeader(MPT_PORT *port, char *buf, int len);
int getIocFacts(MPT_PORT *port, IOCFactsReply_t *rep);
int getPortFacts(MPT_PORT *port, PortFactsReply_t *rep);
int getConfigPageHeader(MPT_PORT *port, int type, int number, int address, ConfigReply_t *repOut);
int getConfigPageAction(MPT_PORT *port, int action, int type, int number, int address, void *page, int pageSize);
int getConfigPage(MPT_PORT *port, int type, int number, int address, void *page, int pageSize);
int setConfigPageAction(MPT_PORT *port, int action, int type, int number, int address, void *page, int pageSize);
int setConfigPage(MPT_PORT *port, int type, int number, int address, void *page, int pageSize);
int showConfigPage(MPT_PORT *port, char *string, void *page);
int mapOsToHwTarget(MPT_PORT *port, int target);
int doInquiry(MPT_PORT *port, int bus, int target, int lun, unsigned char *buf, int len);
int doInquiryVpdPage(MPT_PORT *port, int bus, int target, int lun, int page, unsigned char *buf, int len);
int doReadCapacity(MPT_PORT *port, int bus, int target, int lun, unsigned char *buf, int len);
int doReadBuffer(MPT_PORT *port, int bus, int target, int lun, int mode, unsigned char *buf, int len);
int doWriteBuffer(MPT_PORT *port, int bus, int target, int lun, int mode, unsigned char *buf, int len);
int doRead(MPT_PORT *port, int bus, int target, int lun,
		   unsigned int lbn, int lbns, int mode, unsigned char *buf, int len);
int doWrite(MPT_PORT *port, int bus, int target, int lun,
			unsigned int lbn, int lbns, int mode, unsigned char *buf, int len);
int doScsiIo(MPT_PORT *port, SCSIIORequest_t *req, int reqSize, SCSI_REPLY *rep, int repSize,
			 void *payIn, int payInSize, void *payOut, int payOutSize, int timeOut);
int doFwDownload(MPT_PORT *port, int type, unsigned char *buf, int len, int offset);
int doFwUpload(MPT_PORT *port, int type, unsigned char *buf, int len, int offset, int *outLen);
int doIocInit(MPT_PORT *port, int WhoInit);
int doResetPort(MPT_PORT *port);
int doMptCommand(MPT_PORT *port, void *req, int reqSize, void *rep, int repSize,
				 void *payIn, int payInSize, void *payOut, int payOutSize, int timeOut);
int doMptCommandCheck(MPT_PORT *port, void *req, int reqSize, void *rep, int repSize,
					  void *payIn, int payInSize, void *payOut, int payOutSize, int timeOut);
char *translateIocStatus(int ioc_status);
void dumpMemory(void *buf, int len, char *string);


int
main(int argc, char *argv[])
{
	int				 portNumber;
	MPT_PORT		*port;
	int				 numPorts;
	int				 i;
	int				 arg;
	CMNDLINE_ARGS	 cargs;
	char			*type;
	int				 n;
	int				 t;
	extern int		 optind;

	numFileNames = 0;
	yesFlag = FALSE;

	tagType = MPI_SCSIIO_CONTROL_SIMPLEQ;

	memset(&cargs, 0, sizeof (CMNDLINE_ARGS));

	printf("\nLSI Logic MPT Configuration Utility, %s\n", LSIUTIL_VERSION);

	numPorts = findPorts();

	printf("\n%d MPT Port%s found\n", numPorts, numPorts == 1 ? "" : "s");

	if (numPorts > 0)
	{
		if (argc > 1)
		{
			while ((arg = getopt(argc, argv, "c:df:hil:p:rst:uyz")) != EOF)
			{
				switch (arg)
				{
				case 'h':
					printf(
"\nHelp Usage\n"
"Invoking lsiutil with no arguments will start an interactive session\n\n"
"Display Options\n"
"usage:  lsiutil [ -p portNumber ] [ -u ][ -s ] [ -d ] [ -i ]\n"
"      -p portNumber     Specify the port number to operate on.\n"
"                        If not specified, all ports used.\n"
"      -u                Use untagged, rather than tagged, SCSI commands.\n"
"      -s                Scan for and display all targets.\n"
"      -i                Display port settings.\n"
"      -d                Dump all config pages.\n\n"
"Examples:\n"
"1. to display the port settings and targets for port 1:\n"
"       lsiutil -p 1 -i -s\n"
"2. to display the targets found on all known ports:\n"
"       lsiutil -s\n\n"
"Operational Options\n"
"usage:  lsiutil -p portNumber [ -l linkSpeed ] [ -t topology ] [ -c timeout,depth ] [ -r ]\n"
"      -p portNumber     Specify the port number to operate on.\n"
"                        Required parameter for operational options.\n"
"      -l linkSpeed      Set link speed.  Valid options for linkSpeed are:\n"
"                            'a'     Auto link speed negotiation\n"
"                            '1'     Force 1Gb link speed\n"
"                            '2'     Force 2Gb link speed\n"
"                            '4'     Force 4Gb link speed\n"
"      -t topology       Set topology.  Valid options for topology are:\n"
"                            'a'     Auto topology negotiation\n"
"                            '1'     Force NL_Port topology\n"
"                            '2'     Force N_Port topology\n"
"      -c timeout,depth  Set interrupt coalescing values.\n"
"                        Timeout is a value in microseconds between\n"
"                        1 and 1000.  Depth is a value between 1 and 128.\n"
"                        Setting either or both values to zero will\n"
"                        disable interrupt coalescing for that port.\n"
"      -r                Perform a chip reset on the given port.\n"
"      -z                Perform an FClink  reset on the given port.\n"
"NOTE:  In order for linkSpeed, topology, or interrupt coalescing\n"
"       settings to take effect, a chip reset is necessary.\n\n"
"Examples:\n"
"1. to force linkspeed to 1Gb on port 2:\n"
"       lsiutil -p 2 -l 1\n"
"2. to set interrupt coalescing to a timeout of 200ms with\n"
"   a depth of 9 and to force N_Port topology on port 1:\n"
"       lsiutil -p 1 -c 200,9 -t 2\n\n"
);
					closePorts(numPorts);
					return 0;

				case 'c':
					if (sscanf(optarg, "%d,%d", &cargs.ic_timeout, &cargs.ic_depth) != 2)
					{
						printf("ERROR:  Invalid argument %s for interrupt coalescing.\n", optarg);
						printf("Must specify timeout,depth\n");
						closePorts(numPorts);
						return 0;
					}
					else
					{
						if (cargs.ic_timeout < 0)
							cargs.ic_timeout = 0;
						if (cargs.ic_timeout > 1000)
							cargs.ic_timeout = 1000;
						if (cargs.ic_depth < 0)
							cargs.ic_depth = 0;
						if (cargs.ic_depth > 128)
							cargs.ic_depth = 128;
						
						cargs.coalescing = TRUE;
					}
					break;

				case 'f':
					if (numFileNames < 3)
						fileNames[numFileNames++] = optarg;
					else
					{
						printf("ERROR:  At most, three filenames may be supplied\n");
						closePorts(numPorts);
						return 0;
					}
					break;

				case 'd':
					cargs.dump = TRUE;
					break;

				case 'i':
					cargs.info = TRUE;
					break;

				case 'l':
					if (*optarg == 'a' || *optarg == '1' || *optarg == '2' || *optarg == '4')
						cargs.linkspeed = *optarg;
					else
					{
						printf("ERROR:  Invalid link speed %s.\n", optarg);
						printf("Valid link speeds are: 'a' for Auto, '1', '2', or '4' for 1Gb, 2Gb, or 4Gb\n");
						closePorts(numPorts);
						return 0;
					}
					break;

				case 'p':
					t = 0;
					while (sscanf(optarg, "%d%n", &i, &n) >= 1)
					{
						if (i <= 0)
						{
							cargs.numPortNums = 0;
							break;
						}
						if (t == 1)
						{
							if (i < cargs.portNums[cargs.numPortNums - 1])
							{
								cargs.numPortNums = 0;
								break;
							}
							while (i > cargs.portNums[cargs.numPortNums - 1])
							{
								if (i <= numPorts && cargs.numPortNums < NUM_PORTS)
								{
									cargs.portNums[cargs.numPortNums] = cargs.portNums[cargs.numPortNums - 1] + 1;
									cargs.numPortNums++;
								}
								else
									break;
							}
							t = 0;
						}
						else
						{
							if (i <= numPorts && cargs.numPortNums < NUM_PORTS)
							{
								cargs.portNums[cargs.numPortNums] = i;
								cargs.numPortNums++;
							}
						}
						if (optarg[n] == '\0')
						{
							optarg += n;
							break;
						}
						else if (optarg[n] == ',')
						{
							optarg += n + 1;
						}
						else if (optarg[n] == '-' && t == 0)
						{
							optarg += n + 1;
							t = 1;
						}
						else
						{
							cargs.numPortNums = 0;
							break;
						}
					}
					if (optarg[0] != '\0')
						cargs.numPortNums = 0;
					if (cargs.numPortNums == 0)
					{
						printf("ERROR:  No such port.  Valid ports are:\n");
						printf("\n     Port Name         Chip Vendor/Type    MPT Rev  Firmware Rev\n");
						for (i = 0; i < numPorts; i++)
						{
							port = mptPorts[i];
							printf("%2d.  %-16s  LSI Logic %-8s    %03x      %08x\n",
								   i+1, port->portName, port->chipName, port->mptVersion, port->fwVersion);
						}
						closePorts(numPorts);
						return 0;
					}
					else
					{
					}
					break;

				case 'r':
					cargs.reset = TRUE;
					break;

				case 's':
					cargs.scan = TRUE;
					break;

				case 't':
					if (*optarg == 'a' || *optarg == '1' || *optarg == '2')
						cargs.topology = *optarg;
					else
					{
						printf("ERROR:  Invalid topology %s.\n", optarg);
						printf("Valid topologys are: 'a' for Auto, '1' for NL_Port, or '2' for N_Port\n");
						closePorts(numPorts);
						return 0;
					}
					break;

				case 'u':
					tagType = MPI_SCSIIO_CONTROL_UNTAGGED;
					break;

				case 'y':
					if (numFileNames > 0)
					{
						yesFlag = TRUE;
						break;
					}
					cargs.linkReset = TRUE;
					cargs.linkResetFlag = TRUE;
					break;

				case 'z':
					cargs.linkReset = TRUE;
					cargs.linkResetFlag = FALSE;
					break;

				default:
					closePorts(numPorts);
					return 0;
				}
			}

			if ((cargs.info || cargs.scan || cargs.dump) &&
				(cargs.linkspeed || cargs.topology || cargs.coalescing || cargs.reset || cargs.linkReset))
			{
				printf("ERROR:  -s -i and -d (display options) can't be mixed\n");
				printf("with -l -t and -c (set options) or -r and -z (reset options)\n");
				closePorts(numPorts);
				return 0;
			}

			if (cargs.info || cargs.scan || cargs.dump)
			{
				if (cargs.numPortNums)
				{
					for (i = 0; i < cargs.numPortNums; i++)
					{
						port = mptPorts[cargs.portNums[i] - 1];

						printf("\n============================================================================\n");
						printf("\n%-16s  LSI Logic %-8s    MPT Rev %03x    Firmware Rev %08x\n",
							   port->portName, port->chipName, port->mptVersion, port->fwVersion);

						t = !cargs.info;

						if (cargs.info)
						{
							printf("\n");
							doPortStateSummary(port);
						}

						if (cargs.scan)
						{
							printf("\n");
							doScanForDevices(port, t);
						}

						if (cargs.dump)
						{
							if (t)
								printf("\n");
							doDumpPortState(port, t);
						}
					}
				}
				else
				{
					for (i = 0; i < numPorts; i++)
					{
						port = mptPorts[i];

						printf("\n============================================================================\n");
						printf("\n%-16s  LSI Logic %-8s    MPT Rev %03x    Firmware Rev %08x\n",
							   port->portName, port->chipName, port->mptVersion, port->fwVersion);

						t = !cargs.info;

						if (cargs.info)
						{
							printf("\n");
							doPortStateSummary(port);
						}

						if (cargs.scan)
						{
							printf("\n");
							doScanForDevices(port, t);
						}

						if (cargs.dump)
						{
							if (t)
								printf("\n");
							doDumpPortState(port, t);
						}
					}
				}
			}

			if (cargs.linkspeed)
			{
				if (cargs.numPortNums)
				{
					for (i = 0; i < cargs.numPortNums; i++)
					{
						port = mptPorts[cargs.portNums[i] - 1];
						if (port->portType == MPI_PORTFACTS_PORTTYPE_FC)
						{
							switch (cargs.linkspeed)
							{
							default:
							case 'a':  t = MPI_FCPORTPAGE1_LCONFIG_SPEED_AUTO; type = "Auto";  break;
							case '1':  t = MPI_FCPORTPAGE1_LCONFIG_SPEED_1GIG; type = "1 Gb";  break;
							case '2':  t = MPI_FCPORTPAGE1_LCONFIG_SPEED_2GIG; type = "2 Gb";  break;
							case '4':  t = MPI_FCPORTPAGE1_LCONFIG_SPEED_4GIG; type = "4 Gb";  break;
							}
							printf("Setting link speed to %s on port %d\n", type, cargs.portNums[i]);
							doFcLinkSpeedValue(port, t);
						}
						else
						{
							printf("ERROR:  Link speed supported only on FC ports.\n");
						}
					}
				}
				else
				{
					printf("ERROR:  Must specify a port to set link speed.  Valid ports are:\n");
					printf("\n     Port Name         Chip Vendor/Type    MPT Rev  Firmware Rev\n");
					for (i = 0; i < numPorts; i++)
					{
						port = mptPorts[i];
						printf("%2d.  %-16s  LSI Logic %-8s    %03x      %08x\n",
							   i+1, port->portName, port->chipName, port->mptVersion, port->fwVersion);
					}
					closePorts(numPorts);
					return 0;
				}
			}

			if (cargs.topology)
			{
				if (cargs.numPortNums)
				{
					for (i = 0; i < cargs.numPortNums; i++)
					{
						port = mptPorts[cargs.portNums[i] - 1];
						if (port->portType == MPI_PORTFACTS_PORTTYPE_FC)
						{
							switch (cargs.topology)
							{
							default:
							case 'a':  t = MPI_FCPORTPAGE1_TOPOLOGY_AUTO;   type = "Auto";     break;
							case '1':  t = MPI_FCPORTPAGE1_TOPOLOGY_NLPORT; type = "NL_Port";  break;
							case '2':  t = MPI_FCPORTPAGE1_TOPOLOGY_NPORT;  type = "N_Port";   break;
							}
							printf("Setting topology to %s on port %d\n", type, cargs.portNums[i]);
							doFcTopologyValue(port, t);
						}
						else
						{
							printf("ERROR:  Topology change supported only on FC ports.\n");
						}
					}
				}
				else
				{
					printf("ERROR:  Must specify a port to set topology.  Valid ports are:\n");
					printf("\n     Port Name         Chip Vendor/Type    MPT Rev  Firmware Rev\n");
					for (i = 0; i < numPorts; i++)
					{
						port = mptPorts[i];
						printf("%2d.  %-16s  LSI Logic %-8s    %03x      %08x\n",
							   i+1, port->portName, port->chipName, port->mptVersion, port->fwVersion);
					}
					closePorts(numPorts);
					return 0;
				}
			}

			if (cargs.coalescing)
			{
				if (cargs.numPortNums)
				{
					for (i = 0; i < cargs.numPortNums; i++)
					{
						if (cargs.ic_timeout != 0 && cargs.ic_depth != 0)
						{
							printf("Setting interrupt coalescing to timeout of %d microseconds\n",
								   cargs.ic_timeout);
							printf("with depth of %d on port %d\n", cargs.ic_depth, cargs.portNums[i]);
						}
						else
							printf("Disabling interrupt coalescing on port %d\n", cargs.portNums[i]);

						port = mptPorts[cargs.portNums[i] - 1];
						doInterruptCoalescingValues(port, cargs.ic_timeout, cargs.ic_depth);
					}
				}
				else
				{
					printf("ERROR:  Must specify a port to set interrupt coalescing.  Valid ports are:\n");
					printf("\n     Port Name         Chip Vendor/Type    MPT Rev  Firmware Rev\n");
					for (i = 0; i < numPorts; i++)
					{
						port = mptPorts[i];
						printf("%2d.  %-16s  LSI Logic %-8s    %03x      %08x\n",
							   i+1, port->portName, port->chipName, port->mptVersion, port->fwVersion);
					}
					closePorts(numPorts);
					return 0;
				}
			}

			if (cargs.reset)
			{
				if (cargs.numPortNums)
				{
					for (i = 0; i < cargs.numPortNums; i++)
					{
						port = mptPorts[cargs.portNums[i] - 1];
						doResetPort(port);
					}
				}
				else
				{
					printf("ERROR:  Must specify a port to reset.  Valid ports are:\n");
					printf("\n     Port Name         Chip Vendor/Type    MPT Rev  Firmware Rev\n");
					for (i = 0; i < numPorts; i++)
					{
						port = mptPorts[i];
						printf("%2d.  %-16s  LSI Logic %-8s    %03x      %08x\n",
							   i+1, port->portName, port->chipName, port->mptVersion, port->fwVersion);
					}
					closePorts(numPorts);
					return 0;
				}
			}

			if (cargs.linkReset)
			{
				if (cargs.numPortNums)
				{
					for (i = 0; i < cargs.numPortNums; i++)
					{
						port = mptPorts[cargs.portNums[i] - 1];
						if (port->portType == MPI_PORTFACTS_PORTTYPE_FC)
						{
							doResetFcLink(port, cargs.linkResetFlag);
						}
						else
						{
							printf("ERROR:  Link reset supported only on FC ports.\n");
						}
					}
				}
				else
				{
					printf("ERROR:  Must specify a port to reset.  Valid ports are:\n");
					printf("\n     Port Name         Chip Vendor/Type    MPT Rev  Firmware Rev\n");
					for (i = 0; i < numPorts; i++)
					{
						port = mptPorts[i];
						printf("%2d.  %-16s  LSI Logic %-8s    %03x      %08x\n",
							   i+1, port->portName, port->chipName, port->mptVersion, port->fwVersion);
					}
					closePorts(numPorts);
					return 0;
				}
			}

			if (cargs.info || cargs.scan || cargs.dump ||
				cargs.linkspeed || cargs.topology || cargs.coalescing || cargs.reset || cargs.linkReset)
			{
				closePorts(numPorts);
				return 0;
			}

			if (cargs.numPortNums)
			{
				if (optind < argc)
				{
					while (optind < argc)
					{
						if (sscanf(argv[optind], "%d", &t) == 1)
						for (i = 0; i < cargs.numPortNums; i++)
						{
							printf("\n     Port Name         Chip Vendor/Type    MPT Rev  Firmware Rev\n");

							port = mptPorts[cargs.portNums[i] - 1];

							printf("%2d.  %-16s  LSI Logic %-8s    %03x      %08x\n",
								   cargs.portNums[i], port->portName, port->chipName, port->mptVersion, port->fwVersion);

							doPortOption(port, t);
						}
						else
							printf("\n%s: invalid argument -- %s\n", argv[0], argv[optind]);
						optind++;
					}
				}
				else
				{
					for (i = 0; i < cargs.numPortNums; i++)
					{
						printf("\n     Port Name         Chip Vendor/Type    MPT Rev  Firmware Rev\n");

						port = mptPorts[cargs.portNums[i] - 1];

						printf("%2d.  %-16s  LSI Logic %-8s    %03x      %08x\n",
							   cargs.portNums[i], port->portName, port->chipName, port->mptVersion, port->fwVersion);

						doPort(port);
					}
				}

				closePorts(numPorts);
				return 0;
			}

			if (optind < argc)
			{
				while (optind < argc)
				{
					if (sscanf(argv[optind], "%d", &t) == 1)
					for (i = 0; i < numPorts; i++)
					{
						printf("\n     Port Name         Chip Vendor/Type    MPT Rev  Firmware Rev\n");

						port = mptPorts[i];

						printf("%2d.  %-16s  LSI Logic %-8s    %03x      %08x\n",
							   i+1, port->portName, port->chipName, port->mptVersion, port->fwVersion);

						doPortOption(port, t);
					}
					else
						printf("\n%s: invalid argument -- %s\n", argv[0], argv[optind]);
					optind++;
				}

				closePorts(numPorts);
				return 0;
			}
		}

		while (TRUE)
		{
			printf("\n     Port Name         Chip Vendor/Type    MPT Rev  Firmware Rev\n");
			for (i = 0; i < numPorts; i++)
			{
				port = mptPorts[i];

				printf("%2d.  %-16s  LSI Logic %-8s    %03x      %08x\n",
					   i+1, port->portName, port->chipName, port->mptVersion, port->fwVersion);
			}

			printf("\nSelect a device:  [1-%d or 0 to quit] ", numPorts);
			portNumber = getNumberAnswer(0, numPorts, -1);

			if (portNumber < 0)
				continue;

			if (portNumber == 0)
				break;

			doPort(mptPorts[portNumber-1]);
		}
	}

	closePorts(numPorts);

	return 0;
}


int
findPorts(void)
{
#if WIN32
	int				 status;
	HKEY			 hKeyScsi;
	HKEY			 hKeyPort;
	const LPSTR		 NamesKey = TEXT("HARDWARE\\DEVICEMAP\\Scsi");
	DWORD			 portIndex;
	DWORD			 portNameSize;
	DWORD			 valueType;
	DWORD			 driverNameSize;
	char			 portName[16];
	char			 driverName[16];
	char			 fileName[16];
	FILETIME		 lastWriteTime;
	int				 portNumber;
	HANDLE			 fileHandle;
	MPT_PORT		*port;
	int				 numPorts;

	status = RegOpenKeyEx(HKEY_LOCAL_MACHINE, NamesKey, 0L, KEY_READ, &hKeyScsi);

	// if can't open this registry key, there are no scsi drivers installed
	if (status != ERROR_SUCCESS)
	{
		printf("Couldn't get Scsi key from registry\n");
		return 0;
	}

	// enumerate all SCSI ports under the Scsi key
	portIndex = 0;
	numPorts = 0;
	while (TRUE)
	{
		portNameSize = sizeof portName;
		status = RegEnumKeyEx(hKeyScsi, portIndex++, portName,
							  &portNameSize, NULL, NULL, NULL, &lastWriteTime);

		if (status != ERROR_SUCCESS)
			break;

		// open this port key to check the driver name
		status = RegOpenKeyEx(hKeyScsi, portName, 0L, KEY_READ, &hKeyPort);

		if (status == ERROR_SUCCESS)
		{
			driverNameSize = sizeof driverName;
			status = RegQueryValueEx(hKeyPort, "Driver", NULL, &valueType,
									 driverName, &driverNameSize);

			if (status == ERROR_SUCCESS)
			{
				if (strcasecmp("symmpi", driverName) == 0 ||
					strcasecmp("LSImpt", driverName) == 0 ||
					strcasecmp("dcssp", driverName) == 0)
				{
					portNumber = atoi(&portName[10]);
					sprintf(fileName, "\\\\.\\Scsi%d:", portNumber);
					fileHandle = CreateFile(fileName, GENERIC_READ | GENERIC_WRITE,
											FILE_SHARE_WRITE, NULL, OPEN_EXISTING, 0, NULL);
					if (fileHandle != INVALID_HANDLE_VALUE)
					{
						port = malloc(sizeof *port);

						port->portNumber	= portNumber;
						port->fileHandle	= fileHandle;
						port->ioctlValue	= IOCTL_SCSI_MINIPORT;
						sprintf(port->portName, "Scsi Port %d", portNumber);

						if (getPortInfo(port) == 1)
						{
							mptPorts[numPorts++] = port;
						}
						else
						{
							port->ioctlValue |= 0x2000;
							if (getPortInfo(port) == 1)
							{
								mptPorts[numPorts++] = port;
							}
							else
							{
								CloseHandle(fileHandle);
								free(port);
							}
						}
					}
				}
			}

			// close SCSI port key
			RegCloseKey(hKeyPort);
		}
	}

	// close Scsi key
	RegCloseKey(hKeyScsi);

	for (portNumber = 0; portNumber < 32; portNumber++)
	{
		sprintf(fileName, "\\\\.\\MPTstm%d", portNumber);
		fileHandle = CreateFile(fileName, GENERIC_READ | GENERIC_WRITE,
								FILE_SHARE_WRITE, NULL, OPEN_EXISTING, 0, NULL);
		if (fileHandle != INVALID_HANDLE_VALUE)
		{
			port = malloc(sizeof *port);

			port->portNumber	= portNumber;
			port->fileHandle	= fileHandle;
			port->ioctlValue	= IOCTL_SCSI_MINIPORT;
			sprintf(port->portName, "MPTstm%d", portNumber);

			if (getPortInfo(port) == 1)
			{
				mptPorts[numPorts++] = port;
			}
			else
			{
				CloseHandle(fileHandle);
				free(port);
			}
		}
	}

	for (portNumber = 0; portNumber < 32; portNumber++)
	{
		sprintf(fileName, "\\\\.\\MPTstm%02x", portNumber);
		fileHandle = CreateFile(fileName, GENERIC_READ | GENERIC_WRITE,
								FILE_SHARE_WRITE, NULL, OPEN_EXISTING, 0, NULL);
		if (fileHandle != INVALID_HANDLE_VALUE)
		{
			port = malloc(sizeof *port);

			port->portNumber	= portNumber;
			port->fileHandle	= fileHandle;
			port->ioctlValue	= IOCTL_SCSI_MINIPORT;
			sprintf(port->portName, "MPTstm%02x", portNumber);

			if (getPortInfo(port) == 1)
			{
				mptPorts[numPorts++] = port;
			}
			else
			{
				CloseHandle(fileHandle);
				free(port);
			}
		}
	}

	return numPorts;
#endif
#if __linux__
	int							 numPorts;
	int							 status;
	HANDLE						 fileHandle;
	FILE						*portFile;
	char						 portName[16];
	int							 portNumber;
	MPT_PORT					*port;
	struct mpt_ioctl_eventquery	*eventquery;

	eventquery = (struct mpt_ioctl_eventquery *)malloc(sizeof *eventquery);

	memset(eventquery, 0, sizeof *eventquery);

	eventquery->hdr.maxDataSize = sizeof *eventquery;

	if ((fileHandle = open("/dev/mptctl", O_RDWR)) < 0)
	{
		printf("Couldn't open /dev/mptctl!\n");
		perror("Error is");
		return 0;
	}

	numPorts = 0;
	for (portNumber = 0; portNumber < NUM_PORTS; portNumber++)
	{
		sprintf(portName, "/proc/mpt/ioc%d", portNumber);
		portFile = fopen(portName, "r");
		if (portFile == NULL)
			continue;
		fclose(portFile);

		eventquery->hdr.iocnum = portNumber;

		status = ioctl(fileHandle, MPTEVENTQUERY, eventquery);

		if (status == 0)
		{
			port = (MPT_PORT *)malloc(sizeof *port);

			port->portNumber	= portNumber;
			port->fileHandle	= fileHandle;
			strcpy(port->portName, portName);

			if (getPortInfo(port) == 1)
			{
				mptPorts[numPorts++] = port;
			}
			else
			{
				free(port);
			}
		}
	}

	free(eventquery);

	if (numPorts == 0)
		close(fileHandle);

	return numPorts;
#endif
#if __sparc__
	int			 numPorts;
	FILE		*pathFile;
	char		 pathEntry[512];
	char		*o;
	char		*p;
	char		*q;
	char		 path[512];
	HANDLE		 fileHandle;
	int			 portNumber;
	MPT_PORT	*port;
	int			 i;
	int			 j;

	pathFile = fopen("/etc/path_to_inst", "r");
	if (pathFile == NULL)
	{
		printf("Couldn't open /etc/path_to_inst!\n");
		perror("Error is");
		return 0;
	}

	numPorts = 0;
	while (fgets(pathEntry, sizeof pathEntry, pathFile) != NULL)
	{
		if ((o = strstr(pathEntry, "\"itmpt\"")) == NULL &&
			(o = strstr(pathEntry, "\"itmptfc\"")) == NULL)
			continue;

		p = strchr(pathEntry, (int)'"');
		if (p == NULL)
			continue;
		p += 1;
		if (strncmp(p, "/node@", 6) == 0) {
			p += 6;
			p = strchr(p, '/');
			if (p == NULL)
				continue;
		}
		q = strchr(p, '"');
		if (q == NULL)
			continue;
		*q = '\0';
		q += 1;

		portNumber = atoi(q);
		sprintf(path, "/devices%s:devctl", p);
		if ((fileHandle = open(path, O_RDWR)) < 0)
			continue;

		o += 1;
		q = strchr(o, '"');
		*q = '\0';

		port = (MPT_PORT *)malloc(sizeof *port);

		port->portNumber	= portNumber;
		port->fileHandle	= fileHandle;
		sprintf(port->portName, "%s%d", o, portNumber);

		if (getPortInfo(port) == 1)
		{
			mptPorts[numPorts++] = port;
		}
		else
		{
			close(fileHandle);
			free(port);
		}
	}

	for (i = 0; i < numPorts - 1; i++)
		for (j = i + 1; j < numPorts; j++)
			if (strcmp(mptPorts[i]->portName, mptPorts[j]->portName) > 0)
			{
				port = mptPorts[i];
				mptPorts[i] = mptPorts[j];
				mptPorts[j] = port;
			}

	fclose(pathFile);

	return numPorts;
#endif
#if __irix__
	int			 numPorts;
	HANDLE		 fileHandle;
	char		 portName[24];
	int			 portNumber;
	MPT_PORT	*port;

	numPorts = 0;
	for (portNumber = 0; portNumber < NUM_PORTS; portNumber++)
	{
		sprintf(portName, "/hw/scsi_ctlr/%d/bus", portNumber);
		if ((fileHandle = open(portName, O_RDWR)) < 0)
			continue;

		port = (MPT_PORT *)malloc(sizeof *port);

		port->portNumber	= portNumber;
		port->fileHandle	= fileHandle;
		sprintf(port->portName, "scsi_ctlr%d", portNumber);

		if (getPortInfo(port) == 1)
		{
			mptPorts[numPorts++] = port;
		}
		else
		{
			free(port);
		}
	}

	return numPorts;
#endif
#if __alpha__
	int			 numPorts;
	HANDLE		 fileHandle;
	int			 portNumber;
	MPT_PORT	*port;

	if ((fileHandle = open("/dev/itmpt", O_RDWR)) < 0)
	{
		printf("Couldn't open /dev/itmpt!\n");
		perror("Error is");
		return 0;
	}

	numPorts = 0;
	for (portNumber = 0; portNumber < 8; portNumber++)
	{
		port = (MPT_PORT *)malloc(sizeof *port);

		port->portNumber	= portNumber;
		port->fileHandle	= fileHandle;
		sprintf(port->portName, "itmpt%d", portNumber);

		if (getPortInfo(port) == 1)
		{
			mptPorts[numPorts++] = port;
		}
		else
		{
			free(port);
		}
	}

	if (numPorts == 0)
		close(fileHandle);

	return numPorts;
#endif
}


int
closePorts(int numPorts)
{
	MPT_PORT	*port;
	int			 i;

	for (i = 0; i < numPorts; i++)
	{
		port = mptPorts[i];

#if WIN32
		CloseHandle(port->fileHandle);
#endif
#if __linux__ || __alpha__
		if (i == 0)
			close(port->fileHandle);
#endif
#if __sparc__ || __irix__
		close(port->fileHandle);
#endif
		free(port);
	}

	return 1;
}


int
getFileName(char *buf, int len, FILE *file, char *fileString, int fileType)
{
	int		 n;

	if (numFileNames > fileType)
	{
		strncpy(buf, fileNames[fileType], len);
	}
	else
	{
		printf("Enter %s filename: ", fileString);

		if (fgets(buf, len, file) == NULL)
			exit(0);
	}

	buf[len - 1] = '\0';
	n = strlen(buf);

	while (n != 0 && (buf[n-1] == ' ' || buf[n-1] == '\n'))
		buf[--n] = '\0';

	return n;
}


int
getString(char *buf, int len, FILE *file)
{
	int		 n;

	if (fgets(buf, len, file) == NULL)
		exit(0);

	buf[len - 1] = '\0';
	n = strlen(buf);

	while (n != 0 && (buf[n-1] == ' ' || buf[n-1] == '\n'))
		buf[--n] = '\0';

	return n;
}


int
getYesNoAnswer(int defvalue)
{
	char	 buf[16];
	int		 n;

	n = getString(buf, sizeof buf, stdin);

	if (n == 0)
		return defvalue;

	if (strncasecmp(buf, "no", n) == 0)
		return 0;

	if (strncasecmp(buf, "yes", n) == 0)
		return 1;

	return defvalue;
}


int
getNumberAnswer(int low, int high, int defvalue)
{
	char	 buf[16];
	int		 i;
	int		 n;
	int		 answer;

	while (TRUE)
	{
		n = getString(buf, sizeof buf, stdin);

		if (n == 0)
			return defvalue;

		for (i = 0; i < n; i++)
			if (!isdigit((int)buf[i]))
				break;

		if (i == n)
		{
			if (sscanf(buf, "%d", &answer) == 1)
				if (answer >= low && answer <= high)
					return answer;
		}

		printf("Invalid response, try again: ");
	}
}


int
getHexNumberAnswer(U32 *value)
{
	char	 buf[16];
	int		 i;
	int		 n;
	U32		 answer;

	while (TRUE)
	{
		n = getString(buf, sizeof buf, stdin);

		if (n == 0)
			return 0;

		if (n > 0)
		{
			for (i = 0; i < n; i++)
				if (!isxdigit((int)buf[i]))
					break;

			if (i == n)
			{
				if (sscanf(buf, "%x", &answer) == 1)
				{
					*value = answer;
					return 1;
				}
			}
		}

		printf("Invalid answer, try again: ");
	}
}


int
getHexDoubleNumberAnswer(U32 *value1, U32 *value2)
{
	char	 buf[24];
	int		 i;
	int		 n;
	U32		 answer1;
	U32		 answer2;

	while (TRUE)
	{
		n = getString(buf, sizeof buf, stdin);

		if (n == 0)
			return 0;

		if (n == 16)
		{
			for (i = 0; i < n; i++)
				if (!isxdigit((int)buf[i]))
					break;

			if (i == n)
			{
				if (sscanf(buf, "%8x%8x", &answer1, &answer2) == 2)
				{
					*value1 = answer1;
					*value2 = answer2;
					return 1;
				}
			}
		}

		printf("Invalid answer, try again: ");
	}
}


int
readFile(char *name, unsigned char **outBuf, int *outLen)
{
	int				 file;
	struct stat		 stat;
	unsigned char	*buf = NULL;
	int				 len;
	int				 t;

	file = open(name, O_RDONLY | O_BINARY);
	if (file < 0)
	{
		printf("Open failure for file %s\n", name);
		perror("Error is");
		return 0;
	}

	t = fstat(file, &stat);
	if (t < 0)
	{
		printf("Couldn't get size of file %s\n", name);
		perror("Error is");
		close(file);
		return 0;
	}

	len = stat.st_size;

	buf = (unsigned char *)malloc(len);

	t = read(file, buf, len);
	if (t != len)
	{
		printf("Read failed for file %s\n", name);
		perror("Error is");
		free(buf);
		close(file);
		return 0;
	}

	close(file);

	*outBuf = buf;
	*outLen = len;

	return 1;
}


int
printWhatString(char *type, unsigned char *buf, int len)
{
	int		 i;
	int		 j;
	char	 c = 0;

	for (i = 0; i < len; i++)
		if (buf[i] == '@' && buf[i+1] == '(' && buf[i+2] == '#' && buf[i+3] == ')')
			break;

	if (i >= len)
	{
		if (buf[1] == 0xaa && buf[0] == 0x55 &&
			buf[5] == 0x0e && buf[4] == 0xf1)  /* "efi", how cute */
		{
			PCIR			*pcir;
			int				 n;
			unsigned short	 rev;

			n = (buf[0x19]<<8) + buf[0x18];

			if (n + (int)sizeof *pcir < len)
			{
				pcir = (PCIR *)(buf + n);

				if (pcir->signature[0] == 'P' &&
					pcir->signature[1] == 'C' &&
					pcir->signature[2] == 'I' &&
					pcir->signature[3] == 'R')
				{
					if (pcir->type == 3)
					{
						rev = get16(pcir->imageRevision);
						printf("%s image's version is %d.%02d.%02d.%02d\n", type,
							   (rev >> 13) & 0x7, (rev >> 8) & 0x1f, (rev >> 4) & 0xf, (rev >> 0) & 0xf);
					}
				}
			}
		}
		return 0;
	}

	for (j = i + 4; j < len; j++)
		if ((c = buf[j]) == '\0' || c == '"' || c == '>' || c == '\n')
			break;

	buf[j] = '\0';

	printf("%s image's version is %s\n", type, buf + i + 4);

	buf[j] = c;

	return len;
}


int
getCompatible(int deviceId, int type)
{
	switch (deviceId)
	{
	case MPI_MANUFACTPAGE_DEVICEID_FC909:      return 1;  break;
	case MPI_MANUFACTPAGE_DEVICEID_FC919:      return 2;  break;
	case MPI_MANUFACTPAGE_DEVICEID_FC929:      return 2;  break;
	case MPI_MANUFACTPAGE_DEVICEID_FC919X:     return 2;  break;
	case MPI_MANUFACTPAGE_DEVICEID_FC929X:     return 2;  break;
	case MPI_MANUFACTPAGE_DEVICEID_FC939X:     return 2;  break;
	case MPI_MANUFACTPAGE_DEVICEID_FC949X:     return 2;  break;
	case MPI_MANUFACTPAGE_DEVID_53C1030:       return 3;  break;
	case MPI_MANUFACTPAGE_DEVID_1030_53C1035:  return 3;  break;
	case MPI_MANUFACTPAGE_DEVID_SAS1064:       return 4;  break;
	case MPI_MANUFACTPAGE_DEVID_SAS1064E:      return 4;  break;
	case MPI_MANUFACTPAGE_DEVID_SAS1066:       return 4;  break;
	case MPI_MANUFACTPAGE_DEVID_SAS1066E:      return 4;  break;
	case MPI_MANUFACTPAGE_DEVID_SAS1068:       return 4;  break;
	case MPI_MANUFACTPAGE_DEVID_SAS1068E:      return 4;  break;
	}

	return deviceId + 100;
}


int
checkCompatible(int deviceId1, int deviceId2, int type)
{
	int		 id1;
	int		 id2;

	id1 = getCompatible(deviceId1, type);
	id2 = getCompatible(deviceId2, type);

	if (type == 3)
	{
		/* everything is compatible (one image serves all) */
		return 1;
	}

	return id1 == id2;
}


int
doPort(MPT_PORT *port)
{
	int		 option;

	printf("\n");
	option = -1;
	while (TRUE)
	{
		if (option < 0)
		{
			printf(" 1.  Identify firmware, BIOS, and/or FCode\n");
			printf(" 2.  Download firmware (update the FLASH)\n");
			printf(" 3.  Upload firmware\n");
			printf(" 4.  Download BIOS and/or FCode (update the FLASH)\n");
			printf(" 5.  Upload BIOS and/or FCode\n");
			printf(" 6.  Download SEEPROM\n");
			printf(" 7.  Upload SEEPROM\n");
			printf(" 8.  Scan for devices\n");
			printf(" 9.  Read/change configuration pages\n");
			printf("10.  Change IOC settings (interrupt coalescing)\n");
			if (port->portType == MPI_PORTFACTS_PORTTYPE_SCSI)
			{
				printf("11.  Change SCSI Initiator settings\n");
				printf("12.  Change SCSI Target settings\n");
			}
			if (port->portType == MPI_PORTFACTS_PORTTYPE_FC)
			{
				printf("13.  Change FC Port settings\n");
				printf("14.  Change (enable/disable) multi-pathing\n");
				printf("15.  Change persistent mappings\n");
				printf("16.  Display logged-in devices\n");
				printf("17.  Show port aliases\n");
			}
			if (port->portType == MPI_PORTFACTS_PORTTYPE_SAS)
			{
				printf("13.  Change SAS IO Unit settings\n");
				printf("15.  Change persistent mappings\n");
				printf("16.  Display attached devices\n");
			}
			printf("19.  Test configuration page actions\n");
			printf("20.  Diagnostics\n");
			printf("21.  RAID actions\n");
			printf("22.  Reset bus\n");
			printf("23.  Reset target\n");
			if (port->portType == MPI_PORTFACTS_PORTTYPE_FC)
			{
				printf("80.  Set port offline\n");
				printf("81.  Set port online\n");
//				printf("82.  Set port to NL_Port\n");
//				printf("83.  Set port to N_Port\n");
				printf("94.  Send GID_FT\n");
				printf("95.  Send GA_NXT\n");
				printf("96.  Send ELS\n");
				printf("97.  Reset FC link, Maintain Logins\n");
				printf("98.  Reset FC link\n");
			}
			if (port->portType == MPI_PORTFACTS_PORTTYPE_SAS)
			{
				printf("97.  Reset SAS phy\n");
				printf("98.  Reset SAS link\n");
			}
			printf("99.  Reset port\n");
			printf("\n");
		}

		printf("Select an option:  [1-99 or 0 to quit] ");
		option = getNumberAnswer(0, 100, -1);
		if (option < 0)
		{
			printf("\n");
			continue;
		}

		if (option == 0)
			break;

		doPortOption(port, option);

		printf("\n");
	}

	return 1;
}


int
doPortOption(MPT_PORT *port, int option)
{
	printf("\n");
	switch (option)
	{
	case 1:
		doIdentify(port);
		break;
	case 2:
		doFirmwareDownload(port);
		break;
	case 3:
		doFirmwareUpload(port);
		break;
	case 4:
		doBiosFcodeDownload(port);
		break;
	case 5:
		doBiosFcodeUpload(port, NULL, NULL, 0);
		break;
	case 6:
		doSeepromDownload(port);
		break;
	case 7:
		doSeepromUpload(port);
		break;
	case 8:
		doScanForDevices(port, 1);
		break;
	case 9:
		doConfigPage(port);
		break;
	case 10:
		doIocSettings(port);
		break;
	case 11:
		if (port->portType == MPI_PORTFACTS_PORTTYPE_SCSI)
		{
			doScsiInitiatorSettings(port);
			break;
		}
		printf("Invalid selection!\n");
		break;
	case 12:
		if (port->portType == MPI_PORTFACTS_PORTTYPE_SCSI)
		{
			doScsiTargetSettings(port);
			break;
		}
		printf("Invalid selection!\n");
		break;
	case 13:
		if (port->portType == MPI_PORTFACTS_PORTTYPE_FC)
		{
			doFcPortSettings(port);
			break;
		}
		if (port->portType == MPI_PORTFACTS_PORTTYPE_SAS)
		{
			doSasIoUnitSettings(port);
			break;
		}
		printf("Invalid selection!\n");
		break;
	case 14:
		if (port->portType == MPI_PORTFACTS_PORTTYPE_FC)
		{
			doMultiPathing(port);
			break;
		}
		printf("Invalid selection!\n");
		break;
	case 15:
		if (port->portType == MPI_PORTFACTS_PORTTYPE_FC ||
			port->portType == MPI_PORTFACTS_PORTTYPE_SAS)
		{
			option = -1;
			while (TRUE)
			{
				if (option < 0)
				{
					printf(" 1.  Show persistent mappings\n");
					printf(" 2.  Automatically add persistent mappings for ALL targets\n");
					printf(" 3.  Automatically add persistent mappings for SOME targets\n");
					printf(" 4.  Delete persistent mappings for ALL targets\n");
					printf(" 5.  Delete persistent mappings for SOME targets\n");
					printf(" 6.  Manually add persistent mappings for targets\n");
					if (port->portType == MPI_PORTFACTS_PORTTYPE_FC)
					{
						printf(" 7.  Automatically add persistent mappings for ALL LoopIDs\n");
					}
					if (port->portType == MPI_PORTFACTS_PORTTYPE_SAS)
					{
						printf("10.  Clear all persistent mappings\n");
						printf("11.  Clear all non-present persistent mappings\n");
						printf("12.  Change (enable/disable) persistence\n");
					}
					printf("\n");
				}

				printf("Select an option:  [1-99 or 0 to quit] ");
				option = getNumberAnswer(0, 99, -1);
				if (option < 0)
				{
					printf("\n");
					continue;
				}

				if (option == 0)
					break;

				printf("\n");
				if (port->portType == MPI_PORTFACTS_PORTTYPE_FC)
				{
					if (option <= 7)
					{
						doFcPersistentMappings(port, option);
					}
					else
					{
						printf("Invalid selection!\n");
					}
				}
				if (port->portType == MPI_PORTFACTS_PORTTYPE_SAS)
				{
					if (option <= 6 || (option >= 10 && option <= 12))
					{
						doSasPersistentMappings(port, option);
					}
					else
					{
						printf("Invalid selection!\n");
					}
				}
				printf("\n");
			}
			option = -1;
			break;
		}
		printf("Invalid selection!\n");
		break;
	case 16:
		if (port->portType == MPI_PORTFACTS_PORTTYPE_FC)
		{
			doDisplayLoggedInDevices(port);
			break;
		}
		if (port->portType == MPI_PORTFACTS_PORTTYPE_SAS)
		{
			doDisplayAttachedDevices(port);
			break;
		}
		printf("Invalid selection!\n");
		break;
	case 17:
		if (port->portType == MPI_PORTFACTS_PORTTYPE_FC)
		{
			doShowPortAliases(port);
			break;
		}
		printf("Invalid selection!\n");
		break;
	case 19:
		doTestConfigPageActions(port);
		break;
	case 20:
		option = -1;
		while (TRUE)
		{
			if (option < 0)
			{
				printf(" 1.  Inquiry Test\n");
				printf(" 2.  WriteBuffer/ReadBuffer/Compare Test\n");
				printf(" 3.  Read Test\n");
				printf(" 4.  Write/Read/Compare Test\n");
				printf(" 5.  Write Test\n");
				printf(" 6.  Read/Compare Test\n");
				if (port->portType == MPI_PORTFACTS_PORTTYPE_FC)
				{
					printf("10.  Echo ELS Test\n");
					printf("11.  Read Link Error Status ELS Test\n");
					printf("12.  Display port counters\n");
					printf("13.  Clear port counters\n");
				}
				printf("\n");
			}

			printf("Select an option:  [1-99 or 0 to quit] ");
			option = getNumberAnswer(0, 99, -1);
			if (option < 0)
			{
				printf("\n");
				continue;
			}

			if (option == 0)
				break;

			printf("\n");
			doDiagnostics(port, option);
			printf("\n");
		}
		option = -1;
		break;
	case 21:
		option = -1;
		while (TRUE)
		{
			if (option < 0)
			{
				printf(" 1.  Show volumes\n");
				printf(" 2.  Show physical disks\n");
				printf(" 3.  Get volume state\n");
				printf("10.  Disable volume\n");
				printf("11.  Enable volume\n");
				printf("12.  Inactivate volume\n");
				printf("13.  Activate volume\n");
				printf("20.  Offline physical disk\n");
				printf("21.  Online physical disk\n");
				printf("22.  Fail physical disk\n");
				printf("23.  Replace physical disk\n");
				printf("24.  Quiesce physical disk I/Os\n");
				printf("25.  Unquiesce physical disk I/Os\n");
				printf("30.  Create volume\n");
				printf("31.  Delete volume\n");
				printf("32.  Change volume settings\n");
//				printf("40.  Create physical disk\n");
				printf("41.  Delete physical disk\n");
				printf("42.  Change physical disk settings\n");
				printf("\n");
			}

			printf("Select an option:  [1-99 or 0 to quit] ");
			option = getNumberAnswer(0, 99, -1);
			if (option < 0)
			{
				printf("\n");
				continue;
			}

			if (option == 0)
				break;

			printf("\n");
			doRaidActions(port, option);
			printf("\n");
		}
		option = -1;
		break;
	case 22:
		doResetBus(port);
		break;
	case 23:
		doResetTarget(port);
		break;
	case 80:
		if (port->portType == MPI_PORTFACTS_PORTTYPE_FC)
		{
			doFcPortOffline(port);
			break;
		}
		printf("Invalid selection!\n");
		break;
	case 81:
		if (port->portType == MPI_PORTFACTS_PORTTYPE_FC)
		{
			doFcPortOnline(port);
			break;
		}
		printf("Invalid selection!\n");
		break;
	case 82:
		if (port->portType == MPI_PORTFACTS_PORTTYPE_FC)
		{
			doFcTopologyNLPort(port);
			break;
		}
		printf("Invalid selection!\n");
		break;
	case 83:
		if (port->portType == MPI_PORTFACTS_PORTTYPE_FC)
		{
			doFcTopologyNPort(port);
			break;
		}
		printf("Invalid selection!\n");
		break;
	case 94:
		if (port->portType == MPI_PORTFACTS_PORTTYPE_FC)
		{
			doGID_FT(port);
			break;
		}
		printf("Invalid selection!\n");
		break;
	case 95:
		if (port->portType == MPI_PORTFACTS_PORTTYPE_FC)
		{
			doGA_NXT(port);
			break;
		}
		printf("Invalid selection!\n");
		break;
	case 96:
		if (port->portType == MPI_PORTFACTS_PORTTYPE_FC)
		{
			doExLinkServiceSend(port);
			break;
		}
		printf("Invalid selection!\n");
		break;
	case 97:
		if (port->portType == MPI_PORTFACTS_PORTTYPE_FC)
		{
			doResetFcLink(port, 1);
			break;
		}
		if (port->portType == MPI_PORTFACTS_PORTTYPE_SAS)
		{
			doResetSasPhy(port);
			break;
		}
		printf("Invalid selection!\n");
		break;
	case 98:
		if (port->portType == MPI_PORTFACTS_PORTTYPE_FC)
		{
			doResetFcLink(port, 0);
			break;
		}
		if (port->portType == MPI_PORTFACTS_PORTTYPE_SAS)
		{
			doResetSasLink(port);
			break;
		}
		printf("Invalid selection!\n");
		break;
	case 99:
		doResetPort(port);
		break;
	case 100:
		doDumpPortState(port, 1);
		break;
	default:
		printf("Invalid selection!\n");
		break;
	}

	return 1;
}


int
doIdentify(MPT_PORT *port)
{
	unsigned char	*imageBuf;
	int				 imageLen;
	int				 actualImageLen;
	int				 offset;
	unsigned char	*buf;
	int				 len;
	int				 i;
	int				 n;
	PCIR			*pcir;
	char			*type;

	imageLen = 0x10000;
	imageBuf = (unsigned char *)malloc(imageLen);

	if (doFwUpload(port, MPI_FW_UPLOAD_ITYPE_FW_FLASH, imageBuf, imageLen, 0, &actualImageLen) == 1)
	{
		printWhatString("Firmware", imageBuf, imageLen);
	}

	offset = 0;
	while (offset < 0x40000)
	{
		if (doFwUpload(port, MPI_FW_UPLOAD_ITYPE_BIOS_FLASH, imageBuf, imageLen, offset, &actualImageLen) != 1)
			break;

		for (i = 0; i < imageLen; i += 512)
		{
			buf = imageBuf + i;

			n = (buf[0x01]<<8) + buf[0x00];
			if (n != 0xaa55)
				continue;

			n = (buf[0x19]<<8) + buf[0x18];

			if (i + n + (int)sizeof *pcir > imageLen)
			{
				// not all of the image is in the buffer, so quit now and read more of the image
				break;
			}

			pcir = (PCIR *)(buf + n);

			if (pcir->signature[0] != 'P' ||
				pcir->signature[1] != 'C' ||
				pcir->signature[2] != 'I' ||
				pcir->signature[3] != 'R')
			{
				printf("Image's PCIR signature is invalid!\n");
				type = "Unknown";
				len = 512;
			}
			else
			{
				len = get16(pcir->imageLength) * 512;
				if (i + len > imageLen)
				{
					// not all of the image is in the buffer, so quit now and read more of the image
					if (len > imageLen)
					{
						free(imageBuf);
						imageLen = len;
						imageBuf = (unsigned char *)malloc(len);
					}
					break;
				}

				switch (pcir->type)
				{
				case 0:   type = "x86 BIOS";  break;
				case 1:   type = "FCode";     break;
				case 3:   type = "EFI BIOS";  break;
				default:  type = "Unknown";   break;
				}

				if (pcir->indicator & 0x80)
				{
					// last image, so make sure we quit after this
					i += 0x40000;
				}
			}

			printWhatString(type, buf, len);
			i += len - 512;
		}

		offset += i;
	}

	free(imageBuf);

	return 1;
}


int
doFirmwareDownload(MPT_PORT *port)
{
	char				 name[256];
	unsigned char		*imageBuf = NULL;
	int					 imageLen;
	int					 n;
	int					 warn = 0;
	MpiFwHeader_t		*fwHeader;
	MpiExtImageHeader_t	*fwExtImageHeader;
	U32					 fwNextImage;
	int					 i;
	U32					 checksum;

	n = getFileName(name, sizeof name, stdin, "firmware", 0);
	if (n > 0)
	{
		if (readFile(name, &imageBuf, &imageLen) != 1)
			return 0;
	}
	else
	{
		printf("Image won't be downloaded\n");
		return 1;
	}

	fwHeader = (MpiFwHeader_t *)imageBuf;

	printf("Image's version is %s\n", fwHeader->VersionName);

	if (get16(fwHeader->ProductId) != port->productId)
	{
		printf("Image's Product ID appears to be wrong!\n");
		printf("  Current firmware Product ID is %04x\n", port->productId);
		printf("  Image's firmware Product ID is %04x\n", get16(fwHeader->ProductId));
		warn = 1;
	}

	checksum = 0;
	for (i = 0; i < imageLen / 4; i++)
		checksum += get32(((U32 *)imageBuf)[i]);

	if (checksum != 0)
	{
		printf("Image's checksum is invalid!\n");
		printf("  The image appears to be corrupted, proceed with caution\n");
		warn = 1;
	}

	fwNextImage = get32(fwHeader->NextImageHeaderOffset);
	while (fwNextImage != 0)
	{
		fwExtImageHeader = (MpiExtImageHeader_t *)(imageBuf + fwNextImage);

		if (fwExtImageHeader->ImageType == MPI_EXT_IMAGE_TYPE_NVDATA)
		{
			if (get32(fwExtImageHeader->ImageSize) <= sizeof *fwExtImageHeader)
			{
				printf("Image's attached NVDATA is invalid!\n");
				printf("  Using this image is likely to cause pain and suffering\n");
				warn = 1;
			}
		}

		fwNextImage = get32(fwExtImageHeader->NextImageHeaderOffset);
	}

	if (port->portType == MPI_PORTFACTS_PORTTYPE_SAS)
	{
		i = get32(fwHeader->SeqCodeVersion);

		if (port->deviceId == MPI_MANUFACTPAGE_DEVID_SAS1064)
		{
			if (port->revisionId == 0 && i != 0x1064a1)
			{
				printf("Image is for %X, port is 1064A1, image is not compatible!\n", i);
				warn = 1;
			}
			if (port->revisionId == 1 && i != 0x1064a2)
			{
				printf("Image is for %X, port is 1064A2, image is not compatible!\n", i);
				warn = 1;
			}
		}
	}

	if (warn || yesFlag == FALSE)
	{
		if (warn)
			printf("\nAre you sure you want to continue?  [Yes or No, default is No] ");
		else
			printf("\nDo you want to continue?  [Yes or No, default is No] ");

		if (getYesNoAnswer(0) != 1)
		{
			free(imageBuf);
			return 0;
		}
	}

	doFwDownload(port, MPI_FW_DOWNLOAD_ITYPE_FW, imageBuf, imageLen, 0);

	free(imageBuf);

	return 1;
}


int
doFirmwareUpload(MPT_PORT *port)
{
	char			 name[256];
	int				 file;
	unsigned char	*imageBuf = NULL;
	int				 imageLen;
	int				 actualImageLen;
	int				 offset;
	int				 n;
	int				 t;

	n = getFileName(name, sizeof name, stdin, "firmware", 0);
	if (n > 0)
	{
		file = open(name, O_CREAT | O_TRUNC | O_RDWR | O_BINARY, 0644);
		if (file < 0)
		{
			printf("Open failure for file %s\n", name);
			perror("Error is");
			return 0;
		}
	}
	else
	{
		printf("Image won't be uploaded\n");
		return 1;
	}

	imageLen = 0x10000;
	imageBuf = (unsigned char *)malloc(imageLen);

	offset = 0;
	while (TRUE)
	{
		if (doFwUpload(port, MPI_FW_UPLOAD_ITYPE_FW_FLASH, imageBuf, imageLen, offset, &actualImageLen) != 1)
			break;

		if (offset + imageLen > actualImageLen)
			imageLen = actualImageLen - offset;

		t = write(file, imageBuf, imageLen);
		if (t != imageLen)
		{
			printf("Write failed for file %s, t = %x\n", name, t);
			perror("Error is");
			break;
		}

		offset += imageLen;
		if (offset >= actualImageLen)
			break;
	}

	printf("\nWrote %d bytes to file %s\n", offset, name);

	close(file);

	free(imageBuf);

	return 1;
}


int
doBiosFcodeDownload(MPT_PORT *port)
{
	char			 name[256];
	unsigned char	*imageBuf = NULL;
	unsigned char	*image1Buf = NULL;
	unsigned char	*image2Buf = NULL;
	unsigned char	*image3Buf = NULL;
	unsigned char	*image4Buf = NULL;
	unsigned char	*image5Buf = NULL;
	int				 imageLen;
	int				 image1Len;
	int				 image2Len;
	int				 image3Len;
	int				 image4Len = 0;
	int				 image5Len = 0;
	int				 imageOff;
	int				 n;
	int				 warn = 0;
	int				 last;

	n = getFileName(name, sizeof name, stdin, "x86 BIOS", 0);
	if (n > 0)
	{
		if (readFile(name, &image1Buf, &image1Len) != 1)
			return 0;

		printWhatString("x86 BIOS", image1Buf, image1Len);

		warn |= verifyBiosFcodeImage(port, image1Buf, image1Len, 0);
	}
	else
	{
		if (doBiosFcodeUpload(port, &image1Buf, &image1Len, 0) == 1 && image1Buf != NULL)
		{
			printWhatString("Current x86 BIOS", image1Buf, image1Len);

			if (yesFlag == FALSE)
			{
				printf("\nDo you want to preserve the current x86 BIOS?  [Yes or No, default is Yes] ");

				if (getYesNoAnswer(1) != 1)
				{
					free(image1Buf);
					image1Buf = NULL;
				}
			}
		}
		else
		{
			printf("No x86 BIOS image exists in FLASH, and image won't be downloaded\n");
		}
	}

	printf("\n");

	n = getFileName(name, sizeof name, stdin, "FCode", 1);
	if (n > 0)
	{
		if (readFile(name, &image2Buf, &image2Len) != 1)
			return 0;

		printWhatString("FCode", image2Buf, image2Len);

		warn |= verifyBiosFcodeImage(port, image2Buf, image2Len, 1);
	}
	else
	{
		if (doBiosFcodeUpload(port, &image2Buf, &image2Len, 1) == 1 && image2Buf != NULL)
		{
			printWhatString("Current FCode", image2Buf, image2Len);

			if (yesFlag == FALSE)
			{
				printf("\nDo you want to preserve the current FCode?  [Yes or No, default is Yes] ");

				if (getYesNoAnswer(1) != 1)
				{
					free(image2Buf);
					image2Buf = NULL;
				}
			}
		}
		else
		{
			printf("No FCode image exists in FLASH, and image won't be downloaded\n");
		}
	}

	printf("\n");

	n = getFileName(name, sizeof name, stdin, "EFI BIOS", 2);
	if (n > 0)
	{
		if (readFile(name, &image3Buf, &image3Len) != 1)
			return 0;

		printWhatString("EFI BIOS", image3Buf, image3Len);

		warn |= verifyBiosFcodeImage(port, image3Buf, image3Len, 3);
	}
	else
	{
		if (doBiosFcodeUpload(port, &image3Buf, &image3Len, 3) == 1 && image3Buf != NULL)
		{
			printWhatString("Current EFI BIOS", image3Buf, image3Len);

			if (yesFlag == FALSE)
			{
				printf("\nDo you want to preserve the current EFI BIOS?  [Yes or No, default is Yes] ");

				if (getYesNoAnswer(1) != 1)
				{
					free(image3Buf);
					image3Buf = NULL;
				}
			}
		}
		else
		{
			printf("No EFI BIOS image exists in FLASH, and image won't be downloaded\n");
		}
	}

	if (image1Buf != NULL)
	{
		last = image2Buf == NULL && image3Buf == NULL;
		splitBiosImage(port, &image1Buf, &image1Len, &image4Buf, &image4Len);
		fixupBiosFcodeImage(port, image1Buf, image1Len, last);
	}
	else
		image1Len = 0;

	if (image2Buf != NULL)
	{
		last = image3Buf == NULL;
		fixupBiosFcodeImage(port, image2Buf, image2Len, last);
	}
	else
		image2Len = 0;

	if (image3Buf != NULL)
	{
		last = 1;
		splitBiosImage(port, &image3Buf, &image3Len, &image5Buf, &image5Len);
		fixupBiosFcodeImage(port, image3Buf, image3Len, last);
	}
	else
		image3Len = 0;

	imageLen = image1Len + image2Len + image3Len + image4Len + image5Len;
	if (imageLen == 0)
		return 1;

	imageBuf = (unsigned char *)malloc(imageLen);
	imageOff = 0;
	if (image1Buf != NULL)
	{
		memcpy(imageBuf + imageOff, image1Buf, image1Len);
		imageOff += image1Len;
		free(image1Buf);
	}
	if (image2Buf != NULL)
	{
		memcpy(imageBuf + imageOff, image2Buf, image2Len);
		imageOff += image2Len;
		free(image2Buf);
	}
	if (image3Buf != NULL)
	{
		memcpy(imageBuf + imageOff, image3Buf, image3Len);
		imageOff += image3Len;
		free(image3Buf);
	}
	if (image4Buf != NULL)
	{
		memcpy(imageBuf + imageOff, image4Buf, image4Len);
		imageOff += image4Len;
		free(image4Buf);
	}
	if (image5Buf != NULL)
	{
		memcpy(imageBuf + imageOff, image5Buf, image5Len);
		imageOff += image5Len;
		free(image5Buf);
	}

	if (warn || yesFlag == FALSE)
	{
		if (warn)
			printf("\nAre you sure you want to continue?  [Yes or No, default is No] ");
		else
			printf("\nDo you want to continue?  [Yes or No, default is No] ");

		if (getYesNoAnswer(0) != 1)
		{
			free(imageBuf);
			return 0;
		}
	}

	doFwDownload(port, MPI_FW_DOWNLOAD_ITYPE_BIOS, imageBuf, imageLen, 0);

	free(imageBuf);

	return 1;
}


int
doBiosFcodeUpload(MPT_PORT *port, unsigned char **outBuf, int *outLen, int type)
{
	char			 name[4][256];
	int				 file[4];
	int				 who;
	unsigned char	*imageBuf = NULL;
	int				 imageLen;
	int				 actualImageLen;
	int				 offset;
	unsigned char	*buf;
	int				 len;
	int				 i;
	int				 n;
	PCIR			*pcir;
	int				 t;

	if (outBuf == NULL || outLen == NULL)
		printf("Searching for BIOS and/or FCode images\n");

	for (i = 0; i < 4; i++)
		file[i] = -1;

	imageLen = 0x10000;
	imageBuf = (unsigned char *)malloc(imageLen);

	offset = 0;
	while (offset < 0x40000)
	{
		if (doFwUpload(port, MPI_FW_UPLOAD_ITYPE_BIOS_FLASH, imageBuf, imageLen, offset, &actualImageLen) != 1)
			break;

		buf = imageBuf;

		n = (buf[0x01]<<8) + buf[0x00];
		if (n != 0xaa55 && n != 0xbb55)
		{
			break;
		}

		n = (buf[0x19]<<8) + buf[0x18];

		if (n + (int)sizeof *pcir >= imageLen)
		{
			// not all of the image is in the buffer, so quit now and read more of the image
			free(imageBuf);
			imageLen = n + (int)sizeof *pcir;
			imageBuf = (unsigned char *)malloc(imageLen);
			continue;
		}

		pcir = (PCIR *)(buf + n);

		if (pcir->signature[0] != 'P' ||
			pcir->signature[1] != 'C' ||
			pcir->signature[2] != 'I' ||
			pcir->signature[3] != 'R')
		{
			break;
		}

		len = get16(pcir->imageLength) * 512;
		if (len > imageLen)
		{
			// not all of the image is in the buffer, so quit now and read more of the image
			free(imageBuf);
			imageLen = len;
			imageBuf = (unsigned char *)malloc(imageLen);
			continue;
		}

		if (outBuf == NULL || outLen == NULL)
		{
			printf("\n");

			switch (pcir->type)
			{
			case 0:
				n = getFileName(name[0], sizeof name[0], stdin, "x86 BIOS", 0);
				who = 0;
				break;
			case 1:
				n = getFileName(name[1], sizeof name[1], stdin, "FCode", 1);
				who = 1;
				break;
			case 3:
				n = getFileName(name[2], sizeof name[2], stdin, "EFI BIOS", 2);
				who = 2;
				break;
			case 255:
				if (buf[4] == 'L' && buf[5] == 'S' && buf[6] == 'I' && buf[7] == 'L' && buf[0x34] == 0x02)
				{
					if ((buf[0x3b] & 0xf0) == 0x10)
					{
						printf("Found x86 BIOS extended image\n");
						n = -1;
						who = 0;
						break;
					}
					if ((buf[0x3b] & 0xf0) == 0x30)
					{
						printf("Found EFI BIOS extended image\n");
						n = -1;
						who = 2;
						break;
					}
				}
			default:
				printf("Image type is unknown, enter filename: ");
				n = getString(name[3], sizeof name[3], stdin);
				who = 3;
				break;
			}

			if (n > 0)
			{
				fixupBiosFcodeImage(port, buf, len, 1);

				file[who] = open(name[who], O_CREAT | O_TRUNC | O_RDWR | O_BINARY, 0644);
				if (file[who] < 0)
				{
					printf("Open failure for file %s\n", name[who]);
					perror("Error is");
				}

				t = write(file[who], buf, len);
				if (t != len)
				{
					printf("Write failed for file %s, t = %x\n", name[who], t);
					perror("Error is");
				}
				else
					printf("\nWrote %d bytes to file %s\n", len, name[who]);
			}

			else if (n < 0)
			{
				if (file[who] >= 0)
				{
					fixupBiosFcodeImage(port, buf, len, 0);

					t = write(file[who], buf, len);
					if (t != len)
					{
						printf("Write failed for file %s, t = %x\n", name[who], t);
						perror("Error is");
					}
					else
						printf("\nWrote %d bytes to file %s\n", len, name[who]);
				}
			}

			else
			{
				printf("Image won't be uploaded\n");
			}
		}
		else
		{
			if (pcir->type == type)
			{
				*outBuf = (unsigned char *)malloc(len);
				*outLen = len;
				memcpy(*outBuf, buf, len);
			}
			else if (pcir->type == 255)
			{
				if (buf[4] == 'L' && buf[5] == 'S' && buf[6] == 'I' && buf[7] == 'L' && buf[0x34] == 0x02)
				{
					if (type == 0 && (buf[0x3b] & 0xf0) == 0x10)
					{
						*outBuf = (unsigned char *)realloc(*outBuf, *outLen + len);
						memcpy(*outBuf + *outLen, buf, len);
						*outLen = *outLen + len;
					}
					if (type == 3 && (buf[0x3b] & 0xf0) == 0x30)
					{
						*outBuf = (unsigned char *)realloc(*outBuf, *outLen + len);
						memcpy(*outBuf + *outLen, buf, len);
						*outLen = *outLen + len;
					}
				}
			}
		}

		if (pcir->indicator & 0x80)
		{
			// last image, so make sure we quit after this
			if (port->portType != MPI_PORTFACTS_PORTTYPE_SAS)
				break;
		}

		offset += len;

		if (offset >= actualImageLen)
			break;
	}

	for (i = 0; i < 4; i++)
		if (file[i] >= 0)
			close(file[i]);

	free(imageBuf);

	return 1;
}


int
verifyBiosFcodeImage(MPT_PORT *port, unsigned char *buf, int len, int type)
{
	int		 n;
	int		 warn = 0;
	PCIR	*pcir;
	int		 i;
	U8		 checksum;

	n = (buf[0x01]<<8) + buf[0x00];
	if (n != 0xaa55)
	{
		if (n == 0xbb55)
		{
			printf("\nThis appears to be the special non-functional (blank) image!\n");
			return 0;
		}
		printf("Image's ROM signature %04x is invalid!\n", n);
		warn = 1;
	}

	/* if there's a what string, we will check the checksum even if it's not BIOS */
	for (i = 0; i < len; i++)
		if (buf[i] == '@' && buf[i+1] == '(' && buf[i+2] == '#' && buf[i+3] == ')')
			break;

	if (type == 0 || i < len)
	{
		checksum = 0;
		for (i = 0; i < len; i++)
			checksum += buf[i];

		if (checksum != 0)
		{
			printf("Image's checksum is invalid!\n");
			printf("  The image appears to be corrupted, proceed with caution\n");
			warn = 1;
		}
	}

	n = (buf[0x19]<<8) + buf[0x18];

	if (n + (int)sizeof *pcir < len)
	{
		pcir = (PCIR *)(buf + n);

		if (pcir->signature[0] != 'P' ||
			pcir->signature[1] != 'C' ||
			pcir->signature[2] != 'I' ||
			pcir->signature[3] != 'R')
		{
			printf("Image's PCIR signature is invalid!\n");
			warn = 1;
		}

		if (checkCompatible(get16(pcir->deviceId), port->deviceId, type) != 1)
		{
			printf("Image's PCI Device ID %04x is not compatible!\n", get16(pcir->deviceId));
			warn = 1;
		}

		if (pcir->type != type)
		{
			printf("Image's PCI Type %d is not correct!\n", pcir->type);
			warn = 1;
		}
	}
	else
	{
		printf("Image's PCIR offset %04x is invalid!\n", n);
		warn = 1;
	}

	return warn;
}


int
splitBiosImage(MPT_PORT *port, unsigned char **buf1, int *len1, unsigned char **buf2, int *len2)
{
	int				 n;
	PCIR			*pcir;

	n = ((*buf1)[0x19]<<8) + (*buf1)[0x18];

	if (n + (int)sizeof *pcir < *len1)
	{
		pcir = (PCIR *)(*buf1 + n);

		if (pcir->signature[0] == 'P' &&
			pcir->signature[1] == 'C' &&
			pcir->signature[2] == 'I' &&
			pcir->signature[3] == 'R')
		{
			n = get16(pcir->imageLength) * 512;
			if (n < *len1)
			{
				*buf2 = (unsigned char *)malloc(*len1 - n);
				*len2 = *len1 - n;
				memcpy(*buf2, *buf1 + n, *len1 - n);
				*buf1 = (unsigned char *)realloc(*buf1, n);
				*len1 = n;
			}
		}
	}

	return 1;
}


int
fixupBiosFcodeImage(MPT_PORT *port, unsigned char *buf, int len, int last)
{
	int				 n;
	PCIR			*pcir;
	int				 i;
	U8				 checksum;

	n = (buf[0x19]<<8) + buf[0x18];

	if (n + (int)sizeof *pcir < len)
	{
		pcir = (PCIR *)(buf + n);

		if (pcir->signature[0] == 'P' &&
			pcir->signature[1] == 'C' &&
			pcir->signature[2] == 'I' &&
			pcir->signature[3] == 'R')
		{
			if (pcir->type != 255)
				pcir->deviceId = set16(port->deviceId);

			if (last)
				pcir->indicator |= 0x80;
			else
				pcir->indicator &= ~0x80;
		}
	}

	checksum = 0;
	for (i = 0; i < get16(pcir->imageLength) * 512 - 1; i++)
		checksum += buf[i];
	buf[i] = -checksum;

	return 1;
}


int
doSeepromDownload(MPT_PORT *port)
{
	char			 name[256];
	unsigned char	*imageBuf = NULL;
	int				 imageLen;
	int				 n;

	n = getFileName(name, sizeof name, stdin, "SEEPROM", 0);
	if (n > 0)
	{
		if (readFile(name, &imageBuf, &imageLen) != 1)
			return 0;
	}
	else
	{
		printf("Image won't be downloaded\n");
		return 1;
	}

	if (doFwDownload(port, MPI_FW_DOWNLOAD_ITYPE_NVDATA, imageBuf, imageLen, 0) == 1)
	{
		if (port->portType == MPI_PORTFACTS_PORTTYPE_FC)
		{
			printf("\n");
			printf("WARNING!  The SEEPROM contains information (World Wide Names) that must\n");
			printf("          be unique for each port.  Each port on a host adapter must have\n");
			printf("          its own WWNs, not copied from another host adapter.  Please use\n");
			printf("          Manufacturing Page 3 to verify that the WWNs assigned to this\n");
			printf("          port are unique, and modify them if necessary.\n");
		}
	}

	free(imageBuf);

	return 1;
}


int
doSeepromUpload(MPT_PORT *port)
{
	char			 name[256];
	int				 file;
	unsigned char	*imageBuf = NULL;
	int				 imageLen;
	int				 actualImageLen;
	int				 offset;
	int				 n;
	int				 t;

	n = getFileName(name, sizeof name, stdin, "SEEPROM", 0);
	if (n > 0)
	{
		file = open(name, O_CREAT | O_TRUNC | O_RDWR | O_BINARY, 0644);
		if (file < 0)
		{
			printf("Open failure for file %s\n", name);
			perror("Error is");
			return 0;
		}
	}
	else
	{
		printf("Image won't be uploaded\n");
		return 1;
	}

	imageLen = 0x4000;
	imageBuf = (unsigned char *)malloc(imageLen);

	offset = 0;
	while (TRUE)
	{
		if (doFwUpload(port, MPI_FW_UPLOAD_ITYPE_NVDATA, imageBuf, imageLen, offset, &actualImageLen) != 1)
			break;

		if (offset + imageLen > actualImageLen)
			imageLen = actualImageLen - offset;

		t = write(file, imageBuf, imageLen);
		if (t != imageLen)
		{
			printf("Write failed for file %s, t = %x\n", name, t);
			perror("Error is");
			break;
		}

		offset += imageLen;
		if (offset >= actualImageLen)
			break;
	}

	printf("\nWrote %d bytes to file %s\n", offset, name);

	close(file);

	free(imageBuf);

	return 1;
}


char *deviceType[32] = {
	"Disk",
	"Tape",
	"Printer",
	"Processor",
	"WriteOnce",
	"CDROM",
	"Scanner",
	"Optical",
	"Jukebox",
	"Comm",
	"0Ah",
	"0Bh",
	"RAIDArray",
	"EnclServ",
	"0Eh",
	"0Fh",
	"10h",
	"11h",
	"12h",
	"13h",
	"14h",
	"15h",
	"16h",
	"17h",
	"18h",
	"19h",
	"1Ah",
	"1Bh",
	"1Ch",
	"1Dh",
	"1Eh",
	""
};


int
doScanForDevices(MPT_PORT *port, int flag)
{
	int				 bus;
	int				 target;
	int				 lun;
	unsigned char	 inq[36];
	char			 buf[32];
	int				 i;
	int				 version;
	int				 max_lun;

	if (flag)
		showPortInfoHeader(port);

	getDeviceInfoHeader(port, buf, sizeof buf);

	printf(" B___T___L  Type       Vendor   Product          Rev   %s\n", buf);

	for (bus = 0; bus < port->maxBuses; bus++)
	{
		for (target = 0; target < port->maxTargets; target++)
		{
			max_lun = 1;

			for (lun = 0; lun < max_lun; lun++)
			{
				if (doInquiry(port, bus, target, lun, inq, sizeof inq) != 1)
				{
					if (lun == 0)
						break;
					else
						continue;
				}

				if (lun == 0)
				{
					getDeviceInfo(port, bus, target, buf, sizeof buf);

					version = inq[2] & 0x07;
					if (version > 1)
						max_lun = 8;
					if (version > 2)
						max_lun = 64;
				}
				else
				{
					if ((inq[0] & 0x1f) == 0x1f)
						continue;

					if ((inq[0] & 0xe0) == 0x60)
						continue;

					if ((inq[0] & 0x1f) == 0x0d)
						continue;
				}

				for (i = 8; i < 36; i++)
					if (!isprint(inq[i]))
						inq[i] = ' ';

				printf("%2d %3d %3d  %-9s  %8.8s %16.16s %4.4s  %s\n",
					   bus, target, lun, deviceType[inq[0] & 0x1f],
					   inq+8, inq+16, inq+32, lun == 0 ? buf : "");
			}
		}
	}

	showPortInfo(port);

	return 1;
}


int
doConfigPage(MPT_PORT *port)
{
	ConfigReply_t		 rep;
	U32					 buf[256];
	int					 type;
	int					 number;
	U32					 address;
	U32					 offset;
	U32					 value;
	int					 i;
	int					 n;
	int					 t;
	int					 attributes;
	int					 action;
	int					 changed;

	while (TRUE)
	{
		printf("Enter page type:  [0-255 or RETURN to quit] ");
		type = getNumberAnswer(0, 255, -1);
		if (type == -1)
			break;

		printf("Enter page number:  [0-255 or RETURN to quit] ");
		number = getNumberAnswer(0, 255, -1);
		if (number == -1)
			break;

		if ((type == MPI_CONFIG_PAGETYPE_SCSI_DEVICE && (number == 0 || number == 1)) ||
			(type == MPI_CONFIG_PAGETYPE_FC_PORT && (number == 3 || number == 5)) ||
			(type == MPI_CONFIG_PAGETYPE_FC_DEVICE && number == 0) ||
			(type == MPI_CONFIG_PAGETYPE_RAID_VOLUME && number == 0) ||
			(type == MPI_CONFIG_PAGETYPE_RAID_PHYSDISK && number == 0) ||
			(type == MPI_CONFIG_EXTPAGETYPE_SAS_EXPANDER && (number == 0 || number == 1)) ||
			(type == MPI_CONFIG_EXTPAGETYPE_SAS_DEVICE && (number == 0 || number == 1 || number == 2)) ||
			(type == MPI_CONFIG_EXTPAGETYPE_SAS_PHY && (number == 0 || number == 1)))
		{
			printf("Enter page address:  [00000000-FFFFFFFF or RETURN to quit] ");
			if (getHexNumberAnswer(&address) == 0)
				break;
		}
		else
			address = 0;

		if (getConfigPageHeader(port, type, number, address, &rep) != 1)
		{
			printf("\nFailed to read page header -- that page might not exist\n\n");
			continue;
		}

		attributes = rep.Header.PageType & MPI_CONFIG_PAGEATTR_MASK;

		if (attributes == MPI_CONFIG_PAGEATTR_PERSISTENT ||
			attributes == MPI_CONFIG_PAGEATTR_RO_PERSISTENT)
		{
			printf("Read NVRAM or current values?  [0=NVRAM, 1=Current, default is 0] ");
			t = getNumberAnswer(0, 1, 0);
			if (t == 0)
				action = MPI_CONFIG_ACTION_PAGE_READ_NVRAM;
			else
				action = MPI_CONFIG_ACTION_PAGE_READ_CURRENT;
		}
		else
			action = MPI_CONFIG_ACTION_PAGE_READ_CURRENT;

		if (getConfigPageAction(port, action, type, number, address, buf, sizeof buf) == 1)
		{
			if (type > MPI_CONFIG_PAGETYPE_EXTENDED)
				n = get16(((ConfigExtendedPageHeader_t *)buf)->ExtPageLength);
			else
				n = ((ConfigPageHeader_t *)buf)->PageLength;

			printf("\n");
			for (i = 0; i < n; i++)
				printf("%04x : %08x\n", i*4, get32(buf[i]));

			changed = FALSE;

			if (attributes == MPI_CONFIG_PAGEATTR_CHANGEABLE ||
				attributes == MPI_CONFIG_PAGEATTR_PERSISTENT ||
				attributes == MPI_CONFIG_PAGEATTR_RO_PERSISTENT)
			{
				printf("\nDo you want to make changes?  [Yes or No, default is No] ");
				if (getYesNoAnswer(0) == 1)
				{
					if (type > MPI_CONFIG_PAGETYPE_EXTENDED)
						i = sizeof (ConfigExtendedPageHeader_t);
					else
						i = sizeof (ConfigPageHeader_t);
					while (TRUE)
					{
						printf("Enter offset of value to change:  [%04x-%04x or RETURN to quit] ", i, (n - 1) * 4);
						while (TRUE)
						{
							t = getHexNumberAnswer(&offset);
							if (t == 0)
								break;

							if ((offset % 4) == 0)
							{
								offset /= 4;
								if (offset >= 1 && offset < (U32)n)
									break;
							}

							printf("Invalid answer, try again: ");
						}
						if (t == 0)
							break;

						printf("Enter value:  [00000000-FFFFFFFF or RETURN to quit] ");
						if (getHexNumberAnswer(&value) == 0)
							break;

						buf[offset] = set32(value);

						changed = TRUE;
					}
				}
			}

			if (changed == TRUE)
			{
				printf("\nDo you want to write your changes?  [Yes or No, default is No] ");
				if (getYesNoAnswer(0) == 1)
				{
					if (action == MPI_CONFIG_ACTION_PAGE_READ_NVRAM)
						action = MPI_CONFIG_ACTION_PAGE_WRITE_NVRAM;
					else
						action = MPI_CONFIG_ACTION_PAGE_WRITE_CURRENT;

					if (type == MPI_CONFIG_PAGETYPE_MANUFACTURING && number == 2)
					{
						U8	 checksum = 0xa5;
						U8	*p = (U8 *)buf;

						p += 8;
						t = n * 4 - 8;
						switch (port->deviceId)
						{
						case MPI_MANUFACTPAGE_DEVICEID_FC919:   t -= 4;  break;
						case MPI_MANUFACTPAGE_DEVICEID_FC929:   t -= 4;  break;
						case MPI_MANUFACTPAGE_DEVICEID_FC919X:  t -= 3;  break;
						case MPI_MANUFACTPAGE_DEVICEID_FC929X:  t -= 3;  break;
						case MPI_MANUFACTPAGE_DEVICEID_FC939X:  t -= 3;  break;
						case MPI_MANUFACTPAGE_DEVICEID_FC949X:  t -= 3;  break;
						default:                                t = 0;   break;
						}
						if (t != 0)
						{
							for (i = 0; i < t; i++)
							{
								checksum += *p++;
							}
							*p = -checksum;
						}
					}

					if (type == MPI_CONFIG_PAGETYPE_MANUFACTURING)
						doIocInit(port, MPI_WHOINIT_MANUFACTURER);

					if (setConfigPageAction(port, action, type, number, address, buf, sizeof buf) != 1)
						printf("Failed to write changes!\n");
					else
						printf("Changes have been written\n");

					if (type == MPI_CONFIG_PAGETYPE_MANUFACTURING)
						doIocInit(port, port->whoInit);
				}
			}
		}

		printf("\n");
	}

	return 1;
}


int
doInterruptCoalescingValues(MPT_PORT *port, int timeout, int depth)
{
	IOCPage1_t	 IOCPage1;

	if (getConfigPage(port, MPI_CONFIG_PAGETYPE_IOC, 1, 0, &IOCPage1, sizeof IOCPage1) != 1)
		return 0;

	if (timeout != 0 && depth != 0)
	{
		IOCPage1.Flags |= set32(MPI_IOCPAGE1_REPLY_COALESCING);
		IOCPage1.CoalescingTimeout = set32(timeout);
		IOCPage1.CoalescingDepth = depth;
	}
	else
	{
		IOCPage1.Flags &= ~set32(MPI_IOCPAGE1_REPLY_COALESCING);
		IOCPage1.CoalescingTimeout = 0;
		IOCPage1.CoalescingDepth = 0;
	}

	if (setConfigPage(port, MPI_CONFIG_PAGETYPE_IOC, 1, 0, &IOCPage1, sizeof IOCPage1) != 1)
	{
		printf("Failed to save changes to NVRAM!\n");
		return 0;
	}

	return 1;
}


int
doIocSettings(MPT_PORT *port)
{
	IOCPage1_t	 IOCPage1;
	int			 flags;
	int			 timeout;
	int			 depth;
	int			 on;

	if (getConfigPage(port, MPI_CONFIG_PAGETYPE_IOC, 1, 0, &IOCPage1, sizeof IOCPage1) != 1)
		return 0;

	flags = get32(IOCPage1.Flags) & MPI_IOCPAGE1_REPLY_COALESCING;
	timeout = get32(IOCPage1.CoalescingTimeout);
	depth = IOCPage1.CoalescingDepth;

	if (timeout < 0)
		timeout = 0;
	if (timeout > 1000)
		timeout = 1000;
	if (depth < 0)
		depth = 0;
	if (depth > 128)
		depth = 128;

	on = flags != 0 && timeout != 0 && depth != 0;
	if (on)
		printf("Interrupt Coalescing is enabled, timeout is %d microseconds, depth is %d\n",
			   timeout, depth);
	else
		printf("Interrupt Coalescing is disabled\n");

	printf("Enable interrupt coalescing:  [0=No, 1=Yes, default is %d] ", on);
	on = getNumberAnswer(0, 1, on);

	if (on)
	{
		printf("Enter timeout:  [1-1000, 0=disable, default is %d] ", timeout);
		timeout = getNumberAnswer(0, 1000, timeout);

		printf("Enter depth:  [1-128, 0=disable, default is %d] ", depth);
		depth = getNumberAnswer(0, 128, depth);
	}
	else
	{
		timeout = 0;
		depth = 0;
	}

	if (on && timeout != 0 && depth != 0)
	{
		IOCPage1.Flags |= set32(MPI_IOCPAGE1_REPLY_COALESCING);
		IOCPage1.CoalescingTimeout = set32(timeout);
		IOCPage1.CoalescingDepth = depth;
	}
	else
	{
		IOCPage1.Flags &= ~set32(MPI_IOCPAGE1_REPLY_COALESCING);
		IOCPage1.CoalescingTimeout = 0;
		IOCPage1.CoalescingDepth = 0;
	}

	if (setConfigPage(port, MPI_CONFIG_PAGETYPE_IOC, 1, 0, &IOCPage1, sizeof IOCPage1) != 1)
	{
		printf("Failed to save changes to NVRAM!\n");
		return 0;
	}

	return 1;
}


int
doScsiInitiatorSettings(MPT_PORT *port)
{
	SCSIPortPage1_t		 SCSIPortPage1;
	SCSIPortPage2_t		 SCSIPortPage2;
	int					 flags;
	int					 settings;
	int					 id;
	int					 t;

	if (getConfigPage(port, MPI_CONFIG_PAGETYPE_SCSI_PORT, 2, 0, &SCSIPortPage2, sizeof SCSIPortPage2) != 1)
		return 0;

	flags = get32(SCSIPortPage2.PortFlags);
	settings = get32(SCSIPortPage2.PortSettings);

	id = settings & MPI_SCSIPORTPAGE2_PORT_HOST_ID_MASK;
	printf("Host SCSI ID:  [0-15, default is %d] ", id);
	id = getNumberAnswer(0, 15, id);
	settings &= ~MPI_SCSIPORTPAGE2_PORT_HOST_ID_MASK;
	settings |= id;

	t = (flags & MPI_SCSIPORTPAGE2_PORT_FLAGS_SCAN_HIGH_TO_LOW) != 0;
	printf("Bus scan order:  [0=LowToHigh, 1=HighToLow, default is %d] ", t);
	t = getNumberAnswer(0, 1, t);
	if (t == 0)
		flags &= ~MPI_SCSIPORTPAGE2_PORT_FLAGS_SCAN_HIGH_TO_LOW;
	else
		flags |= MPI_SCSIPORTPAGE2_PORT_FLAGS_SCAN_HIGH_TO_LOW;

	t = (flags & MPI_SCSIPORTPAGE2_PORT_FLAGS_AVOID_SCSI_RESET) != 0;
	printf("Avoid SCSI bus reset:  [0=No, 1=Yes, default is %d] ", t);
	t = getNumberAnswer(0, 1, t);
	if (t == 0)
		flags &= ~MPI_SCSIPORTPAGE2_PORT_FLAGS_AVOID_SCSI_RESET;
	else
		flags |= MPI_SCSIPORTPAGE2_PORT_FLAGS_AVOID_SCSI_RESET;

	t = (flags & MPI_SCSIPORTPAGE2_PORT_FLAGS_ALTERNATE_CHS) != 0;
	printf("CHS mapping:  [0=PlugAndPlay, 1=AlternateCHS, default is %d] ", t);
	t = getNumberAnswer(0, 1, t);
	if (t == 0)
		flags &= ~MPI_SCSIPORTPAGE2_PORT_FLAGS_ALTERNATE_CHS;
	else
		flags |= MPI_SCSIPORTPAGE2_PORT_FLAGS_ALTERNATE_CHS;

	t = (settings & MPI_SCSIPORTPAGE2_PORT_REMOVABLE_MEDIA) >> 6;
	printf("Removable media support:  [0=None, 1=BootDrive, 2=AnyWithMedia, default is %d] ", t);
	t = getNumberAnswer(0, 2, t);
	settings &= ~MPI_SCSIPORTPAGE2_PORT_REMOVABLE_MEDIA;
	settings |= t << 6;  // what, no nice symbolic name I can use here?

	t = (settings & MPI_SCSIPORTPAGE2_PORT_SPINUP_DELAY_MASK) >> 8;
	printf("Spinup delay (in seconds):  [0-15, default is %d] ", t);
	t = getNumberAnswer(0, 15, t);
	settings &= ~MPI_SCSIPORTPAGE2_PORT_SPINUP_DELAY_MASK;
	settings |= t << 8;  // what, no nice symbolic name I can use here?

	SCSIPortPage2.PortFlags = set32(flags);
	SCSIPortPage2.PortSettings = set32(settings);

	if (setConfigPage(port, MPI_CONFIG_PAGETYPE_SCSI_PORT, 2, 0, &SCSIPortPage2, sizeof SCSIPortPage2) != 1)
	{
		printf("Failed to save changes to NVRAM!\n");
		return 0;
	}

	if (getConfigPage(port, MPI_CONFIG_PAGETYPE_SCSI_PORT, 1, 0, &SCSIPortPage1, sizeof SCSIPortPage1) != 1)
		return 0;

	SCSIPortPage1.Configuration = set32((1 << (id + 16)) | id);

	if (setConfigPage(port, MPI_CONFIG_PAGETYPE_SCSI_PORT, 1, 0, &SCSIPortPage1, sizeof SCSIPortPage1) != 1)
	{
		printf("Failed to save changes to NVRAM!\n");
		return 0;
	}

	return 1;
}


char *
syncToMt(int sync)
{
	if (sync == 0)
		return "Async";
	if (sync == 8)
		return "160";
	if (sync == 9)
		return "80";
	if (sync == 10)
		return "40";
	if (sync == 12)
		return "20";
	if (sync == 25)
		return "10";
	if (sync == 50)
		return "5";
	return "";
}


char *
syncToMb(int sync, int wide)
{
	if (wide == 0)
		return syncToMt(sync);
	if (sync == 0)
		return "Async";
	if (sync == 8)
		return "320";
	if (sync == 9)
		return "160";
	if (sync == 10)
		return "80";
	if (sync == 12)
		return "40";
	if (sync == 25)
		return "20";
	if (sync == 50)
		return "10";
	return "";
}


int
doScsiTargetSettings(MPT_PORT *port)
{
	SCSIPortPage2_t		 SCSIPortPage2;
	MPI_DEVICE_INFO		 settings[16];
	int					 id;
	int					 sync;
	int					 i;
	int					 t;

	if (getConfigPage(port, MPI_CONFIG_PAGETYPE_SCSI_PORT, 2, 0, &SCSIPortPage2, sizeof SCSIPortPage2) != 1)
		return 0;

	id = get32(SCSIPortPage2.PortSettings) & MPI_SCSIPORTPAGE2_PORT_HOST_ID_MASK;

	for (i = 0; i < 16; i++)
	{
		settings[i].Timeout		= SCSIPortPage2.DeviceSettings[i].Timeout;
		settings[i].SyncFactor	= SCSIPortPage2.DeviceSettings[i].SyncFactor;
		settings[i].DeviceFlags	= get16(SCSIPortPage2.DeviceSettings[i].DeviceFlags);
	}

	while (TRUE)
	{
		printf("Target MB/sec | MT/sec Wide ScanID ScanLUNs Disconnect Timeout QueueTag Boot\n");
		for (i = 0; i < 16; i++)
			printf("  %2d   %5s  | %5s  %3s   %3s     %3s       %3s       %3d     %3s    %3s\n",
				   i,
				   syncToMb(settings[i].SyncFactor,
							settings[i].DeviceFlags & MPI_SCSIPORTPAGE2_DEVICE_WIDE_DISABLE ? 0 : 1),
				   syncToMt(settings[i].SyncFactor),
				   settings[i].DeviceFlags & MPI_SCSIPORTPAGE2_DEVICE_WIDE_DISABLE ? "No" : "Yes",
				   settings[i].DeviceFlags & MPI_SCSIPORTPAGE2_DEVICE_ID_SCAN_ENABLE ? "Yes" : "No",
				   settings[i].DeviceFlags & MPI_SCSIPORTPAGE2_DEVICE_LUN_SCAN_ENABLE ? "Yes" : "No",
				   settings[i].DeviceFlags & MPI_SCSIPORTPAGE2_DEVICE_DISCONNECT_ENABLE ? "Yes" : "No",
				   settings[i].Timeout,
				   settings[i].DeviceFlags & MPI_SCSIPORTPAGE2_DEVICE_TAG_QUEUE_ENABLE ? "Yes" : "No",
				   settings[i].DeviceFlags & MPI_SCSIPORTPAGE2_DEVICE_BOOT_CHOICE ? "Yes" : "No");
		printf("\nSelect a Target:  [0-15, %d=AllTargets, RETURN to quit] ", id);
		i = getNumberAnswer(0, 15, -1);

		if (i < 0)
			break;

		switch (settings[i].SyncFactor)
		{
		case 8:   sync = 160;  break;
		case 9:   sync = 80;   break;
		case 10:  sync = 40;   break;
		case 12:  sync = 20;   break;
		case 25:  sync = 10;   break;
		case 50:  sync = 5;    break;
		default:
		case 0:   sync = 0;    break;
		}
		printf("\nMT/sec:  [160, 80, 40, 20, 10, 5, 0=Async, default is %d] ", sync);
		while (TRUE)
		{
			t = getNumberAnswer(0, 160, sync);
			switch (t)
			{
			case 160:  t = 8;   break;
			case 80:   t = 9;   break;
			case 40:   t = 10;  break;
			case 20:   t = 12;  break;
			case 10:   t = 25;  break;
			case 5:    t = 50;  break;
			case 0:    t = 0;   break;
			default:
				printf("Invalid response, try again: ");
				t = -1;
				break;
			}
			if (t >= 0)
				break;
		}
		settings[i].SyncFactor = t;

		if (settings[i].SyncFactor > 9 || settings[i].SyncFactor == 0)
		{
			t = (settings[i].DeviceFlags & MPI_SCSIPORTPAGE2_DEVICE_WIDE_DISABLE) == 0;
			printf("Enable Wide:  [0=No, 1=Yes, default is %d] ", t);
			t = getNumberAnswer(0, 1, t);
			if (t == 1)
				settings[i].DeviceFlags &= ~MPI_SCSIPORTPAGE2_DEVICE_WIDE_DISABLE;
			else
				settings[i].DeviceFlags |= MPI_SCSIPORTPAGE2_DEVICE_WIDE_DISABLE;
		}
		else
			settings[i].DeviceFlags &= ~MPI_SCSIPORTPAGE2_DEVICE_WIDE_DISABLE;

		t = (settings[i].DeviceFlags & MPI_SCSIPORTPAGE2_DEVICE_ID_SCAN_ENABLE) != 0;
		printf("Enable ScanID:  [0=No, 1=Yes, default is %d] ", t);
		t = getNumberAnswer(0, 1, t);
		if (t == 0)
			settings[i].DeviceFlags &= ~MPI_SCSIPORTPAGE2_DEVICE_ID_SCAN_ENABLE;
		else
			settings[i].DeviceFlags |= MPI_SCSIPORTPAGE2_DEVICE_ID_SCAN_ENABLE;

		t = (settings[i].DeviceFlags & MPI_SCSIPORTPAGE2_DEVICE_LUN_SCAN_ENABLE) != 0;
		printf("Enable ScanLUNs:  [0=No, 1=Yes, default is %d] ", t);
		t = getNumberAnswer(0, 1, t);
		if (t == 0)
			settings[i].DeviceFlags &= ~MPI_SCSIPORTPAGE2_DEVICE_LUN_SCAN_ENABLE;
		else
			settings[i].DeviceFlags |= MPI_SCSIPORTPAGE2_DEVICE_LUN_SCAN_ENABLE;

		t = (settings[i].DeviceFlags & MPI_SCSIPORTPAGE2_DEVICE_DISCONNECT_ENABLE) != 0;
		printf("Enable Disconnect:  [0=No, 1=Yes, default is %d] ", t);
		t = getNumberAnswer(0, 1, t);
		if (t == 0)
			settings[i].DeviceFlags &= ~MPI_SCSIPORTPAGE2_DEVICE_DISCONNECT_ENABLE;
		else
			settings[i].DeviceFlags |= MPI_SCSIPORTPAGE2_DEVICE_DISCONNECT_ENABLE;

		t = settings[i].Timeout;
		printf("Timeout:  [0-255, default is %d] ", t);
		t = getNumberAnswer(0, 255, t);
		settings[i].Timeout = t;

		t = (settings[i].DeviceFlags & MPI_SCSIPORTPAGE2_DEVICE_TAG_QUEUE_ENABLE) != 0;
		printf("Enable QueueTag:  [0=No, 1=Yes, default is %d] ", t);
		t = getNumberAnswer(0, 1, t);
		if (t == 0)
			settings[i].DeviceFlags &= ~MPI_SCSIPORTPAGE2_DEVICE_TAG_QUEUE_ENABLE;
		else
			settings[i].DeviceFlags |= MPI_SCSIPORTPAGE2_DEVICE_TAG_QUEUE_ENABLE;

		t = (settings[i].DeviceFlags & MPI_SCSIPORTPAGE2_DEVICE_BOOT_CHOICE) != 0;
		printf("Enable Boot:  [0=No, 1=Yes, default is %d] ", t);
		t = getNumberAnswer(0, 1, t);
		if (t == 0)
			settings[i].DeviceFlags &= ~MPI_SCSIPORTPAGE2_DEVICE_BOOT_CHOICE;
		else
			settings[i].DeviceFlags |= MPI_SCSIPORTPAGE2_DEVICE_BOOT_CHOICE;

		if (i == id)
			for (i = 0; i < 16; i++)
				if (i != id)
					settings[i] = settings[id];

		printf("\n");
		}

	for (i = 0; i < 16; i++)
	{
		SCSIPortPage2.DeviceSettings[i].Timeout		= settings[i].Timeout;
		SCSIPortPage2.DeviceSettings[i].SyncFactor	= settings[i].SyncFactor;
		SCSIPortPage2.DeviceSettings[i].DeviceFlags	= get16(settings[i].DeviceFlags);
	}

	if (setConfigPage(port, MPI_CONFIG_PAGETYPE_SCSI_PORT, 2, 0, &SCSIPortPage2, sizeof SCSIPortPage2) != 1)
	{
		printf("Failed to save changes to NVRAM!\n");
		return 0;
	}

	return 1;
}


int
doFcLinkSpeedValue(MPT_PORT *port, int t)
{
	FCPortPage0_t	 FCPortPage0;
	FCPortPage1_t	 FCPortPage1;
	int				 speeds;

	if (getConfigPage(port, MPI_CONFIG_PAGETYPE_FC_PORT, 0, 0, &FCPortPage0, sizeof FCPortPage0) != 1)
		return 0;

	speeds = get32(FCPortPage0.SupportedSpeeds);

	switch (t)
	{
	case MPI_FCPORTPAGE1_LCONFIG_SPEED_1GIG:
		speeds &= MPI_FCPORTPAGE0_SUPPORT_1GBIT_SPEED;
		break;
	case MPI_FCPORTPAGE1_LCONFIG_SPEED_2GIG:
		speeds &= MPI_FCPORTPAGE0_SUPPORT_2GBIT_SPEED;
		break;
	case MPI_FCPORTPAGE1_LCONFIG_SPEED_4GIG:
		speeds &= MPI_FCPORTPAGE0_SUPPORT_4GBIT_SPEED;
		break;
	}

	if (speeds == 0)
	{
		printf("That link speed is not supported on this port!\n");
		return 0;
	}

	if (getConfigPage(port, MPI_CONFIG_PAGETYPE_FC_PORT, 1, 0, &FCPortPage1, sizeof FCPortPage1) != 1)
		return 0;

	FCPortPage1.LinkConfig &= ~MPI_FCPORTPAGE1_LCONFIG_SPEED_MASK;
	FCPortPage1.LinkConfig |= t;

	if (setConfigPage(port, MPI_CONFIG_PAGETYPE_FC_PORT, 1, 0, &FCPortPage1, sizeof FCPortPage1) != 1)
	{
		printf("Failed to save changes to NVRAM!\n");
		return 0;
	}

	return 1;
}


int
doFcTopologyValue(MPT_PORT *port, int t)
{
	FCPortPage1_t	 FCPortPage1;

	if (getConfigPage(port, MPI_CONFIG_PAGETYPE_FC_PORT, 1, 0, &FCPortPage1, sizeof FCPortPage1) != 1)
		return 0;

	FCPortPage1.TopologyConfig &= ~MPI_FCPORTPAGE1_TOPOLOGY_MASK;
	FCPortPage1.TopologyConfig |= t;

	if (setConfigPage(port, MPI_CONFIG_PAGETYPE_FC_PORT, 1, 0, &FCPortPage1, sizeof FCPortPage1) != 1)
	{
		printf("Failed to save changes to NVRAM!\n");
		return 0;
	}

	return 1;
}


int
doFcPortOffline(MPT_PORT *port)
{
	FCPortPage1_t	 FCPortPage1;

	printf("Setting port offline\n");

	if (getConfigPageAction(port, MPI_CONFIG_ACTION_PAGE_READ_CURRENT, MPI_CONFIG_PAGETYPE_FC_PORT, 1, 0, &FCPortPage1, sizeof FCPortPage1) != 1)
		return 0;

	FCPortPage1.Flags |= set32(MPI_FCPORTPAGE1_FLAGS_PORT_OFFLINE);

	if (setConfigPageAction(port, MPI_CONFIG_ACTION_PAGE_WRITE_CURRENT, MPI_CONFIG_PAGETYPE_FC_PORT, 1, 0, &FCPortPage1, sizeof FCPortPage1) != 1)
	{
		printf("Failed to write changes!\n");
		return 0;
	}

	return 1;
}


int
doFcPortOnline(MPT_PORT *port)
{
	FCPortPage1_t	 FCPortPage1;

	printf("Setting port online\n");

	if (getConfigPageAction(port, MPI_CONFIG_ACTION_PAGE_READ_CURRENT, MPI_CONFIG_PAGETYPE_FC_PORT, 1, 0, &FCPortPage1, sizeof FCPortPage1) != 1)
		return 0;

	FCPortPage1.Flags &= ~set32(MPI_FCPORTPAGE1_FLAGS_PORT_OFFLINE);

	if (setConfigPageAction(port, MPI_CONFIG_ACTION_PAGE_WRITE_CURRENT, MPI_CONFIG_PAGETYPE_FC_PORT, 1, 0, &FCPortPage1, sizeof FCPortPage1) != 1)
	{
		printf("Failed to write changes!\n");
		return 0;
	}

	return 1;
}


int
doFcTopologyNLPort(MPT_PORT *port)
{
	FCPortPage1_t	 FCPortPage1;

	printf("Setting port to NL_Port\n");

	if (getConfigPageAction(port, MPI_CONFIG_ACTION_PAGE_READ_CURRENT, MPI_CONFIG_PAGETYPE_FC_PORT, 1, 0, &FCPortPage1, sizeof FCPortPage1) != 1)
		return 0;

	FCPortPage1.TopologyConfig &= ~MPI_FCPORTPAGE1_TOPOLOGY_MASK;
	FCPortPage1.TopologyConfig |= MPI_FCPORTPAGE1_TOPOLOGY_NLPORT;

	if (setConfigPageAction(port, MPI_CONFIG_ACTION_PAGE_WRITE_CURRENT, MPI_CONFIG_PAGETYPE_FC_PORT, 1, 0, &FCPortPage1, sizeof FCPortPage1) != 1)
	{
		printf("Failed to write changes!\n");
		return 0;
	}

	return 1;
}


int
doFcTopologyNPort(MPT_PORT *port)
{
	FCPortPage1_t	 FCPortPage1;

	printf("Setting port to N_Port\n");

	if (getConfigPageAction(port, MPI_CONFIG_ACTION_PAGE_READ_CURRENT, MPI_CONFIG_PAGETYPE_FC_PORT, 1, 0, &FCPortPage1, sizeof FCPortPage1) != 1)
		return 0;

	FCPortPage1.TopologyConfig &= ~MPI_FCPORTPAGE1_TOPOLOGY_MASK;
	FCPortPage1.TopologyConfig |= MPI_FCPORTPAGE1_TOPOLOGY_NPORT;

	if (setConfigPageAction(port, MPI_CONFIG_ACTION_PAGE_WRITE_CURRENT, MPI_CONFIG_PAGETYPE_FC_PORT, 1, 0, &FCPortPage1, sizeof FCPortPage1) != 1)
	{
		printf("Failed to write changes!\n");
		return 0;
	}

	return 1;
}


int
doFcPortSettings(MPT_PORT *port)
{
	FCPortPage0_t	 FCPortPage0;
	FCPortPage1_t	 FCPortPage1;
	int				 flags;
	int				 speed;
	int				 speeds;
	int				 t;
	U32				 alpa;

	if (getConfigPage(port, MPI_CONFIG_PAGETYPE_FC_PORT, 0, 0, &FCPortPage0, sizeof FCPortPage0) != 1)
		return 0;

	speeds = get32(FCPortPage0.SupportedSpeeds);

	if (getConfigPage(port, MPI_CONFIG_PAGETYPE_FC_PORT, 1, 0, &FCPortPage1, sizeof FCPortPage1) != 1)
		return 0;

	t = FCPortPage1.TopologyConfig & MPI_FCPORTPAGE1_TOPOLOGY_MASK;
	if (t == MPI_FCPORTPAGE1_TOPOLOGY_AUTO)
		t = 0;
	printf("Link topology:  [0=Auto, 1=NL_Port, 2=N_Port, default is %d] ", t);
	t = getNumberAnswer(0, 2, t);
	if (t == 0)
		t = MPI_FCPORTPAGE1_TOPOLOGY_AUTO;
	FCPortPage1.TopologyConfig &= ~MPI_FCPORTPAGE1_TOPOLOGY_MASK;
	FCPortPage1.TopologyConfig |= t;

	t = FCPortPage1.LinkConfig & MPI_FCPORTPAGE1_LCONFIG_SPEED_MASK;
	if (t == MPI_FCPORTPAGE1_LCONFIG_SPEED_AUTO)
		speed = 0;
	else
		speed = t + 1;
	if (speeds & MPI_FCPORTPAGE0_SUPPORT_10GBIT_SPEED)
	{
		if (speed == 4)
			speed = 10;
		else if (speed == 3)
			speed = 4;
		else if (speed > 3)
			speed = 0;
		printf("Link speed:  [0=Auto, 1=1Gb, 2=2Gb, 4=4Gb, 10=10Gb, default is %d] ", speed);
		while (TRUE)
		{
			t = getNumberAnswer(0, 10, speed);
			if (t < 3 || t == 4 || t == 10)
				break;
			printf("Invalid response, try again: ");
		}
		if (t == 10)
			t = 4;
		else if (t == 4)
			t = 3;
	}
	else if (speeds & MPI_FCPORTPAGE0_SUPPORT_4GBIT_SPEED)
	{
		if (speed == 3)
			speed = 4;
		else if (speed > 3)
			speed = 0;
		printf("Link speed:  [0=Auto, 1=1Gb, 2=2Gb, 4=4Gb, default is %d] ", speed);
		while (TRUE)
		{
			t = getNumberAnswer(0, 4, speed);
			if (t < 3 || t == 4)
				break;
			printf("Invalid response, try again: ");
		}
		if (t == 4)
			t = 3;
	}
	else if (speeds & MPI_FCPORTPAGE0_SUPPORT_2GBIT_SPEED)
	{
		if (speed > 2)
			speed = 0;
		printf("Link speed:  [0=Auto, 1=1Gb, 2=2Gb, default is %d] ", speed);
		t = getNumberAnswer(0, 2, speed);
	}
	else if (speeds & MPI_FCPORTPAGE1_LCONFIG_SPEED_1GIG)
	{
		t = 1;
	}
	else
	{
		t = 0;
	}
	if (t == 0)
		t = MPI_FCPORTPAGE1_LCONFIG_SPEED_AUTO;
	else
		t--;
	FCPortPage1.LinkConfig &= ~MPI_FCPORTPAGE1_LCONFIG_SPEED_MASK;
	FCPortPage1.LinkConfig |= t;

	flags = get32(FCPortPage1.Flags);

	t = (flags & MPI_FCPORTPAGE1_FLAGS_PROT_FCP_INIT) != 0;
	printf("FCP Initiator protocol:  [0=Disabled, 1=Enabled, default is %d] ", t);
	t = getNumberAnswer(0, 1, t);
	if (t == 0)
		flags &= ~MPI_FCPORTPAGE1_FLAGS_PROT_FCP_INIT;
	else
		flags |= MPI_FCPORTPAGE1_FLAGS_PROT_FCP_INIT;

	t = (flags & MPI_FCPORTPAGE1_FLAGS_PROT_FCP_TARG) != 0;
	printf("FCP Target protocol:  [0=Disabled, 1=Enabled, default is %d] ", t);
	t = getNumberAnswer(0, 1, t);
	if (t == 0)
		flags &= ~MPI_FCPORTPAGE1_FLAGS_PROT_FCP_TARG;
	else
		flags |= MPI_FCPORTPAGE1_FLAGS_PROT_FCP_TARG;

	t = (flags & MPI_FCPORTPAGE1_FLAGS_PROT_LAN) != 0;
	printf("LAN protocol:  [0=Disabled, 1=Enabled, default is %d] ", t);
	t = getNumberAnswer(0, 1, t);
	if (t == 0)
		flags &= ~MPI_FCPORTPAGE1_FLAGS_PROT_LAN;
	else
		flags |= MPI_FCPORTPAGE1_FLAGS_PROT_LAN;

	t = (flags & MPI_FCPORTPAGE1_FLAGS_SORT_BY_DID) != 0;
	printf("Assignment of Bus/Target IDs:  [0=SortByWWN, 1=SortByDID, default is %d] ", t);
	t = getNumberAnswer(0, 1, t);
	if (t == 0)
		flags &= ~MPI_FCPORTPAGE1_FLAGS_SORT_BY_DID;
	else
		flags |= MPI_FCPORTPAGE1_FLAGS_SORT_BY_DID;

	t = (flags & MPI_FCPORTPAGE1_FLAGS_IMMEDIATE_ERROR_REPLY) != 0;
	printf("Immediate Error Reply:  [0=Disabled, 1=Enabled, default is %d] ", t);
	t = getNumberAnswer(0, 1, t);
	if (t == 0)
		flags &= ~MPI_FCPORTPAGE1_FLAGS_IMMEDIATE_ERROR_REPLY;
	else
		flags |= MPI_FCPORTPAGE1_FLAGS_IMMEDIATE_ERROR_REPLY;

	t = (flags & MPI_FCPORTPAGE1_FLAGS_MAINTAIN_LOGINS) != 0;
	printf("Maintain Logins:  [0=Disabled, 1=Enabled, default is %d] ", t);
	t = getNumberAnswer(0, 1, t);
	if (t == 0)
		flags &= ~MPI_FCPORTPAGE1_FLAGS_MAINTAIN_LOGINS;
	else
		flags |= MPI_FCPORTPAGE1_FLAGS_MAINTAIN_LOGINS;

	FCPortPage1.Flags = set32(flags);

	t = FCPortPage1.HardALPA;
	printf("Hard AL_PA:  [01 to EF, or FF for Soft AL_PA, default is %02x] ", t);
	alpa = t;
	while (TRUE)
	{
		t = getHexNumberAnswer(&alpa);
		if (t == 0)
			break;

		if (alpa == 0xff)
			break;

		if (alpa > 0x00 && alpa < 0xff && AlpaToLoopId[alpa] != 0xff)
			break;

		printf("Invalid answer, try again: ");
	}
	FCPortPage1.HardALPA = (U8)alpa;

	t = FCPortPage1.InitiatorDeviceTimeout;
	if (t == 0)
		t = 60;
	printf("Initiator Device Timeout:  [0 to 127, default is %d] ", t);
	t = getNumberAnswer(0, 127, t);
	if (t == 60)
		t = 0;
	FCPortPage1.InitiatorDeviceTimeout = t;

	t = FCPortPage1.InitiatorIoPendTimeout;
	if (t == 0)
		t = 8;
	printf("Initiator I/O Pending Timeout:  [0 to 127, default is %d] ", t);
	t = getNumberAnswer(0, 127, t);
	if (t == 8)
		t = 0;
	FCPortPage1.InitiatorIoPendTimeout = t;

	if (setConfigPage(port, MPI_CONFIG_PAGETYPE_FC_PORT, 1, 0, &FCPortPage1, sizeof FCPortPage1) != 1)
	{
		printf("Failed to save changes to NVRAM!\n");
		return 0;
	}

	return 1;
}


int
doSasIoUnitSettings(MPT_PORT *port)
{
	SasIOUnitPage1_t	*SASIOUnitPage1;
	ConfigReply_t		 rep;
	int					 length;
	int					 flags;
	int					 min;
	int					 max;
	int					 dev_info;
	int					 i;
	int					 t;

	if (getConfigPageHeader(port, MPI_CONFIG_EXTPAGETYPE_SAS_IO_UNIT, 1, 0, &rep) != 1)
		return 0;

	length = get16(rep.ExtPageLength) * 4;
	SASIOUnitPage1 = malloc(length);

	if (getConfigPage(port, MPI_CONFIG_EXTPAGETYPE_SAS_IO_UNIT, 1, 0, SASIOUnitPage1, length) != 1)
		return 0;

	t = SASIOUnitPage1->SATAMaxQDepth;
	printf("SATA Maximum Queue Depth:  [0 to 127, default is %d] ", t);
	t = getNumberAnswer(0, 127, t);
	SASIOUnitPage1->SATAMaxQDepth = t;

	while (TRUE)
	{
		printf("\nPhyNum  Link      MinRate  MaxRate  Initiator  Target\n");
		for (i = 0; i < SASIOUnitPage1->NumPhys; i++)
		{
			flags = SASIOUnitPage1->PhyData[i].PhyFlags & MPI_SAS_IOUNIT1_PHY_FLAGS_PHY_DISABLE;
			min = SASIOUnitPage1->PhyData[i].MaxMinLinkRate & MPI_SAS_IOUNIT1_MIN_RATE_MASK;
			max = SASIOUnitPage1->PhyData[i].MaxMinLinkRate & MPI_SAS_IOUNIT1_MAX_RATE_MASK;
			dev_info = get32(SASIOUnitPage1->PhyData[i].ControllerPhyDeviceInfo);
			printf("  %2d    %s    %s      %s    %s   %s\n", i,
				   flags ? "Disabled" : "Enabled ",
				   min == MPI_SAS_IOUNIT1_MIN_RATE_3_0 ? "3.0" : "1.5",
				   max == MPI_SAS_IOUNIT1_MAX_RATE_3_0 ? "3.0" : "1.5",
				   dev_info & MPI_SAS_DEVICE_INFO_SSP_INITIATOR ? "Enabled " : "Disabled",
				   dev_info & MPI_SAS_DEVICE_INFO_SSP_TARGET ? "Enabled " : "Disabled");
		}
		printf("\nSelect a Phy:  [0-%d, %d=AllPhys, RETURN to quit] ", SASIOUnitPage1->NumPhys - 1, SASIOUnitPage1->NumPhys);
		i = getNumberAnswer(0, SASIOUnitPage1->NumPhys, -1);

		if (i < 0)
			break;

		if (i < SASIOUnitPage1->NumPhys)
		{
			flags = SASIOUnitPage1->PhyData[i].PhyFlags & MPI_SAS_IOUNIT1_PHY_FLAGS_PHY_DISABLE;
			min = SASIOUnitPage1->PhyData[i].MaxMinLinkRate & MPI_SAS_IOUNIT1_MIN_RATE_MASK;
			max = SASIOUnitPage1->PhyData[i].MaxMinLinkRate & MPI_SAS_IOUNIT1_MAX_RATE_MASK;
			dev_info = get32(SASIOUnitPage1->PhyData[i].ControllerPhyDeviceInfo);

			t = flags == 0;
			printf("Link:  [0=Disabled, 1=Enabled, default is %d] ", t);
			t = getNumberAnswer(0, 1, t);
			if (t == 1)
				SASIOUnitPage1->PhyData[i].PhyFlags &= ~MPI_SAS_IOUNIT1_PHY_FLAGS_PHY_DISABLE;
			else
				SASIOUnitPage1->PhyData[i].PhyFlags |= MPI_SAS_IOUNIT1_PHY_FLAGS_PHY_DISABLE;

			t = min == MPI_SAS_IOUNIT1_MIN_RATE_3_0;
			printf("MinRate:  [0=1.5 Gbps, 1=3.0 Gbps, default is %d] ", t);
			t = getNumberAnswer(0, 1, t);
			SASIOUnitPage1->PhyData[i].MaxMinLinkRate &= ~MPI_SAS_IOUNIT1_MIN_RATE_MASK;
			if (t == 0)
				SASIOUnitPage1->PhyData[i].MaxMinLinkRate |= MPI_SAS_IOUNIT1_MIN_RATE_1_5;
			else
				SASIOUnitPage1->PhyData[i].MaxMinLinkRate |= MPI_SAS_IOUNIT1_MIN_RATE_3_0;

			if (t == 0)
			{
				t = max == MPI_SAS_IOUNIT1_MAX_RATE_3_0;
				printf("MaxRate:  [0=1.5 Gbps, 1=3.0 Gbps, default is %d] ", t);
				t = getNumberAnswer(0, 1, t);
			}
			else
			{
				t = 1;
			}
			SASIOUnitPage1->PhyData[i].MaxMinLinkRate &= ~MPI_SAS_IOUNIT1_MAX_RATE_MASK;
			if (t == 0)
				SASIOUnitPage1->PhyData[i].MaxMinLinkRate |= MPI_SAS_IOUNIT1_MAX_RATE_1_5;
			else
				SASIOUnitPage1->PhyData[i].MaxMinLinkRate |= MPI_SAS_IOUNIT1_MAX_RATE_3_0;

			t = (dev_info & MPI_SAS_DEVICE_INFO_SSP_INITIATOR) != 0;
			printf("Initiator:  [0=Disabled, 1=Enabled, default is %d] ", t);
			t = getNumberAnswer(0, 1, t);
			if (t == 1)
				dev_info |= (MPI_SAS_DEVICE_INFO_SSP_INITIATOR |
							 MPI_SAS_DEVICE_INFO_STP_INITIATOR |
							 MPI_SAS_DEVICE_INFO_SMP_INITIATOR);
			else
				dev_info &= ~(MPI_SAS_DEVICE_INFO_SSP_INITIATOR |
							  MPI_SAS_DEVICE_INFO_STP_INITIATOR |
							  MPI_SAS_DEVICE_INFO_SMP_INITIATOR);

			t = (dev_info & MPI_SAS_DEVICE_INFO_SSP_TARGET) != 0;
			printf("Target:  [0=Disabled, 1=Enabled, default is %d] ", t);
			t = getNumberAnswer(0, 1, t);
			if (t == 1)
				dev_info |= MPI_SAS_DEVICE_INFO_SSP_TARGET;
			else
				dev_info &= ~MPI_SAS_DEVICE_INFO_SSP_TARGET;

			SASIOUnitPage1->PhyData[i].ControllerPhyDeviceInfo = set32(dev_info);
		}
		else
		{
			printf("Link:  [0=Disabled, 1=Enabled, default is 1] ");
			flags = getNumberAnswer(0, 1, 1);

			printf("MinRate:  [0=1.5 Gbps, 1=3.0 Gbps, default is 0] ");
			min = getNumberAnswer(0, 1, 0);

			if (min == 0)
			{
				printf("MaxRate:  [0=1.5 Gbps, 1=3.0 Gbps, default is 1] ");
				max = getNumberAnswer(0, 1, 1);
			}
			else
			{
				max = 1;
			}

			dev_info = 0;

			printf("Initiator:  [0=Disabled, 1=Enabled, default is 1] ");
			t = getNumberAnswer(0, 1, 1);
			if (t == 1)
				dev_info |= (MPI_SAS_DEVICE_INFO_SSP_INITIATOR |
							 MPI_SAS_DEVICE_INFO_STP_INITIATOR |
							 MPI_SAS_DEVICE_INFO_SMP_INITIATOR);

			printf("Target:  [0=Disabled, 1=Enabled, default is 0] ");
			t = getNumberAnswer(0, 1, 0);
			if (t == 1)
				dev_info |= MPI_SAS_DEVICE_INFO_SSP_TARGET;

			for (i = 0; i < SASIOUnitPage1->NumPhys; i++)
			{
				if (flags == 1)
					SASIOUnitPage1->PhyData[i].PhyFlags &= ~MPI_SAS_IOUNIT1_PHY_FLAGS_PHY_DISABLE;
				else
					SASIOUnitPage1->PhyData[i].PhyFlags |= MPI_SAS_IOUNIT1_PHY_FLAGS_PHY_DISABLE;

				SASIOUnitPage1->PhyData[i].MaxMinLinkRate &= ~MPI_SAS_IOUNIT1_MIN_RATE_MASK;
				if (min == 0)
					SASIOUnitPage1->PhyData[i].MaxMinLinkRate |= MPI_SAS_IOUNIT1_MIN_RATE_1_5;
				else
					SASIOUnitPage1->PhyData[i].MaxMinLinkRate |= MPI_SAS_IOUNIT1_MIN_RATE_3_0;

				SASIOUnitPage1->PhyData[i].MaxMinLinkRate &= ~MPI_SAS_IOUNIT1_MAX_RATE_MASK;
				if (max == 0)
					SASIOUnitPage1->PhyData[i].MaxMinLinkRate |= MPI_SAS_IOUNIT1_MAX_RATE_1_5;
				else
					SASIOUnitPage1->PhyData[i].MaxMinLinkRate |= MPI_SAS_IOUNIT1_MAX_RATE_3_0;

				t = get32(SASIOUnitPage1->PhyData[i].ControllerPhyDeviceInfo);
				t &= ~(MPI_SAS_DEVICE_INFO_SSP_INITIATOR |
					   MPI_SAS_DEVICE_INFO_STP_INITIATOR |
					   MPI_SAS_DEVICE_INFO_SMP_INITIATOR |
					   MPI_SAS_DEVICE_INFO_SSP_TARGET);
				t |= dev_info;
				SASIOUnitPage1->PhyData[i].ControllerPhyDeviceInfo = set32(t);
			}
		}
	}

	if (setConfigPage(port, MPI_CONFIG_EXTPAGETYPE_SAS_IO_UNIT, 1, 0, SASIOUnitPage1, length) != 1)
	{
		printf("Failed to save changes to NVRAM!\n");
		return 0;
	}

	free(SASIOUnitPage1);

	return 1;
}


int
doMultiPathing(MPT_PORT *port)
{
	IOUnitPage1_t	 IOUnitPage1;
	int				 flags;
	int				 t;

	if (getConfigPage(port, MPI_CONFIG_PAGETYPE_IO_UNIT, 1, 0, &IOUnitPage1, sizeof IOUnitPage1) != 1)
		return 0;

	flags = get32(IOUnitPage1.Flags);

	t = (flags & MPI_IOUNITPAGE1_MULTI_PATHING) != 0;
	printf("Multi-pathing:  [0=Disabled, 1=Enabled, default is %d] ", t);
	t = getNumberAnswer(0, 1, t);
	if (t == 0)
		flags &= ~MPI_IOUNITPAGE1_MULTI_PATHING;
	else
		flags |= MPI_IOUNITPAGE1_MULTI_PATHING;

	IOUnitPage1.Flags = set32(flags);

	if (setConfigPage(port, MPI_CONFIG_PAGETYPE_IO_UNIT, 1, 0, &IOUnitPage1, sizeof IOUnitPage1) != 1)
	{
		printf("Failed to save changes to NVRAM!\n");
		return 0;
	}

	return 1;
}


int
doFcPersistentMappings(MPT_PORT *port, int command)
{
	FCPortPage1_t	 FCPortPage1;
	FCPortPage3_t	*FCPortPage3;
	FCDevicePage0_t	 FCDevicePage0;
	int				 sort_by_did;
	int				 do_by_entry;
	int				 i;
	int				 j;
	int				 loop_id;
	int				 t;
	int				 flags;
	int				 bus;
	int				 target;
	int				 b_t;
	int				 n;
	int				*changed;
	int				 n_changed;
	char			*type;
	U32				 d_id;
	U32				 wwnn_l;
	U32				 wwnn_h;
	U32				 wwpn_l;
	U32				 wwpn_h;

	if (getConfigPage(port, MPI_CONFIG_PAGETYPE_FC_PORT, 1, 0, &FCPortPage1, sizeof FCPortPage1) != 1)
		return 0;

	sort_by_did = get32(FCPortPage1.Flags) & MPI_FCPORTPAGE1_FLAGS_SORT_BY_DID;

	n = port->maxPersistentIds * sizeof (PersistentData_t) + sizeof (ConfigPageHeader_t);
	FCPortPage3 = (FCPortPage3_t *)malloc(n);
	changed = (int *)malloc(port->maxPersistentIds * sizeof *changed);

	if (n > 255 * 4)
	{
		FCPortPage3_t	tempFCPortPage3;

		do_by_entry = 1;

		for (i = 0; i < port->maxPersistentIds; i++)
		{
			if (getConfigPage(port, MPI_CONFIG_PAGETYPE_FC_PORT, 3,
							  MPI_FC_PORT_PGAD_FORM_INDEX + i,
							  &tempFCPortPage3, sizeof tempFCPortPage3) != 1)
				return 0;

			FCPortPage3->Entry[i] = tempFCPortPage3.Entry[0];
		}

		FCPortPage3->Header = tempFCPortPage3.Header;

		n = port->maxPersistentIds;
	}
	else
	{
		do_by_entry = 0;

		if (getConfigPage(port, MPI_CONFIG_PAGETYPE_FC_PORT, 3, 0, FCPortPage3, n) != 1)
			return 0;

		n = (FCPortPage3->Header.PageLength * 4 - sizeof (ConfigPageHeader_t)) / sizeof (PersistentData_t);
	}
	memset(changed, 0, port->maxPersistentIds * sizeof *changed);
	n_changed = 0;

	if (command == 1 || command == 4 || command == 5)
	{
		j = 0;
		for (i = 0; i < n; i++)
		{
			flags = get16(FCPortPage3->Entry[i].Flags);
			if (flags & MPI_PERSISTENT_FLAGS_ENTRY_VALID)
			{
				j++;
				if (flags & MPI_PERSISTENT_FLAGS_BY_DID)
				{
					printf("Persistent entry %d is valid, Bus %d Target %d is DID %06x\n", i,
						   FCPortPage3->Entry[i].Bus,
						   FCPortPage3->Entry[i].TargetID,
						   get32(FCPortPage3->Entry[i].PhysicalIdentifier.Did));
					if (!sort_by_did)
						printf("  Since the port is in SortByWWN mode, this entry is being ignored\n");
				}
				else
				{
					printf("Persistent entry %d is valid, Bus %d Target %d is WWN %08x%08x\n", i,
						   FCPortPage3->Entry[i].Bus,
						   FCPortPage3->Entry[i].TargetID,
						   get32(FCPortPage3->Entry[i].PhysicalIdentifier.WWN.WWPN.High),
						   get32(FCPortPage3->Entry[i].PhysicalIdentifier.WWN.WWPN.Low));
					if (sort_by_did)
						printf("  Since the port is in SortByDID mode, this entry is being ignored\n");
				}
				if (command == 5)
				{
					printf("  Delete the entry for this target?  [Yes or No, default is No] ");
					if (getYesNoAnswer(0) != 1)
						continue;
				}
				if (command == 4 || command == 5)
				{
					printf("  Deleting persistent entry %d\n", i);
					FCPortPage3->Entry[i].Flags &= ~set16(MPI_PERSISTENT_FLAGS_ENTRY_VALID);
					changed[i] = 1;
					n_changed++;
				}
			}
		}

		if (j == 0)
			printf("No persistent entries found\n");
	}

	if (command == 2 || command == 3)
	{
		for (bus = 0; bus < port->maxBuses; bus++)
		{
			for (target = 0; target < port->maxTargets; target++)
			{
				b_t = (bus << 8) + target;

				if (getConfigPage(port, MPI_CONFIG_PAGETYPE_FC_DEVICE, 0,
								  MPI_FC_DEVICE_PGAD_FORM_BUS_TID + b_t,
								  &FCDevicePage0, sizeof FCDevicePage0) != 1)
					continue;

				if (sort_by_did)
					printf("Bus %d Target %d is DID %06x\n", bus, target,
						   get32(FCDevicePage0.PortIdentifier));
				else
					printf("Bus %d Target %d is WWN %08x%08x\n", bus, target,
						   get32(FCDevicePage0.WWPN.High), get32(FCDevicePage0.WWPN.Low));
				for (i = 0; i < n; i++)
				{
					if (get16(FCPortPage3->Entry[i].Flags) & MPI_PERSISTENT_FLAGS_ENTRY_VALID)
					{
						if (FCPortPage3->Entry[i].Bus      != bus ||
							FCPortPage3->Entry[i].TargetID != target)
						{
							continue;
						}

						if (sort_by_did)
						{
							if (FCPortPage3->Entry[i].PhysicalIdentifier.Did == FCDevicePage0.PortIdentifier)
							{
								break;
							}
						}
						else
						{
							if (FCPortPage3->Entry[i].PhysicalIdentifier.WWN.WWPN.High == FCDevicePage0.WWPN.High &&
								FCPortPage3->Entry[i].PhysicalIdentifier.WWN.WWPN.Low  == FCDevicePage0.WWPN.Low &&
								FCPortPage3->Entry[i].PhysicalIdentifier.WWN.WWNN.High == FCDevicePage0.WWNN.High &&
								FCPortPage3->Entry[i].PhysicalIdentifier.WWN.WWNN.Low  == FCDevicePage0.WWNN.Low)
							{
								break;
							}
						}
					}
				}
				if (i < n)
				{
					printf("  Persistent entry %d is already valid for this target!\n", i);
					continue;
				}

				for (i = 0; i < n; i++)
				{
					if (!(get16(FCPortPage3->Entry[i].Flags) & MPI_PERSISTENT_FLAGS_ENTRY_VALID))
					{
						break;
					}
				}
				if (i < n)
				{
					if (command == 3)
					{
						printf("  Add an entry for this target?  [Yes or No, default is No] ");
						if (getYesNoAnswer(0) != 1)
							continue;
					}

					printf("  Adding persistent entry %d\n", i);

					FCPortPage3->Entry[i].Flags		 = set16(MPI_PERSISTENT_FLAGS_ENTRY_VALID);
					FCPortPage3->Entry[i].Bus		 = bus;
					FCPortPage3->Entry[i].TargetID	 = target;
					if (sort_by_did)
					{
						FCPortPage3->Entry[i].Flags |= set16(MPI_PERSISTENT_FLAGS_BY_DID);
						FCPortPage3->Entry[i].PhysicalIdentifier.Did = FCDevicePage0.PortIdentifier;
					}
					else
					{
						FCPortPage3->Entry[i].PhysicalIdentifier.WWN.WWPN.High = FCDevicePage0.WWPN.High;
						FCPortPage3->Entry[i].PhysicalIdentifier.WWN.WWPN.Low  = FCDevicePage0.WWPN.Low;
						FCPortPage3->Entry[i].PhysicalIdentifier.WWN.WWNN.High = FCDevicePage0.WWNN.High;
						FCPortPage3->Entry[i].PhysicalIdentifier.WWN.WWNN.Low  = FCDevicePage0.WWNN.Low;
					}
					changed[i] = 1;
					n_changed++;
				}
				else
				{
					printf("  No persistent entry available for this target!\n");
				}
			}
		}
	}

	if (command == 6)
	{
		if (sort_by_did)
		{
			printf("The port is in SortByDID mode, enter Port IDs\n");
			type = "DID";
		}
		else
		{
			printf("The port is in SortByWWN mode, enter World Wide Node and Port Names\n");
			type = "WWN";
		}

		for (i = 0; i < n; i++)
		{
			if (get16(FCPortPage3->Entry[i].Flags) & MPI_PERSISTENT_FLAGS_ENTRY_VALID)
				continue;

			printf("\n");

			if (sort_by_did)
			{
				printf("Enter DID:  [000000-FFFFFF or RETURN to quit] ");
				while (TRUE)
				{
					t = getHexNumberAnswer(&d_id);
					if (t == 0)
						break;

					if (d_id <= 0xffffff)
						break;

					printf("Invalid answer, try again: ");
				}
				if (t == 0)
					break;
			}
			else
			{
				printf("Enter WWNN:  [16 hex digits or RETURN to quit] ");
				t = getHexDoubleNumberAnswer(&wwnn_h, &wwnn_l);
				if (t == 0)
					break;

				printf("Enter WWPN:  [16 hex digits or RETURN to quit] ");
				t = getHexDoubleNumberAnswer(&wwpn_h, &wwpn_l);
				if (t == 0)
					break;
			}

			if (port->maxBuses > 1)
			{
				printf("Enter desired Bus for this %s:  [0-%d or RETURN to quit] ", type, port->maxBuses - 1);
				bus = getNumberAnswer(0, port->maxBuses - 1, -1);
				if (bus == -1)
					break;
			}
			else
			{
				bus = 0;
			}

			printf("Enter desired Target for this %s:  [0-%d or RETURN to quit] ", type, port->maxTargets - 1);
			target = getNumberAnswer(0, port->maxTargets - 1, -1);
			if (target == -1)
				break;

			for (j = 0; j < n; j++)
			{
				if (i == j)
					continue;

				if (get16(FCPortPage3->Entry[j].Flags) & MPI_PERSISTENT_FLAGS_ENTRY_VALID)
				{
					if (sort_by_did)
					{
						if (FCPortPage3->Entry[j].PhysicalIdentifier.Did == d_id)
						{
							printf("\nPersistent entry %d is already valid for this DID!\n", j);
							break;
						}
					}
					else
					{
						if (get32(FCPortPage3->Entry[j].PhysicalIdentifier.WWN.WWPN.High) == wwpn_h &&
							get32(FCPortPage3->Entry[j].PhysicalIdentifier.WWN.WWPN.Low)  == wwpn_l &&
							get32(FCPortPage3->Entry[j].PhysicalIdentifier.WWN.WWNN.High) == wwnn_h &&
							get32(FCPortPage3->Entry[j].PhysicalIdentifier.WWN.WWNN.Low)  == wwnn_l)
						{
							printf("\nPersistent entry %d is already valid for this WWN!\n", j);
							break;
						}
					}

					if (FCPortPage3->Entry[j].Bus      == bus &&
						FCPortPage3->Entry[j].TargetID == target)
					{
						printf("\nPersistent entry %d is already valid for this Bus & Target!\n", j);
						break;
					}
				}
			}
			if (j < n)
			{
				i--;
				continue;
			}

			printf("\nAdding persistent entry %d\n", i);

			FCPortPage3->Entry[i].Flags		 = set16(MPI_PERSISTENT_FLAGS_ENTRY_VALID);
			FCPortPage3->Entry[i].Bus		 = bus;
			FCPortPage3->Entry[i].TargetID	 = target;
			if (sort_by_did)
			{
				FCPortPage3->Entry[i].Flags |= set16(MPI_PERSISTENT_FLAGS_BY_DID);
				FCPortPage3->Entry[i].PhysicalIdentifier.Did = set32(d_id);
			}
			else
			{
				FCPortPage3->Entry[i].PhysicalIdentifier.WWN.WWPN.High = set32(wwpn_h);
				FCPortPage3->Entry[i].PhysicalIdentifier.WWN.WWPN.Low  = set32(wwpn_l);
				FCPortPage3->Entry[i].PhysicalIdentifier.WWN.WWNN.High = set32(wwnn_h);
				FCPortPage3->Entry[i].PhysicalIdentifier.WWN.WWNN.Low  = set32(wwnn_l);
			}
			changed[i] = 1;
			n_changed++;
		}
	}

	if (command == 7)
	{
		if (!sort_by_did)
		{
			printf("The port is not in SortByDID mode!\n");

			printf("\nDo you want to switch to SortByDID mode?  [Yes or No, default is No] ");

			if (getYesNoAnswer(0) == 1)
			{
				FCPortPage1.Flags |= set32(MPI_FCPORTPAGE1_FLAGS_SORT_BY_DID);

				if (setConfigPage(port, MPI_CONFIG_PAGETYPE_FC_PORT, 1, 0, &FCPortPage1, sizeof FCPortPage1) != 1)
				{
					printf("Failed to save changes to NVRAM!\n");
					return 0;
				}
				else
				{
					sort_by_did = 1;
					printf("\n");
				}
			}
		}

		loop_id = 0;
		for (i = 0; i < n; i++)
		{
			if (get16(FCPortPage3->Entry[i].Flags) & MPI_PERSISTENT_FLAGS_ENTRY_VALID)
				continue;

			d_id = LoopIdToAlpa[loop_id];
			bus = 0;
			target = loop_id;

			t = n;
			for (j = 0; j < n; j++)
			{
				if (i == j)
					continue;

				if (get16(FCPortPage3->Entry[j].Flags) & MPI_PERSISTENT_FLAGS_ENTRY_VALID)
				{
					if (FCPortPage3->Entry[j].PhysicalIdentifier.Did == d_id &&
						FCPortPage3->Entry[j].Bus                    == bus &&
						FCPortPage3->Entry[j].TargetID               == target)
					{
						printf("Persistent entry %d exists for LoopID %d: Bus %d Target %d is DID %02x\n",
							   j, loop_id, bus, target, d_id);
						t = j;
						break;
					}
					else
					{
						if (FCPortPage3->Entry[j].PhysicalIdentifier.Did == d_id)
						{
							printf("\nPersistent entry %d is already valid for DID %02x!\n", j, d_id);
							break;
						}

						if (FCPortPage3->Entry[j].Bus      == bus &&
							FCPortPage3->Entry[j].TargetID == target)
						{
							printf("\nPersistent entry %d is already valid for Bus %d Target %d!\n", j, bus, target);
							break;
						}
					}
				}
			}
			if (j < n)
			{
				if (t == n)
				{
					printf("\nDo you want to continue?  [Yes or No, default is No] ");

					if (getYesNoAnswer(0) != 1)
						break;

					printf("\n");
				}
				i--;
			}
			else
			{
				printf("Adding persistent entry %d for LoopID %d: Bus %d Target %d is DID %02x\n",
					   i, loop_id, bus, target, d_id);

				FCPortPage3->Entry[i].Flags		 = set16(MPI_PERSISTENT_FLAGS_ENTRY_VALID);
				FCPortPage3->Entry[i].Bus		 = bus;
				FCPortPage3->Entry[i].TargetID	 = target;
				FCPortPage3->Entry[i].Flags |= set16(MPI_PERSISTENT_FLAGS_BY_DID);
				FCPortPage3->Entry[i].PhysicalIdentifier.Did = set32(d_id);
				changed[i] = 1;
				n_changed++;
			}

			loop_id++;
			if (loop_id == 126)
				break;
		}
	}

	if (command != 1 && n_changed != 0)
	{
		if (do_by_entry)
		{
			FCPortPage3_t	tempFCPortPage3;

			tempFCPortPage3.Header = FCPortPage3->Header;

			for (i = 0; i < port->maxPersistentIds; i++)
			{
				if (changed[i])
				{
					tempFCPortPage3.Entry[0] = FCPortPage3->Entry[i];

					if (setConfigPage(port, MPI_CONFIG_PAGETYPE_FC_PORT, 3,
									  MPI_FC_PORT_PGAD_FORM_INDEX + i,
									  &tempFCPortPage3, sizeof tempFCPortPage3) != 1)
					{
						printf("Failed to save changes to NVRAM!\n");
						return 0;
					}
				}
			}
		}
		else
		{
			n = FCPortPage3->Header.PageLength * 4;
			if (setConfigPage(port, MPI_CONFIG_PAGETYPE_FC_PORT, 3, 0, FCPortPage3, n) != 1)
			{
				printf("Failed to save changes to NVRAM!\n");
				return 0;
			}
		}
	}

	free(FCPortPage3);
	free(changed);

	return 1;
}


int
doSasPersistentMappings(MPT_PORT *port, int command)
{
	SasDevicePage0_t	 SASDevicePage0;
	SasDevicePage2_t	*SASDevicePage2;
	int					 i;
	int					 j;
	int					 t;
	int					 bus;
	int					 target;
	int					 b_t;
	int					 max_b_t;
	int					 n;
	int					*changed;
	int					 n_changed;
	int					*i_to_b_t;
	U32					 physid_l;
	U32					 physid_h;

	if (command == 10 || command == 11)
	{
		SasIoUnitControlRequest_t	 req;
		SasIoUnitControlReply_t		 rep;

		memset(&req, 0, sizeof req);
		memset(&rep, 0, sizeof rep);

		req.Function			= MPI_FUNCTION_SAS_IO_UNIT_CONTROL;

		if (command == 10)
		{
			req.Operation		= MPI_SAS_OP_CLEAR_ALL_PERSISTENT;

			printf("Clearing all persistent entries...\n");
		}
		if (command == 11)
		{
			req.Operation		= MPI_SAS_OP_CLEAR_NOT_PRESENT;

			printf("Clearing all non-present persistent entries...\n");
		}

		if (doMptCommandCheck(port, &req, sizeof req, &rep, sizeof rep, NULL, 0, NULL, 0, 10) != 1)
		{
			printf("Clear failed!\n");
			return 0;
		}

		return 1;
	}

	if (command == 12)
	{
		SasIOUnitPage2_t	 SASIOUnitPage2;
		int					 flags;
		int					 t;

		if (getConfigPage(port, MPI_CONFIG_EXTPAGETYPE_SAS_IO_UNIT, 2, 0,
						  &SASIOUnitPage2, sizeof SASIOUnitPage2) != 1)
			return 0;

		flags = SASIOUnitPage2.Flags;

		t = (flags & MPI_SAS_IOUNIT2_FLAGS_DISABLE_PERSISTENT_MAPPINGS) == 0;
		printf("Persistence:  [0=Disabled, 1=Enabled, default is %d] ", t);
		t = getNumberAnswer(0, 1, t);
		if (t == 1)
			flags &= ~MPI_SAS_IOUNIT2_FLAGS_DISABLE_PERSISTENT_MAPPINGS;
		else
			flags |= MPI_SAS_IOUNIT2_FLAGS_DISABLE_PERSISTENT_MAPPINGS;

		SASIOUnitPage2.Flags = flags;

		if (setConfigPage(port, MPI_CONFIG_EXTPAGETYPE_SAS_IO_UNIT, 2, 0,
						  &SASIOUnitPage2, sizeof SASIOUnitPage2) != 1)
		{
			printf("Failed to save changes to NVRAM!\n");
			return 0;
		}

		return 1;
	}

	max_b_t = (port->maxBuses << 8) + port->maxTargets;
	n = port->maxPersistentIds;
	SASDevicePage2 = (SasDevicePage2_t *)malloc(n * sizeof *SASDevicePage2);
	changed = (int *)malloc(n * sizeof *changed);
	i_to_b_t = (int *)malloc(n * sizeof *i_to_b_t);

	for (i = 0, j = 0; i < max_b_t && j < n; i++)
	{
		if (getConfigPage(port, MPI_CONFIG_EXTPAGETYPE_SAS_DEVICE, 2,
						  (MPI_SAS_DEVICE_PGAD_FORM_BUS_TARGET_ID
						   <<MPI_SAS_DEVICE_PGAD_FORM_SHIFT) + i,
						  &SASDevicePage2[j], sizeof *SASDevicePage2) != 1)
			return 0;

		if (SASDevicePage2[j].PhysicalIdentifier.High != 0 ||
			SASDevicePage2[j].PhysicalIdentifier.Low  != 0)
		{
			i_to_b_t[j] = i;
			j++;
		}
	}
	memset(changed, 0, n * sizeof *changed);
	n_changed = 0;

	if (command == 1 || command == 4 || command == 5)
	{
		for (i = 0; i < j; i++)
		{
			b_t = i_to_b_t[i];
			printf("Persistent entry %d is valid, Bus %d Target %d is PhysId %08x%08x\n",
				   i, b_t >> 8, b_t & 255,
				   get32(SASDevicePage2[i].PhysicalIdentifier.High),
				   get32(SASDevicePage2[i].PhysicalIdentifier.Low));
			if (command == 5)
			{
				printf("  Delete the entry for this target?  [Yes or No, default is No] ");
				if (getYesNoAnswer(0) != 1)
					continue;
			}
			if (command == 4 || command == 5)
			{
				printf("  Deleting persistent entry\n");
				SASDevicePage2[i].PhysicalIdentifier.High = 0;
				SASDevicePage2[i].PhysicalIdentifier.Low  = 0;
				changed[i] = 1;
				n_changed++;
			}
		}

		if (j == 0)
			printf("No persistent entries found\n");
	}

	if (command == 2 || command == 3)
	{
		for (bus = 0; bus < port->maxBuses; bus++)
		{
			for (target = 0; target < port->maxTargets; target++)
			{
				b_t = (bus << 8) + target;

				if (getConfigPage(port, MPI_CONFIG_EXTPAGETYPE_SAS_DEVICE, 0,
								  (MPI_SAS_DEVICE_PGAD_FORM_BUS_TARGET_ID
								   <<MPI_SAS_DEVICE_PGAD_FORM_SHIFT) + b_t,
								  &SASDevicePage0, sizeof SASDevicePage0) != 1)
					continue;

				printf("Bus %d Target %d is PhysId %08x%08x\n", bus, target,
					   get32(SASDevicePage0.SASAddress.High), get32(SASDevicePage0.SASAddress.Low));

				if (get32(SASDevicePage0.DeviceInfo) & (MPI_SAS_DEVICE_INFO_SATA_HOST | MPI_SAS_DEVICE_INFO_SATA_DEVICE))
				{
					printf("  This target is SATA and cannot be set persistent manually!\n");
					continue;
				}

				for (i = 0; i < j; i++)
				{
					if (SASDevicePage2[i].PhysicalIdentifier.High == SASDevicePage0.SASAddress.High &&
						SASDevicePage2[i].PhysicalIdentifier.Low  == SASDevicePage0.SASAddress.Low)
					{
						break;
					}
				}
				if (i < j)
				{
					printf("  Persistent entry %d is already valid for this target!\n", i);
					continue;
				}

				if (j < n)
				{
					if (command == 3)
					{
						printf("  Add an entry for this target?  [Yes or No, default is No] ");
						if (getYesNoAnswer(0) != 1)
							continue;
					}

					printf("  Adding persistent entry %d\n", j);

					SASDevicePage2[j].PhysicalIdentifier.High = SASDevicePage0.SASAddress.High;
					SASDevicePage2[j].PhysicalIdentifier.Low  = SASDevicePage0.SASAddress.Low;
					changed[j] = 1;
					n_changed++;
					i_to_b_t[j] = i;
					j++;
				}
				else
				{
					printf("  No persistent entry available for this target!\n");
				}
			}
		}
	}

	if (command == 6)
	{
		while (j < n)
		{
			printf("Enter PhysId:  [16 hex digits or RETURN to quit] ");
			t = getHexDoubleNumberAnswer(&physid_h, &physid_l);
			if (t == 0)
				break;

			if (port->maxBuses > 1)
			{
				printf("Enter desired Bus for this PhysId:  [0-%d or RETURN to quit] ", port->maxBuses - 1);
				bus = getNumberAnswer(0, port->maxBuses - 1, -1);
				if (bus == -1)
					break;
			}
			else
			{
				bus = 0;
			}

			printf("Enter desired Target for this PhysId:  [0-%d or RETURN to quit] ", port->maxTargets - 1);
			target = getNumberAnswer(0, port->maxTargets - 1, -1);
			if (target == -1)
				break;

			b_t = (bus << 8) + target;

			for (i = 0; i < j; i++)
			{
				if (get32(SASDevicePage2[i].PhysicalIdentifier.High) == physid_h &&
					get32(SASDevicePage2[i].PhysicalIdentifier.Low)  == physid_l)
				{
					printf("\nPersistent entry %d is already valid for this PhysId!\n\n", i);
					break;
				}

				if (i_to_b_t[i] == b_t)
				{
					printf("\nPersistent entry %d is already valid for this Bus & Target!\n\n", i);
					break;
				}
			}
			if (i < j)
				continue;

			printf("\nAdding persistent entry %d\n\n", j);

			SASDevicePage2[j].PhysicalIdentifier.High = set32(physid_h);
			SASDevicePage2[j].PhysicalIdentifier.Low  = set32(physid_l);
			changed[j] = 1;
			n_changed++;
			i_to_b_t[j] = i;
			j++;
		}
	}

	if (command != 1 && n_changed != 0)
	{
		for (i = 0; i < j; i++)
		{
			if (changed[i])
			{
				if (setConfigPage(port, MPI_CONFIG_EXTPAGETYPE_SAS_DEVICE, 2,
								  (MPI_SAS_DEVICE_PGAD_FORM_BUS_TARGET_ID
								   <<MPI_SAS_DEVICE_PGAD_FORM_SHIFT) + i_to_b_t[i],
								  &SASDevicePage2[i], sizeof *SASDevicePage2) != 1)
				{
					printf("Failed to save changes to NVRAM!\n");
					return 0;
				}
			}
		}
	}

	free(SASDevicePage2);
	free(changed);
	free(i_to_b_t);

	return 1;
}


int
doDisplayLoggedInDevices(MPT_PORT *port)
{
	FCDevicePage0_t	 FCDevicePage0;
	U32				 d_id;

	showPortInfoHeader(port);

	printf(" B___T        WWNN              WWPN        PortId\n");

	d_id = 0xffffff;
	while (TRUE)
	{
		if (getConfigPage(port, MPI_CONFIG_PAGETYPE_FC_DEVICE, 0, d_id,
						  &FCDevicePage0, sizeof FCDevicePage0) != 1)
			break;

		d_id = get32(FCDevicePage0.PortIdentifier);

		if (FCDevicePage0.Flags & MPI_FC_DEVICE_PAGE0_FLAGS_TARGETID_BUS_VALID)
		{
			printf("%2d %3d  %08x%08x  %08x%08x  %06x\n",
				   FCDevicePage0.CurrentBus, FCDevicePage0.CurrentTargetID,
				   get32(FCDevicePage0.WWNN.High), get32(FCDevicePage0.WWNN.Low),
				   get32(FCDevicePage0.WWPN.High), get32(FCDevicePage0.WWPN.Low),
				   d_id);
		}
		else
		{
			printf("        %08x%08x  %08x%08x  %06x\n",
				   get32(FCDevicePage0.WWNN.High), get32(FCDevicePage0.WWNN.Low),
				   get32(FCDevicePage0.WWPN.High), get32(FCDevicePage0.WWPN.Low),
				   d_id);
		}
	}

	return 1;
}


int
doDisplayAttachedDevices(MPT_PORT *port)
{
	SasDevicePage0_t	 SASDevicePage0;
	U32					 handle;
	int					 dev_info;
	int					 flags;
	int					 type;
	char				 info[80];

	showPortInfoHeader(port);

	printf(" B___T     SASAddress     PhyNum  Handle  Parent  Type\n");

	handle = 0xffff;
	while (TRUE)
	{
		if (getConfigPage(port, MPI_CONFIG_EXTPAGETYPE_SAS_DEVICE, 0, handle,
						  &SASDevicePage0, sizeof SASDevicePage0) != 1)
			break;

		handle = get16(SASDevicePage0.DevHandle);
		dev_info = get32(SASDevicePage0.DeviceInfo);
		flags = get16(SASDevicePage0.Flags);

		info[0] = 0;
		type = dev_info & MPI_SAS_DEVICE_INFO_MASK_DEVICE_TYPE;

		if (type == MPI_SAS_DEVICE_INFO_EDGE_EXPANDER)
			strcat(info, ", Edge Expander");
		if (type == MPI_SAS_DEVICE_INFO_FANOUT_EXPANDER)
			strcat(info, ", FanOut Expander");

		if (dev_info & MPI_SAS_DEVICE_INFO_SSP_INITIATOR)
			if (dev_info & MPI_SAS_DEVICE_INFO_SSP_TARGET)
				strcat(info, ", SAS Initiator and Target");
			else
				strcat(info, ", SAS Initiator");
		else if (dev_info & MPI_SAS_DEVICE_INFO_SSP_TARGET)
			strcat(info, ", SAS Target");

		if (dev_info & MPI_SAS_DEVICE_INFO_SATA_HOST)
			if (dev_info & MPI_SAS_DEVICE_INFO_SATA_DEVICE)
				strcat(info, ", SATA Initiator and Target");
			else
				strcat(info, ", SATA Initiator");
		else if (dev_info & MPI_SAS_DEVICE_INFO_SATA_DEVICE)
			strcat(info, ", SATA Target");

		if (dev_info & MPI_SAS_DEVICE_INFO_ATAPI_DEVICE)
			strcat(info, ", ATAPI");

		if (!(flags & MPI_SAS_DEVICE0_FLAGS_DEVICE_PRESENT))
			strcat(info, ", not present");

		if (SASDevicePage0.ParentDevHandle == 0)
		{
			printf("        %08x%08x           %04x           %s\n",
				   get32(SASDevicePage0.SASAddress.High), get32(SASDevicePage0.SASAddress.Low),
				   handle,
				   info + 2);
		}
		else if (get16(SASDevicePage0.Flags) & MPI_SAS_DEVICE0_FLAGS_DEVICE_MAPPED)
		{
			printf("%2d %3d  %08x%08x    %2d     %04x    %04x   %s\n",
				   SASDevicePage0.Bus, SASDevicePage0.TargetID,
				   get32(SASDevicePage0.SASAddress.High), get32(SASDevicePage0.SASAddress.Low),
				   SASDevicePage0.PhyNum, handle, get16(SASDevicePage0.ParentDevHandle),
				   info + 2);
		}
		else
		{
			printf("        %08x%08x    %2d     %04x    %04x   %s\n",
				   get32(SASDevicePage0.SASAddress.High), get32(SASDevicePage0.SASAddress.Low),
				   SASDevicePage0.PhyNum, handle, get16(SASDevicePage0.ParentDevHandle),
				   info + 2);
		}
	}

	return 1;
}


int
doShowPortAliases(MPT_PORT *port)
{
	FCPortPage0_t	 FCPortPage0;
	FCPortPage1_t	 FCPortPage1;
	FCPortPage5_t	 FCPortPage5;
	int	 i;

	if (getConfigPageAction(port, MPI_CONFIG_ACTION_PAGE_READ_CURRENT, MPI_CONFIG_PAGETYPE_FC_PORT, 0, 0, &FCPortPage0, sizeof FCPortPage0) != 1)
		return 0;

	if (getConfigPageAction(port, MPI_CONFIG_ACTION_PAGE_READ_CURRENT, MPI_CONFIG_PAGETYPE_FC_PORT, 1, 0, &FCPortPage1, sizeof FCPortPage1) != 1)
		return 0;

	if (FCPortPage0.MaxAliasesSupported == 0)
	{
		printf("Aliases are not supported on this port\n");
		return 1;
	}

	printf("%d aliases requested, %d aliases active\n", FCPortPage1.NumRequestedAliases, FCPortPage0.NumCurrentAliases);

	if (FCPortPage1.NumRequestedAliases == 0)
		return 1;

	printf("\n           WWNN              WWPN        PortId  Flags\n");

	for (i = 1; i <= FCPortPage1.NumRequestedAliases; i++)
	{
		if (getConfigPageAction(port, MPI_CONFIG_ACTION_PAGE_READ_CURRENT, MPI_CONFIG_PAGETYPE_FC_PORT, 5, 0x01000000 + i, &FCPortPage5, sizeof FCPortPage5) != 1)
			continue;

		printf("%3d  %08x%08x  %08x%08x  0000%02x  %02x (%s%s)\n", i,
			get32(FCPortPage5.AliasInfo.AliasWWNN.High), get32(FCPortPage5.AliasInfo.AliasWWNN.Low),
			get32(FCPortPage5.AliasInfo.AliasWWPN.High), get32(FCPortPage5.AliasInfo.AliasWWPN.Low),
			FCPortPage5.AliasInfo.AliasAlpa,
			FCPortPage5.AliasInfo.Flags,
			(FCPortPage5.AliasInfo.Flags & MPI_FCPORTPAGE5_FLAGS_DISABLE) ? "Disabled, " : "",
			(FCPortPage5.AliasInfo.Flags & MPI_FCPORTPAGE5_FLAGS_ALPA_ACQUIRED) ? "Active" : "Inactive");
	}

	return 1;
}


int
doTestConfigPageActions(MPT_PORT *port)
{
	Config_t			 req;
	ConfigReply_t		 rep;
	char				 name[256];
	FILE				*in_file;
	FILE				*out_file;
	char				 str[80];
	char				 act[80];
	U32					 buf[256];
	int					 action;
	int					 type;
	int					 number;
	int					 length;
	int					 ioc_status;
	U32					 address;
	U32					 offset;
	U32					 value;
	int					 i;
	int					 j;
	int					 n;
	int					 t;
	char				*action_string[7] = {"H", "RC", "WC", "D", "WN", "RD", "RN"};

	printf("Enter input file, or RETURN to enter commands interactively: ");
	n = getString(name, sizeof name, stdin);
	
	if (n > 0)
	{
		in_file = fopen(name, "r");
		if (in_file == NULL)
		{
			printf("Open failure for file %s\n", name);
			perror("Error is");
			return 0;
		}
	}
	else
	{
		in_file = stdin;
	}

	printf("Enter output file, or RETURN to send output to the screen: ");
	n = getString(name, sizeof name, stdin);

	if (n > 0)
	{
		out_file = fopen(name, "w");
		if (out_file == NULL)
		{
			printf("Open failure for file %s\n", name);
			perror("Error is");
			if (in_file != stdin)
				fclose(in_file);
			return 0;
		}
	}
	else
	{
		out_file = stdout;
	}

	while (TRUE)
	{
		while (TRUE)
		{
			if (in_file == stdin)
			{
				printf("\nEnter command, or ? for help, or RETURN to quit: ");
				n = getString(str, sizeof str, stdin);
			}
			else
			{
				if (fgets(str, sizeof str, in_file) == NULL)
				{
					n = 0;
				}
				else
				{
					str[sizeof str - 1] = '\0';
					n = strlen(str);
					if (n >= 1 && str[n-1] == '\n')
					{
						str[n-1] = '\0';
						n -= 1;
					}

					if (n == 0)
						continue;

					printf("Executing \"%s\"...\n", str);
				}
			}

			if (n == 0)
			{
				if (in_file != stdin)
					fclose(in_file);
				if (out_file != stdout)
					fclose(out_file);
				return 1;
			}

			act[0]	= '\0';
			type	= 0;
			number	= 0;
			address	= 0;
			i = sscanf(str, "%s %d %d %x%n\n", act, &type, &number, &address, &j);
			if (i >= 4 && j < n && str[j] != ' ')
				i = 0;

			if (i >= 3)
			{
				if (sscanf(act, "%d", &action) == 1)
					;
				else if (strcasecmp(act, "h") == 0)
					action = 0;
				else if (strcasecmp(act, "rc") == 0)
					action = 1;
				else if (strcasecmp(act, "wc") == 0)
					action = 2;
				else if (strcasecmp(act, "d") == 0)
					action = 3;
				else if (strcasecmp(act, "wn") == 0)
					action = 4;
				else if (strcasecmp(act, "rd") == 0)
					action = 5;
				else if (strcasecmp(act, "rn") == 0)
					action = 6;

				if (action >= 0 && action <= 6   &&
					type   >= 0 && type   <= 255 &&
					number >= 0 && number <= 255)
				{
					break;
				}
			}

			if (in_file == stdin)
			{
				if (n != 1 || str[0] != '?')
					printf("\nInvalid response!\n");
				printf("\nValid input is:  <Action> <PageType> <PageNumber> [<PageAddress>]\n");
				printf("  Action is:\n");
				printf("    0 or H   display page Header only\n");
				printf("    1 or RC  display page after Read Current\n");
				printf("    2 or WC  enter page and do Write Current\n");
				printf("    3 or D   set page to Default values\n");
				printf("    4 or WN  enter page and do Write NVRAM\n");
				printf("    5 or RD  display page after Read Default\n");
				printf("    6 or RN  display page after Read NVRAM\n");
				printf("  PageType is a decimal number beteen 0 and 255:\n");
				printf("  PageNumber is a decimal number between 0 and 255\n");
				printf("  PageAddress is an optional hex number\n");
			}
		}

		memset(&req, 0, sizeof req);
		memset(&rep, 0, sizeof rep);

		req.Function			= MPI_FUNCTION_CONFIG;
		req.Action				= MPI_CONFIG_ACTION_PAGE_HEADER;
		if (type > MPI_CONFIG_PAGETYPE_EXTENDED)
		{
			req.Header.PageType	= MPI_CONFIG_PAGETYPE_EXTENDED;
			req.ExtPageType		= type;
		}
		else
		{
			req.Header.PageType	= type;
		}
		req.Header.PageNumber	= number;
		req.PageAddress			= set32(address);

		if (doMptCommand(port, &req, sizeof req - sizeof req.PageBufferSGE, &rep, sizeof rep,
						 NULL, 0, NULL, 0, 10) != 1)
		{
			printf("Failed to get Header for page %d/%d/%x\n\n", type, number, address);
			continue;
		}

		if (action == 0)
		{
			ioc_status = get16(rep.IOCStatus) & MPI_IOCSTATUS_MASK;

			fprintf(out_file, "\n%s %d %d %x -- IOCStatus = %04x (%s)\n",
					action_string[action], type, number, address,
					ioc_status, translateIocStatus(ioc_status));

			if (ioc_status == MPI_IOCSTATUS_SUCCESS)
			{
				fprintf(out_file, "0000 : %08x\n", get32(*(U32 *)&rep.Header));
				if (type > MPI_CONFIG_PAGETYPE_EXTENDED)
					fprintf(out_file, "0004 : 00%02x%04x\n", rep.ExtPageType, get16(rep.ExtPageLength));
			}
		}

		if (action == 1 || action == 5 || action == 6)
		{
			req.Action			= action;
			req.ExtPageType		= rep.ExtPageType;
			req.ExtPageLength	= rep.ExtPageLength;
			req.Header			= rep.Header;

			if (doMptCommand(port, &req, sizeof req - sizeof req.PageBufferSGE, &rep, sizeof rep,
							 buf, sizeof buf, NULL, 0, 10) != 1)
			{
				printf("Failed to read page %d/%d/%x\n\n", type, number, address);
				continue;
			}

			ioc_status = get16(rep.IOCStatus) & MPI_IOCSTATUS_MASK;

			fprintf(out_file, "\n%s %d %d %x -- IOCStatus = %04x (%s)\n",
					action_string[action], type, number, address,
					ioc_status, translateIocStatus(ioc_status));

			if (ioc_status == MPI_IOCSTATUS_SUCCESS)
			{
				if (type > MPI_CONFIG_PAGETYPE_EXTENDED)
					n = get16(rep.ExtPageLength);
				else
					n = rep.Header.PageLength;

				for (i = 0; i < n; i++)
					fprintf(out_file, "%04x : %08x\n", i*4, get32(buf[i]));
			}
		}

		if (action == 2 || action == 4)
		{
			ioc_status = get16(rep.IOCStatus) & MPI_IOCSTATUS_MASK;

			if (ioc_status == MPI_IOCSTATUS_SUCCESS)
			{
				if (type > MPI_CONFIG_PAGETYPE_EXTENDED)
					length = get16(rep.ExtPageLength) * 4;
				else
					length = rep.Header.PageLength * 4;
			}
			else
				length = sizeof buf;

			memset (buf, 0, sizeof buf);
			((ConfigPageHeader_t *)buf)->PageVersion	= rep.Header.PageVersion;
			((ConfigPageHeader_t *)buf)->PageLength		= rep.Header.PageLength;
			((ConfigPageHeader_t *)buf)->PageNumber		= rep.Header.PageNumber;
			((ConfigPageHeader_t *)buf)->PageType		= rep.Header.PageType;
			if (type > MPI_CONFIG_PAGETYPE_EXTENDED)
			{
				((ConfigExtendedPageHeader_t *)buf)->ExtPageLength	= rep.ExtPageLength;
				((ConfigExtendedPageHeader_t *)buf)->ExtPageType	= rep.ExtPageType;
				t = 8;
			}
			else
				t = 4;

			while (TRUE)
			{
				if (in_file == stdin)
				{
					printf("Enter offset (%04x) and value, or ? for help, or RETURN to quit: ", t);
					n = getString(str, sizeof str, stdin);
				}
				else
				{
					if (fgets(str, sizeof str, in_file) == NULL)
					{
						n = 0;
					}
					else
					{
						str[sizeof str - 1] = '\0';
						n = strlen(str);
						if (n >= 1 && str[n-1] == '\n')
						{
							str[n-1] = '\0';
							n -= 1;
						}
					}
				}

				if (n == 0)
					break;

				offset	= 0;
				value	= 0;
				i = sscanf(str, "%x : %x%n\n", &offset, &value, &j);
				if (i == 1 && strchr(str, ':') == NULL)
				{
					i = sscanf(str, "%x %x%n\n", &offset, &value, &j);
					if (i == 1 && strchr(str, ' ') == NULL)
					{
						offset = t;
						sscanf(str, "%x%n", &value, &j);
						i = 2;
					}
				}

				if (i == 2 && j == n)
				{
					if (offset >= 0 && offset < (U32)length && (offset % 4) == 0)
					{
						buf[offset / 4] = value;
						t = offset + 4;

						if (t == length)
							break;

						continue;
					}
				}

				if (in_file == stdin)
				{
					if (n != 1 || str[0] != '?')
						printf("\nInvalid response!\n");
					printf("\nValid input is:  [<Offset> [:]] <Value>\n");
					printf("  Offset is an optional hex number between 0000 and %04x (multiple of 4 only)\n", length - 4);
					printf("  Value is a hex number\n\n");
				}
			}
		}

		if (action == 2 || action == 3 || action == 4)
		{
			req.Action			= action;
			req.ExtPageType		= rep.ExtPageType;
			req.ExtPageLength	= rep.ExtPageLength;
			req.Header			= rep.Header;

			if (action != 3 && type == MPI_CONFIG_PAGETYPE_MANUFACTURING && number == 2)
			{
				U8	 checksum = 0xa5;
				U8	*p = (U8 *)buf;

				p += 8;
				t = n * 4 - 8;
				switch (port->deviceId)
				{
				case MPI_MANUFACTPAGE_DEVICEID_FC919:   t -= 4;  break;
				case MPI_MANUFACTPAGE_DEVICEID_FC929:   t -= 4;  break;
				case MPI_MANUFACTPAGE_DEVICEID_FC919X:  t -= 3;  break;
				case MPI_MANUFACTPAGE_DEVICEID_FC929X:  t -= 3;  break;
				case MPI_MANUFACTPAGE_DEVICEID_FC939X:  t -= 3;  break;
				case MPI_MANUFACTPAGE_DEVICEID_FC949X:  t -= 3;  break;
				default:                                t = 0;   break;
				}
				if (t != 0)
				{
					for (i = 0; i < t; i++)
					{
						checksum += *p++;
					}
					*p = -checksum;
				}
			}

			if (type == MPI_CONFIG_PAGETYPE_MANUFACTURING)
				doIocInit(port, MPI_WHOINIT_MANUFACTURER);

			if (doMptCommand(port, &req, sizeof req - sizeof req.PageBufferSGE, &rep, sizeof rep,
							 NULL, 0, buf, sizeof buf, 10) != 1)
			{
				if (type == MPI_CONFIG_PAGETYPE_MANUFACTURING)
					doIocInit(port, port->whoInit);

				printf("Failed to write page %d/%d/%x\n\n", type, number, address);
				continue;
			}

			if (type == MPI_CONFIG_PAGETYPE_MANUFACTURING)
				doIocInit(port, port->whoInit);

			ioc_status = get16(rep.IOCStatus) & MPI_IOCSTATUS_MASK;

			fprintf(out_file, "\n%s %d %d %x -- IOCStatus = %04x (%s)\n",
					action_string[action], type, number, address,
					ioc_status, translateIocStatus(ioc_status));
		}
	}

	if (in_file != stdin)
		fclose(in_file);
	if (out_file != stdout)
		fclose(out_file);

	return 1;
}


int
doDiagnostics(MPT_PORT *port, int command)
{
	switch (command)
	{
	case 1:
		doInquiryTest(port);
		break;
	case 2:
		doWriteBufferReadBufferCompareTest(port);
		break;
	case 3:
		doReadTest(port);
		break;
	case 4:
		doWriteReadCompareTest(port);
		break;
	case 5:
		doWriteTest(port);
		break;
	case 6:
		doReadCompareTest(port);
		break;
	case 10:
		if (port->portType == MPI_PORTFACTS_PORTTYPE_FC)
		{
			doEchoTest(port);
			break;
		}
		printf("Invalid selection!\n");
		break;
	case 11:
		if (port->portType == MPI_PORTFACTS_PORTTYPE_FC)
		{
			doReadLinkErrorStatusTest(port);
			break;
		}
		printf("Invalid selection!\n");
		break;
	case 12:
		if (port->portType == MPI_PORTFACTS_PORTTYPE_FC)
		{
			doDisplayPortCounters(port);
			break;
		}
		printf("Invalid selection!\n");
		break;
	case 13:
		if (port->portType == MPI_PORTFACTS_PORTTYPE_FC)
		{
			doClearPortCounters(port);
			break;
		}
		printf("Invalid selection!\n");
		break;
	default:
		printf("Invalid selection!\n");
		break;
	}

	return 1;
}


void
generatePattern(int pattern, void *buf, int len)
{
	int				 i;
	int				 j;
	unsigned char	*buf1 = (unsigned char *)buf;
	unsigned short	*buf2 = (unsigned short *)buf;

	switch (pattern)
	{
	case 1:
		for (i = 0; i < len / 2; i++)
		{
			*buf1++ = 0x00;
			*buf1++ = 0xff;
		}
		break;

	case 2:
		for (i = 0; i < len / 2; i++)
		{
			*buf1++ = 0x55;
			*buf1++ = 0xaa;
		}
		break;

	case 3:
		for (i = 0; i < len; i++)
		{
			*buf1++ = i;
		}
		break;

	case 4:
		for (i = 0; i < len / 2; i++)
		{
			*buf1++ = 1 << (i & 7);
			*buf1++ = ~(1 << (i & 7));
		}
		break;

	case 5:
		for (i = 0; i < len / 4; i++)
		{
			*buf2++ = 0x0000;
			*buf2++ = 0xffff;
		}
		break;

	case 6:
		for (i = 0; i < len / 4; i++)
		{
			*buf2++ = 0x5555;
			*buf2++ = 0xaaaa;
		}
		break;

	case 7:
		for (i = 0; i < len / 2; i++)
		{
			*buf2++ = i;
		}
		break;

	case 8:
		for (i = 0; i < len / 4; i++)
		{
			*buf2++ = 1 << (i & 15);
			*buf2++ = ~(1 << (i & 15));
		}
		break;
	case 9:
		for (j = 0; j < len / 0x200; j++)
		{
			for (i = 0x0; i < 0xc0; i++)
				*buf1++ = 0x32;
			for (i = 0xc0; i < 0x1fe; i++)
				*buf1++ = 0x4a;
			for (i = 0x1fe; i < 0x200; i++)
				*buf1++ = 0x0;
		}
		break;
	case 10:
		for (j = 0; j < len / 0x200; j++)
		{
			for (i = 0x0; i < 0x200; i++)
				*buf1++ = 0xb5;
		}
		break;
	case 11:
		for (j = 0; j < len / 0x200; j++)
		{
			for (i = 0x0; i < 0x200; i++)
				*buf1++ = 0x4a;
		}
		break;
	}
}


void
format64bitDecimal(uint64_t number, char *buf, int len)
{
	int		i;
	int		part;
	char	temp[4];

	memset(buf, ' ', len);

	i = len - 1;

	while (TRUE)
	{
		part = (int)(number % 1000);
		number /= 1000;

		if (number == 0)
		{
			sprintf(temp, "%3d", part);
			buf[--i] = temp[2];
			buf[--i] = temp[1];
			buf[--i] = temp[0];
			break;
		}

		sprintf(temp, "%03d", part);

		buf[--i] = temp[2];
		buf[--i] = temp[1];
		buf[--i] = temp[0];
		buf[--i] = ',';
	}

	buf[len - 1] = '\0';
}


int
doInquiryTest(MPT_PORT *port)
{
	int				 bus;
	int				 target;
	int				 lun;
	int				 vpd;
	int				 page;
	unsigned char	 inq[255];
	int				 i;
	int				 j;
	int				 n;
	int				 t;
	char			 c[16];

	while (TRUE)
	{
		if (port->maxBuses > 1)
		{
			printf("Bus:  [0-%d or RETURN to quit] ", port->maxBuses - 1);
			bus = getNumberAnswer(0, port->maxBuses - 1, -1);
			if (bus < 0)
				return 1;
		}
		else
		{
			bus = 0;
		}

		printf("Target:  [0-%d or RETURN to quit] ", port->maxTargets - 1);
		target = getNumberAnswer(0, port->maxTargets - 1, -1);
		if (target < 0)
			return 1;

		printf("LUN:  [0-%d or RETURN to quit] ", port->maxLuns - 1);
		lun = getNumberAnswer(0, port->maxLuns - 1, -1);
		if (lun < 0)
			return 1;

		printf("VPD Page:  [00-FF or RETURN for normal Inquiry] ");
		while (TRUE)
		{
			t = getHexNumberAnswer(&page);
			if (t == 0)
			{
				vpd = 0;
				break;
			}

			if (page >= 0x00 && page <= 0xff)
			{
				vpd = 1;
				break;
			}

			printf("Invalid answer, try again: ");
		}

		printf("\n");

		if (vpd)
		{
			t = doInquiryVpdPage(port, bus, target, lun, page, inq, sizeof inq);
			n = inq[3] + 4;
		}
		else
		{
			t = doInquiry(port, bus, target, lun, inq, sizeof inq);
			n = inq[4] + 5;
		}

		if (t == 1)
		{
			if (vpd)
			{
				printf(" B___T___L  Page\n");
				printf("%2d %3d %3d   %02x\n\n",
					   bus, target, lun, page);
			}
			else
			{
				for (i = 8; i < 36; i++)
					if (!isprint(inq[i]))
						inq[i] = ' ';

				printf(" B___T___L  Type       Vendor   Product          Rev\n");
				printf("%2d %3d %3d  %-9s  %8.8s %16.16s %4.4s\n\n",
					   bus, target, lun, deviceType[inq[0] & 0x1f],
					   inq+8, inq+16, inq+32);
			}

			printf("%d bytes of Inquiry Data returned\n\n", n);

			for (i = 0, j = 0; i < n; i++, j++)
			{
				if (j == 0)
					printf("%04x : ", i);

				printf("%02x ", inq[i]);

				if (!isprint(inq[i]))
					c[j] = ' ';
				else
					c[j] = inq[i];

				if (j == 15)
				{
					printf("   ");
					for (j = 0; j < 16; j++)
					{
						printf("%c", c[j]);
					}
					printf("\n");
					j = -1;
				}
			}

			if (j != 0)
			{
				for (i = j; i < 16; i++)
					printf("   ");

				printf("   ");
				for (i = 0; i < j; i++)
				{
					printf("%c", c[i]);
				}
				printf("\n");
			}
		}
		else
		{
			printf("Inquiry failed\n");
		}

		printf("\n");
	}
}


int
doWriteBufferReadBufferCompareTest(MPT_PORT *port)
{
	int				 bus;
	int				 target;
	unsigned char	 inq[36];
	unsigned char	 buf[4];
	unsigned char	 *data_in;
	unsigned char	 *data_out;
	int				 i;
	int				 n;
	int				 mode;
	unsigned int	 size;
	DIAG_TARGET		 diag_targets[99];
	int				 pattern;
	int				 passes;
	int				 progress;

	showPortInfoHeader(port);

	printf("     B___T  Type       Vendor   Product          Rev     Buffer & Size\n");

	n = 0;
	for (bus = 0; bus < port->maxBuses; bus++)
	{
		for (target = 0; target < port->maxTargets; target++)
		{
			if (doInquiry(port, bus, target, 0, inq, sizeof inq) != 1)
				continue;

			if ((inq[0] & 0x1f) == 0x1f)
				continue;

			if ((inq[0] & 0xe0) == 0x20)
				continue;

			if ((inq[0] & 0xe0) == 0x60)
				continue;

			for (i = 8; i < 36; i++)
				if (!isprint(inq[i]))
					inq[i] = ' ';

			mode = 0x2;
			if (doReadBuffer(port, bus, target, 0, mode + 1, buf, sizeof buf) == 1)
			{
				size = min(0x4000, (buf[1] << 16) | (buf[2] << 8) | buf[3]);
				if (size < 4)
					continue;

				diag_targets[n].bus		= bus;
				diag_targets[n].target	= target;
				diag_targets[n].lun		= 0;
				diag_targets[n].mode	= mode;
				diag_targets[n].size	= size;

				n++;

				printf("%2d. %2d %3d  %-9s  %8.8s %16.16s %4.4s     Data  %5d\n",
					   n, bus, target, deviceType[inq[0] & 0x1f],
					   inq+8, inq+16, inq+32, size);

				if (n == 99)
					break;
			}

			mode = 0xa;
			if (doReadBuffer(port, bus, target, 0, mode + 1, buf, sizeof buf) == 1)
			{
				size = min(0x4000, (buf[1] << 16) | (buf[2] << 8) | buf[3]);
				if (size < 4)
					continue;

				diag_targets[n].bus		= bus;
				diag_targets[n].target	= target;
				diag_targets[n].lun		= 0;
				diag_targets[n].mode	= mode;
				diag_targets[n].size	= size;

				n++;

				printf("%2d. %2d %3d  %-9s  %8.8s %16.16s %4.4s     Echo  %5d\n",
					   n, bus, target, deviceType[inq[0] & 0x1f],
					   inq+8, inq+16, inq+32, size);
				
				if (n == 99)
					break;
			}
		}
		if (n == 99)
			break;
	}

	if (n == 0)
	{
		printf("\nNo devices are available for this test\n");
		return 1;
	}

	printf("\nSelect a device:  [1-%d or RETURN to quit] ", n);
	n = getNumberAnswer(1, n, 0);
	if (n == 0)
		return 1;
	n--;

	bus		= diag_targets[n].bus;
	target	= diag_targets[n].target;
	mode	= diag_targets[n].mode;

	while (TRUE)
	{
		size = diag_targets[n].size;

		printf("\n");
		printf(" 1.  Alternating, 8-Bit, 00 and FF\n");
		printf(" 2.  Alternating, 8-Bit, 55 and AA\n");
		printf(" 3.  Incrementing, 8-Bit\n");
		printf(" 4.  Walking 1s and 0s, 8-Bit\n");
		printf(" 5.  Alternating, 16-Bit, 0000 and FFFF\n");
		printf(" 6.  Alternating, 16-Bit, 5555 and AAAA\n");
		printf(" 7.  Incrementing, 16-Bit\n");
		printf(" 8.  Walking 1s and 0s, 16-Bit\n");
		printf("\nSelect a data pattern:  [1-8 or RETURN to quit] ");
		pattern = getNumberAnswer(1, 8, 0);
		if (pattern == 0)
			return 1;

		printf("Pattern length:  [1-%d or RETURN to quit] ", size);
		size = getNumberAnswer(1, size, 0);
		if (size == 0)
			return 1;

		printf("Number of iterations:  [1-1000000 or 0 for infinite or RETURN to quit] ");
		passes = getNumberAnswer(0, 1000000, -1);
		if (passes == -1)
			return 1;

		data_in = malloc(size);
		data_out = malloc(size);

		generatePattern(pattern, data_out, size);

		progress = max(10, passes / 10);

		printf("Testing started...\n");
		for (i = 1; i <= passes || passes == 0; i++)
		{
			if (doWriteBuffer(port, bus, target, 0, mode, data_out, size) != 1)
			{
				printf("W");
				continue;
			}

			if (doReadBuffer(port, bus, target, 0, mode, data_in, size) != 1)
			{
				printf("R");
				continue;
			}

			if (memcmp(data_in, data_out, size) != 0)
			{
				printf("C");
				continue;
			}

//			printf(".");

			if (passes == 0)
			{
				if (i % 100000 == 0)
				{
					printf(" %d", i);
					if (i == 1000000)
					{
						i = 0;
						printf("\n");
					}
					fflush(stdout);
				}
			}
			else if (i % progress == 0)
			{
				printf(" %d%% ", i * 100 / passes);
				fflush(stdout);
			}
		}
		printf("\nTesting ended...\n");

		free(data_in);
		free(data_out);
	}
}


int
doReadTest(MPT_PORT *port)
{
	int				 bus;
	int				 target;
	int				 lun;
	unsigned char	 inq[36];
	unsigned char	 cap[8];
	unsigned char	 *data_in;
	int				 i;
	int				 n;
	unsigned int	 mode;
	unsigned int	 size;
	DIAG_TARGET		 diag_targets[99];
	unsigned int	 lbn;
	int				 lbns;
	int				 len;
	int				 passes;
	int				 random;
	int				 progress;
	int				 eedp_mode = 0;

	showPortInfoHeader(port);

	printf("     B___T___L  Type       Vendor   Product          Rev   Mode  Disk Blocks\n");

	n = 0;
	for (bus = 0; bus < port->maxBuses; bus++)
	{
		for (target = 0; target < port->maxTargets; target++)
		{
			for (lun = 0; lun < port->maxLuns; lun++)
			{
				if (doInquiry(port, bus, target, lun, inq, sizeof inq) != 1)
				{
					if (lun == 0)
						break;
					else
						continue;
				}

				if ((inq[0] & 0x1f) == 0x1f)
					continue;

				if ((inq[0] & 0xe0) == 0x20)
					continue;

				if ((inq[0] & 0xe0) == 0x60)
					continue;

				for (i = 8; i < 36; i++)
					if (!isprint(inq[i]))
						inq[i] = ' ';

				if (doReadCapacity(port, bus, target, lun, cap, sizeof cap) != 1)
					continue;

				mode = (cap[4] << 24) | (cap[5] << 16) | (cap[6] << 8) | cap[7];
				size = (cap[0] << 24) | (cap[1] << 16) | (cap[2] << 8) | cap[3];

				diag_targets[n].bus		= bus;
				diag_targets[n].target	= target;
				diag_targets[n].lun		= lun;
				diag_targets[n].mode	= mode;
				diag_targets[n].size	= size;

				n++;

				printf("%2d. %2d %3d %3d  %-9s  %8.8s %16.16s %4.4s  %4d   %10d\n",
					   n, bus, target, lun, deviceType[inq[0] & 0x1f],
					   inq+8, inq+16, inq+32, mode, size + 1);

				if (n == 99)
					break;
			}
			if (n == 99)
				break;
		}
		if (n == 99)
			break;
	}

	if (n == 0)
	{
		printf("\nNo devices are available for this test\n");
		return 1;
	}

	printf("\nSelect a device:  [1-%d or RETURN to quit] ", n);
	n = getNumberAnswer(1, n, 0);
	if (n == 0)
		return 1;
	n--;

	bus		= diag_targets[n].bus;
	target	= diag_targets[n].target;
	lun		= diag_targets[n].lun;
	mode	= diag_targets[n].mode;
	size	= diag_targets[n].size;

	while (TRUE)
	{
		printf("\n");
		printf("Number of blocks per I/O:  [1-64 or RETURN to quit] ");
		lbns = getNumberAnswer(1, 64, 0);
		if (lbns == 0)
			return 1;

		printf("Number of iterations:  [1-1000000 or 0 for infinite or RETURN to quit] ");
		passes = getNumberAnswer(0, 1000000, -1);
		if (passes == -1)
			return 1;

		printf("Type of I/O:  [0=Sequential, 1=Random, default is 0] ");
		random = getNumberAnswer(0, 1, 0);

		len = lbns * ((eedp_mode == 0) ? mode : (mode & ~255));

		data_in = malloc(len);

		progress = max(10, passes / 10);

		lbn = 128;
		printf("Testing started...\n");
		for (i = 1; i <= passes || passes == 0; i++, lbn += lbns)
		{
			if (random)
			{
				lbn = rand();
				while (lbn + lbns >= size)
				{
					lbn /= 2;
				}
			}
			else
			{
				if (lbn + lbns >= size)
				{
					lbn = 128;
				}
			}

			if (doRead(port, bus, target, lun, lbn, lbns, eedp_mode, data_in, len) != 1)
			{
				printf("R");
				continue;
			}

//			printf(".");

			if (passes == 0)
			{
				if (i % 100000 == 0)
				{
					printf(" %d", i);
					if (i == 1000000)
					{
						i = 0;
						printf("\n");
					}
					fflush(stdout);
				}
			}
			else if (i % progress == 0)
			{
				printf(" %d%% ", i * 100 / passes);
				fflush(stdout);
			}
		}
		printf("\nTesting ended...\n");

		free(data_in);
	}
}


int
doWriteReadCompareTest(MPT_PORT *port)
{
	int				 bus;
	int				 target;
	int				 lun;
	unsigned char	 inq[36];
	unsigned char	 cap[8];
	unsigned char	 *data_in;
	unsigned char	 *data_out;
	int				 i;
	int				 n;
	unsigned int	 mode;
	unsigned int	 size;
	DIAG_TARGET		 diag_targets[99];
	int				 pattern;
	unsigned int	 lbn;
	int				 lbns;
	int				 len;
	int				 passes;
	int				 random;
	int				 progress;
	int				 eedp_mode = 0;

	showPortInfoHeader(port);

	printf("     B___T___L  Type       Vendor   Product          Rev   Mode  Disk Blocks\n");

	n = 0;
	for (bus = 0; bus < port->maxBuses; bus++)
	{
		for (target = 0; target < port->maxTargets; target++)
		{
			for (lun = 0; lun < port->maxLuns; lun++)
			{
				if (doInquiry(port, bus, target, lun, inq, sizeof inq) != 1)
				{
					if (lun == 0)
						break;
					else
						continue;
				}

				if ((inq[0] & 0x1f) == 0x1f)
					continue;

				if ((inq[0] & 0xe0) == 0x20)
					continue;

				if ((inq[0] & 0xe0) == 0x60)
					continue;

				for (i = 8; i < 36; i++)
					if (!isprint(inq[i]))
						inq[i] = ' ';

				if (doReadCapacity(port, bus, target, lun, cap, sizeof cap) != 1)
					continue;

				mode = (cap[4] << 24) | (cap[5] << 16) | (cap[6] << 8) | cap[7];
				size = (cap[0] << 24) | (cap[1] << 16) | (cap[2] << 8) | cap[3];

				diag_targets[n].bus		= bus;
				diag_targets[n].target	= target;
				diag_targets[n].lun		= lun;
				diag_targets[n].mode	= mode;
				diag_targets[n].size	= size;

				n++;

				printf("%2d. %2d %3d %3d  %-9s  %8.8s %16.16s %4.4s  %4d   %10d\n",
					   n, bus, target, lun, deviceType[inq[0] & 0x1f],
					   inq+8, inq+16, inq+32, mode, size + 1);

				if (n == 99)
					break;
			}
			if (n == 99)
				break;
		}
		if (n == 99)
			break;
	}

	if (n == 0)
	{
		printf("\nNo devices are available for this test\n");
		return 1;
	}

	printf("\nSelect a device:  [1-%d or RETURN to quit] ", n);
	n = getNumberAnswer(1, n, 0);
	if (n == 0)
		return 1;
	n--;

	bus		= diag_targets[n].bus;
	target	= diag_targets[n].target;
	lun		= diag_targets[n].lun;
	mode	= diag_targets[n].mode;
	size	= diag_targets[n].size;

	while (TRUE)
	{
		printf("\n");
		printf(" 1.  Alternating, 8-Bit, 00 and FF\n");
		printf(" 2.  Alternating, 8-Bit, 55 and AA\n");
		printf(" 3.  Incrementing, 8-Bit\n");
		printf(" 4.  Walking 1s and 0s, 8-Bit\n");
		printf(" 5.  Alternating, 16-Bit, 0000 and FFFF\n");
		printf(" 6.  Alternating, 16-Bit, 5555 and AAAA\n");
		printf(" 7.  Incrementing, 16-Bit\n");
		printf(" 8.  Walking 1s and 0s, 16-Bit\n");
		printf(" 9:  NCR's pattern\n");
		printf("10:  All B5\n");
		printf("11:  All 4A\n");
		printf("12:  Incrementing across iterations (00 through FF)\n");
		printf("\nSelect a data pattern:  [1-12 or RETURN to quit] ");
		pattern = getNumberAnswer(1, 12, 0);
		if (pattern == 0)
			return 1;

		printf("Number of blocks per I/O:  [1-64 or RETURN to quit] ");
		lbns = getNumberAnswer(1, 64, 0);
		if (lbns == 0)
			return 1;

		if (pattern == 12)
			printf("Number of iterations per pattern:  [1-1000000 or RETURN to quit] ");
		else
			printf("Number of iterations:  [1-1000000 or 0 for infinite or RETURN to quit] ");
		passes = getNumberAnswer(0, 1000000, -1);
		if (passes == -1)
			return 1;
		if (passes == 0 && pattern == 12)
			return 1;

		printf("Type of I/O:  [0=Sequential, 1=Random, default is 0] ");
		random = getNumberAnswer(0, 1, 0);

		len = lbns * ((eedp_mode == 0) ? mode : (mode & ~255));

		data_in = malloc(len);
		data_out = malloc(len);

		if (pattern == 12)
		{
			int		 j;
			int		 quit;

			printf("Stop pattern on Write, Read, or Compare error?  [Yes or No, default is Yes] ");
			quit = getYesNoAnswer(1);

			lbn = 128;
			printf("Testing started...\n");
			for (j = 0x00; j <= 0xFF; j++)
			{
				FCPortPage6_t	 FCPortPage6;
				uint64_t		 LipCount = 0;
				uint64_t		 LossOfSyncCount = 0;
				uint64_t		 LipCountDiff;
				uint64_t		 LossOfSyncCountDiff;

				if (getConfigPage(port, MPI_CONFIG_PAGETYPE_FC_PORT, 6, 0, &FCPortPage6, sizeof FCPortPage6) == 1)
				{
					LipCount = get64(FCPortPage6.LipCount);
					LossOfSyncCount = get64(FCPortPage6.LossOfSyncCount);
				}

				printf(" %02X ", j);
				fflush(stdout);

				memset(data_out, j, len);

				for (i = 1; i <= passes; i++, lbn += lbns)
				{
					if (random)
					{
						lbn = rand();
						while (lbn + lbns >= size)
						{
							lbn /= 2;
						}
					}
					else
					{
						if (lbn + lbns >= size)
						{
							lbn = 128;
						}
					}

					if (doWrite(port, bus, target, lun, lbn, lbns, eedp_mode, data_out, len) != 1)
					{
						if (quit)
						{
							printf("W at pass %d", i);
							break;
						}
						else
						{
							printf("W");
							continue;
						}
					}

					if (doRead(port, bus, target, lun, lbn, lbns, eedp_mode, data_in, len) != 1)
					{
						if (quit)
						{
							printf("R at pass %d", i);
							break;
						}
						else
						{
							printf("R");
							continue;
						}
					}

					if (memcmp(data_in, data_out, len) != 0)
					{
						if (quit)
						{
							printf("C at pass %d", i);
							break;
						}
						else
						{
							printf("C");
							continue;
						}
					}
				}

				if (getConfigPage(port, MPI_CONFIG_PAGETYPE_FC_PORT, 6, 0, &FCPortPage6, sizeof FCPortPage6) == 1)
				{
					LipCountDiff = get64(FCPortPage6.LipCount) - LipCount;
					LossOfSyncCountDiff = get64(FCPortPage6.LossOfSyncCount) - LossOfSyncCount;
					if (LipCountDiff != 0 || LossOfSyncCountDiff != 0)
					{
						printf("\nPattern %02X caused %s%d LIP%s and %s%d Loss%sOfSync\n", j,
							LipCountDiff > 1000 ? ">" : "", 
							LipCountDiff > 1000 ? 1000 : (int)LipCountDiff,
							LipCountDiff != 1 ? "s" : "",
							LossOfSyncCountDiff > 1000 ? ">" : "", 
							LossOfSyncCountDiff > 1000 ? 1000 : (int)LossOfSyncCountDiff,
							LossOfSyncCountDiff != 1 ? "es" : "");
					}
				}
			}
			printf("\nTesting ended...\n");
		}
		else
		{
			generatePattern(pattern, data_out, len);

			progress = max(10, passes / 10);

			lbn = 128;
			printf("Testing started...\n");
			for (i = 1; i <= passes || passes == 0; i++, lbn += lbns)
			{
				if (random)
				{
					lbn = rand();
					while (lbn + lbns >= size)
					{
						lbn /= 2;
					}
				}
				else
				{
					if (lbn + lbns >= size)
					{
						lbn = 128;
					}
				}

				if (doWrite(port, bus, target, lun, lbn, lbns, eedp_mode, data_out, len) != 1)
				{
					printf("W");
					continue;
				}

				if (doRead(port, bus, target, lun, lbn, lbns, eedp_mode, data_in, len) != 1)
				{
					printf("R");
					continue;
				}

				if (memcmp(data_in, data_out, len) != 0)
				{
					printf("C");
					continue;
				}

//				printf(".");

				if (passes == 0)
				{
					if (i % 100000 == 0)
					{
						printf(" %d", i);
						if (i == 1000000)
						{
							i = 0;
							printf("\n");
						}
						fflush(stdout);
					}
				}
				else if (i % progress == 0)
				{
					printf(" %d%% ", i * 100 / passes);
					fflush(stdout);
				}
			}
			printf("\nTesting ended...\n");
		}

		free(data_in);
		free(data_out);
	}
}


int
doWriteTest(MPT_PORT *port)
{
	int				 bus;
	int				 target;
	int				 lun;
	unsigned char	 inq[36];
	unsigned char	 cap[8];
	unsigned char	 *data_in;
	unsigned char	 *data_out;
	int				 i;
	int				 j;
	int				 n;
	unsigned int	 mode;
	unsigned int	 size;
	DIAG_TARGET		 diag_targets[99];
	U32				 tag;
	unsigned int	 lbn;
	int				 lbns;
	int				 len;
	int				 progress;
	int				 eedp_mode = 0;
	int				 quit;
	U32				*p;

	showPortInfoHeader(port);

	printf("     B___T___L  Type       Vendor   Product          Rev   Mode  Disk Blocks\n");

	n = 0;
	for (bus = 0; bus < port->maxBuses; bus++)
	{
		for (target = 0; target < port->maxTargets; target++)
		{
			for (lun = 0; lun < port->maxLuns; lun++)
			{
				if (doInquiry(port, bus, target, lun, inq, sizeof inq) != 1)
				{
					if (lun == 0)
						break;
					else
						continue;
				}

				if ((inq[0] & 0x1f) == 0x1f)
					continue;

				if ((inq[0] & 0xe0) == 0x20)
					continue;

				if ((inq[0] & 0xe0) == 0x60)
					continue;

				for (i = 8; i < 36; i++)
					if (!isprint(inq[i]))
						inq[i] = ' ';

				if (doReadCapacity(port, bus, target, lun, cap, sizeof cap) != 1)
					continue;

				mode = (cap[4] << 24) | (cap[5] << 16) | (cap[6] << 8) | cap[7];
				size = (cap[0] << 24) | (cap[1] << 16) | (cap[2] << 8) | cap[3];

				diag_targets[n].bus		= bus;
				diag_targets[n].target	= target;
				diag_targets[n].lun		= lun;
				diag_targets[n].mode	= mode;
				diag_targets[n].size	= size;

				n++;

				printf("%2d. %2d %3d %3d  %-9s  %8.8s %16.16s %4.4s  %4d   %10d\n",
					   n, bus, target, lun, deviceType[inq[0] & 0x1f],
					   inq+8, inq+16, inq+32, mode, size + 1);

				if (n == 99)
					break;
			}
			if (n == 99)
				break;
		}
		if (n == 99)
			break;
	}

	if (n == 0)
	{
		printf("\nNo devices are available for this test\n");
		return 1;
	}

	printf("\nSelect a device:  [1-%d or RETURN to quit] ", n);
	n = getNumberAnswer(1, n, 0);
	if (n == 0)
		return 1;
	n--;

	bus		= diag_targets[n].bus;
	target	= diag_targets[n].target;
	lun		= diag_targets[n].lun;
	mode	= diag_targets[n].mode;
	size	= diag_targets[n].size;

	printf("Tagging data value:  [00000000-FFFFFFFF or RETURN to quit] ");
	if (getHexNumberAnswer(&tag) == 0)
		return 1;

	printf("Number of blocks per I/O:  [1-64 or RETURN to quit] ");
	lbns = getNumberAnswer(1, 64, 0);
	if (lbns == 0)
		return 1;

	len = lbns * ((eedp_mode == 0) ? mode : (mode & ~255));

	data_in = malloc(len);
	data_out = malloc(len);

	printf("Stop pattern on Write error?  [Yes or No, default is Yes] ");
	quit = getYesNoAnswer(1);

	progress = ((size - 128) / 10) / lbns * lbns;

	printf("Testing started...\n");
	for (lbn = 128; lbn < size; i++, lbn += lbns)
	{
		p = (U32 *)data_out;

		for (i = 0; i < lbns; i++)
		{
			*p++ = set32(tag);
			for (j = 4; j < (int)((eedp_mode == 0) ? mode : (mode & ~255)); j += 4)
			{
				*p++ = set32(lbn + i);
			}
		}

		if (doWrite(port, bus, target, lun, lbn, lbns, eedp_mode, data_out, len) != 1)
		{
			if (quit)
			{
				printf("W at lbn %d", lbn);
				break;
			}
			else
			{
				printf("W");
				continue;
			}
		}

//		printf(".");

		if (lbn > 128 && (lbn - 128) % progress == 0)
		{
			printf(" %d%% ", (lbn - 128) * 10 / progress);
			fflush(stdout);
		}
	}
	printf("\nTesting ended...\n");

	free(data_in);
	free(data_out);

	return 1;
}


int
doReadCompareTest(MPT_PORT *port)
{
	int				 bus;
	int				 target;
	int				 lun;
	unsigned char	 inq[36];
	unsigned char	 cap[8];
	unsigned char	 *data_in;
	unsigned char	 *data_out;
	int				 i;
	int				 j;
	int				 n;
	unsigned int	 mode;
	unsigned int	 size;
	DIAG_TARGET		 diag_targets[99];
	U32				 tag;
	unsigned int	 lbn;
	int				 lbns;
	int				 len;
	int				 progress;
	int				 eedp_mode = 0;
	int				 quit;
	U32				*p;
	U32				*q;

	showPortInfoHeader(port);

	printf("     B___T___L  Type       Vendor   Product          Rev   Mode  Disk Blocks\n");

	n = 0;
	for (bus = 0; bus < port->maxBuses; bus++)
	{
		for (target = 0; target < port->maxTargets; target++)
		{
			for (lun = 0; lun < port->maxLuns; lun++)
			{
				if (doInquiry(port, bus, target, lun, inq, sizeof inq) != 1)
				{
					if (lun == 0)
						break;
					else
						continue;
				}

				if ((inq[0] & 0x1f) == 0x1f)
					continue;

				if ((inq[0] & 0xe0) == 0x20)
					continue;

				if ((inq[0] & 0xe0) == 0x60)
					continue;

				for (i = 8; i < 36; i++)
					if (!isprint(inq[i]))
						inq[i] = ' ';

				if (doReadCapacity(port, bus, target, lun, cap, sizeof cap) != 1)
					continue;

				mode = (cap[4] << 24) | (cap[5] << 16) | (cap[6] << 8) | cap[7];
				size = (cap[0] << 24) | (cap[1] << 16) | (cap[2] << 8) | cap[3];

				diag_targets[n].bus		= bus;
				diag_targets[n].target	= target;
				diag_targets[n].lun		= lun;
				diag_targets[n].mode	= mode;
				diag_targets[n].size	= size;

				n++;

				printf("%2d. %2d %3d %3d  %-9s  %8.8s %16.16s %4.4s  %4d   %10d\n",
					   n, bus, target, lun, deviceType[inq[0] & 0x1f],
					   inq+8, inq+16, inq+32, mode, size + 1);

				if (n == 99)
					break;
			}
			if (n == 99)
				break;
		}
		if (n == 99)
			break;
	}

	if (n == 0)
	{
		printf("\nNo devices are available for this test\n");
		return 1;
	}

	printf("\nSelect a device:  [1-%d or RETURN to quit] ", n);
	n = getNumberAnswer(1, n, 0);
	if (n == 0)
		return 1;
	n--;

	bus		= diag_targets[n].bus;
	target	= diag_targets[n].target;
	lun		= diag_targets[n].lun;
	mode	= diag_targets[n].mode;
	size	= diag_targets[n].size;

	printf("Tagging data value:  [00000000-FFFFFFFF or RETURN to quit] ");
	if (getHexNumberAnswer(&tag) == 0)
		return 1;

	printf("Number of blocks per I/O:  [1-64 or RETURN to quit] ");
	lbns = getNumberAnswer(1, 64, 0);
	if (lbns == 0)
		return 1;

	len = lbns * ((eedp_mode == 0) ? mode : (mode & ~255));

	data_in = malloc(len);
	data_out = malloc(len);

	printf("Stop pattern on Read or Compare error?  [Yes or No, default is Yes] ");
	quit = getYesNoAnswer(1);

	progress = ((size - 128) / 10) / lbns * lbns;

	printf("Testing started...\n");
	for (lbn = 128; lbn < size; i++, lbn += lbns)
	{
		p = (U32 *)data_out;

		for (i = 0; i < lbns; i++)
		{
			*p++ = set32(tag);
			for (j = 4; j < (int)((eedp_mode == 0) ? mode : (mode & ~255)); j += 4)
			{
				*p++ = set32(lbn + i);
			}
		}

		if (doRead(port, bus, target, lun, lbn, lbns, eedp_mode, data_in, len) != 1)
		{
			if (quit)
			{
				printf("R at lbn %d", lbn);
				break;
			}
			else
			{
				printf("R");
				continue;
			}
		}

		if (memcmp(data_in, data_out, len) != 0)
		{
			if (quit)
			{
				p = (U32 *)data_out;
				q = (U32 *)data_in;

				for (i = 0; i < lbns; i++)
				{
					for (j = 0; j < (int)((eedp_mode == 0) ? mode : (mode & ~255)); j += 4)
					{
						if (*p++ != *q++)
						{
							printf("C at lbn %d offset %x", lbn + i, j);
							i = lbns - 1;
							break;
						}
					}
				}
				break;
			}
			else
			{
				printf("C");
				continue;
			}
		}

//		printf(".");

		if (lbn > 128 && (lbn - 128) % progress == 0)
		{
			printf(" %d%% ", (lbn - 128) * 10 / progress);
			fflush(stdout);
		}
	}
	printf("\nTesting ended...\n");

	free(data_in);
	free(data_out);

	return 1;
}


int
doEchoTest(MPT_PORT *port)
{
	FCPortPage0_t				 FCPortPage0;
	FCDevicePage0_t				 FCDevicePage0;
	ExLinkServiceSendRequest_t	 req;
	ExLinkServiceSendReply_t	 rep;
	U32							 buf_in[32];
	U32							 buf_out[32];
	U32							 els;
	U32							 d_id;
	U32							 port_id;
	int							 len;
	int							 n;
	int							 i;
	char						*type;
	U32							 port_ids[99];
	int							 pattern;
	int							 passes;
	int							 progress;
	int							 ioc_status;
	U32							 code;

	if (getConfigPage(port, MPI_CONFIG_PAGETYPE_FC_PORT, 0, 0, &FCPortPage0, sizeof FCPortPage0) != 1)
		return 0;

	port_id = get32(FCPortPage0.PortIdentifier);

	printf("     Type                            WWNN              WWPN        PortId\n");

	n = 0;
	d_id = 0xffffff;
	while (TRUE)
	{
		if (getConfigPage(port, MPI_CONFIG_PAGETYPE_FC_DEVICE, 0, d_id,
						  &FCDevicePage0, sizeof FCDevicePage0) != 1)
			break;

		d_id = get32(FCDevicePage0.PortIdentifier);

		if (d_id == port_id)
		{
			type = port->chipName;
		}
		else if (FCDevicePage0.Protocol & MPI_FC_DEVICE_PAGE0_PROT_FCP_INITIATOR)
		{
			if (FCDevicePage0.Protocol & MPI_FC_DEVICE_PAGE0_PROT_FCP_TARGET)
				type = "FCP Initiator & Target";
			else
				type = "FCP Initiator";
		}
		else
		{
			if (FCDevicePage0.Protocol & MPI_FC_DEVICE_PAGE0_PROT_FCP_TARGET)
				type = "FCP Target";
			else
				type = "Non-FCP";
		}

		port_ids[n] = d_id;

		n++;

		printf("%2d.  %-24s  %08x%08x  %08x%08x  %06x\n", n, type,
			   get32(FCDevicePage0.WWNN.High), get32(FCDevicePage0.WWNN.Low),
			   get32(FCDevicePage0.WWPN.High), get32(FCDevicePage0.WWPN.Low),
			   d_id);

		if (n == 99)
			break;
	}

	if (n == 0)
	{
		printf("\nNo devices are available for this test\n");
		return 1;
	}

	printf("\nSelect a device:  [1-%d or RETURN to quit] ", n);
	n = getNumberAnswer(1, n, 0);
	if (n == 0)
		return 1;
	n--;

	d_id = port_ids[n];

	els = 0x10;

	while (TRUE)
	{
		printf("\n");
		printf(" 1.  Alternating, 8-Bit, 00 and FF\n");
		printf(" 2.  Alternating, 8-Bit, 55 and AA\n");
		printf(" 3.  Incrementing, 8-Bit\n");
		printf(" 4.  Walking 1s and 0s, 8-Bit\n");
		printf(" 5.  Alternating, 16-Bit, 0000 and FFFF\n");
		printf(" 6.  Alternating, 16-Bit, 5555 and AAAA\n");
		printf(" 7.  Incrementing, 16-Bit\n");
		printf(" 8.  Walking 1s and 0s, 16-Bit\n");
		printf("\nSelect a data pattern:  [1-8 or RETURN to quit] ");
		pattern = getNumberAnswer(1, 8, 0);
		if (pattern == 0)
			return 1;

		printf("Pattern length in words:  [2-32 or RETURN to quit] ");
		len = getNumberAnswer(2, 32, 0);
		if (len == 0)
			return 1;
		len *= 4;

		printf("Number of iterations:  [1-1000000 or 0 for infinite or RETURN to quit] ");
		passes = getNumberAnswer(0, 1000000, -1);
		if (passes == -1)
			return 1;

		memset(&req, 0, sizeof req);
		memset(&rep, 0, sizeof rep);

		req.Function			= MPI_FUNCTION_FC_EX_LINK_SRVC_SEND;
		req.MsgFlags_Did		= set32(d_id);
		req.ElsCommandCode		= set32(els);

		generatePattern(pattern, buf_out, len);

		buf_out[0] = set32(els);

		progress = max(10, passes / 10);

		printf("Testing started...\n");
		for (i = 1; i <= passes || passes == 0; i++)
		{
			buf_in[0] = 0;

			if (doMptCommand(port, &req, sizeof req - sizeof req.SGL, &rep, sizeof rep,
							 buf_in, sizeof buf_in, buf_out, len, 10) != 1)
			{
				printf("E");
				continue;
			}

			ioc_status = get16(rep.IOCStatus) & MPI_IOCSTATUS_MASK;

			if (ioc_status != MPI_IOCSTATUS_SUCCESS)
			{
				printf("\nSendELS failed, IOCStatus = %04x (%s)\n", ioc_status, translateIocStatus(ioc_status));
				return 0;
			}

			code = (get32(swap32(buf_in[0])) >> 24) & 0xff;

			if (code == 0x01)
			{
				printf("\nECHO ELS rejected\n");
				return 0;
			}

			if (code != 0x02)
			{
				printf("\nECHO ELS not accepted, code is %02x\n", code);
				return 0;
			}

			if (memcmp(buf_in + 1, buf_out + 1, len - 4) != 0)
			{
				printf("C");
				continue;
			}

//			printf(".");

			if (passes == 0)
			{
				if (i % 100000 == 0)
				{
					printf(" %d", i);
					if (i == 1000000)
					{
						i = 0;
						printf("\n");
					}
					fflush(stdout);
				}
			}
			else if (i % progress == 0)
			{
				printf(" %d%% ", i * 100 / passes);
				fflush(stdout);
			}
		}
		printf("\nTesting ended...\n");
	}
}


int
doReadLinkErrorStatusTest(MPT_PORT *port)
{
	FCPortPage0_t				 FCPortPage0;
	FCDevicePage0_t				 FCDevicePage0;
	ExLinkServiceSendRequest_t	 req;
	ExLinkServiceSendReply_t	 rep;
	U32							 buf_in[7];
	U32							 buf_out[2];
	U32							 els;
	U32							 d_id;
	U32							 port_id;
	int							 n;
	char						*type;
	U32							 port_ids[99];
	int							 ioc_status;
	U32							 code;

	if (getConfigPage(port, MPI_CONFIG_PAGETYPE_FC_PORT, 0, 0, &FCPortPage0, sizeof FCPortPage0) != 1)
		return 0;

	port_id = get32(FCPortPage0.PortIdentifier);

	printf("     Type                            WWNN              WWPN        PortId\n");

	n = 0;
	d_id = 0xffffff;
	while (TRUE)
	{
		if (getConfigPage(port, MPI_CONFIG_PAGETYPE_FC_DEVICE, 0, d_id,
						  &FCDevicePage0, sizeof FCDevicePage0) != 1)
			break;

		d_id = get32(FCDevicePage0.PortIdentifier);

		if (d_id == port_id)
		{
			type = port->chipName;
		}
		else if (FCDevicePage0.Protocol & MPI_FC_DEVICE_PAGE0_PROT_FCP_INITIATOR)
		{
			if (FCDevicePage0.Protocol & MPI_FC_DEVICE_PAGE0_PROT_FCP_TARGET)
				type = "FCP Initiator & Target";
			else
				type = "FCP Initiator";
		}
		else
		{
			if (FCDevicePage0.Protocol & MPI_FC_DEVICE_PAGE0_PROT_FCP_TARGET)
				type = "FCP Target";
			else
				type = "Non-FCP";
		}

		port_ids[n] = d_id;

		n++;

		printf("%2d.  %-24s  %08x%08x  %08x%08x  %06x\n", n, type,
			   get32(FCDevicePage0.WWNN.High), get32(FCDevicePage0.WWNN.Low),
			   get32(FCDevicePage0.WWPN.High), get32(FCDevicePage0.WWPN.Low),
			   d_id);

		if (n == 99)
			break;
	}

	if (n == 0)
	{
		printf("\nNo devices are available for this test\n");
		return 1;
	}

	printf("\nSelect a device:  [1-%d or RETURN to quit] ", n);
	n = getNumberAnswer(1, n, 0);
	if (n == 0)
		return 1;
	n--;

	printf("\n");

	d_id = port_ids[n];

	els = 0x0f;

	memset(&req, 0, sizeof req);
	memset(&rep, 0, sizeof rep);

	req.Function			= MPI_FUNCTION_FC_EX_LINK_SRVC_SEND;
	req.MsgFlags_Did		= set32(d_id);
	req.ElsCommandCode		= set32(els);

	buf_out[0] = set32(els);
	buf_out[1] = set32(swap32(d_id));

	buf_in[0] = 0;

	if (doMptCommand(port, &req, sizeof req - sizeof req.SGL, &rep, sizeof rep,
					 buf_in, sizeof buf_in, buf_out, sizeof buf_out, 10) == 1)
	{
		char	 buf[32];

		ioc_status = get16(rep.IOCStatus) & MPI_IOCSTATUS_MASK;

		if (ioc_status != MPI_IOCSTATUS_SUCCESS)
		{
			printf("SendELS failed, IOCStatus = %04x (%s)\n", ioc_status, translateIocStatus(ioc_status));
			return 0;
		}

		code = (get32(swap32(buf_in[0])) >> 24) & 0xff;

		if (code == 0x01)
		{
			printf("RLS ELS rejected\n");
			return 0;
		}

		if (code != 0x02)
		{
			printf("RLS ELS not accepted, code is %02x\n", code);
			return 0;
		}

		format64bitDecimal(get32(swap32(buf_in[1])), buf, sizeof buf);
		printf("Link Failure Count              %32s\n", buf);

		format64bitDecimal(get32(swap32(buf_in[2])), buf, sizeof buf);
		printf("Loss Of Sync Count              %32s\n", buf);

		format64bitDecimal(get32(swap32(buf_in[3])), buf, sizeof buf);
		printf("Loss Of Signal Count            %32s\n", buf);

		format64bitDecimal(get32(swap32(buf_in[4])), buf, sizeof buf);
		printf("Primitive Sequence Error Count  %32s\n", buf);

		format64bitDecimal(get32(swap32(buf_in[5])), buf, sizeof buf);
		printf("Invalid Tx Word Count           %32s\n", buf);

		format64bitDecimal(get32(swap32(buf_in[6])), buf, sizeof buf);
		printf("Invalid CRC Count               %32s\n", buf);
	}

	return 1;
}


int
doDisplayPortCounters(MPT_PORT *port)
{
	FCPortPage6_t	 FCPortPage6;

	if (getConfigPage(port, MPI_CONFIG_PAGETYPE_FC_PORT, 6, 0, &FCPortPage6, sizeof FCPortPage6) == 1)
	{
		char		 buf[32];
		uint64_t	 time;
		int			 seconds;
		int			 minutes;
		int			 hours;
		int			 days;

		time = get64(FCPortPage6.TimeSinceReset);
		seconds	= (int)(time / 1000000) % 60;
		minutes	= (int)(time / 1000000 / 60) % 60;
		hours	= (int)(time / 1000000 / 60 / 60) % 24;
		days	= (int)(time / 1000000 / 60 / 60 / 24);
		sprintf(buf, "%d day%s + %02d:%02d:%02d", days, days == 1 ? "" : "s", hours, minutes, seconds);
		printf("Time Since Reset                %32s\n\n", buf);

		format64bitDecimal(get64(FCPortPage6.TxFrames), buf, sizeof buf);
		printf("Tx Frames                       %32s\n", buf);

		format64bitDecimal(get64(FCPortPage6.RxFrames), buf, sizeof buf);
		printf("Rx Frames                       %32s\n\n", buf);

		format64bitDecimal(get64(FCPortPage6.TxWords), buf, sizeof buf);
		printf("Tx Words                        %32s\n", buf);

		format64bitDecimal(get64(FCPortPage6.RxWords), buf, sizeof buf);
		printf("Rx Words                        %32s\n\n", buf);

		format64bitDecimal(get64(FCPortPage6.LipCount), buf, sizeof buf);
		printf("LIP Count                       %32s\n", buf);

		format64bitDecimal(get64(FCPortPage6.NosCount), buf, sizeof buf);
		printf("NOS Count                       %32s\n\n", buf);

		format64bitDecimal(get64(FCPortPage6.ErrorFrames), buf, sizeof buf);
		printf("Error Frames                    %32s\n", buf);

		format64bitDecimal(get64(FCPortPage6.DumpedFrames), buf, sizeof buf);
		printf("Dumped Frames                   %32s\n\n", buf);

		format64bitDecimal(get64(FCPortPage6.LinkFailureCount), buf, sizeof buf);
		printf("Link Failure Count              %32s\n", buf);

		format64bitDecimal(get64(FCPortPage6.LossOfSyncCount), buf, sizeof buf);
		printf("Loss Of Sync Count              %32s\n", buf);

		format64bitDecimal(get64(FCPortPage6.LossOfSignalCount), buf, sizeof buf);
		printf("Loss Of Signal Count            %32s\n", buf);

		format64bitDecimal(get64(FCPortPage6.PrimativeSeqErrCount), buf, sizeof buf);
		printf("Primitive Sequence Error Count  %32s\n", buf);

		format64bitDecimal(get64(FCPortPage6.InvalidTxWordCount), buf, sizeof buf);
		printf("Invalid Tx Word Count           %32s\n", buf);

		format64bitDecimal(get64(FCPortPage6.InvalidCrcCount), buf, sizeof buf);
		printf("Invalid CRC Count               %32s\n\n", buf);

		format64bitDecimal(get64(FCPortPage6.FcpInitiatorIoCount), buf, sizeof buf);
		printf("FCP Initiator I/O Count         %32s\n", buf);
	}

	return 1;
}


int
doClearPortCounters(MPT_PORT *port)
{
	ConfigReply_t	 rep;
	FCPortPage6_t	 FCPortPage6;

	if (getConfigPageHeader(port, MPI_CONFIG_PAGETYPE_FC_PORT, 6, 0, &rep) != 1)
		return 0;

	FCPortPage6.Header = rep.Header;

	if (setConfigPage(port, MPI_CONFIG_PAGETYPE_FC_PORT, 6, 0, &FCPortPage6, sizeof FCPortPage6) != 1)
		printf("Failed to clear port counters!\n");
	else
		printf("Port counters have been cleared\n");

	return 1;
}


int
doRaidActions(MPT_PORT *port, int command)
{
	IOCPage2_t		*IOCPage2;
	ConfigReply_t	 rep;
	int				 length;

	if (getConfigPageHeader(port, MPI_CONFIG_PAGETYPE_IOC, 2, 0, &rep) != 1)
	{
		printf("RAID is not supported on this port\n");
		return 0;
	}

	length = rep.Header.PageLength * 4;
	IOCPage2 = malloc(length);

	if (getConfigPage(port, MPI_CONFIG_PAGETYPE_IOC, 2, 0, IOCPage2, length) != 1)
		return 0;

	if (IOCPage2->MaxVolumes == 0)
	{
		printf("RAID is not supported on this port\n");
		return 0;
	}

	switch (command)
	{
	case 1:
		doShowVolumes(port, IOCPage2);
		break;
	case 2:
		doShowPhysDisks(port, IOCPage2);
		break;
	case 3:
		doGetVolumeState(port, IOCPage2);
		break;
	case 10:
		doModifyVolume(port, IOCPage2, MPI_RAID_ACTION_DISABLE_VOLUME, "disabled");
		break;
	case 11:
		doModifyVolume(port, IOCPage2, MPI_RAID_ACTION_ENABLE_VOLUME, "enabled");
		break;
	case 12:
		doModifyVolume(port, IOCPage2, MPI_RAID_ACTION_INACTIVATE_VOLUME, "inactivated");
		break;
	case 13:
		doModifyVolume(port, IOCPage2, MPI_RAID_ACTION_ACTIVATE_VOLUME, "activated");
		break;
	case 20:
		doModifyPhysDisk(port, IOCPage2, MPI_RAID_ACTION_PHYSDISK_OFFLINE, "offlined");
		break;
	case 21:
		doModifyPhysDisk(port, IOCPage2, MPI_RAID_ACTION_PHYSDISK_ONLINE, "onlined");
		break;
	case 22:
		doModifyPhysDisk(port, IOCPage2, MPI_RAID_ACTION_FAIL_PHYSDISK, "failed");
		break;
	case 23:
		doModifyPhysDisk(port, IOCPage2, MPI_RAID_ACTION_REPLACE_PHYSDISK, "replaced");
		break;
	case 24:
		doModifyPhysDisk(port, IOCPage2, MPI_RAID_ACTION_QUIESCE_PHYS_IO, "quiesced");
		break;
	case 25:
		doModifyPhysDisk(port, IOCPage2, MPI_RAID_ACTION_ENABLE_PHYS_IO, "unquiesced");
		break;
	case 30:
		doCreateVolume(port, IOCPage2);
		break;
	case 31:
		doDeleteVolume(port, IOCPage2);
		break;
	case 32:
		doVolumeSettings(port, IOCPage2);
		break;
	case 40:
		doCreatePhysDisk(port, IOCPage2);
		break;
	case 41:
		doModifyPhysDisk(port, IOCPage2, MPI_RAID_ACTION_DELETE_PHYSDISK, "deleted");
		break;
	case 42:
		doPhysDiskSettings(port, IOCPage2);
		break;
	default:
		printf("Invalid selection!\n");
		break;
	}

	return 1;
}


int
doShowVolumes(MPT_PORT *port, IOCPage2_t *IOCPage2)
{
	IOCPage3_t			*IOCPage3;
	RaidVolumePage0_t	*RaidVolumePage0;
	ConfigReply_t		 rep;
	int					 b_t;
	int					 i;
	int					 j;
	int					 k;
	int					 nv;
	int					 nd;
	int					 t1;
	int					 t2;
	int					 length;

	if (getConfigPageHeader(port, MPI_CONFIG_PAGETYPE_IOC, 3, 0, &rep) != 1)
		return 0;

	length = rep.Header.PageLength * 4;
	IOCPage3 = malloc(length);

	if (getConfigPage(port, MPI_CONFIG_PAGETYPE_IOC, 3, 0, IOCPage3, length) != 1)
		return 0;

	nv = IOCPage2->NumActiveVolumes;
	nd = IOCPage2->NumActivePhysDisks;
	printf("%d volume%s active, %d physical disk%s active\n",
		   nv, nv == 1 ? " is" : "s are", nd, nd == 1 ? " is" : "s are");

	for (i = 0; i < nv; i++)
	{
		printf("\nVolume %d is Bus %d Target %d, Type %s%s\n",
			   i, IOCPage2->RaidVolume[i].VolumeBus, IOCPage2->RaidVolume[i].VolumeID,
			   IOCPage2->RaidVolume[i].VolumeType == MPI_RAID_VOL_TYPE_IS ? "IS (Integrated Striping)" :
			   IOCPage2->RaidVolume[i].VolumeType == MPI_RAID_VOL_TYPE_IME ? "IME (Integrated Mirroring Extended)" :
			   IOCPage2->RaidVolume[i].VolumeType == MPI_RAID_VOL_TYPE_IM ? "IM (Integrated Mirroring)" : "Unknown",
			   IOCPage2->RaidVolume[i].Flags & MPI_IOCPAGE2_FLAG_VOLUME_INACTIVE ? ", inactive" : "");

		if (IOCPage2->RaidVolume[i].VolumePageNumber == 0)
		{
			b_t = (IOCPage2->RaidVolume[i].VolumeBus << 8) + IOCPage2->RaidVolume[i].VolumeID;
			if (getConfigPageHeader(port, MPI_CONFIG_PAGETYPE_RAID_VOLUME, 0, b_t, &rep) != 1)
				continue;

			length = rep.Header.PageLength * 4;
			RaidVolumePage0 = malloc(length);

			if (getConfigPage(port, MPI_CONFIG_PAGETYPE_RAID_VOLUME, 0, b_t, RaidVolumePage0, length) != 1)
				continue;

			t1 = RaidVolumePage0->VolumeStatus.State;
			t2 = RaidVolumePage0->VolumeStatus.Flags;
			printf("  Volume State:  %s%s%s%s\n",
				   t1 == MPI_RAIDVOL0_STATUS_STATE_OPTIMAL ? "optimal" :
				   t1 == MPI_RAIDVOL0_STATUS_STATE_DEGRADED ? "degraded" :
				   t1 == MPI_RAIDVOL0_STATUS_STATE_FAILED ? "failed" : "unknown",
				   t2 & MPI_RAIDVOL0_STATUS_FLAG_ENABLED ? ", enabled" : ", disabled",
				   t2 & MPI_RAIDVOL0_STATUS_FLAG_QUIESCED ? ", quiesced" : "",
				   t2 & MPI_RAIDVOL0_STATUS_FLAG_RESYNC_IN_PROGRESS ? ", resync in progress" : "");

			t1 = get16(RaidVolumePage0->VolumeSettings.Settings);
			t2 = RaidVolumePage0->VolumeSettings.HotSparePool;
			printf("  Volume Settings:  write caching %s%s%s%s\n",
				   t1 & MPI_RAIDVOL0_SETTING_WRITE_CACHING_ENABLE ? "enabled" : "disabled",
				   t1 & MPI_RAIDVOL0_SETTING_OFFLINE_ON_SMART ? ", offline on SMART data" : "",
				   t1 & MPI_RAIDVOL0_SETTING_AUTO_CONFIGURE ? ", auto configure" : "",
				   t1 & MPI_RAIDVOL0_SETTING_PRIORITY_RESYNC ? ", priority resync" : "");
			if (t2 != 0)
				printf("  Volume draws from Hot Spare Pools: %s%s%s%s%s%s%s%s\n",
					   t2 & MPI_RAID_HOT_SPARE_POOL_0 ? " 0" : "",
					   t2 & MPI_RAID_HOT_SPARE_POOL_1 ? " 1" : "",
					   t2 & MPI_RAID_HOT_SPARE_POOL_2 ? " 2" : "",
					   t2 & MPI_RAID_HOT_SPARE_POOL_3 ? " 3" : "",
					   t2 & MPI_RAID_HOT_SPARE_POOL_4 ? " 4" : "",
					   t2 & MPI_RAID_HOT_SPARE_POOL_5 ? " 5" : "",
					   t2 & MPI_RAID_HOT_SPARE_POOL_6 ? " 6" : "",
					   t2 & MPI_RAID_HOT_SPARE_POOL_7 ? " 7" : "");

			printf("  Volume Size %d MB, Stripe Size %d KB, %d Members\n",
				   (get32(RaidVolumePage0->MaxLBA) + 1) / 2048, get32(RaidVolumePage0->StripeSize) / 2,
				   RaidVolumePage0->NumPhysDisks);

			for (j = 0; j < RaidVolumePage0->NumPhysDisks; j++)
			{
				for (k = 0; k < nd; k++)
				{
					if (IOCPage3->PhysDisk[k].PhysDiskNum == RaidVolumePage0->PhysDisk[j].PhysDiskNum)
					{
						if (IOCPage2->RaidVolume[i].VolumeType == MPI_RAID_VOL_TYPE_IM)
							printf("  %s is PhysDisk %d (Bus %d Target %d)\n",
								   RaidVolumePage0->PhysDisk[j].PhysDiskMap == MPI_RAIDVOL0_PHYSDISK_PRIMARY ? "Primary" :
								   RaidVolumePage0->PhysDisk[j].PhysDiskMap == MPI_RAIDVOL0_PHYSDISK_SECONDARY ? "Secondary" :
								   "Member", RaidVolumePage0->PhysDisk[j].PhysDiskNum,
								   IOCPage3->PhysDisk[k].PhysDiskBus, IOCPage3->PhysDisk[k].PhysDiskID);
						else
							printf("  Member %d is PhysDisk %d (Bus %d Target %d)\n",
								   RaidVolumePage0->PhysDisk[j].PhysDiskMap, RaidVolumePage0->PhysDisk[j].PhysDiskNum,
								   IOCPage3->PhysDisk[k].PhysDiskBus, IOCPage3->PhysDisk[k].PhysDiskID);
						break;
					}
				}
				if (k < nd)
					continue;

				if (IOCPage2->RaidVolume[i].VolumeType == MPI_RAID_VOL_TYPE_IM)
					printf("  %s is PhysDisk %d\n",
						   RaidVolumePage0->PhysDisk[j].PhysDiskMap == MPI_RAIDVOL0_PHYSDISK_PRIMARY ? "Primary" :
						   RaidVolumePage0->PhysDisk[j].PhysDiskMap == MPI_RAIDVOL0_PHYSDISK_SECONDARY ? "Secondary" :
						   "Member", RaidVolumePage0->PhysDisk[j].PhysDiskNum);
				else
					printf("  Member %d is PhysDisk %d\n",
						   RaidVolumePage0->PhysDisk[j].PhysDiskMap, RaidVolumePage0->PhysDisk[j].PhysDiskNum);
			}

			free(RaidVolumePage0);
		}
	}

	free(IOCPage2);
	free(IOCPage3);

	return 1;
}


int
doShowPhysDisks(MPT_PORT *port, IOCPage2_t *IOCPage2)
{
	RaidPhysDiskPage0_t	*RaidPhysDiskPage0;
	ConfigReply_t		 rep;
	int					 i;
	int					 nv;
	int					 nd;
	int					 t1;
	int					 t2;
	int					 length;

	nv = IOCPage2->NumActiveVolumes;
	nd = IOCPage2->NumActivePhysDisks;
	printf("%d volume%s active, %d physical disk%s active\n",
		   nv, nv == 1 ? " is" : "s are", nd, nd == 1 ? " is" : "s are");

	for (i = 0; i < nd; i++)
	{
		if (getConfigPageHeader(port, MPI_CONFIG_PAGETYPE_RAID_PHYSDISK, 0, i, &rep) != 1)
			continue;

		length = rep.Header.PageLength * 4;
		RaidPhysDiskPage0 = malloc(length);

		if (getConfigPage(port, MPI_CONFIG_PAGETYPE_RAID_PHYSDISK, 0, i, RaidPhysDiskPage0, length) != 1)
			continue;

		printf("\nPhysDisk %d is Bus %d Target %d\n",
			   i, RaidPhysDiskPage0->PhysDiskBus, RaidPhysDiskPage0->PhysDiskID);

		t1 = RaidPhysDiskPage0->PhysDiskStatus.State;
		t2 = RaidPhysDiskPage0->PhysDiskStatus.Flags;
		printf("  PhysDisk State:  %s%s%s\n",
			   t1 == MPI_PHYSDISK0_STATUS_ONLINE ? "online" :
			   t1 == MPI_PHYSDISK0_STATUS_MISSING ? "missing" :
			   t1 == MPI_PHYSDISK0_STATUS_NOT_COMPATIBLE ? "not compatible" :
			   t1 == MPI_PHYSDISK0_STATUS_FAILED ? "failed" :
			   t1 == MPI_PHYSDISK0_STATUS_INITIALIZING ? "initializing" :
			   t1 == MPI_PHYSDISK0_STATUS_OFFLINE_REQUESTED ? "offline requested" :
			   t1 == MPI_PHYSDISK0_STATUS_FAILED_REQUESTED ? "failed requested" :
			   t1 == MPI_PHYSDISK0_STATUS_OTHER_OFFLINE ? "offline" : "unknown",
			   t2 & MPI_PHYSDISK0_STATUS_FLAG_OUT_OF_SYNC ? ", out of sync" : "",
			   t2 & MPI_PHYSDISK0_STATUS_FLAG_QUIESCED ? ", quiesced" : "");

		t1 = RaidPhysDiskPage0->PhysDiskSettings.PhysDiskSettings;
		t2 = RaidPhysDiskPage0->PhysDiskSettings.HotSparePool;
//		printf("  PhysDisk Settings:  \n");
		if (t2 != 0)
			printf("  PhysDisk belongs to Hot Spare Pools: %s%s%s%s%s%s%s%s\n",
				   t2 & MPI_RAID_HOT_SPARE_POOL_0 ? " 0" : "",
				   t2 & MPI_RAID_HOT_SPARE_POOL_1 ? " 1" : "",
				   t2 & MPI_RAID_HOT_SPARE_POOL_2 ? " 2" : "",
				   t2 & MPI_RAID_HOT_SPARE_POOL_3 ? " 3" : "",
				   t2 & MPI_RAID_HOT_SPARE_POOL_4 ? " 4" : "",
				   t2 & MPI_RAID_HOT_SPARE_POOL_5 ? " 5" : "",
				   t2 & MPI_RAID_HOT_SPARE_POOL_6 ? " 6" : "",
				   t2 & MPI_RAID_HOT_SPARE_POOL_7 ? " 7" : "");

		printf("  PhysDisk Size %d MB, Inquiry Data:  %8.8s %16.16s %4.4s\n",
			   (get32(RaidPhysDiskPage0->MaxLBA) + 1) / 2048, RaidPhysDiskPage0->InquiryData.VendorID,
			   RaidPhysDiskPage0->InquiryData.ProductID, RaidPhysDiskPage0->InquiryData.ProductRevLevel);

		t1 = get16(RaidPhysDiskPage0->ErrorData.ErrorCount);
		t2 = get16(RaidPhysDiskPage0->ErrorData.SmartCount);
		if (t1 != 0)
			printf("  Error Count %d, Last Error:  Command = %02Xh, Key = %d, ASC/ASCQ = %02Xh/%02Xh\n",
				   t1, RaidPhysDiskPage0->ErrorData.ErrorCdbByte, RaidPhysDiskPage0->ErrorData.ErrorSenseKey,
				   RaidPhysDiskPage0->ErrorData.ErrorASC, RaidPhysDiskPage0->ErrorData.ErrorASCQ);
		if (t2 != 0)
			printf("  SMART Data Count %d, Last SMART Data:  ASC/ASCQ = %02Xh/%02Xh\n",
				   t2, RaidPhysDiskPage0->ErrorData.SmartASC, RaidPhysDiskPage0->ErrorData.SmartASCQ);

		free(RaidPhysDiskPage0);
	}

	free(IOCPage2);

	return 1;
}


int
doGetVolumeState(MPT_PORT *port, IOCPage2_t *IOCPage2)
{
	MpiRaidActionRequest_t	 req;
	struct {
		MpiRaidActionReply_t	 rep;
		MpiRaidVolIndicator_t	 data;
	}						 rep;
	MpiRaidVolIndicator_t	*data;
	RaidVol0Status_t		*status;
	int						 volume;
	unsigned int			 size;
	unsigned int			 done;
	int						 t1;
	int						 t2;

	if (IOCPage2->NumActiveVolumes == 0)
	{
		printf("No active volumes\n");
		return 1;
	}

	if (IOCPage2->MaxVolumes > 1)
	{
		printf("Volume:  [0-%d or RETURN to quit] ", IOCPage2->MaxVolumes - 1);
		volume = getNumberAnswer(0, IOCPage2->MaxVolumes - 1, -1);
		if (volume < 0)
			return 1;
	}
	else
	{
		volume = 0;
	}

	memset(&req, 0, sizeof req);
	memset(&rep, 0, sizeof rep);

	req.Function			= MPI_FUNCTION_RAID_ACTION;
	req.Action				= MPI_RAID_ACTION_INDICATOR_STRUCT;
	req.VolumeBus			= IOCPage2->RaidVolume[volume].VolumeBus;
	req.VolumeID			= IOCPage2->RaidVolume[volume].VolumeID;

	if (doMptCommandCheck(port, &req, sizeof req, &rep, sizeof rep, NULL, 0, NULL, 0, 10) == 1)
	{
		data = (MpiRaidVolIndicator_t *)&rep.rep.ActionData;
		status = (RaidVol0Status_t *)&rep.rep.VolumeStatus;

		t1 = status->State;
		t2 = status->Flags;
		printf("Volume %d State:  %s%s%s%s\n", volume,
			   t1 == MPI_RAIDVOL0_STATUS_STATE_OPTIMAL ? "optimal" :
			   t1 == MPI_RAIDVOL0_STATUS_STATE_DEGRADED ? "degraded" :
			   t1 == MPI_RAIDVOL0_STATUS_STATE_FAILED ? "failed" : "unknown",
			   t2 & MPI_RAIDVOL0_STATUS_FLAG_ENABLED ? ", enabled" : ", disabled",
			   t2 & MPI_RAIDVOL0_STATUS_FLAG_QUIESCED ? ", quiesced" : "",
			   t2 & MPI_RAIDVOL0_STATUS_FLAG_RESYNC_IN_PROGRESS ? ", resync in progress" : "");

		if (t2 & MPI_RAIDVOL0_STATUS_FLAG_RESYNC_IN_PROGRESS)
		{
			size = get32(data->TotalBlocks.Low);
			done = get32(data->BlocksRemaining.Low);

			printf("Resync Progress:  total blocks %d, blocks remaining %d, %d%%\n",
				   size, done, done / (size / 100));
		}
	}

	return 1;
}


int
doModifyVolume(MPT_PORT *port, IOCPage2_t *IOCPage2, int action, char *string)
{
	MpiRaidActionRequest_t	 req;
	MpiRaidActionReply_t	 rep;
	int						 volume;

	if (IOCPage2->NumActiveVolumes == 0)
	{
		printf("No active volumes\n");
		return 1;
	}

	if (IOCPage2->MaxVolumes > 1)
	{
		printf("Volume:  [0-%d or RETURN to quit] ", IOCPage2->MaxVolumes - 1);
		volume = getNumberAnswer(0, IOCPage2->MaxVolumes - 1, -1);
		if (volume < 0)
			return 1;
		printf("\n");
	}
	else
	{
		volume = 0;
	}

	memset(&req, 0, sizeof req);
	memset(&rep, 0, sizeof rep);

	req.Function			= MPI_FUNCTION_RAID_ACTION;
	req.Action				= action;
	req.VolumeBus			= IOCPage2->RaidVolume[volume].VolumeBus;
	req.VolumeID			= IOCPage2->RaidVolume[volume].VolumeID;

	printf("Volume %d is being %s\n", volume, string);

	return doMptCommandCheck(port, &req, sizeof req, &rep, sizeof rep, NULL, 0, NULL, 0, 10);
}


int
doCreateVolume(MPT_PORT *port, IOCPage2_t *IOCPage2)
{
	RaidVolumePage0_t		*RaidVolumePage0;
	RaidPhysDiskPage0_t		 RaidPhysDiskPage0;
	ManufacturingPage4_t	 ManufacturingPage4;
	MpiRaidActionRequest_t	 req;
	MpiRaidActionReply_t	 rep;
	ConfigReply_t			 config_rep;
	int						 bus;
	int						 target;
	int						 num_phys_disks;
	int						 phys_disks[256];
	unsigned char			 inq[36];
	unsigned char			 cap[8];
	unsigned int			 size;
	unsigned int			 min_size;
	DIAG_TARGET				 diag_targets[99];
	int						 first_bus;
	int						 first_target;
	int						 length;
	int						 i;
	int						 n;
	int						 t;
	int						 settings;
	int						 flags;

	if (IOCPage2->NumActiveVolumes == IOCPage2->MaxVolumes)
	{
		printf("Cannot create another active volume\n");
		return 1;
	}

	printf("     B___T___L  Type       Vendor   Product          Rev   Disk Blocks  Disk MB\n");

	n = 0;
	for (bus = 0; bus < port->maxBuses; bus++)
	{
		for (target = 0; target < port->maxTargets; target++)
		{
			if (doInquiry(port, bus, target, 0, inq, sizeof inq) != 1)
				continue;

			if ((inq[0] & 0x1f) == 0x1f)
				continue;

			if ((inq[0] & 0xe0) == 0x20)
				continue;

			if ((inq[0] & 0xe0) == 0x60)
				continue;

			for (i = 8; i < 36; i++)
				if (!isprint(inq[i]))
					inq[i] = ' ';

			if (doReadCapacity(port, bus, target, 0, cap, sizeof cap) != 1)
				continue;

			size = (cap[0] << 24) | (cap[1] << 16) | (cap[2] << 8) | cap[3];

			diag_targets[n].bus		= bus;
			diag_targets[n].target	= target;
			diag_targets[n].size	= size;

			n++;

			printf("%2d. %2d %3d %3d  %-9s  %8.8s %16.16s %4.4s   %10d  %7d\n",
				   n, bus, target, 0, deviceType[inq[0] & 0x1f],
				   inq+8, inq+16, inq+32, size + 1, (size + 1) / 2048);

			if (n == 99)
				break;
		}
		if (n == 99)
			break;
	}

	if (n < 2)
	{
		printf("\nNot enough available targets found\n");
		return 1;
	}

	printf("\nTo create a volume, select two or more of the available targets\n\n");

	num_phys_disks = 0;

	if (getConfigPageHeader(port, MPI_CONFIG_PAGETYPE_RAID_PHYSDISK, 0, 0, &config_rep) != 1)
		return 0;

	while (TRUE)
	{
		printf("Select a target:  [1-%d or RETURN to quit] ", n);
		i = getNumberAnswer(1, n, 0);
		if (i == 0)
			break;
		i--;

		bus		= diag_targets[i].bus;
		target	= diag_targets[i].target;
		size	= (diag_targets[i].size - 32) / 2048;

		if (num_phys_disks == 0)
		{
			min_size = size;
		}
		else
		{
			if (size < min_size)
			{
				printf("  reducing size from %d MB to %d MB\n", min_size, size);
				min_size = size;
			}
		}

		memset(&RaidPhysDiskPage0, 0, sizeof RaidPhysDiskPage0);

		RaidPhysDiskPage0.Header.PageType		= config_rep.Header.PageType;
		RaidPhysDiskPage0.Header.PageNumber		= config_rep.Header.PageNumber;
		RaidPhysDiskPage0.Header.PageLength		= config_rep.Header.PageLength;
		RaidPhysDiskPage0.Header.PageVersion	= config_rep.Header.PageVersion;

		RaidPhysDiskPage0.PhysDiskIOC			= port->iocNumber;
		RaidPhysDiskPage0.PhysDiskBus			= bus;
		RaidPhysDiskPage0.PhysDiskID			= target;

		memset(&req, 0, sizeof req);
		memset(&rep, 0, sizeof rep);

		req.Function			= MPI_FUNCTION_RAID_ACTION;
		req.Action				= MPI_RAID_ACTION_CREATE_PHYSDISK;

		if (doMptCommandCheck(port, &req, sizeof req - sizeof req.ActionDataSGE, &rep, sizeof rep,
							  NULL, 0, &RaidPhysDiskPage0, sizeof RaidPhysDiskPage0, 10) != 1)
			return 0;

		phys_disks[num_phys_disks] = get32(rep.ActionData);

//		printf("PhysDisk %d was created\n", phys_disks[num_phys_disks]);

		if (num_phys_disks == 0)
		{
			first_bus = bus;
			first_target = target;
		}
		num_phys_disks++;

		if (num_phys_disks >= n)
			break;

		if (IOCPage2->NumActivePhysDisks + num_phys_disks >= IOCPage2->MaxPhysDisks)
		{
			printf("  no more physical disks can be created, exiting loop\n");
			break;
		}
	}

	printf("\n%d physical disks were created\n\n", num_phys_disks);

	if (num_phys_disks < 2)
	{
		printf("Volumes must have at least 2 physical disks!\n");
		return 0;
	}

	if (getConfigPage(port, MPI_CONFIG_PAGETYPE_MANUFACTURING, 4, 0,
					  &ManufacturingPage4, sizeof ManufacturingPage4) != 1)
		return 0;

	if (getConfigPageHeader(port, MPI_CONFIG_PAGETYPE_RAID_VOLUME, 0, 0, &config_rep) != 1)
		return 0;

	length = sizeof *RaidVolumePage0 + sizeof RaidVolumePage0->PhysDisk * (num_phys_disks - 1);
	if (length < config_rep.Header.PageLength * 4)
		length = config_rep.Header.PageLength * 4;
	RaidVolumePage0 = malloc(length);

	memset(RaidVolumePage0, 0, length);

	RaidVolumePage0->Header.PageType			= config_rep.Header.PageType;
	RaidVolumePage0->Header.PageNumber			= config_rep.Header.PageNumber;
	RaidVolumePage0->Header.PageLength			= length / 4;
	RaidVolumePage0->Header.PageVersion			= config_rep.Header.PageVersion;

	RaidVolumePage0->VolumeIOC					= port->iocNumber;
	RaidVolumePage0->VolumeBus					= first_bus;
	RaidVolumePage0->VolumeID					= first_target;

	RaidVolumePage0->NumPhysDisks				= num_phys_disks;

	flags = get32(IOCPage2->CapabilitiesFlags);
	if (num_phys_disks == 2 && flags & MPI_IOCPAGE2_CAP_FLAGS_IM_SUPPORT)
	{
		RaidVolumePage0->VolumeType = MPI_RAID_VOL_TYPE_IM;

		RaidVolumePage0->PhysDisk[0].PhysDiskNum = phys_disks[0];
		RaidVolumePage0->PhysDisk[0].PhysDiskMap = MPI_RAIDVOL0_PHYSDISK_PRIMARY;

		RaidVolumePage0->PhysDisk[1].PhysDiskNum = phys_disks[1];
		RaidVolumePage0->PhysDisk[1].PhysDiskMap = MPI_RAIDVOL0_PHYSDISK_SECONDARY;

		printf("Select volume size:  [1 to %d MB, default is %d] ", min_size, min_size);
		size = getNumberAnswer(1, min_size, min_size) * 2048;

		RaidVolumePage0->MaxLBA = set32(size - 1);

		settings = ManufacturingPage4.IMVolumeSettings;
	}
	else
	{
		if (flags & MPI_IOCPAGE2_CAP_FLAGS_IME_SUPPORT &&
			flags & MPI_IOCPAGE2_CAP_FLAGS_IS_SUPPORT)
		{
			printf("Select volume type:  [0=Mirroring, 1=Striping, default is 0] ");
			if (getNumberAnswer(0, 1, 0) == 0)
				RaidVolumePage0->VolumeType = MPI_RAID_VOL_TYPE_IME;
			else
				RaidVolumePage0->VolumeType = MPI_RAID_VOL_TYPE_IS;
		}
		else if (flags & MPI_IOCPAGE2_CAP_FLAGS_IME_SUPPORT)
		{
			RaidVolumePage0->VolumeType = MPI_RAID_VOL_TYPE_IME;
		}
		else if (flags & MPI_IOCPAGE2_CAP_FLAGS_IS_SUPPORT)
		{
			RaidVolumePage0->VolumeType = MPI_RAID_VOL_TYPE_IS;
		}

		min_size *= num_phys_disks;
		if (RaidVolumePage0->VolumeType == MPI_RAID_VOL_TYPE_IME)
			min_size /= 2;

		printf("Select volume size:  [1 to %d MB, default is %d] ", min_size, min_size);
		size = getNumberAnswer(1, min_size, min_size) * 2048;

		RaidVolumePage0->MaxLBA = set32(size - 1);

		if (RaidVolumePage0->VolumeType == MPI_RAID_VOL_TYPE_IME)
		{
			settings = ManufacturingPage4.IMEVolumeSettings;
		}

		if (RaidVolumePage0->VolumeType == MPI_RAID_VOL_TYPE_IS)
		{
			printf("Select stripe size:  [4 to 256 KB, default is 64] ");
			RaidVolumePage0->StripeSize = set32(getNumberAnswer(4, 256, 64)) * 2;

			settings = ManufacturingPage4.ISVolumeSettings;
		}

		for (i = 0; i < num_phys_disks; i++)
		{
			RaidVolumePage0->PhysDisk[i].PhysDiskNum = phys_disks[i];
			RaidVolumePage0->PhysDisk[i].PhysDiskMap = i;
		}
	}

	t = settings & MPI_RAIDVOL0_SETTING_WRITE_CACHING_ENABLE ? 1 : 0;
	printf("Enable write caching:  [Yes or No, default is %s] ", t == 0 ? "No" : "Yes");
	if (getYesNoAnswer(t))
		settings |= MPI_RAIDVOL0_SETTING_WRITE_CACHING_ENABLE;
	else
		settings &= ~MPI_RAIDVOL0_SETTING_WRITE_CACHING_ENABLE;

	RaidVolumePage0->VolumeSettings.Settings = set16(settings);

	memset(&req, 0, sizeof req);
	memset(&rep, 0, sizeof rep);

	req.Function			= MPI_FUNCTION_RAID_ACTION;
	req.Action				= MPI_RAID_ACTION_CREATE_VOLUME;
	req.VolumeBus			= first_bus;
	req.VolumeID			= first_target;

	printf("Zero the first and last blocks of the volume?  [Yes or No, default is No] ");
	if (getYesNoAnswer(0) == 1)
		req.ActionDataWord	= MPI_RAID_ACTION_ADATA_LOW_LEVEL_INIT;

	if (doMptCommandCheck(port, &req, sizeof req - sizeof req.ActionDataSGE, &rep, sizeof rep,
						  NULL, 0, RaidVolumePage0, length, 60) != 1)
		return 0;

	printf("\nVolume was created\n");

	free(RaidVolumePage0);

	return 1;
}


int
doDeleteVolume(MPT_PORT *port, IOCPage2_t *IOCPage2)
{
	MpiRaidActionRequest_t	 req;
	MpiRaidActionReply_t	 rep;
	int						 volume;

	if (IOCPage2->NumActiveVolumes == 0)
	{
		printf("No active volumes\n");
		return 1;
	}

	if (IOCPage2->MaxVolumes > 1)
	{
		printf("Volume:  [0-%d or RETURN to quit] ", IOCPage2->MaxVolumes - 1);
		volume = getNumberAnswer(0, IOCPage2->MaxVolumes - 1, -1);
		if (volume < 0)
			return 1;
		printf("\n");
	}
	else
	{
		volume = 0;
	}

	printf("All data on Volume %d will be lost!\n", volume);
	printf("\nAre you sure you want to continue?  [Yes or No, default is No] ");

	if (getYesNoAnswer(0) != 1)
		return 1;

	memset(&req, 0, sizeof req);
	memset(&rep, 0, sizeof rep);

	req.Function			= MPI_FUNCTION_RAID_ACTION;
	req.Action				= MPI_RAID_ACTION_DELETE_VOLUME;
	req.VolumeBus			= IOCPage2->RaidVolume[volume].VolumeBus;
	req.VolumeID			= IOCPage2->RaidVolume[volume].VolumeID;

	printf("\nVolume %d is being deleted\n", volume);

	return doMptCommandCheck(port, &req, sizeof req, &rep, sizeof rep, NULL, 0, NULL, 0, 10);
}


int
doVolumeSettings(MPT_PORT *port, IOCPage2_t *IOCPage2)
{
	RaidVolumePage0_t		 RaidVolumePage0;
	MpiRaidActionRequest_t	 req;
	MpiRaidActionReply_t	 rep;
	int						 b_t;
	int						 volume;
	int						 t1;
	int						 t2;
	int						 t;

	if (IOCPage2->NumActiveVolumes == 0)
	{
		printf("No active volumes\n");
		return 1;
	}

	if (IOCPage2->MaxVolumes > 1)
	{
		printf("Volume:  [0-%d or RETURN to quit] ", IOCPage2->MaxVolumes - 1);
		volume = getNumberAnswer(0, IOCPage2->MaxVolumes - 1, -1);
		if (volume < 0)
			return 1;
		printf("\n");
	}
	else
	{
		volume = 0;
	}

	b_t = (IOCPage2->RaidVolume[volume].VolumeBus << 8) + IOCPage2->RaidVolume[volume].VolumeID;

	if (getConfigPage(port, MPI_CONFIG_PAGETYPE_RAID_VOLUME, 0, b_t,
					  &RaidVolumePage0, sizeof RaidVolumePage0) != 1)
		return 0;

	t1 = get16(RaidVolumePage0.VolumeSettings.Settings);
	t2 = RaidVolumePage0.VolumeSettings.HotSparePool;
	printf("Volume %d Settings:  write caching %s%s%s%s\n", volume,
		   t1 & MPI_RAIDVOL0_SETTING_WRITE_CACHING_ENABLE ? "enabled" : "disabled",
		   t1 & MPI_RAIDVOL0_SETTING_OFFLINE_ON_SMART ? ", offline on SMART data" : "",
		   t1 & MPI_RAIDVOL0_SETTING_AUTO_CONFIGURE ? ", auto configure" : "",
		   t1 & MPI_RAIDVOL0_SETTING_PRIORITY_RESYNC ? ", priority resync" : "");
	if (t2 != 0)
		printf("Volume %d draws from Hot Spare Pools: %s%s%s%s%s%s%s%s\n", volume,
			   t2 & MPI_RAID_HOT_SPARE_POOL_0 ? " 0" : "",
			   t2 & MPI_RAID_HOT_SPARE_POOL_1 ? " 1" : "",
			   t2 & MPI_RAID_HOT_SPARE_POOL_2 ? " 2" : "",
			   t2 & MPI_RAID_HOT_SPARE_POOL_3 ? " 3" : "",
			   t2 & MPI_RAID_HOT_SPARE_POOL_4 ? " 4" : "",
			   t2 & MPI_RAID_HOT_SPARE_POOL_5 ? " 5" : "",
			   t2 & MPI_RAID_HOT_SPARE_POOL_6 ? " 6" : "",
			   t2 & MPI_RAID_HOT_SPARE_POOL_7 ? " 7" : "");
	printf("\n");

	t = t1 & MPI_RAIDVOL0_SETTING_WRITE_CACHING_ENABLE ? 1 : 0;
	printf("Enable write caching:  [Yes or No, default is %s] ", t == 0 ? "No" : "Yes");
	if (getYesNoAnswer(t))
		t1 |= MPI_RAIDVOL0_SETTING_WRITE_CACHING_ENABLE;
	else
		t1 &= ~MPI_RAIDVOL0_SETTING_WRITE_CACHING_ENABLE;

	t = t1 & MPI_RAIDVOL0_SETTING_OFFLINE_ON_SMART ? 1 : 0;
	printf("Offline on SMART data:  [Yes or No, default is %s] ", t == 0 ? "No" : "Yes");
	if (getYesNoAnswer(t))
		t1 |= MPI_RAIDVOL0_SETTING_OFFLINE_ON_SMART;
	else
		t1 &= ~MPI_RAIDVOL0_SETTING_OFFLINE_ON_SMART;

	t = t1 & MPI_RAIDVOL0_SETTING_AUTO_CONFIGURE ? 1 : 0;
	printf("Auto configuration:  [Yes or No, default is %s] ", t == 0 ? "No" : "Yes");
	if (getYesNoAnswer(t))
		t1 |= MPI_RAIDVOL0_SETTING_AUTO_CONFIGURE;
	else
		t1 &= ~MPI_RAIDVOL0_SETTING_AUTO_CONFIGURE;

	t = t1 & MPI_RAIDVOL0_SETTING_PRIORITY_RESYNC ? 1 : 0;
	printf("Priority resync:  [Yes or No, default is %s] ", t == 0 ? "No" : "Yes");
	if (getYesNoAnswer(t))
		t1 |= MPI_RAIDVOL0_SETTING_PRIORITY_RESYNC;
	else
		t1 &= ~MPI_RAIDVOL0_SETTING_PRIORITY_RESYNC;

	printf("Hot Spare Pools (bitmask of pool numbers):  [00 to FF, default is %02x] ", t2);
	while (TRUE)
	{
		t = getHexNumberAnswer(&t2);
		if (t == 0)
			break;

		if (t2 >= 0x00 && t2 <= 0xff)
			break;

		t2 = RaidVolumePage0.VolumeSettings.HotSparePool;

		printf("Invalid answer, try again: ");
	}

	RaidVolumePage0.VolumeSettings.Settings = set16(t1);
	RaidVolumePage0.VolumeSettings.HotSparePool = t2;

	memset(&req, 0, sizeof req);
	memset(&rep, 0, sizeof rep);

	req.Function			= MPI_FUNCTION_RAID_ACTION;
	req.Action				= MPI_RAID_ACTION_CHANGE_VOLUME_SETTINGS;
	req.VolumeBus			= IOCPage2->RaidVolume[volume].VolumeBus;
	req.VolumeID			= IOCPage2->RaidVolume[volume].VolumeID;

	memcpy(&req.ActionDataWord, &RaidVolumePage0.VolumeSettings, sizeof req.ActionDataWord);

	return doMptCommandCheck(port, &req, sizeof req, &rep, sizeof rep, NULL, 0, NULL, 0, 10);
}


int
doModifyPhysDisk(MPT_PORT *port, IOCPage2_t *IOCPage2, int action, char *string)
{
	MpiRaidActionRequest_t	 req;
	MpiRaidActionReply_t	 rep;
	int						 phys_disk;
	int						 phys_disk2;

	if (IOCPage2->NumActivePhysDisks == 0)
	{
		printf("No active physical disks\n");
		return 1;
	}

	printf("PhysDisk:  [0-%d or RETURN to quit] ", IOCPage2->MaxPhysDisks - 1);
	phys_disk = getNumberAnswer(0, IOCPage2->MaxPhysDisks - 1, -1);
	if (phys_disk < 0)
		return 1;

	memset(&req, 0, sizeof req);
	memset(&rep, 0, sizeof rep);

	req.Function			= MPI_FUNCTION_RAID_ACTION;
	req.Action				= action;
	req.PhysDiskNum			= phys_disk;

	if (action == MPI_RAID_ACTION_REPLACE_PHYSDISK)
	{
		printf("Replacement PhysDisk:  [0-%d or RETURN to quit] ", IOCPage2->MaxPhysDisks - 1);
		phys_disk2 = getNumberAnswer(0, IOCPage2->MaxPhysDisks - 1, -1);
		if (phys_disk2 < 0)
			return 1;

		req.ActionDataWord	= phys_disk2;
	}

	printf("\nPhysDisk %d is being %s\n", phys_disk, string);

	return doMptCommandCheck(port, &req, sizeof req, &rep, sizeof rep, NULL, 0, NULL, 0, 10);
}


int
doCreatePhysDisk(MPT_PORT *port, IOCPage2_t *IOCPage2)
{
	RaidPhysDiskPage0_t		 RaidPhysDiskPage0;
	MpiRaidActionRequest_t	 req;
	MpiRaidActionReply_t	 rep;
	ConfigReply_t			 config_rep;
	int						 bus;
	int						 target;
	int						 phys_disk;

	if (getConfigPageHeader(port, MPI_CONFIG_PAGETYPE_RAID_PHYSDISK, 0, 0, &config_rep) != 1)
		return 0;

	while (TRUE)
	{
		if (port->maxBuses > 1)
		{
			printf("Bus:  [0-%d or RETURN to quit] ", port->maxBuses - 1);
			bus = getNumberAnswer(0, port->maxBuses - 1, -1);
			if (bus < 0)
				break;
		}
		else
		{
			bus = 0;
		}

		printf("Target:  [0-%d or RETURN to quit] ", port->maxTargets - 1);
		target = getNumberAnswer(0, port->maxTargets - 1, -1);
		if (target < 0)
			break;

		memset(&RaidPhysDiskPage0, 0, sizeof RaidPhysDiskPage0);

		RaidPhysDiskPage0.Header.PageType		= config_rep.Header.PageType;
		RaidPhysDiskPage0.Header.PageNumber		= config_rep.Header.PageNumber;
		RaidPhysDiskPage0.Header.PageLength		= config_rep.Header.PageLength;
		RaidPhysDiskPage0.Header.PageVersion	= config_rep.Header.PageVersion;

		RaidPhysDiskPage0.PhysDiskIOC			= port->iocNumber;
		RaidPhysDiskPage0.PhysDiskBus			= bus;
		RaidPhysDiskPage0.PhysDiskID			= target;

		memset(&req, 0, sizeof req);
		memset(&rep, 0, sizeof rep);

		req.Function			= MPI_FUNCTION_RAID_ACTION;
		req.Action				= MPI_RAID_ACTION_CREATE_PHYSDISK;

		if (doMptCommandCheck(port, &req, sizeof req - sizeof req.ActionDataSGE, &rep, sizeof rep,
							  NULL, 0, &RaidPhysDiskPage0, sizeof RaidPhysDiskPage0, 10) != 1)
			return 0;

		phys_disk = get32(rep.ActionData);

		printf("PhysDisk %d was created\n", phys_disk);
	}

	return 1;
}


int
doPhysDiskSettings(MPT_PORT *port, IOCPage2_t *IOCPage2)
{
	RaidPhysDiskPage0_t		 RaidPhysDiskPage0;
	MpiRaidActionRequest_t	 req;
	MpiRaidActionReply_t	 rep;
	int						 phys_disk;
	int						 t2;
	int						 t;

	if (IOCPage2->NumActivePhysDisks == 0)
	{
		printf("No active physical disks\n");
		return 1;
	}

	printf("PhysDisk:  [0-%d or RETURN to quit] ", IOCPage2->MaxPhysDisks - 1);
	phys_disk = getNumberAnswer(0, IOCPage2->MaxPhysDisks - 1, -1);
	if (phys_disk < 0)
		return 1;

	if (getConfigPage(port, MPI_CONFIG_PAGETYPE_RAID_PHYSDISK, 0, phys_disk,
					  &RaidPhysDiskPage0, sizeof RaidPhysDiskPage0) != 1)
		return 0;

	t2 = RaidPhysDiskPage0.PhysDiskSettings.HotSparePool;
	if (t2 != 0)
		printf("PhysDisk %d belongs to Hot Spare Pools: %s%s%s%s%s%s%s%s\n", phys_disk,
			   t2 & MPI_RAID_HOT_SPARE_POOL_0 ? " 0" : "",
			   t2 & MPI_RAID_HOT_SPARE_POOL_1 ? " 1" : "",
			   t2 & MPI_RAID_HOT_SPARE_POOL_2 ? " 2" : "",
			   t2 & MPI_RAID_HOT_SPARE_POOL_3 ? " 3" : "",
			   t2 & MPI_RAID_HOT_SPARE_POOL_4 ? " 4" : "",
			   t2 & MPI_RAID_HOT_SPARE_POOL_5 ? " 5" : "",
			   t2 & MPI_RAID_HOT_SPARE_POOL_6 ? " 6" : "",
			   t2 & MPI_RAID_HOT_SPARE_POOL_7 ? " 7" : "");
	printf("\n");

	printf("Hot Spare Pools (bitmask of pool numbers):  [00 to FF, default is %02x] ", t2);
	while (TRUE)
	{
		t = getHexNumberAnswer(&t2);
		if (t == 0)
			break;

		if (t2 >= 0x00 && t2 <= 0xff)
			break;

		t2 = RaidPhysDiskPage0.PhysDiskSettings.HotSparePool;

		printf("Invalid answer, try again: ");
	}

	RaidPhysDiskPage0.PhysDiskSettings.HotSparePool = t2;

	memset(&req, 0, sizeof req);
	memset(&rep, 0, sizeof rep);

	req.Function			= MPI_FUNCTION_RAID_ACTION;
	req.Action				= MPI_RAID_ACTION_CHANGE_VOLUME_SETTINGS;
	req.PhysDiskNum			= phys_disk;

	memcpy(&req.ActionDataWord, &RaidPhysDiskPage0.PhysDiskSettings, sizeof req.ActionDataWord);

	return doMptCommandCheck(port, &req, sizeof req, &rep, sizeof rep, NULL, 0, NULL, 0, 10);
}


int
doResetBus(MPT_PORT *port)
{
#if WIN32
	int			 status;
	SRB_BUFFER	 srb;
	int			 inLen;
	int			 outLen;
	int			 retLen;

	memset(&srb, 0, sizeof srb);

	srb.Sic.Length			= sizeof srb - sizeof srb.Sic;
	srb.Sic.ControlCode		= ISSUE_BUS_RESET;
	srb.Sic.HeaderLength	= sizeof srb.Sic;
	srb.Sic.Timeout			= 30;

	inLen					= sizeof srb;
	outLen					= sizeof srb;
	retLen					= 0;

	printf("Resetting bus...\n");

	status = DeviceIoControl(port->fileHandle, port->ioctlValue,
							 &srb, inLen, &srb, outLen, &retLen, NULL);

	return status;
#else
	SCSITaskMgmt_t			 req;
	SCSITaskMgmtReply_t		 rep;
	int						 bus;

	if (port->maxBuses > 1)
	{
		printf("Bus:  [0-%d or RETURN to quit] ", port->maxBuses - 1);
		bus = getNumberAnswer(0, port->maxBuses - 1, -1);
		if (bus < 0)
			return 1;
	}
	else
	{
		bus = 0;
	}

	memset(&req, 0, sizeof req);
	memset(&rep, 0, sizeof rep);

	req.Function			= MPI_FUNCTION_SCSI_TASK_MGMT;
	req.TaskType			= MPI_SCSITASKMGMT_TASKTYPE_RESET_BUS;

	printf("Resetting bus...\n");

	return doMptCommandCheck(port, &req, sizeof req, &rep, sizeof rep, NULL, 0, NULL, 0, 10);
#endif
}


int
doResetTarget(MPT_PORT *port)
{
	SCSITaskMgmt_t			 req;
	SCSITaskMgmtReply_t		 rep;
	int						 bus;
	int						 target;
	int						 lun;
	int						 type;

	printf(" 1.  Target Reset\n");
	printf(" 2.  Logical Unit Reset\n");
	printf(" 3.  Abort Task Set\n");
	printf(" 4.  Clear Task Set\n");
	printf("\nSelect a reset type:  [1-4 or RETURN to quit] ");
	type = getNumberAnswer(1, 4, 0);
	if (type == 0)
		return 1;

	if (port->maxBuses > 1)
	{
		printf("Bus:  [0-%d or RETURN to quit] ", port->maxBuses - 1);
		bus = getNumberAnswer(0, port->maxBuses - 1, -1);
		if (bus < 0)
			return 1;
	}
	else
	{
		bus = 0;
	}

	printf("Target:  [0-%d or RETURN to quit] ", port->maxTargets - 1);
	target = getNumberAnswer(0, port->maxTargets - 1, -1);
	if (target < 0)
		return 1;

	if (type != 1)
	{
		printf("LUN:  [0-%d or RETURN to quit] ", port->maxLuns - 1);
		lun = getNumberAnswer(0, port->maxLuns - 1, -1);
		if (lun < 0)
			return 1;
	}

	memset(&req, 0, sizeof req);
	memset(&rep, 0, sizeof rep);

	req.Function			= MPI_FUNCTION_SCSI_TASK_MGMT;
	switch(type)
	{
	default:
	case 1:  req.TaskType	= MPI_SCSITASKMGMT_TASKTYPE_TARGET_RESET;        break;
	case 2:  req.TaskType	= MPI_SCSITASKMGMT_TASKTYPE_LOGICAL_UNIT_RESET;  break;
	case 3:  req.TaskType	= MPI_SCSITASKMGMT_TASKTYPE_ABRT_TASK_SET;       break;
	case 4:  req.TaskType	= MPI_SCSITASKMGMT_TASKTYPE_CLEAR_TASK_SET;      break;
	}
	req.Bus					= bus;
	req.TargetID			= mapOsToHwTarget(port, target);
	req.LUN[1]				= lun;

	printf("\nResetting target...\n");

	return doMptCommandCheck(port, &req, sizeof req, &rep, sizeof rep, NULL, 0, NULL, 0, 10);
}


int
doGID_FT(MPT_PORT *port)
{
	FcCommonTransportSendRequest_t	 req;
	FcCommonTransportSendReply_t	 rep;
	U32								 buf_out[256];
	U32								 buf_in[256];
	int								 ioc_status;
	U32								 code;
	int								 i;

	memset(&req, 0, sizeof req);
	memset(&rep, 0, sizeof rep);

	req.Function			= MPI_FUNCTION_FC_COMMON_TRANSPORT_SEND;
	req.MsgFlags_Did		= set32(0xfffffc);
	req.CTCommandCode		= set16(0x0171);
	req.FsType				= 0xfc;

	memset(buf_out, 0, sizeof buf_out);

	buf_out[0]				= set32(swap32(0x01000000));
	buf_out[1]				= set32(swap32(0xfc020000));
	buf_out[2]				= set32(swap32(0x01710208));
	buf_out[3]				= set32(swap32(0x00000000));
	buf_out[4]				= set32(swap32(0x00000008));

	buf_in[0] = 0;

	if (doMptCommand(port, &req, sizeof req - sizeof req.SGL, &rep, sizeof rep,
					 buf_in, sizeof buf_in, buf_out, 20, 10) == 1)
	{
		ioc_status = get16(rep.IOCStatus) & MPI_IOCSTATUS_MASK;

		if (ioc_status != MPI_IOCSTATUS_SUCCESS)
		{
			printf("CTSend failed, IOCStatus = %04x (%s)\n", ioc_status, translateIocStatus(ioc_status));
			return 0;
		}

		code = (get32(swap32(buf_in[2])) >> 16) & 0xffff;

		if (code == 0x8001)
		{
			printf("GID_FT rejected\n\n");
			for (i = 0; i < (int)get32(rep.ResponseLength) / 4; i++)
				printf("Word %04d = %08x\n", i, get32(swap32(buf_in[i])));
			return 0;
		}

		if (code != 0x8002)
		{
			printf("GID_FT not accepted, code is %04x\n\n", code);
			for (i = 0; i < (int)get32(rep.ResponseLength) / 4; i++)
				printf("Word %04d = %08x\n", i, get32(swap32(buf_in[i])));
			return 0;
		}

		for (i = 0; i < (int)get32(rep.ResponseLength) / 4; i++)
			printf("Word %04d = %08x\n", i, get32(swap32(buf_in[i])));
	}

	return 1;
}


int
doGA_NXT(MPT_PORT *port)
{
	FcCommonTransportSendRequest_t	 req;
	FcCommonTransportSendReply_t	 rep;
	U32								 buf_out[256];
	U32								 buf_in[256];
	int								 ioc_status;
	U32								 code;
	int								 i;
	U32								 d_id;

	printf("Enter D_ID:  [000000-FFFFFF or RETURN to quit] ");
	while (TRUE)
	{
		if (getHexNumberAnswer(&d_id) == 0)
			return 0;

		if (d_id <= 0xffffff)
			break;

		printf("Invalid answer, try again: ");
	}

	printf("\n");

	memset(&req, 0, sizeof req);
	memset(&rep, 0, sizeof rep);

	req.Function			= MPI_FUNCTION_FC_COMMON_TRANSPORT_SEND;
	req.MsgFlags_Did		= set32(0xfffffc);
	req.CTCommandCode		= set16(0x0100);
	req.FsType				= 0xfc;

	memset(buf_out, 0, sizeof buf_out);

	buf_out[0]				= set32(swap32(0x01000000));
	buf_out[1]				= set32(swap32(0xfc020000));
	buf_out[2]				= set32(swap32(0x01000208));
	buf_out[3]				= set32(swap32(0x00000000));
	buf_out[4]				= set32(swap32(d_id));

	buf_in[0] = 0;

	if (doMptCommand(port, &req, sizeof req - sizeof req.SGL, &rep, sizeof rep,
					 buf_in, sizeof buf_in, buf_out, 20, 10) == 1)
	{
		ioc_status = get16(rep.IOCStatus) & MPI_IOCSTATUS_MASK;

		if (ioc_status != MPI_IOCSTATUS_SUCCESS)
		{
			printf("CTSend failed, IOCStatus = %04x (%s)\n", ioc_status, translateIocStatus(ioc_status));
			return 0;
		}

		code = (get32(swap32(buf_in[2])) >> 16) & 0xffff;

		if (code == 0x8001)
		{
			printf("GA_NXT rejected\n\n");
			for (i = 0; i < (int)get32(rep.ResponseLength) / 4; i++)
				printf("Word %04d = %08x\n", i, get32(swap32(buf_in[i])));
			return 0;
		}

		if (code != 0x8002)
		{
			printf("GA_NXT not accepted, code is %04x\n\n", code);
			for (i = 0; i < (int)get32(rep.ResponseLength) / 4; i++)
				printf("Word %04d = %08x\n", i, get32(swap32(buf_in[i])));
			return 0;
		}

		for (i = 0; i < (int)get32(rep.ResponseLength) / 4; i++)
			printf("Word %04d = %08x\n", i, get32(swap32(buf_in[i])));
	}

	return 1;
}


int
doExLinkServiceSend(MPT_PORT *port)
{
	ExLinkServiceSendRequest_t	 req;
	ExLinkServiceSendReply_t	 rep;
	U32							 buf_out[256];
	U32							 buf_in[256];
	U32							 els;
	U32							 d_id;
	int							 len;
	U32							 value;
	int							 i;
	int							 j;
	int							 t;
	int							 ioc_status;
	U32							 code;
	int							 passes;
	int							 delay;

	printf("Enter ELS:  [00-FF or RETURN to quit] ");
	while (TRUE)
	{
		if (getHexNumberAnswer(&els) == 0)
			return 0;

		if (els <= 0xff)
			break;

		printf("Invalid answer, try again: ");
	}

	printf("Enter D_ID:  [000000-FFFFFF or RETURN to quit] ");
	while (TRUE)
	{
		if (getHexNumberAnswer(&d_id) == 0)
			return 0;

		if (d_id <= 0xffffff)
			break;

		printf("Invalid answer, try again: ");
	}

	printf("Enter length in words:  [1-256 or 0 to quit] ");
	len = getNumberAnswer(0, 256, -1);
	if (len < 0)
		return 0;

	memset(&req, 0, sizeof req);
	memset(&rep, 0, sizeof rep);

	req.Function			= MPI_FUNCTION_FC_EX_LINK_SRVC_SEND;
	req.MsgFlags_Did		= set32(d_id);
	req.ElsCommandCode		= set32(els);

	memset(buf_out, 0, sizeof buf_out);

	buf_out[0] = set32(els);

	if (len > 1)
	{
		while (TRUE)
		{
			printf("\n");
			for (i = 0; i < len; i++)
				printf("Word %04d = %08x\n", i, get32(swap32(buf_out[i])));
			printf("\n");

			printf("Enter word to change:  [1-%d or RETURN to quit] ", len - 1);
			while (TRUE)
			{
				t = getHexNumberAnswer(&value);
				i = value;
				if (t == 0)
					break;

				if (i >= 1 && i < len)
					break;

				printf("Invalid answer, try again: ");
			}
			if (t == 0)
				break;

			printf("Enter value:  [00000000-FFFFFFFF or RETURN to quit] ");
			if (getHexNumberAnswer(&value) == 0)
				break;

			buf_out[i] = set32(swap32(value));
		}
	}

	printf("\nNumber of iterations:  [1-1000000, default is 1] ");
	passes = getNumberAnswer(1, 1000000, 1);
	if (passes > 1)
	{
		printf("Delay in seconds between iterations:  [1-6000, default is 30] ");
		delay = getNumberAnswer(1, 6000, 30);
	}
	else
		delay = 0;

	for (j = 1; j <= passes; j++)
	{
		if (passes > 1)
		{
			if (j > 1)
				sleep(delay);
			printf("\nPass %d...\n", j);
		}

		printf("\n");

		buf_in[0] = 0;

		if (doMptCommand(port, &req, sizeof req - sizeof req.SGL, &rep, sizeof rep,
						 buf_in, sizeof buf_in, buf_out, len * 4, 10) == 1)
		{
			ioc_status = get16(rep.IOCStatus) & MPI_IOCSTATUS_MASK;

			if (ioc_status != MPI_IOCSTATUS_SUCCESS)
			{
				printf("SendELS failed, IOCStatus = %04x (%s)\n", ioc_status, translateIocStatus(ioc_status));
#if WIN32
				continue;
#else
				return 0;
#endif
			}

			code = (get32(swap32(buf_in[0])) >> 24) & 0xff;

			if (code == 0x01)
			{
				printf("ELS rejected\n");
#if WIN32
				continue;
#else
				return 0;
#endif
			}

			if (code != 0x02)
			{
				printf("ELS not accepted, code is %02x\n", code);
#if WIN32
				continue;
#else
				return 0;
#endif
			}

			for (i = 0; i < (int)get32(rep.ResponseLength) / 4; i++)
				printf("Word %04d = %08x\n", i, get32(swap32(buf_in[i])));
		}
	}

	return 1;
}


int
doResetFcLink(MPT_PORT *port, int flag)
{
	FcPrimitiveSendRequest_t	 req;
	FcPrimitiveSendReply_t		 rep;

	memset(&req, 0, sizeof req);
	memset(&rep, 0, sizeof rep);

	req.Function			= MPI_FUNCTION_FC_PRIMITIVE_SEND;
	if (flag)
		req.SendFlags		= MPI_FC_PRIM_SEND_FLAGS_ML_RESET_LINK;
	else
		req.SendFlags		= MPI_FC_PRIM_SEND_FLAGS_RESET_LINK;

	printf("Resetting FC link...\n");

	return doMptCommandCheck(port, &req, sizeof req, &rep, sizeof rep, NULL, 0, NULL, 0, 10);
}


int
doResetSasPhy(MPT_PORT *port)
{
	SasIoUnitControlRequest_t	 req;
	SasIoUnitControlReply_t		 rep;
	SasIOUnitPage1_t			*SASIOUnitPage1;
	ConfigReply_t				 config_rep;
	int							 length;
	int							 phy;

	if (getConfigPageHeader(port, MPI_CONFIG_EXTPAGETYPE_SAS_IO_UNIT, 1, 0, &config_rep) != 1)
		return 0;

	length = get16(config_rep.ExtPageLength) * 4;
	SASIOUnitPage1 = malloc(length);

	if (getConfigPage(port, MPI_CONFIG_EXTPAGETYPE_SAS_IO_UNIT, 1, 0, SASIOUnitPage1, length) != 1)
		return 0;

	printf("Select a Phy:  [0-%d, RETURN to quit] ", SASIOUnitPage1->NumPhys - 1);
	phy = getNumberAnswer(0, SASIOUnitPage1->NumPhys, -1);

	free(SASIOUnitPage1);

	if (phy < 0)
		return 0;

	memset(&req, 0, sizeof req);
	memset(&rep, 0, sizeof rep);

	req.Function			= MPI_FUNCTION_SAS_IO_UNIT_CONTROL;
	req.Operation			= MPI_SAS_OP_PHY_HARD_RESET;
	req.PhyNum				= phy;

	printf("\nResetting SAS phy...\n");

	return doMptCommandCheck(port, &req, sizeof req, &rep, sizeof rep, NULL, 0, NULL, 0, 10);
}


int
doResetSasLink(MPT_PORT *port)
{
	SasIoUnitControlRequest_t	 req;
	SasIoUnitControlReply_t		 rep;
	SasIOUnitPage1_t			*SASIOUnitPage1;
	ConfigReply_t				 config_rep;
	int							 length;
	int							 phy;

	if (getConfigPageHeader(port, MPI_CONFIG_EXTPAGETYPE_SAS_IO_UNIT, 1, 0, &config_rep) != 1)
		return 0;

	length = get16(config_rep.ExtPageLength) * 4;
	SASIOUnitPage1 = malloc(length);

	if (getConfigPage(port, MPI_CONFIG_EXTPAGETYPE_SAS_IO_UNIT, 1, 0, SASIOUnitPage1, length) != 1)
		return 0;

	printf("Select a Phy:  [0-%d, RETURN to quit] ", SASIOUnitPage1->NumPhys - 1);
	phy = getNumberAnswer(0, SASIOUnitPage1->NumPhys, -1);

	free(SASIOUnitPage1);

	if (phy < 0)
		return 0;

	memset(&req, 0, sizeof req);
	memset(&rep, 0, sizeof rep);

	req.Function			= MPI_FUNCTION_SAS_IO_UNIT_CONTROL;
	req.Operation			= MPI_SAS_OP_PHY_LINK_RESET;
	req.PhyNum				= phy;

	printf("\nResetting SAS link...\n");

	return doMptCommandCheck(port, &req, sizeof req, &rep, sizeof rep, NULL, 0, NULL, 0, 10);
}


int
doDumpPortState(MPT_PORT *port, int flag)
{
	U32		 buf[255];
	char	 name[32];
	int		 i;

	if (flag)
		doIdentify(port);

	if (getConfigPage(port, MPI_CONFIG_PAGETYPE_MANUFACTURING, 0, 0, buf, sizeof buf) == 1)
	{
		showConfigPage(port, "ManufacturingPage0", buf);
	}

	if (getConfigPage(port, MPI_CONFIG_PAGETYPE_MANUFACTURING, 1, 0, buf, sizeof buf) == 1)
	{
		showConfigPage(port, "ManufacturingPage1", buf);
	}

	if (getConfigPage(port, MPI_CONFIG_PAGETYPE_MANUFACTURING, 2, 0, buf, sizeof buf) == 1)
	{
		showConfigPage(port, "ManufacturingPage2", buf);
	}

	if (getConfigPage(port, MPI_CONFIG_PAGETYPE_MANUFACTURING, 3, 0, buf, sizeof buf) == 1)
	{
		showConfigPage(port, "ManufacturingPage3", buf);
	}

	if (getConfigPage(port, MPI_CONFIG_PAGETYPE_MANUFACTURING, 4, 0, buf, sizeof buf) == 1)
	{
		showConfigPage(port, "ManufacturingPage4", buf);
	}

	if (getConfigPage(port, MPI_CONFIG_PAGETYPE_MANUFACTURING, 5, 0, buf, sizeof buf) == 1)
	{
		showConfigPage(port, "ManufacturingPage5", buf);
	}

	if (getConfigPage(port, MPI_CONFIG_PAGETYPE_MANUFACTURING, 6, 0, buf, sizeof buf) == 1)
	{
		showConfigPage(port, "ManufacturingPage6", buf);
	}

	if (getConfigPage(port, MPI_CONFIG_PAGETYPE_IO_UNIT, 0, 0, buf, sizeof buf) == 1)
	{
		showConfigPage(port, "IOUnitPage0", buf);
	}

	if (getConfigPage(port, MPI_CONFIG_PAGETYPE_IO_UNIT, 1, 0, buf, sizeof buf) == 1)
	{
		showConfigPage(port, "IOUnitPage1", buf);
	}

	if (getConfigPage(port, MPI_CONFIG_PAGETYPE_IO_UNIT, 2, 0, buf, sizeof buf) == 1)
	{
		showConfigPage(port, "IOUnitPage2", buf);
	}

	if (getConfigPage(port, MPI_CONFIG_PAGETYPE_IOC, 0, 0, buf, sizeof buf) == 1)
	{
		showConfigPage(port, "IOCPage0", buf);
	}

	if (getConfigPage(port, MPI_CONFIG_PAGETYPE_IOC, 1, 0, buf, sizeof buf) == 1)
	{
		showConfigPage(port, "IOCPage1", buf);
	}

	if (getConfigPage(port, MPI_CONFIG_PAGETYPE_IOC, 2, 0, buf, sizeof buf) == 1)
	{
		showConfigPage(port, "IOCPage2", buf);
	}

	if (getConfigPage(port, MPI_CONFIG_PAGETYPE_IOC, 3, 0, buf, sizeof buf) == 1)
	{
		showConfigPage(port, "IOCPage3", buf);
	}

	if (getConfigPage(port, MPI_CONFIG_PAGETYPE_IOC, 4, 0, buf, sizeof buf) == 1)
	{
		showConfigPage(port, "IOCPage4", buf);
	}

	if (getConfigPage(port, MPI_CONFIG_PAGETYPE_IOC, 5, 0, buf, sizeof buf) == 1)
	{
		showConfigPage(port, "IOCPage5", buf);
	}

	if (port->portType == MPI_PORTFACTS_PORTTYPE_SCSI)
	{
		if (getConfigPage(port, MPI_CONFIG_PAGETYPE_SCSI_PORT, 0, 0, buf, sizeof buf) == 1)
		{
			showConfigPage(port, "SCSIPortPage0", buf);
		}

		if (getConfigPage(port, MPI_CONFIG_PAGETYPE_SCSI_PORT, 1, 0, buf, sizeof buf) == 1)
		{
			showConfigPage(port, "SCSIPortPage1", buf);
		}

		if (getConfigPage(port, MPI_CONFIG_PAGETYPE_SCSI_PORT, 2, 0, buf, sizeof buf) == 1)
		{
			showConfigPage(port, "SCSIPortPage2", buf);
		}

		for (i = 0; i < 16; i++)
		{
			if (getConfigPage(port, MPI_CONFIG_PAGETYPE_SCSI_DEVICE, 0, i, buf, sizeof buf) == 1)
			{
				sprintf(name, "SCSIDevicePage0 / %d", i);
				showConfigPage(port, name, buf);
			}

			if (getConfigPage(port, MPI_CONFIG_PAGETYPE_SCSI_DEVICE, 1, i, buf, sizeof buf) == 1)
			{
				sprintf(name, "SCSIDevicePage1 / %d", i);
				showConfigPage(port, name, buf);
			}

			if (getConfigPage(port, MPI_CONFIG_PAGETYPE_SCSI_DEVICE, 2, i, buf, sizeof buf) == 1)
			{
				sprintf(name, "SCSIDevicePage2 / %d", i);
				showConfigPage(port, name, buf);
			}

			if (getConfigPage(port, MPI_CONFIG_PAGETYPE_SCSI_DEVICE, 3, i, buf, sizeof buf) == 1)
			{
				sprintf(name, "SCSIDevicePage3 / %d", i);
				showConfigPage(port, name, buf);
			}
		}
	}

	if (port->portType == MPI_PORTFACTS_PORTTYPE_FC)
	{
		if (getConfigPage(port, MPI_CONFIG_PAGETYPE_FC_PORT, 0, 0, buf, sizeof buf) == 1)
		{
			showConfigPage(port, "FCPortPage0", buf);
		}

		if (getConfigPage(port, MPI_CONFIG_PAGETYPE_FC_PORT, 1, 0, buf, sizeof buf) == 1)
		{
			showConfigPage(port, "FCPortPage1", buf);
		}

		if (getConfigPage(port, MPI_CONFIG_PAGETYPE_FC_PORT, 2, 0, buf, sizeof buf) == 1)
		{
			showConfigPage(port, "FCPortPage2", buf);
		}

		if (port->maxPersistentIds * sizeof (PersistentData_t) + sizeof (ConfigPageHeader_t) > sizeof buf)
		{
			for (i = 0; i < port->maxPersistentIds; i++)
			{
				if (getConfigPage(port, MPI_CONFIG_PAGETYPE_FC_PORT, 3,
								  MPI_FC_PORT_PGAD_FORM_INDEX + i, buf, sizeof buf) == 1)
				{
					sprintf(name, "FCPortPage3 / %d", i);
					showConfigPage(port, name, buf);
				}
			}
		}
		else
		{
			if (getConfigPage(port, MPI_CONFIG_PAGETYPE_FC_PORT, 3, 0, buf, sizeof buf) == 1)
			{
				showConfigPage(port, "FCPortPage3", buf);
			}
		}

		if (getConfigPage(port, MPI_CONFIG_PAGETYPE_FC_PORT, 4, 0, buf, sizeof buf) == 1)
		{
			showConfigPage(port, "FCPortPage4", buf);
		}

		if (getConfigPage(port, MPI_CONFIG_PAGETYPE_FC_PORT, 6, 0, buf, sizeof buf) == 1)
		{
			showConfigPage(port, "FCPortPage6", buf);
		}

		if (getConfigPage(port, MPI_CONFIG_PAGETYPE_FC_PORT, 7, 0, buf, sizeof buf) == 1)
		{
			showConfigPage(port, "FCPortPage7", buf);
		}

		i = 0xffffff;
		while (TRUE)
		{
			if (getConfigPage(port, MPI_CONFIG_PAGETYPE_FC_DEVICE, 0, i, buf, sizeof buf) == 1)
			{
				i = get32(((FCDevicePage0_t *)buf)->PortIdentifier);
				sprintf(name, "FCDevicePage0 / %06x", i);
				showConfigPage(port, name, buf);
			}
			else
				break;
		}
	}

	if (port->portType == MPI_PORTFACTS_PORTTYPE_SAS)
	{
		if (getConfigPage(port, MPI_CONFIG_EXTPAGETYPE_SAS_IO_UNIT, 0, 0, buf, sizeof buf) == 1)
		{
			showConfigPage(port, "SASIOUnitPage0", buf);
		}

		if (getConfigPage(port, MPI_CONFIG_EXTPAGETYPE_SAS_IO_UNIT, 1, 0, buf, sizeof buf) == 1)
		{
			showConfigPage(port, "SASIOUnitPage1", buf);
		}

		if (getConfigPage(port, MPI_CONFIG_EXTPAGETYPE_SAS_IO_UNIT, 2, 0, buf, sizeof buf) == 1)
		{
			showConfigPage(port, "SASIOUnitPage2", buf);
		}

		if (getConfigPage(port, MPI_CONFIG_EXTPAGETYPE_SAS_IO_UNIT, 3, 0, buf, sizeof buf) == 1)
		{
			showConfigPage(port, "SASIOUnitPage3", buf);
		}

		i = 0xffff;
		while (TRUE)
		{
			if (getConfigPage(port, MPI_CONFIG_EXTPAGETYPE_SAS_EXPANDER, 0, i, buf, sizeof buf) == 1)
			{
				i = get16(((SasExpanderPage0_t *)buf)->DevHandle);
				sprintf(name, "SASExpanderPage0 / %04x", i);
				showConfigPage(port, name, buf);
			}
			else
				break;
		}

		i = 0xffff;
		while (TRUE)
		{
			if (getConfigPage(port, MPI_CONFIG_EXTPAGETYPE_SAS_DEVICE, 0, i, buf, sizeof buf) == 1)
			{
				i = get16(((SasDevicePage0_t *)buf)->DevHandle);
				sprintf(name, "SASDevicePage0 / %04x", i);
				showConfigPage(port, name, buf);

				if (getConfigPage(port, MPI_CONFIG_EXTPAGETYPE_SAS_DEVICE, 1, i + 0x20000000, buf, sizeof buf) == 1)
				{
					sprintf(name, "SASDevicePage1 / %04x", i);
					showConfigPage(port, name, buf);
				}
			}
			else
				break;
		}

		for (i = 0; i < 256; i++)
		{
			if (getConfigPage(port, MPI_CONFIG_EXTPAGETYPE_SAS_PHY, 0, i, buf, sizeof buf) == 1)
			{
				sprintf(name, "SASPhyPage0 / %02x", i);
				showConfigPage(port, name, buf);
			}
			else
				break;
		}
	}

	return 1;
}


int
doPortStateSummary(MPT_PORT *port)
{
	char				*temp;
	FCPortPage1_t		 FCPortPage1;
	int					 flags;
	int					 t;
	IOCPage1_t			 IOCPage1;
	int					 timeout;
	int					 depth;
	int					 on;
	IOUnitPage1_t		 IOUnitPage1;
	SCSIPortPage2_t		 SCSIPortPage2;
	int					 settings;
	int					 id;
	SasIOUnitPage1_t	*SASIOUnitPage1;
	ConfigReply_t		 rep;
	int					 length;
	int					 i;

	printf("Current Port State\n------------------\n");
	showPortInfoHeader(port);

	printf("Software Version Information\n----------------------------\n");
	doIdentify(port);

	printf("\nFirmware Settings\n-----------------\n");
	if (port->portType == MPI_PORTFACTS_PORTTYPE_FC)
	{
		if (getConfigPage(port, MPI_CONFIG_PAGETYPE_FC_PORT, 1, 0, &FCPortPage1, sizeof FCPortPage1) != 1)
			return 0;

		t = FCPortPage1.TopologyConfig & MPI_FCPORTPAGE1_TOPOLOGY_MASK;
		switch (t)
		{
		case MPI_FCPORTPAGE1_TOPOLOGY_AUTO:    temp = "Auto";     break;
		case MPI_FCPORTPAGE1_TOPOLOGY_NLPORT:  temp = "NL_Port";  break;
		case MPI_FCPORTPAGE1_TOPOLOGY_NPORT:   temp = "N_Port";   break;
		default:                               temp = "Unknown";  break;
		}
		printf("Link Topology:\t\t\t%s\n", temp);

		if (port->mptVersion < 0x0101)
			temp = "1 Gb";
		else
		{
			t = FCPortPage1.LinkConfig & MPI_FCPORTPAGE1_LCONFIG_SPEED_MASK;
			switch (t)
			{
			case MPI_FCPORTPAGE1_LCONFIG_SPEED_AUTO:   temp = "Auto";     break;
			case MPI_FCPORTPAGE1_LCONFIG_SPEED_1GIG:   temp = "1 Gb";     break;
			case MPI_FCPORTPAGE1_LCONFIG_SPEED_2GIG:   temp = "2 Gb";     break;
			case MPI_FCPORTPAGE1_LCONFIG_SPEED_4GIG:   temp = "4 Gb";     break;
			case MPI_FCPORTPAGE1_LCONFIG_SPEED_10GIG:  temp = "10 Gb";    break;
			default:                                   temp = "Unknown";  break;
			}
		}
		printf("Link Speed:\t\t\t%s\n", temp);

		flags = get32(FCPortPage1.Flags);

		t = (flags & MPI_FCPORTPAGE1_FLAGS_PROT_FCP_INIT) != 0;
		printf("FCP Initiator protocol:\t\t%s\n", t == 0 ? "Disabled" : "Enabled");

		t = (flags & MPI_FCPORTPAGE1_FLAGS_PROT_FCP_TARG) != 0;
		printf("FCP Target protocol:\t\t%s\n", t == 0 ? "Disabled" : "Enabled");

		t = (flags & MPI_FCPORTPAGE1_FLAGS_PROT_LAN) != 0;
		printf("LAN protocol:\t\t\t%s\n", t == 0 ? "Disabled" : "Enabled");

		t = (flags & MPI_FCPORTPAGE1_FLAGS_SORT_BY_DID) != 0;
		printf("Assignment of Bus/Target IDs:\t%s\n", t == 0 ? "SortByWWN" : "SortByDID");

		t = (flags & MPI_FCPORTPAGE1_FLAGS_IMMEDIATE_ERROR_REPLY) != 0;
		printf("Immediate Error Reply:\t\t%s\n", t == 0 ? "Disabled" : "Enabled");
	
		t = (flags & MPI_FCPORTPAGE1_FLAGS_MAINTAIN_LOGINS) != 0;
		printf("Maintain Logins:\t\t%s\n", t == 0 ? "Disabled" : "Enabled");

		t = FCPortPage1.HardALPA;
		printf("Hard AL_PA:\t\t\t%02x\n", t);

		t = FCPortPage1.InitiatorDeviceTimeout;
		if (t == 0)
			t = 60;
		printf("Initiator Device Timeout:\t%d\n", t);

		t = FCPortPage1.InitiatorIoPendTimeout;
		if (t == 0)
			t = 8;
		printf("Initiator I/O Pending Timeout:\t%d\n", t);

		if (getConfigPage(port, MPI_CONFIG_PAGETYPE_IO_UNIT, 1, 0, &IOUnitPage1, sizeof IOUnitPage1) != 1)
			return 0;

		flags = get32(IOUnitPage1.Flags);

		t = (flags & MPI_IOUNITPAGE1_MULTI_PATHING) != 0;
		printf("Multi-pathing:\t\t\t%s\n", t == 0 ? "Disabled" : "Enabled");
	}
	
	if (port->portType == MPI_PORTFACTS_PORTTYPE_SCSI)
	{
		if (getConfigPage(port, MPI_CONFIG_PAGETYPE_SCSI_PORT, 2, 0, &SCSIPortPage2, sizeof SCSIPortPage2) != 1)
			return 0;

		flags = get32(SCSIPortPage2.PortFlags);
		settings = get32(SCSIPortPage2.PortSettings);

		id = settings & MPI_SCSIPORTPAGE2_PORT_HOST_ID_MASK;
		printf("Host SCSI ID:\t\t\t%d\n", id);

		t = (flags & MPI_SCSIPORTPAGE2_PORT_FLAGS_SCAN_HIGH_TO_LOW) != 0;
		printf("Bus scan order:\t\t\t%s\n", t == 0 ? "LowToHigh" : "HighToLow");

		t = (flags & MPI_SCSIPORTPAGE2_PORT_FLAGS_AVOID_SCSI_RESET) != 0;
		printf("Avoid SCSI bus reset:\t\t%s\n", t == 0 ? "No" : "Yes");

		t = (flags & MPI_SCSIPORTPAGE2_PORT_FLAGS_ALTERNATE_CHS) != 0;
		printf("CHS mapping:\t\t\t%s\n", t == 0 ? "PlugAndPlay" : "AlternateCHS");

		t = (settings & MPI_SCSIPORTPAGE2_PORT_REMOVABLE_MEDIA) >> 6;
		switch (t)
		{
		case 0:   temp = "None";          break;
		case 1:   temp = "BootDrive";     break;
		case 2:   temp = "AnyWithMedia";  break;
		default:  temp = "Unknown";       break;
		}
		printf("Removable media support:\t%s\n", temp);

		t = (settings & MPI_SCSIPORTPAGE2_PORT_SPINUP_DELAY_MASK) >> 8;
		printf("Spinup delay (in seconds):\t%d\n", t);
	}

	if (port->portType == MPI_PORTFACTS_PORTTYPE_SAS)
	{
		if (getConfigPageHeader(port, MPI_CONFIG_EXTPAGETYPE_SAS_IO_UNIT, 1, 0, &rep) != 1)
			return 0;

		length = get16(rep.ExtPageLength) * 4;
		SASIOUnitPage1 = malloc(length);

		if (getConfigPage(port, MPI_CONFIG_EXTPAGETYPE_SAS_IO_UNIT, 1, 0, SASIOUnitPage1, length) != 1)
			return 0;

		t = SASIOUnitPage1->SATAMaxQDepth;
		printf("SATA Maximum Queue Depth:\t%d\n", t);

		printf("Phy Parameters for Phynum:\t");
		for (i = 0; i < SASIOUnitPage1->NumPhys; i++)
			printf("%-5d", i);
		printf("\n");

		printf("  Link Enabled:\t\t\t");
		for (i = 0; i < SASIOUnitPage1->NumPhys; i++)
		{
			t = SASIOUnitPage1->PhyData[i].PhyFlags & MPI_SAS_IOUNIT1_PHY_FLAGS_PHY_DISABLE;
			printf("%-5s", t ? "No" : "Yes");
		}
		printf("\n");

		printf("  Link Min Rate:\t\t");
		for (i = 0; i < SASIOUnitPage1->NumPhys; i++)
		{
			t = SASIOUnitPage1->PhyData[i].MaxMinLinkRate & MPI_SAS_IOUNIT1_MIN_RATE_MASK;
			printf("%-5s", t == MPI_SAS_IOUNIT1_MIN_RATE_3_0 ? "3.0" : "1.5");
		}
		printf("\n");

		printf("  Link Max Rate:\t\t");
		for (i = 0; i < SASIOUnitPage1->NumPhys; i++)
		{
			t = SASIOUnitPage1->PhyData[i].MaxMinLinkRate & MPI_SAS_IOUNIT1_MAX_RATE_MASK;
			printf("%-5s", t == MPI_SAS_IOUNIT1_MAX_RATE_3_0 ? "3.0" : "1.5");
		}
		printf("\n");

		printf("  SSP Initiator Enabled:\t");
		for (i = 0; i < SASIOUnitPage1->NumPhys; i++)
		{
			t = get32(SASIOUnitPage1->PhyData[i].ControllerPhyDeviceInfo) & MPI_SAS_DEVICE_INFO_SSP_INITIATOR;
			printf("%-5s", t ? "No" : "Yes");
		}
		printf("\n");

		printf("  SSP Target Enabled:\t\t");
		for (i = 0; i < SASIOUnitPage1->NumPhys; i++)
		{
			t = get32(SASIOUnitPage1->PhyData[i].ControllerPhyDeviceInfo) & MPI_SAS_DEVICE_INFO_SSP_TARGET;
			printf("%-5s", t ? "No" : "Yes");
		}
		printf("\n");
	}

	if (getConfigPage(port, MPI_CONFIG_PAGETYPE_IOC, 1, 0, &IOCPage1, sizeof IOCPage1) != 1)
		return 0;

	flags = get32(IOCPage1.Flags) & MPI_IOCPAGE1_REPLY_COALESCING;
	timeout = get32(IOCPage1.CoalescingTimeout);
	depth = IOCPage1.CoalescingDepth;

	on = flags != 0 && timeout != 0 && depth != 0;
	if (on)
		printf("Interrupt Coalescing:\t\tEnabled, timeout is %d us, depth is %d\n",
			   timeout, depth);
	else
		printf("Interrupt Coalescing:\t\tDisabled\n");

	if (port->portType == MPI_PORTFACTS_PORTTYPE_FC)
	{
		printf("\nPersistent Mappings\n-------------------\n");
		doFcPersistentMappings(port, 1);
	}

	if (port->portType == MPI_PORTFACTS_PORTTYPE_SAS)
	{
		printf("\nPersistent Mappings\n-------------------\n");
		doSasPersistentMappings(port, 1);
	}

	return 1;
}


int
getPortInfo(MPT_PORT *port)
{
	IOCFactsReply_t		 IOCFacts;
	PortFactsReply_t	 PortFacts;
	IOCPage0_t			 IOCPage0;
	char				*chipName;
	int					 family;
	int					 revision;

	port->payOff = 0;

	if (getIocFacts(port, &IOCFacts) != 1)
		return 0;

//	dumpMemory(&IOCFacts, sizeof IOCFacts, "IOCFactsReply");

	port->iocNumber = IOCFacts.IOCNumber;
	port->mptVersion = get16(IOCFacts.MsgVersion);
	port->whoInit = IOCFacts.WhoInit;
	port->productId = get16(IOCFacts.ProductID);
	port->payOff = get16(IOCFacts.CurReplyFrameSize);
	port->maxBuses = IOCFacts.MaxBuses;
	if (port->maxBuses == 0)
		port->maxBuses = 1;
	port->maxTargets = IOCFacts.MaxDevices;
	if (port->maxTargets == 0)
		port->maxTargets = 255;  /* Linux limit! */
	port->maxLuns = 64;

	if (getConfigPage(port, MPI_CONFIG_PAGETYPE_IOC, 0, 0, &IOCPage0, sizeof IOCPage0) != 1)
		return 0;

	port->deviceId = get16(IOCPage0.DeviceID);
	port->revisionId = IOCPage0.RevisionID;

	family = port->productId & MPI_FW_HEADER_PID_FAMILY_MASK;
	revision = port->revisionId;
	switch (port->deviceId)
	{
	case MPI_MANUFACTPAGE_DEVICEID_FC909:                     chipName = "FC909";     break;
	case MPI_MANUFACTPAGE_DEVICEID_FC919:                     chipName = "FC919";     break;
	case MPI_MANUFACTPAGE_DEVICEID_FC929:                     chipName = "FC929";     break;
	case MPI_MANUFACTPAGE_DEVICEID_FC919X:                    chipName = "FC919X";
		if (family == MPI_FW_HEADER_PID_FAMILY_919XL_FC)      chipName = "FC919XL";   break;
	case MPI_MANUFACTPAGE_DEVICEID_FC929X:		              chipName = "FC929X";
		if (family == MPI_FW_HEADER_PID_FAMILY_919XL_FC)      chipName = "FC929XL";   break;
	case MPI_MANUFACTPAGE_DEVICEID_FC939X:		              chipName = "FC939X";    break;
	case MPI_MANUFACTPAGE_DEVICEID_FC949X:		              chipName = "FC949X";    break;
	case MPI_MANUFACTPAGE_DEVID_53C1030:                      chipName = "53C1030";
		if (family == MPI_FW_HEADER_PID_FAMILY_1030TA0_SCSI)  chipName = "53C1030T";
		if (family == MPI_FW_HEADER_PID_FAMILY_1020TA0_SCSI)  chipName = "53C1020A";
		else if (revision >= 0xc0)                            chipName = "53C1020A";  break;
	case MPI_MANUFACTPAGE_DEVID_1030_53C1035:                 chipName = "53C1035";   break;
	case MPI_MANUFACTPAGE_DEVID_SAS1064:                      chipName = "SAS1064";   break;
	case MPI_MANUFACTPAGE_DEVID_SAS1064E:                     chipName = "SAS1064E";  break;
	case MPI_MANUFACTPAGE_DEVID_SAS1066:                      chipName = "SAS1066";   break;
	case MPI_MANUFACTPAGE_DEVID_SAS1066E:                     chipName = "SAS1066E";  break;
	case MPI_MANUFACTPAGE_DEVID_SAS1068:                      chipName = "SAS1068";   break;
	case MPI_MANUFACTPAGE_DEVID_SAS1068E:                     chipName = "SAS1068E";  break;
	default:                                                  chipName = "Unknown";   break;
	}

	port->chipName = chipName;

	if (port->mptVersion < 0x0102)
		port->fwVersion = get16(IOCFacts.Reserved_0101_FWVersion);
	else
		port->fwVersion = get32(*(U32 *)&IOCFacts.FWVersion);

	if (getPortFacts(port, &PortFacts) != 1)
		return 0;

//	dumpMemory(&PortFacts, sizeof PortFacts, "PortFactsReply");
	
	port->portType = PortFacts.PortType;
	port->maxPersistentIds = get16(PortFacts.MaxPersistentIDs);
	port->hostScsiId = get16(PortFacts.PortSCSIID);

	if (port->maxTargets > get16(PortFacts.MaxDevices))
		port->maxTargets = get16(PortFacts.MaxDevices);

	return 1;
}


int
showPortInfo(MPT_PORT *port)
{
	FCPortPage0_t	 FCPortPage0;
	FCDevicePage0_t	 FCDevicePage0;
	int				 i;

	if (port->portType == MPI_PORTFACTS_PORTTYPE_FC)
	{
		char			 buf[16];
		int				 portId;

		if (getConfigPage(port, MPI_CONFIG_PAGETYPE_FC_PORT, 0, 0, &FCPortPage0, sizeof FCPortPage0) != 1)
			return 0;

		portId = get32(FCPortPage0.PortIdentifier);

		sprintf(buf, "%s Port", port->chipName);
		printf("            %-16s                           %08x%08x  %06x\n", buf,
			   get32(FCPortPage0.WWPN.High), get32(FCPortPage0.WWPN.Low), portId);

		i = 0xffffff;
		while (TRUE)
		{
			if (getConfigPage(port, MPI_CONFIG_PAGETYPE_FC_DEVICE, 0, i,
							  &FCDevicePage0, sizeof FCDevicePage0) != 1)
				break;

			i = get32(FCDevicePage0.PortIdentifier);

			if (i == portId)
				continue;

			if (FCDevicePage0.Protocol & MPI_FC_DEVICE_PAGE0_PROT_FCP_TARGET)
				continue;

			if (FCDevicePage0.Protocol & MPI_FC_DEVICE_PAGE0_PROT_FCP_INITIATOR)
				printf("            FCP Initiator                              %08x%08x  %06x\n",
					   get32(FCDevicePage0.WWPN.High), get32(FCDevicePage0.WWPN.Low), i);
			else
				printf("            Non-FCP                                    %08x%08x  %06x\n",
					   get32(FCDevicePage0.WWPN.High), get32(FCDevicePage0.WWPN.Low), i);
		}
		return 1;
	}

	return 0;
}


int
showPortInfoHeader(MPT_PORT *port)
{
	FCPortPage0_t		 FCPortPage0;
	SasIOUnitPage0_t	*SASIOUnitPage0;
	int					 i;

	if (port->portType == MPI_PORTFACTS_PORTTYPE_SCSI)
	{
		printf("%s's host SCSI ID is %d\n\n", port->chipName, port->hostScsiId);
		return 1;
	}

	if (port->portType == MPI_PORTFACTS_PORTTYPE_FC)
	{
		char			*attach;
		char			*speed;

		if (getConfigPage(port, MPI_CONFIG_PAGETYPE_FC_PORT, 0, 0, &FCPortPage0, sizeof FCPortPage0) != 1)
			return 0;

		switch (get32(FCPortPage0.Flags) & MPI_FCPORTPAGE0_FLAGS_ATTACH_TYPE_MASK)
		{
		case MPI_FCPORTPAGE0_FLAGS_ATTACH_NO_INIT:         attach = NULL;                    break;
		case MPI_FCPORTPAGE0_FLAGS_ATTACH_POINT_TO_POINT:  attach = "point to point";        break;
		case MPI_FCPORTPAGE0_FLAGS_ATTACH_PRIVATE_LOOP:    attach = "private loop";          break;
		case MPI_FCPORTPAGE0_FLAGS_ATTACH_FABRIC_DIRECT:   attach = "fabric direct attach";  break;
		case MPI_FCPORTPAGE0_FLAGS_ATTACH_PUBLIC_LOOP:     attach = "public loop";           break;
		default:                                           attach = "unknown";               break;
		}

		if (port->mptVersion < 0x0101)
			speed = "1 Gbaud";
		else
			switch (get32(FCPortPage0.CurrentSpeed))
			{
			case MPI_FCPORTPAGE0_CURRENT_SPEED_1GBIT:   speed = "1 Gbaud";   break;
			case MPI_FCPORTPAGE0_CURRENT_SPEED_2GBIT:   speed = "2 Gbaud";   break;
			case MPI_FCPORTPAGE0_CURRENT_SPEED_4GBIT:   speed = "4 Gbaud";   break;
			case MPI_FCPORTPAGE0_CURRENT_SPEED_10GBIT:  speed = "10 Gbaud";  break;
			default:                                    speed = "unknown";   break;
			}

		if (attach != NULL)
			printf("%s's link is online, type is %s, speed is %s\n\n", port->chipName, attach, speed);
		else
			printf("%s's link is offline\n\n", port->chipName);
		return 1;
	}

	if (port->portType == MPI_PORTFACTS_PORTTYPE_SAS)
	{
		ConfigReply_t	 rep;
		int				 length;

		if (getConfigPageHeader(port, MPI_CONFIG_EXTPAGETYPE_SAS_IO_UNIT, 0, 0, &rep) != 1)
			return 0;

		length = get16(rep.ExtPageLength) * 4;
		SASIOUnitPage0 = malloc(length);

		if (getConfigPage(port, MPI_CONFIG_EXTPAGETYPE_SAS_IO_UNIT, 0, 0, SASIOUnitPage0, length) != 1)
			return 0;

		if (SASIOUnitPage0->NumPhys == 1)
			printf("%s's link is ", port->chipName);
		else
			printf("%s's links are ", port->chipName);

		for (i = 0; i < SASIOUnitPage0->NumPhys; i++)
		{
			if (i != 0)
				printf(", ");

			switch (SASIOUnitPage0->PhyData[i].NegotiatedLinkRate)
			{
			case MPI_SAS_IOUNIT0_RATE_PHY_DISABLED:
				printf("disabled");
				break;

			case MPI_SAS_IOUNIT0_RATE_FAILED_SPEED_NEGOTIATION:
				printf("failed");
				break;

			case MPI_SAS_IOUNIT0_RATE_1_5:
				printf("1.5 Gbps");
				break;

			case MPI_SAS_IOUNIT0_RATE_3_0:
				printf("3.0 Gbps");
				break;

			default:
				printf("unknown");
				break;
			}
		}

		printf("\n\n");

		free(SASIOUnitPage0);

		return 1;
	}

	return 0;
}


int
getDeviceInfo(MPT_PORT *port, int bus, int target, char *buf, int len)
{
	SCSIDevicePage0_t	 SCSIDevicePage0;
	FCDevicePage0_t		 FCDevicePage0;
	SasDevicePage0_t	 SASDevicePage0;
	int					 b_t;

	buf[0] = '\0';

	b_t = (bus << 8) + mapOsToHwTarget(port, target);

	if (port->portType == MPI_PORTFACTS_PORTTYPE_SCSI)
	{
		if (getConfigPage(port, MPI_CONFIG_PAGETYPE_SCSI_DEVICE, 0, b_t,
						  &SCSIDevicePage0, sizeof SCSIDevicePage0) == 1)
		{
			int		 parameters = get32(SCSIDevicePage0.NegotiatedParameters);
			int		 speed = parameters & MPI_SCSIDEVPAGE0_NP_NEG_SYNC_PERIOD_MASK;
			int		 width = parameters & MPI_SCSIDEVPAGE0_NP_WIDE;
			int		 mbps;
			char	*speed_string;
			char	*width_string;

			if ((parameters & MPI_SCSIDEVPAGE0_NP_NEG_SYNC_OFFSET_MASK) == 0)
				speed = 0;

			if (speed == 0)
			{
				speed_string = "Async";
				mbps = 0;
			}
			else if (speed <= 0x800)
			{
				speed_string = "Ultra4";
				mbps = 160;
			}
			else if (speed <= 0x900)
			{
				speed_string = "Ultra3";
				mbps = 80;
			}
			else if (speed <= 0xa00)
			{
				speed_string = "Ultra2";
				mbps = 40;
			}
			else if (speed <= 0xc00)
			{
				speed_string = "Ultra";
				mbps = 20;
			}
			else if (speed <= 0x1900)
			{
				speed_string = "Fast";
				mbps = 64000 / speed;
			}
			else
			{
				speed_string = "Sync";
				mbps = 64000 / speed;
			}

			if (width == 0)
			{
				width_string = "Narrow";
			}
			else
			{
				width_string = "Wide";
				mbps *= 2;
			}

			if (get32(SCSIDevicePage0.Information) & MPI_SCSIDEVPAGE0_INFO_PARAMS_NEGOTIATED)
			{
				sprintf(buf, "%s %s", speed_string, width_string);
				if (mbps > 0)
					sprintf(buf+strlen(buf), ", %d MB/sec", mbps);
				return 1;
			}
		}
	}

	if (port->portType == MPI_PORTFACTS_PORTTYPE_FC)
	{
		if (getConfigPage(port, MPI_CONFIG_PAGETYPE_FC_DEVICE, 0,
						  MPI_FC_DEVICE_PGAD_FORM_BUS_TID + b_t,
						  &FCDevicePage0, sizeof FCDevicePage0) == 1)
		{
			sprintf(buf, "%08x%08x  %06x",
					get32(FCDevicePage0.WWPN.High), get32(FCDevicePage0.WWPN.Low),
					get32(FCDevicePage0.PortIdentifier));
			return 1;
		}
	}

	if (port->portType == MPI_PORTFACTS_PORTTYPE_SAS)
	{
		if (getConfigPage(port, MPI_CONFIG_EXTPAGETYPE_SAS_DEVICE, 0,
						  (MPI_SAS_DEVICE_PGAD_FORM_BUS_TARGET_ID
						   <<MPI_SAS_DEVICE_PGAD_FORM_SHIFT) + b_t,
						  &SASDevicePage0, sizeof SASDevicePage0) == 1)
		{
			sprintf(buf, "%08x%08x    %2d",
					get32(SASDevicePage0.SASAddress.High),
					get32(SASDevicePage0.SASAddress.Low),
					SASDevicePage0.PhyNum);
			return 1;
		}
	}

	return 0;
}


int
getDeviceInfoHeader(MPT_PORT *port, char *buf, int len)
{
	buf[0] = '\0';

	if (port->portType == MPI_PORTFACTS_PORTTYPE_SCSI)
	{
		sprintf(buf, "Negotiated Speed & Width");
		return 1;
	}

	if (port->portType == MPI_PORTFACTS_PORTTYPE_FC)
	{
		sprintf(buf, "      WWPN        PortId");
		return 1;
	}

	if (port->portType == MPI_PORTFACTS_PORTTYPE_SAS)
	{
		sprintf(buf, "   SASAddress     PhyNum");
		return 1;
	}

	return 0;
}


int
getIocFacts(MPT_PORT *port, IOCFactsReply_t *rep)
{
	IOCFacts_t	 req;

	memset(&req, 0, sizeof req);
	memset(rep, 0, sizeof *rep);

	req.Function			= MPI_FUNCTION_IOC_FACTS;

	return doMptCommand(port, &req, sizeof req, rep, sizeof *rep, NULL, 0, NULL, 0, 10);
}


int
getPortFacts(MPT_PORT *port, PortFactsReply_t *rep)
{
	IOCFacts_t	 req;

	memset(&req, 0, sizeof req);
	memset(rep, 0, sizeof *rep);

	req.Function			= MPI_FUNCTION_PORT_FACTS;

	return doMptCommand(port, &req, sizeof req, rep, sizeof *rep, NULL, 0, NULL, 0, 10);
}


int
getConfigPageHeader(MPT_PORT *port, int type, int number, int address, ConfigReply_t *repOut)
{
	Config_t		 req;
	ConfigReply_t	 rep;

	memset(&req, 0, sizeof req);
	memset(&rep, 0, sizeof rep);

	req.Function			= MPI_FUNCTION_CONFIG;
	req.Action				= MPI_CONFIG_ACTION_PAGE_HEADER;
	if (type > MPI_CONFIG_PAGETYPE_EXTENDED)
	{
		req.Header.PageType	= MPI_CONFIG_PAGETYPE_EXTENDED;
		req.ExtPageType		= type;
	}
	else
	{
		req.Header.PageType	= type;
	}
	req.Header.PageNumber	= number;
	req.PageAddress			= set32(address);

	if (doMptCommand(port, &req, sizeof req - sizeof req.PageBufferSGE, &rep, sizeof rep,
					 NULL, 0, NULL, 0, 10) != 1)
		return 0;

	if (get16(rep.IOCStatus) != MPI_IOCSTATUS_SUCCESS)
		return 0;

	if (repOut != NULL)
		memcpy(repOut, &rep, sizeof rep);

	return 1;
}


int
getConfigPageAction(MPT_PORT *port, int action, int type, int number, int address, void *page, int pageSize)
{
	Config_t			 req;
	ConfigReply_t		 rep;
	ConfigPageHeader_t	 header;
	int					 length;

	memset(&req, 0, sizeof req);
	memset(&rep, 0, sizeof rep);

	if (getConfigPageHeader(port, type, number, address, &rep) != 1)
		return 0;

	header = rep.Header;
	length = get16(rep.ExtPageLength);

	req.Function			= MPI_FUNCTION_CONFIG;
	if (action != -1)
		req.Action			= action;
	else if ((rep.Header.PageType & MPI_CONFIG_PAGEATTR_MASK) == MPI_CONFIG_PAGEATTR_PERSISTENT ||
			 (rep.Header.PageType & MPI_CONFIG_PAGEATTR_MASK) == MPI_CONFIG_PAGEATTR_RO_PERSISTENT)
		req.Action			= MPI_CONFIG_ACTION_PAGE_READ_NVRAM;
	else
		req.Action			= MPI_CONFIG_ACTION_PAGE_READ_CURRENT;
	req.ExtPageType			= rep.ExtPageType;
	req.ExtPageLength		= rep.ExtPageLength;
	req.Header				= rep.Header;
	req.PageAddress			= set32(address);

	if (doMptCommand(port, &req, sizeof req - sizeof req.PageBufferSGE, &rep, sizeof rep,
					 page, pageSize, NULL, 0, 10) != 1)
		return 0;

	if (get16(rep.IOCStatus) == MPI_IOCSTATUS_CONFIG_INVALID_DATA)
	{
		if (action == MPI_CONFIG_ACTION_PAGE_READ_NVRAM)
		{
			printf("\nNon-volatile storage for this page is invalid!\n");
#if 1
			printf("The current values for this page will be used instead\n");
			req.Action = MPI_CONFIG_ACTION_PAGE_READ_NVRAM;
#else
			return 0;
#endif
		}

		if (req.Action == MPI_CONFIG_ACTION_PAGE_READ_NVRAM)
		{
			req.Action		= MPI_CONFIG_ACTION_PAGE_READ_CURRENT;

			if (doMptCommand(port, &req, sizeof req - sizeof req.PageBufferSGE, &rep, sizeof rep,
							 page, pageSize, NULL, 0, 10) != 1)
				return 0;
		}
	}

	if (get16(rep.IOCStatus) != MPI_IOCSTATUS_SUCCESS)
		return 0;

	if (type > MPI_CONFIG_PAGETYPE_EXTENDED)
	{
		if (get16(rep.ExtPageLength) == 0)
			return 0;
		if (memcmp(&header, &rep.Header, sizeof header) != 0)
			printf("warning, header in HEADER reply does not match header in READ reply (%08x vs. %08x)\n",
				   get32(*(U32 *)&header), get32(*(U32 *)&rep.Header));
		if (length != get16(rep.ExtPageLength))
			printf("warning, length in HEADER reply does not match length in READ reply (%d vs. %d)\n",
				   length, get16(rep.ExtPageLength));
		if (get16(rep.ExtPageLength) != get16(((ConfigExtendedPageHeader_t *)page)->ExtPageLength))
			printf("warning, page length in reply does not match page length in buffer (%d vs. %d)\n",
				   get16(rep.ExtPageLength), get16(((ConfigExtendedPageHeader_t *)page)->ExtPageLength));
	}
	else
	{
		if (rep.Header.PageLength == 0)
			return 0;
		if (memcmp(&header, &rep.Header, sizeof header) != 0)
			printf("warning, header in HEADER reply does not match header in READ reply (%08x vs. %08x)\n",
				   get32(*(U32 *)&header), get32(*(U32 *)&rep.Header));
		if (rep.Header.PageLength != ((ConfigPageHeader_t *)page)->PageLength)
			printf("warning, page length in reply does not match page length in buffer (%d vs. %d)\n",
				   rep.Header.PageLength, ((ConfigPageHeader_t *)page)->PageLength);
	}

	return 1;
}


int
getConfigPage(MPT_PORT *port, int type, int number, int address, void *page, int pageSize)
{
	return getConfigPageAction(port, -1, type, number, address, page, pageSize);
}


int
setConfigPageAction(MPT_PORT *port, int action, int type, int number, int address, void *page, int pageSize)
{
	Config_t			 req;
	ConfigReply_t		 rep;
	ConfigPageHeader_t	 header;
	int					 length;

	memset(&req, 0, sizeof req);
	memset(&rep, 0, sizeof rep);

	if (getConfigPageHeader(port, type, number, address, &rep) != 1)
		return 0;

	header = rep.Header;
	length = get16(rep.ExtPageLength);

	req.Function			= MPI_FUNCTION_CONFIG;
	if (action != -1)
		req.Action			= action;
	else if ((rep.Header.PageType & MPI_CONFIG_PAGEATTR_MASK) == MPI_CONFIG_PAGEATTR_PERSISTENT ||
			 (rep.Header.PageType & MPI_CONFIG_PAGEATTR_MASK) == MPI_CONFIG_PAGEATTR_RO_PERSISTENT)
		req.Action			= MPI_CONFIG_ACTION_PAGE_WRITE_NVRAM;
	else
		req.Action			= MPI_CONFIG_ACTION_PAGE_WRITE_CURRENT;
	req.ExtPageType			= rep.ExtPageType;
	req.ExtPageLength		= rep.ExtPageLength;
	req.Header				= rep.Header;
	req.PageAddress			= set32(address);

	if (doMptCommand(port, &req, sizeof req - sizeof req.PageBufferSGE, &rep, sizeof rep,
					 NULL, 0, page, pageSize, 10) != 1)
		return 0;

	if (get16(rep.IOCStatus) != MPI_IOCSTATUS_SUCCESS)
		return 0;

	if (type > MPI_CONFIG_PAGETYPE_EXTENDED)
	{
		if (get16(rep.ExtPageLength) == 0)
			return 0;
		if (memcmp(&header, &rep.Header, sizeof header) != 0)
			printf("warning, header in HEADER reply does not match header in READ reply (%08x vs. %08x)\n",
				   get32(*(U32 *)&header), get32(*(U32 *)&rep.Header));
		if (length != get16(rep.ExtPageLength))
			printf("warning, length in HEADER reply does not match length in READ reply (%d vs. %d)\n",
				   length, get16(rep.ExtPageLength));
		if (get16(rep.ExtPageLength) != get16(((ConfigExtendedPageHeader_t *)page)->ExtPageLength))
			printf("warning, page length in reply does not match page length in buffer (%d vs. %d)\n",
				   get16(rep.ExtPageLength), get16(((ConfigExtendedPageHeader_t *)page)->ExtPageLength));
	}
	else
	{
		if (rep.Header.PageLength == 0)
			return 0;
		if (memcmp(&header, &rep.Header, sizeof header) != 0)
			printf("warning, header in HEADER reply does not match header in READ reply (%08x vs. %08x)\n",
				   get32(*(U32 *)&header), get32(*(U32 *)&rep.Header));
		if (rep.Header.PageLength != ((ConfigPageHeader_t *)page)->PageLength)
			printf("warning, page length in reply does not match page length in buffer (%d vs. %d)\n",
				   rep.Header.PageLength, ((ConfigPageHeader_t *)page)->PageLength);
	}

	return 1;
}


int
setConfigPage(MPT_PORT *port, int type, int number, int address, void *page, int pageSize)
{
	return setConfigPageAction(port, -1, type, number, address, page, pageSize);
}


int
showConfigPage(MPT_PORT *port, char *string, void *page)
{
	ConfigPageHeader_t	*header;
	U32					*buf;
	int					 i;
	int					 n;

	header = (ConfigPageHeader_t *)page;
	buf = (U32 *)page;

	if ((header->PageType & MPI_CONFIG_PAGETYPE_MASK) == MPI_CONFIG_PAGETYPE_EXTENDED)
		n = get16(((ConfigExtendedPageHeader_t *)header)->ExtPageLength);
	else
		n = header->PageLength;

	printf("\n%s\n", string);
	for (i = 0; i < n; i++)
		printf("%04x : %08x\n", i*4, get32(buf[i]));

	return 1;
}


#if __sparc__
int
getProperty(MPT_PORT *port, char *name, char *buf, int bufLen)
{
		SYM_GET_PROPERTY	 getProperty;

		getProperty.PtrName			= (UINT64)(UINT32)name;
		getProperty.PtrBuffer		= (UINT64)(UINT32)buf; 
		getProperty.NameLen			= strlen(name);
		getProperty.BufferLen		= bufLen;
		getProperty.PropertyLen		= 0;

		if (ioctl(port->fileHandle, SYMIOCTL_GET_PROPERTY, &getProperty) == 0)
			return getProperty.PropertyLen;

		return 0;
}
#endif


int
mapOsToHwTarget(MPT_PORT *port, int target)
{
#if __sparc__ && !i386
	if (port->portType == MPI_PORTFACTS_PORTTYPE_FC)
	{
		char				 name[32];
		char				 buffer[16];
		U32					 wwnh;
		U32					 wwnl;
		U32					 port_id;
		FCDevicePage0_t		 FCDevicePage0;
		int					 i;
		int					 t;

		if (port == mappedPort && target == mappedTarget)
			return mappedValue;

		mappedPort = port;
		mappedTarget = target;
		mappedValue = 15;

		sprintf(name, "target-%d-wwn-nv", target);
		t = getProperty(port, name, buffer, sizeof buffer);

		if (t != 8)
		{
			sprintf(name, "target-%d-wwn-cf", target);
			t = getProperty(port, name, buffer, sizeof buffer);
		}

		if (t != 8)
		{
			sprintf(name, "target-%d-wwn-hw", target);
			t = getProperty(port, name, buffer, sizeof buffer);
		}

		if (t == 8)
		{
			wwnh = ((U32 *)buffer)[0];
			wwnl = ((U32 *)buffer)[1];

			for (i = 0; i < 256; i++)
			{
				if (getConfigPage(port, MPI_CONFIG_PAGETYPE_FC_DEVICE, 0,
								  MPI_FC_DEVICE_PGAD_FORM_BUS_TID + i,
								  &FCDevicePage0, sizeof FCDevicePage0) == 1)
				{
					if (wwnh == get32(FCDevicePage0.WWPN.High) &&
						wwnl == get32(FCDevicePage0.WWPN.Low))
					{
						mappedValue = i;
						return i;
					}
				}
			}

			return 15;
		}

		sprintf(name, "target-%d-did-nv", target);
		t = getProperty(port, name, buffer, sizeof buffer);

		if (t != 4)
		{
			sprintf(name, "target-%d-did-cf", target);
			t = getProperty(port, name, buffer, sizeof buffer);
		}

		if (t != 4)
		{
			sprintf(name, "target-%d-did-hw", target);
			t = getProperty(port, name, buffer, sizeof buffer);
		}

		if (t == 4)
		{
			port_id = ((U32 *)buffer)[0];

			for (i = 0; i < 256; i++)
			{
				if (getConfigPage(port, MPI_CONFIG_PAGETYPE_FC_DEVICE, 0,
								  MPI_FC_DEVICE_PGAD_FORM_BUS_TID + i,
								  &FCDevicePage0, sizeof FCDevicePage0) == 1)
				{
					if (port_id == get32(FCDevicePage0.PortIdentifier))
					{
						mappedValue = i;
						return i;
					}
				}
			}

			return 15;
		}

		target = 15;
	}
#endif
	return target;
}


int
doInquiry(MPT_PORT *port, int bus, int target, int lun, unsigned char *buf, int len)
{
	SCSIIORequest_t	 req;
	SCSI_REPLY		 rep;

	memset(&req, 0, sizeof req);
	memset(&rep, 0, sizeof rep);

	req.Function			= MPI_FUNCTION_SCSI_IO_REQUEST;
	req.Bus					= bus;
	req.TargetID			= mapOsToHwTarget(port, target);
	req.CDBLength			= 6;
	req.LUN[1]				= lun;
	req.Control				= set32(MPI_SCSIIO_CONTROL_READ | tagType);
	req.CDB[0]				= 0x12;
	req.CDB[1]				= 0;
	req.CDB[2]				= 0;
	req.CDB[3]				= 0;
	req.CDB[4]				= len;
	req.CDB[5]				= 0;
	req.DataLength			= set32(len);

	return doScsiIo(port, &req, sizeof req - sizeof req.SGL, &rep, sizeof rep, buf, len, NULL, 0, 10);
}


int
doInquiryVpdPage(MPT_PORT *port, int bus, int target, int lun, int page, unsigned char *buf, int len)
{
	SCSIIORequest_t	 req;
	SCSI_REPLY		 rep;

	memset(&req, 0, sizeof req);
	memset(&rep, 0, sizeof rep);

	req.Function			= MPI_FUNCTION_SCSI_IO_REQUEST;
	req.Bus					= bus;
	req.TargetID			= mapOsToHwTarget(port, target);
	req.CDBLength			= 6;
	req.LUN[1]				= lun;
	req.Control				= set32(MPI_SCSIIO_CONTROL_READ | tagType);
	req.CDB[0]				= 0x12;
	req.CDB[1]				= 1;
	req.CDB[2]				= page;
	req.CDB[3]				= 0;
	req.CDB[4]				= len;
	req.CDB[5]				= 0;
	req.DataLength			= set32(len);

	return doScsiIo(port, &req, sizeof req - sizeof req.SGL, &rep, sizeof rep, buf, len, NULL, 0, 10);
}


int
doReadCapacity(MPT_PORT *port, int bus, int target, int lun, unsigned char *buf, int len)
{
	SCSIIORequest_t	 req;
	SCSI_REPLY		 rep;

	memset(&req, 0, sizeof req);
	memset(&rep, 0, sizeof rep);

	req.Function			= MPI_FUNCTION_SCSI_IO_REQUEST;
	req.Bus					= bus;
	req.TargetID			= mapOsToHwTarget(port, target);
	req.CDBLength			= 10;
	req.LUN[1]				= lun;
	req.Control				= set32(MPI_SCSIIO_CONTROL_READ | tagType);
	req.CDB[0]				= 0x25;
	req.CDB[1]				= 0;
	req.CDB[2]				= 0;
	req.CDB[3]				= 0;
	req.CDB[4]				= 0;
	req.CDB[5]				= 0;
	req.CDB[6]				= 0;
	req.CDB[7]				= 0;
	req.CDB[8]				= 0;
	req.CDB[9]				= 0;
	req.DataLength			= set32(len);

	return doScsiIo(port, &req, sizeof req - sizeof req.SGL, &rep, sizeof rep, buf, len, NULL, 0, 10);
}


int
doReadBuffer(MPT_PORT *port, int bus, int target, int lun, int mode, unsigned char *buf, int len)
{
	SCSIIORequest_t	 req;
	SCSI_REPLY		 rep;

	memset(&req, 0, sizeof req);
	memset(&rep, 0, sizeof rep);

	req.Function			= MPI_FUNCTION_SCSI_IO_REQUEST;
	req.Bus					= bus;
	req.TargetID			= mapOsToHwTarget(port, target);
	req.CDBLength			= 10;
	req.LUN[1]				= lun;
	req.Control				= set32(MPI_SCSIIO_CONTROL_READ | tagType);
	req.CDB[0]				= 0x3c;
	req.CDB[1]				= mode;
	req.CDB[2]				= 0;
	req.CDB[3]				= 0;
	req.CDB[4]				= 0;
	req.CDB[5]				= 0;
	req.CDB[6]				= len >> 16;
	req.CDB[7]				= len >> 8;
	req.CDB[8]				= len;
	req.CDB[9]				= 0;
	req.DataLength			= set32(len);

	return doScsiIo(port, &req, sizeof req - sizeof req.SGL, &rep, sizeof rep, buf, len, NULL, 0, 10);
}


int
doWriteBuffer(MPT_PORT *port, int bus, int target, int lun, int mode, unsigned char *buf, int len)
{
	SCSIIORequest_t	 req;
	SCSI_REPLY		 rep;

	memset(&req, 0, sizeof req);
	memset(&rep, 0, sizeof rep);

	req.Function			= MPI_FUNCTION_SCSI_IO_REQUEST;
	req.Bus					= bus;
	req.TargetID			= mapOsToHwTarget(port, target);
	req.CDBLength			= 10;
	req.LUN[1]				= lun;
	req.Control				= set32(MPI_SCSIIO_CONTROL_WRITE | tagType);
	req.CDB[0]				= 0x3b;
	req.CDB[1]				= mode;
	req.CDB[2]				= 0;
	req.CDB[3]				= 0;
	req.CDB[4]				= 0;
	req.CDB[5]				= 0;
	req.CDB[6]				= len >> 16;
	req.CDB[7]				= len >> 8;
	req.CDB[8]				= len;
	req.CDB[9]				= 0;
	req.DataLength			= set32(len);

	return doScsiIo(port, &req, sizeof req - sizeof req.SGL, &rep, sizeof rep, NULL, 0, buf, len, 10);
}


int
doRead(MPT_PORT *port, int bus, int target, int lun,
	   unsigned int lbn, int lbns, int mode, unsigned char *buf, int len)
{
	SCSIIORequest_t	 req;
	SCSI_REPLY		 rep;

	memset(&req, 0, sizeof req);
	memset(&rep, 0, sizeof rep);

	req.Function			= MPI_FUNCTION_SCSI_IO_REQUEST;
	req.Bus					= bus;
	req.TargetID			= mapOsToHwTarget(port, target);
	req.CDBLength			= 10;
	req.LUN[1]				= lun;
	req.Control				= set32(MPI_SCSIIO_CONTROL_READ | tagType);
	req.CDB[0]				= 0x28;
	req.CDB[1]				= 0;
	req.CDB[2]				= lbn >> 24;
	req.CDB[3]				= lbn >> 16;
	req.CDB[4]				= lbn >> 8;
	req.CDB[5]				= lbn;
	req.CDB[6]				= 0;
	req.CDB[7]				= lbns >> 8;
	req.CDB[8]				= lbns;
	req.CDB[9]				= 0;
	req.DataLength			= set32(len);

	return doScsiIo(port, &req, sizeof req - sizeof req.SGL, &rep, sizeof rep, buf, len, NULL, 0, 10);
}


int
doWrite(MPT_PORT *port, int bus, int target, int lun,
		unsigned int lbn, int lbns, int mode, unsigned char *buf, int len)
{
	SCSIIORequest_t	 req;
	SCSI_REPLY		 rep;

	memset(&req, 0, sizeof req);
	memset(&rep, 0, sizeof rep);

	req.Function			= MPI_FUNCTION_SCSI_IO_REQUEST;
	req.Bus					= bus;
	req.TargetID			= mapOsToHwTarget(port, target);
	req.CDBLength			= 10;
	req.LUN[1]				= lun;
	req.Control				= set32(MPI_SCSIIO_CONTROL_WRITE | tagType);
	req.CDB[0]				= 0x2a;
	req.CDB[1]				= 0;
	req.CDB[2]				= lbn >> 24;
	req.CDB[3]				= lbn >> 16;
	req.CDB[4]				= lbn >> 8;
	req.CDB[5]				= lbn;
	req.CDB[6]				= 0;
	req.CDB[7]				= lbns >> 8;
	req.CDB[8]				= lbns;
	req.CDB[9]				= 0;
	req.DataLength			= set32(len);

	return doScsiIo(port, &req, sizeof req - sizeof req.SGL, &rep, sizeof rep, NULL, 0, buf, len, 10);
}


int
doScsiIo(MPT_PORT *port, SCSIIORequest_t *req, int reqSize, SCSI_REPLY *rep, int repSize,
		 void *payIn, int payInSize, void *payOut, int payOutSize, int timeOut)
{
	int		 ioc_status;

	if (doMptCommand(port, req, reqSize, rep, repSize, payIn, payInSize, payOut, payOutSize, timeOut) != 1)
		return 0;

	ioc_status = get16(rep->reply.IOCStatus) & MPI_IOCSTATUS_MASK;

	if (ioc_status == MPI_IOCSTATUS_SCSI_DEVICE_NOT_THERE)
		return 0;

	if (ioc_status == MPI_IOCSTATUS_BUSY ||
		ioc_status == MPI_IOCSTATUS_SCSI_IOC_TERMINATED ||
		rep->reply.SCSIStatus == MPI_SCSI_STATUS_CHECK_CONDITION ||
		rep->reply.SCSIStatus == MPI_SCSI_STATUS_BUSY ||
		rep->reply.SCSIStatus == MPI_SCSI_STATUS_TASK_SET_FULL)
	{
		if (doMptCommand(port, req, reqSize, rep, repSize, payIn, payInSize, payOut, payOutSize, timeOut) != 1)
			return 0;

		ioc_status = get16(rep->reply.IOCStatus) & MPI_IOCSTATUS_MASK;
	}

	if (ioc_status != MPI_IOCSTATUS_SUCCESS &&
		ioc_status != MPI_IOCSTATUS_SCSI_DATA_UNDERRUN &&
		ioc_status != MPI_IOCSTATUS_SCSI_RESIDUAL_MISMATCH)
	{
		printf("ScsiIo failed, IOCStatus = %04x (%s)\n", ioc_status, translateIocStatus(ioc_status));
		return 0;
	}

	if (rep->reply.SCSIStatus != MPI_SCSI_STATUS_SUCCESS)
	{
		if (rep->reply.SCSIStatus == MPI_SCSI_STATUS_CHECK_CONDITION)
		{
			if (rep->sense[2] == 5 && rep->sense[12] == 0x20 && rep->sense[13] == 0x00 && payInSize == 36)
				return 0;

			if (rep->sense[2] == 5 && rep->sense[12] == 0x24 && rep->sense[13] == 0x00 && payInSize == 36)
				return 0;

			if (rep->sense[2] == 5 && rep->sense[12] == 0x25 && rep->sense[13] == 0x00 && payInSize == 36)
				return 0;

			if (rep->sense[2] == 0 && rep->sense[12] == 0x00 && rep->sense[13] == 0x00)
				return 0;

			printf("ScsiIo failed, Check Condition, Key = %d, ASC/ASCQ = %02Xh/%02Xh\n",
				   rep->sense[2], rep->sense[12], rep->sense[13]);
		}
		else
		{
			printf("ScsiIo failed, SCSIStatus = %02x\n", rep->reply.SCSIStatus);
		}
		return 0;
	}

	return 1;
}


int
doFwDownload(MPT_PORT *port, int type, unsigned char *buf, int len, int offset)
{
	FWDownload_t		 req;
	FWDownloadReply_t	 rep;
	FWDownloadTCSGE_t	*tc;
	int					 size;
	int					 ioc_status;

	memset(&req, 0, sizeof req);
	memset(&rep, 0, sizeof rep);

	req.Function			= MPI_FUNCTION_FW_DOWNLOAD;
	req.ImageType			= type;

	tc = (FWDownloadTCSGE_t *)&req.SGL;

	tc->ContextSize			= 0;
	tc->DetailsLength		= 12;
	tc->Flags				= MPI_SGE_FLAGS_TRANSACTION_ELEMENT;

	if (port->mptVersion < 0x0101 && type == MPI_FW_DOWNLOAD_ITYPE_BIOS)
	{
		int		 i;
		U32		 data_l;
		U32		 data_h;
		U32		*p_data_l;
		U32		*p_data_h;

		p_data_l = (U32 *)&buf[0];
		p_data_h = (U32 *)&buf[len] - 1;

		for (i = 0; i < len / 8; i++)
		{
			data_l = *p_data_l;
			data_h = *p_data_h;
			*p_data_l++ = data_h;
			*p_data_h-- = data_l;
		}

		offset += 0x100000;
		buf += len;
	}

	printf("\nDownloading image...\n");

	while (len > 0)
	{
		if (port->mptVersion < 0x0101)
		{
			int		 i;
			U32		 checksum;

			size = min(len, 0x10000);

			if (type == MPI_FW_DOWNLOAD_ITYPE_BIOS)
			{
				offset -= size;
				buf -= size;
			}

			checksum = 0;
			for (i = 0; i < size / 4; i++)
				checksum -= get32(((U32 *)buf)[i]);

			tc->Reserved_0100_Checksum = set32(checksum);
		}
		else
		{
			size = min(len, 0x4000);
			if (size == len)
				req.MsgFlags = MPI_FW_DOWNLOAD_MSGFLGS_LAST_SEGMENT;
		}

		tc->ImageSize		= set32(size);
		tc->ImageOffset		= set32(offset);

		if (doMptCommand(port, &req, sizeof req - sizeof req.SGL + sizeof *tc, &rep, sizeof rep,
						 NULL, 0, buf, size, 60) != 1)
		{
			printf("Download failed\n");
			return 0;
		}

		ioc_status = get16(rep.IOCStatus) & MPI_IOCSTATUS_MASK;

		if (ioc_status != MPI_IOCSTATUS_SUCCESS)
		{
			printf("Download failed, IOCStatus = %04x (%s)\n", ioc_status, translateIocStatus(ioc_status));
			return 0;
		}

		len -= size;

		if (!(port->mptVersion < 0x0101 && type == MPI_FW_DOWNLOAD_ITYPE_BIOS))
		{
			buf += size;
			offset += size;
		}
	}

	printf("Download succeeded\n");

	return 1;
}


int
doFwUpload(MPT_PORT *port, int type, unsigned char *buf, int len, int offset, int *outLen)
{
	FWUpload_t			 req;
	FWUploadReply_t		 rep;
	FWUploadTCSGE_t		*tc;
	int					 size;
	int					 ioc_status;

	memset(&req, 0, sizeof req);
	memset(&rep, 0, sizeof rep);

	req.Function			= MPI_FUNCTION_FW_UPLOAD;
	req.ImageType			= type;

	tc = (FWUploadTCSGE_t *)&req.SGL;

	tc->ContextSize			= 0;
	tc->DetailsLength		= 12;
	tc->Flags				= MPI_SGE_FLAGS_TRANSACTION_ELEMENT;

	while (len > 0)
	{
		size = min(len, 0x4000);

		tc->ImageSize		= set32(size);
		tc->ImageOffset		= set32(offset);

		if (doMptCommand(port, &req, sizeof req - sizeof req.SGL + sizeof *tc, &rep, sizeof rep,
						 buf, size, NULL, 0, 60) != 1)
		{
			printf("Upload failed\n");
			return 0;
		}

		ioc_status = get16(rep.IOCStatus) & MPI_IOCSTATUS_MASK;

		if (ioc_status != MPI_IOCSTATUS_SUCCESS)
		{
			if (ioc_status == MPI_IOCSTATUS_INVALID_FIELD)
			{
				if (offset + size > (int)get32(rep.ActualImageSize))
				{
					len = get32(rep.ActualImageSize) - offset;
					continue;
				}
			}

			printf("Upload failed, IOCStatus = %04x (%s)\n", ioc_status, translateIocStatus(ioc_status));
			return 0;
		}

		buf += size;
		len -= size;
		offset += size;
	}

	*outLen = get32(rep.ActualImageSize);

	return 1;
}


int
doIocInit(MPT_PORT *port, int WhoInit)
{
	IOCInit_t		 req;
	IOCInitReply_t	 rep;
	IOCFactsReply_t	 IOCFacts;

	if (getIocFacts(port, &IOCFacts) != 1)
		return 0;

	memset(&req, 0, sizeof req);
	memset(&rep, 0, sizeof rep);

	req.Function					= MPI_FUNCTION_IOC_INIT;
	req.WhoInit						= WhoInit;
	req.Flags						= 0;
	req.MaxDevices					= IOCFacts.MaxDevices;
	req.MaxBuses					= IOCFacts.MaxBuses;
	req.ReplyFrameSize				= IOCFacts.CurReplyFrameSize;
	req.HostMfaHighAddr				= IOCFacts.CurrentHostMfaHighAddr;
	req.SenseBufferHighAddr			= IOCFacts.CurrentSenseBufferHighAddr;
	req.ReplyFifoHostSignalingAddr	= IOCFacts.ReplyFifoHostSignalingAddr;
	req.HostPageBufferSGE			= IOCFacts.HostPageBufferSGE;
	req.MsgVersion					= MPI_VERSION_01_05;
	req.HeaderVersion				= MPI_HEADER_VERSION;

	if (IOCFacts.Flags & MPI_IOCFACTS_FLAGS_REPLY_FIFO_HOST_SIGNAL)
		req.Flags |= MPI_IOCINIT_FLAGS_REPLY_FIFO_HOST_SIGNAL;

#if __linux__
	// make the Linux IOCTL driver a bit happier
	if (req.MaxDevices == 0)
		req.MaxDevices = 255;
#endif

	return doMptCommandCheck(port, &req, sizeof req, &rep, sizeof rep, NULL, 0, NULL, 0, 10);
}


int
doResetPort(MPT_PORT *port)
{
#if WIN32
	int					 status;
	MPI_DIAG_RESET_SRB	 srb;
	int					 inLen;
	int					 outLen;
	int					 retLen;

	memset(&srb, 0, sizeof srb);

	srb.Sic.Length			= sizeof srb - sizeof srb.Sic;
	srb.Sic.ControlCode		= MPI_DIAG_RESET;
	srb.Sic.HeaderLength	= sizeof srb.Sic;
	srb.Sic.Timeout			= 30;

	memcpy((char *)&srb.Sic.Signature, "v93Ap6Q4", 8);

	inLen					= sizeof srb;
	outLen					= sizeof srb;
	retLen					= 0;

	printf("Resetting port...\n");

	status = DeviceIoControl(port->fileHandle, port->ioctlValue,
							 &srb, inLen, &srb, outLen, &retLen, NULL);

	return status;
#endif
#if __linux__ || __alpha__
	int								 status;
	struct mpt_ioctl_diag_reset		*diag_reset;

	diag_reset = (struct mpt_ioctl_diag_reset *)malloc(sizeof *diag_reset);

	memset(diag_reset, 0, sizeof *diag_reset);

	diag_reset->hdr.iocnum = port->portNumber;

	printf("Resetting port...\n");

	status = ioctl(port->fileHandle, MPTHARDRESET, diag_reset);

	free(diag_reset);

	return status == 0;
#endif
#if __sparc__ || __irix__
	int					 status;
#if __irix__
	struct scsi_ha_op	 scsiioctl;
#endif

	printf("Resetting port...\n");

#if __sparc__
	status = ioctl(port->fileHandle, SYMIOCTL_RESET_ADAPTER);
#endif
#if __irix__
	scsiioctl.sb_opt	= SYMIOCTL_RESET_ADAPTER;
	status = ioctl(port->fileHandle, SOP_GETDATA, &scsiioctl);
#endif

	return status == 0;
#endif
}


int
doMptCommand(MPT_PORT *port, void *req, int reqSize, void *rep, int repSize,
			 void *payIn, int payInSize, void *payOut, int payOutSize, int timeOut)
{
#if WIN32
	int			 cmd;
	int			 status;
	SRB_BUFFER	 srb_buffer;
	SRB_BUFFER	*srb;
	int			 inLen;
	int			 outLen;
	int			 retLen;
	int			 size;

	/*
	 * Add 8 for IoctlDetails, 128 for maximum request and reply,
	 * and 256 for maximum sense data (for SCSI I/O requests).
	 */
	size = 8 + 128 + 256 + max(payInSize, payOutSize);
	if (size > sizeof srb->Buf)
		srb = malloc(sizeof srb->Sic + size);
	else
		srb = &srb_buffer;

	memset(srb, 0, sizeof srb->Sic + size);

	srb->Sic.Length				= size;
	srb->Sic.ControlCode		= MPI_MSG_IOCTL;
	srb->Sic.HeaderLength		= sizeof srb->Sic;
	srb->Sic.Timeout			= timeOut;

	memcpy((char *)&srb->Sic.Signature, "4.00    ", 8);

	if (payInSize != 0 && payOutSize != 0)
		cmd = DUAL_SGLS;		// data in and data out
	else if (payInSize != 0)
		cmd = 0;				// data in
	else if (payOutSize != 0)
		cmd = DATA_FROM_APP;	// data out
	else
		cmd = 0;				// no data transfer, just say Read

	if (((MPIHeader_t *)req)->Function == MPI_FUNCTION_SCSI_IO_REQUEST)
	{
		cmd |= SCSI_IO;			// flag SCSI I/O, potentially get sense data
		((MPIHeader_t *)req)->MsgContext = 0xbadbad;
	}

	*(PUSHORT)&srb->Buf[0]		= cmd;
	*(PUSHORT)&srb->Buf[2]		= reqSize / 4;
	*(PUSHORT)&srb->Buf[4]		= payInSize;
	if (cmd & DUAL_SGLS)
		*(PUSHORT)&srb->Buf[6]	= payOutSize;
	else if (cmd & 0x0001)
		*(PUSHORT)&srb->Buf[4]	= payOutSize;

	memcpy(&srb->Buf[8], req, reqSize);
	if (payOutSize != 0)
		memcpy(&srb->Buf[8+reqSize], payOut, payOutSize);

	inLen						= sizeof srb->Sic + size;
	outLen						= sizeof srb->Sic + size;
	retLen						= 0;

	status = DeviceIoControl(port->fileHandle, port->ioctlValue,
							 srb, inLen, srb, outLen, &retLen, NULL);

	if (status == 1)
	{
		memcpy(rep, &srb->Buf[0], repSize);
		if (payInSize != 0)
			memcpy(payIn, &srb->Buf[port->payOff], payInSize);

		if (((MPIHeader_t *)req)->Function == MPI_FUNCTION_SCSI_IO_REQUEST)
		{
			SCSIIORequest_t		*scsiReq = (SCSIIORequest_t *)req;
			SCSIIOReply_t		*scsiRep = (SCSIIOReply_t *)rep;

			/*
			 * If the SCSI I/O is successful, no actual reply
			 * gets written, so detect this fact and create the
			 * missing reply.
			 */
			if (*(U32 *)rep == 0xbadbad)
			{
				memset(scsiRep, 0, sizeof *scsiRep);
				scsiRep->Function		= scsiReq->Function;
				scsiRep->MsgLength		= sizeof *scsiRep / 4;
				scsiRep->Bus			= scsiReq->Bus;
				scsiRep->TargetID		= scsiReq->TargetID;
				scsiRep->CDBLength		= scsiReq->CDBLength;
				scsiRep->IOCStatus		= MPI_IOCSTATUS_SUCCESS;
				scsiRep->SCSIStatus		= MPI_SCSI_STATUS_SUCCESS;
				scsiRep->TransferCount	= scsiReq->DataLength;
			}

			/* copy the sense data to where it is expected to be */
			memcpy(scsiRep + 1, &srb->Buf[port->payOff + payInSize], repSize - sizeof *scsiRep);
		}
	}

	if (size > sizeof srb->Buf)
		free(srb);

	return status;
#endif
#if __linux__ || __alpha__
	int							 status;
	int							 extra;
	struct mpt_ioctl_command	*command;

	extra = max(0, reqSize - sizeof command->MF);
	command = (struct mpt_ioctl_command *)malloc(sizeof *command + extra);

	memset(command, 0, sizeof *command);

	command->hdr.iocnum	= port->portNumber;

	command->timeout			= timeOut;
	command->replyFrameBufPtr	= rep;
	command->dataInBufPtr		= payIn;
	command->dataOutBufPtr		= payOut;
	command->maxReplyBytes		= repSize;
	command->dataInSize			= payInSize;
	command->dataOutSize		= payOutSize;
	command->dataSgeOffset		= reqSize / 4;

	memcpy(command->MF, req, reqSize);

	status = ioctl(port->fileHandle, MPTCOMMAND, command);

	free(command);

	return status == 0;
#endif
#if __sparc__ || __irix__
	int					 status;
	SYM_PASS_THRU		 passThru;
#if __irix__
	struct scsi_ha_op	 scsiioctl;
#endif

	passThru.PtrRequest			= (UINT64)(UINT32)req;
	passThru.RequestSize		= reqSize;
	passThru.PtrReply			= (UINT64)(UINT32)rep;
	passThru.ReplySize			= repSize;
	if (payInSize != 0 && payOutSize != 0)
	{
		passThru.PtrData		= (UINT64)(UINT32)payIn;
		passThru.DataSize		= payInSize;
		passThru.PtrDataOut		= (UINT64)(UINT32)payOut;
		passThru.DataOutSize	= payOutSize;
		passThru.DataDirection	= SYM_PASS_THRU_BOTH;
	}
	else if (payInSize != 0)
	{
		passThru.PtrData		= (UINT64)(UINT32)payIn;
		passThru.DataSize		= payInSize;
		passThru.DataDirection	= SYM_PASS_THRU_READ;
	}
	else if (payOutSize != 0)
	{
		passThru.PtrData		= (UINT64)(UINT32)payOut;
		passThru.DataSize		= payOutSize;
		passThru.DataDirection	= SYM_PASS_THRU_WRITE;
	}
	else
	{
		passThru.PtrData		= NULL;
		passThru.DataSize		= 0;
		passThru.PtrDataOut		= NULL;
		passThru.DataOutSize	= 0;
		passThru.DataDirection	= SYM_PASS_THRU_NONE;
	}

#if __sparc__
	status = ioctl(port->fileHandle, SYMIOCTL_PASS_THRU, &passThru);
#endif
#if __irix__
	scsiioctl.sb_opt	= SYMIOCTL_PASS_THRU;
	scsiioctl.sb_addr	= (uintptr_t)&passThru;
	status = ioctl(port->fileHandle, SOP_GETDATA, &scsiioctl);
#endif

	return status == 0;
#endif
}


int
doMptCommandCheck(MPT_PORT *port, void *req, int reqSize, void *rep, int repSize,
				  void *payIn, int payInSize, void *payOut, int payOutSize, int timeOut)
{
	MPIDefaultReply_t	*defaultReply;
	int					 ioc_status;

	if (doMptCommand(port, req, reqSize, rep, repSize, payIn, payInSize, payOut, payOutSize, timeOut) != 1)
	{
		printf("\nFailed to issue command\n");
		return 0;
	}

	defaultReply = (MPIDefaultReply_t *)rep;
	ioc_status = get16(defaultReply->IOCStatus) & MPI_IOCSTATUS_MASK;

	if (ioc_status != MPI_IOCSTATUS_SUCCESS)
	{
		printf("\nCommand failed with IOCStatus = %04x (%s)\n", ioc_status, translateIocStatus(ioc_status));

		return 0;
	}
	
	return 1;
}


char *
translateIocStatus(int ioc_status)
{
	switch (ioc_status)
	{
	case MPI_IOCSTATUS_SUCCESS:                     return "Success";
	case MPI_IOCSTATUS_INVALID_FUNCTION:            return "Invalid Function";
	case MPI_IOCSTATUS_BUSY:                        return "IOC Busy";
	case MPI_IOCSTATUS_INVALID_SGL:                 return "Invalid SGL";
	case MPI_IOCSTATUS_INTERNAL_ERROR:              return "Internal Error";
	case MPI_IOCSTATUS_INSUFFICIENT_RESOURCES:      return "Insufficient Resources";
	case MPI_IOCSTATUS_INVALID_FIELD:               return "Invalid Field";
	case MPI_IOCSTATUS_INVALID_STATE:               return "Invalid State";
	case MPI_IOCSTATUS_OP_STATE_NOT_SUPPORTED:      return "Operational State Not Supported";
	case MPI_IOCSTATUS_CONFIG_INVALID_ACTION:       return "Invalid Action";
	case MPI_IOCSTATUS_CONFIG_INVALID_TYPE:         return "Invalid Type";
	case MPI_IOCSTATUS_CONFIG_INVALID_PAGE:         return "Invalid Page";
	case MPI_IOCSTATUS_CONFIG_INVALID_DATA:         return "Invalid Data";
	case MPI_IOCSTATUS_CONFIG_NO_DEFAULTS:          return "No Defaults";
	case MPI_IOCSTATUS_CONFIG_CANT_COMMIT:          return "Can't Commit";
	case MPI_IOCSTATUS_SCSI_RECOVERED_ERROR:        return "Recovered Error";
	case MPI_IOCSTATUS_SCSI_INVALID_BUS:            return "Invalid Bus";
	case MPI_IOCSTATUS_SCSI_INVALID_TARGETID:       return "Invalid Target";
	case MPI_IOCSTATUS_SCSI_DEVICE_NOT_THERE:       return "Device Not There";
	case MPI_IOCSTATUS_SCSI_DATA_OVERRUN:           return "Data Overrun";
	case MPI_IOCSTATUS_SCSI_DATA_UNDERRUN:          return "Data Underrun";
	case MPI_IOCSTATUS_SCSI_IO_DATA_ERROR:          return "I/O Data Error";
	case MPI_IOCSTATUS_SCSI_PROTOCOL_ERROR:         return "Protocol Error";
	case MPI_IOCSTATUS_SCSI_TASK_TERMINATED:        return "Task Terminated";
	case MPI_IOCSTATUS_SCSI_RESIDUAL_MISMATCH:      return "Residual Mismatch";
	case MPI_IOCSTATUS_SCSI_TASK_MGMT_FAILED:       return "Task Managment Failed";
	case MPI_IOCSTATUS_SCSI_IOC_TERMINATED:         return "IOC Terminated";
	case MPI_IOCSTATUS_SCSI_EXT_TERMINATED:         return "Externally Terminated";
	case MPI_IOCSTATUS_EEDP_GUARD_ERROR:            return "EEDP Guard Error";
	case MPI_IOCSTATUS_EEDP_REF_TAG_ERROR:          return "EEDP Reference Tag Error";
	case MPI_IOCSTATUS_EEDP_APP_TAG_ERROR:          return "EEDP Application Tag Error";
	case MPI_IOCSTATUS_TARGET_PRIORITY_IO:          return "Target Priority I/O";
	case MPI_IOCSTATUS_TARGET_INVALID_PORT:         return "Invalid Port";
	case MPI_IOCSTATUS_TARGET_INVALID_IO_INDEX:     return "Invalid I/O Index";
	case MPI_IOCSTATUS_TARGET_ABORTED:              return "Target Aborted";
	case MPI_IOCSTATUS_TARGET_NO_CONN_RETRYABLE:    return "No Connection, Retryable";
	case MPI_IOCSTATUS_TARGET_NO_CONNECTION:        return "No Connection";
	case MPI_IOCSTATUS_TARGET_XFER_COUNT_MISMATCH:  return "Transfer Count Mismatch";
	case MPI_IOCSTATUS_TARGET_STS_DATA_NOT_SENT:    return "Status Data Not Sent";
	case MPI_IOCSTATUS_TARGET_DATA_OFFSET_ERROR:    return "Data Offset Error";
	case MPI_IOCSTATUS_TARGET_TOO_MUCH_WRITE_DATA:  return "Too Much Write Data";
	case MPI_IOCSTATUS_TARGET_IU_TOO_SHORT:         return "Target IU Too Short";
	case MPI_IOCSTATUS_FC_ABORTED:                  return "FC Aborted";
	case MPI_IOCSTATUS_FC_RX_ID_INVALID:            return "RX_ID Invalid";
	case MPI_IOCSTATUS_FC_DID_INVALID:              return "D_ID Invalid";
	case MPI_IOCSTATUS_FC_NODE_LOGGED_OUT:          return "Node Logged Out";
	case MPI_IOCSTATUS_FC_EXCHANGE_CANCELED:        return "Exchange Canceled";
	case MPI_IOCSTATUS_LAN_DEVICE_NOT_FOUND:        return "LAN Device Not Found";
	case MPI_IOCSTATUS_LAN_DEVICE_FAILURE:          return "LAN Device Failure";
	case MPI_IOCSTATUS_LAN_TRANSMIT_ERROR:          return "LAN Transmit Error";
	case MPI_IOCSTATUS_LAN_TRANSMIT_ABORTED:        return "LAN Transmit Aborted";
	case MPI_IOCSTATUS_LAN_RECEIVE_ERROR:           return "LAN Receive Error";
	case MPI_IOCSTATUS_LAN_RECEIVE_ABORTED:         return "LAN Receive Aborted";
	case MPI_IOCSTATUS_LAN_PARTIAL_PACKET:          return "LAN Partial Packet";
	case MPI_IOCSTATUS_LAN_CANCELED:                return "LAN Canceled";
	case MPI_IOCSTATUS_SAS_SMP_REQUEST_FAILED:      return "SMP Request Failed";
	case MPI_IOCSTATUS_SAS_SMP_DATA_OVERRUN:        return "SMP Data Overrun";
	case MPI_IOCSTATUS_INBAND_ABORTED:              return "Inband Aborted";
	case MPI_IOCSTATUS_INBAND_NO_CONNECTION:        return "Inband No Connection";
	case MPI_IOCSTATUS_DIAGNOSTIC_RELEASED:         return "Diagnostic Buffer Released";
	}

	return "";
}


void
dumpMemory(void *buf, int len, char *string)
{
	U32		*p = (U32 *)buf;
	int		 i;

	printf("\n%s\n", string);
	for (i = 0; i < len; i += 4)
		printf("%04x : %08x\n", i, *p++);
}
