/*****************************************************************/
/*                                                               */
/*              CSMI TEST TOOL FOR LINUX                         */
/*                                                               */
/*                                                               */
/* Revision History                                              */
/* ----------------                                              */
/* 1.0-1   AJ   03/16/04     Initial Release                     */
/*****************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <fcntl.h>
#include <sys/errno.h>
#include <string.h>
#include <getopt.h>
#include <sys/ioctl.h>
#include <sys/stat.h>
#include <unistd.h>
#include "csmisas.h"

#define TOOL_MAJOR_VERSION	1
#define TOOL_MINOR_VERSION	0
#define TOOL_RELEASE_NUMBER	1

// Function declaration
void DisplayIoctlHeader(IOCTL_HEADER Hdr);
void DisplayFirmwareDownload(CSMI_SAS_FIRMWARE_DOWNLOAD);
void DisplayDriverInfo(CSMI_SAS_DRIVER_INFO Info);
void DisplayCntlrConfig(CSMI_SAS_CNTLR_CONFIG Config);
void DisplayCntlrStatus(CSMI_SAS_CNTLR_STATUS Status);
void DisplayRaidInfo(CSMI_SAS_RAID_INFO Info);
void DisplayRaidConfig(CSMI_SAS_RAID_CONFIG * Config);
void DisplayPhyInfo(int fd, CSMI_SAS_PHY_INFO Info);
void DisplaySASIdentify(CSMI_SAS_IDENTIFY Identify);
void DisplayErrorInfo(CSMI_SAS_LINK_ERRORS Info);
void DisplaySignature(CSMI_SAS_SATA_SIGNATURE Signature);
void DisplaySCSIAddr(CSMI_SAS_GET_SCSI_ADDRESS_BUFFER ScsiAddrBuff);
void DisplayDeviceAddress(CSMI_SAS_GET_DEVICE_ADDRESS_BUFFER DeviceAddress);
void DisplayGetLocation(CSMI_SAS_GET_LOCATION_BUFFER GetLocation);
void DisplayPassthruStatus(CSMI_SAS_SSP_PASSTHRU_STATUS Status );
void DisplayStpPassthruStatus(CSMI_SAS_STP_PASSTHRU_STATUS Status );
void DisplayPhyControl(CSMI_SAS_PHY_CONTROL_BUFFER PhyControl);

void PrintUsage(void);
void PrintDirection(int Direction);
void PrintReturnCode(int ReturnCode);
void PrintRaidType(unsigned char bRaidType);
void PrintRaidStatus(unsigned char bStatus);
void PrintRaidStatusMoreInfo(unsigned char bStatus, unsigned char bInfo);
void PrintDriveStatus(unsigned char bDriveStatus);
void PrintDriveUsage(unsigned char bDriveUsage);
void PrintProtocol(unsigned char bProtocol);
void PrintLinkRate(unsigned char bLinkRate);
void PrintAutoDiscoverState(unsigned char bAutoDiscover);

void issue_firmware_download(int fd, int IOControllerNumber, char* fname);
void issue_get_driver_info(int fd, int IOControllerNumber);
void issue_get_ctlr_config(int fd, int IOControllerNumber);
void issue_get_ctlr_status(int fd, int IOControllerNumber);
void issue_get_raid_info(int fd, int IOControllerNumber);
void issue_get_raid_config(int fd, int IOControllerNumber);
void issue_get_phy_info(int fd, int menu, int IOControllerNumber);
void issue_get_link_error(int fd, unsigned char bPhyIdentifier, int IOControllerNumber);
void issue_get_sata_signature(int fd, unsigned char bPhyIdentifier, int IOControllerNumber);
void issue_get_scsi_address(int fd, int TargetId, int LunId, int IOControllerNumber);
void issue_get_device_address(int fd, int HostId, int PathId, int TargetId, int LunId, int IOControllerNumber);
void issue_get_location(int fd, int HostId, int PathId, int TargetId, int LunId, int IOControllerNumber);
void issue_tm_passthru(int fd, int TargetId, int LunId, int IOControllerNumber);
void issue_set_phy_info(int fd, int PhyId, int NegLinkRate, int MinLinkRate,
    int MaxLinkRate,int IOControllerNumber);
void issue_ssp_passthru(int fd, int IOControllerNumber);
void issue_smp_passthru(int fd, int IOControllerNumber);
void issue_stp_passthru(int fd, int IOControllerNumber);
void issue_tm_passthru(int fd, int TargetId, int LunId, int IOControllerNumber);
void issue_phy_control(int fd, int IOControllerNumber, int PhyId);
void menu_driven_testing(int fd);
void ShowBuf(char *titleP, void *dataBufP, int count);

extern int errno;

int main (int argc, char *argv[])
{
	int c;
	extern char *optarg;
	char dev_node[20];
	int fd, get_driver_info=0, get_ctlr_config=0, get_ctlr_status=0 ;
	int get_raid_info=0, get_raid_config=0, get_phy_info=0;
	int print_usage=0, drive_by_menu=0;
	int IOControllerNumber;

	printf("\n"
		"Common Serial Attached SCSI Management IOCTL Test for Linux\n"
		"Version %2d.%02d-%02d \n",
		TOOL_MAJOR_VERSION, TOOL_MINOR_VERSION, TOOL_RELEASE_NUMBER);

	if(argc == 1){
		PrintUsage();
		exit(EXIT_FAILURE);
	}

	while ((c = getopt(argc, argv, "d:c:uam")) != -1) {
		switch(c){
			case 'c':
				  IOControllerNumber = atoi(optarg);
				  break;

			case 'd':
				// printf("Device Node <%s> \n", optarg);
				strcpy(dev_node, optarg);
				break;
			case 'u':
				print_usage=1;
				break;

			case 'a':
				get_driver_info=1;
				get_ctlr_config=1;
				get_ctlr_status=1;
				get_raid_info=1;
				get_raid_config=1;
				get_phy_info=1;
				break;

			case 'm':
				drive_by_menu=1;
		}
	}

	if(print_usage == 1){
		PrintUsage();
		exit(EXIT_SUCCESS);
	}

	fd = open(dev_node, O_RDWR);
	if (fd < 0){
		printf("Open of device node <%s> failed with errno=%d\n", dev_node, errno);
		exit(EXIT_FAILURE);
	}

	if(drive_by_menu == 1)
		menu_driven_testing(fd);
	if(get_driver_info == 1)
		issue_get_driver_info(fd,IOControllerNumber);
	if(get_ctlr_config == 1)
		issue_get_ctlr_config(fd, IOControllerNumber);
	if(get_ctlr_status == 1)
		issue_get_ctlr_status(fd, IOControllerNumber);
	if(get_raid_info == 1)
		issue_get_raid_info(fd, IOControllerNumber);
	if(get_raid_config == 1)
		issue_get_raid_config(fd, IOControllerNumber);
	if(get_phy_info == 1)
		issue_get_phy_info(fd, drive_by_menu, IOControllerNumber);
return 0;
}
void PrintUsage(void)
{
	printf("Usage : \n");

	printf(" \t./testioc \n");
	printf(" \t\t -d <device name>, device node to open \n");
	printf(" \t\t -c <controller number>, controller number \n");
	printf(" \t\t -u, to print this usage \n");
	printf(" \t\t -a, exercise all possible IOCTL's \n");
	printf(" \t\t -m, menu driven to test specific IOCTL \n");
}

void PrintDirection(int Direction)
{
	switch(Direction){
		case CSMI_SAS_DATA_READ:
			printf("Data Read\n");
			break;
		case CSMI_SAS_DATA_WRITE:
			printf("Data Write\n");
			break;
	}
}
void PrintReturnCode(int ReturnCode)
{
	switch(ReturnCode){
		case CSMI_SAS_PHY_INFO_CHANGED:
			printf("Success \n");
			break;
		case CSMI_SAS_PHY_INFO_NOT_CHANGEABLE:
			printf("SAS Phy Info Not Changeable\n");
			break;
		case CSMI_SAS_LINK_RATE_OUT_OF_RANGE:
			printf("SAS Link Rate out of range\n");
			break;
		case CSMI_SAS_PHY_DOES_NOT_EXIST:
			printf("SAS Phy Does not exist\n");
			break;
		case CSMI_SAS_PHY_DOES_NOT_MATCH_PORT:
			printf("SAS Phy does not match port\n");
			break;
		case CSMI_SAS_PHY_CANNOT_BE_SELECTED:
			printf("SAS Phy cannot be selected\n");
			break;
		case CSMI_SAS_SELECT_PHY_OR_PORT:
			printf("Select SAS Phy or port \n");
			break;
		case CSMI_SAS_PORT_DOES_NOT_EXIST:
			printf("SAS Port does not exist \n");
			break;
		case CSMI_SAS_PORT_CANNOT_BE_SELECTED:
			printf("SAS Port cannot be selected \n");
			break;
		case CSMI_SAS_CONNECTION_FAILED:
			printf("SAS Connection failed \n");
			break;
		case CSMI_SAS_NO_SATA_DEVICE:
			printf("No SATA Device \n");
			break;
		case CSMI_SAS_NO_SATA_SIGNATURE:
			printf("No SATA Signature\n");
			break;
		case CSMI_SAS_SCSI_EMULATION:
			printf("SCSI Emulation\n");
			break;
		case CSMI_SAS_NOT_AN_END_DEVICE:
			printf("Not an end device\n");
			break;
		case CSMI_SAS_NO_SCSI_ADDRESS:
			printf("No SCSI Address \n");
			break;
		case CSMI_SAS_NO_DEVICE_ADDRESS:
			printf("No device address\n");
			break;
		case CSMI_SAS_STATUS_INVALID_PARAMETER:
			printf("CSMI_SAS_STATUS_INVALID_PARAMETER\n");
			break;
		case CSMI_SAS_STATUS_FAILED:
			printf("CSMI_SAS_STATUS_FAILED\n");
			break;
		case CSMI_SAS_STATUS_BAD_CNTL_CODE:
			printf("CSMI_SAS_STATUS_BAD_CNTL_CODE\n");
			break;
		case CSMI_SAS_STATUS_WRITE_ATTEMPTED:
			printf("CSMI_SAS_STATUS_WRITE_ATTEMPTED\n");
			break;
		default:
			printf("Unknown code = %d\n",ReturnCode);
			break;
	}
}
void PrintRaidType(unsigned char bRaidType)
{
	switch(bRaidType){
		case CSMI_SAS_RAID_TYPE_NONE:
			printf("Raid set is composed of a single drive\n");
			break;
		case CSMI_SAS_RAID_TYPE_0:
			printf("No Fault Tolerance\n");
			break;
		case CSMI_SAS_RAID_TYPE_1:
			printf("RAID 1. Mirrored Set.\n");
			break;
		case CSMI_SAS_RAID_TYPE_10:
			printf("RAID 1+0. Striped Mirrored Set.\n");
			break;
		case CSMI_SAS_RAID_TYPE_5:
			printf("RAID 5. Parity Set.\n");
			break;
		case CSMI_SAS_RAID_TYPE_15:
			printf("Advanced Parity Set.\n");
			break;
		case CSMI_SAS_RAID_TYPE_OTHER:
			printf("Non-Standard RAID type.\n");
			break;
		default:
			printf("Undefined: Invalid RAID type.\n");
			break;
	}
}
void PrintRaidStatus(unsigned char bStatus)
{
	switch(bStatus){
		case CSMI_SAS_RAID_SET_STATUS_OK:
			printf("OK.\n");
			break;
		case CSMI_SAS_RAID_SET_STATUS_DEGRADED:
			printf("Degraded.\n");
			break;
		case CSMI_SAS_RAID_SET_STATUS_REBUILDING:
			printf("Rebuilding.\n");
			break;
		case CSMI_SAS_RAID_SET_STATUS_FAILED:
			printf("Failed.\n");
			break;
		default:
			printf("Undefined: Invalid RAID Status.\n");
			break;
	}
}
void PrintRaidStatusMoreInfo(unsigned char bStatus, unsigned char bInfo)
{
	switch(bStatus){
		case CSMI_SAS_RAID_SET_STATUS_OK:
			printf("%d (No Info)\n", bInfo);
			break;
		case CSMI_SAS_RAID_SET_STATUS_DEGRADED:
			printf("Drive #%d Failed.\n", bInfo);
			break;
		case CSMI_SAS_RAID_SET_STATUS_REBUILDING:
			printf("%d rebuild complete.\n", bInfo);
			break;
		case CSMI_SAS_RAID_SET_STATUS_FAILED:
			printf("%d (Vendor specific)\n", bInfo);
			break;
		default:
			printf("Undefined.\n");
			break;
	}
}
void PrintDriveStatus(unsigned char bDriveStatus)
{
	switch(bDriveStatus){
		case CSMI_SAS_DRIVE_STATUS_OK:
			printf("OK.\n");
			break;
		case CSMI_SAS_DRIVE_STATUS_REBUILDING:
			printf("Rebuilding.\n");
			break;
		case CSMI_SAS_DRIVE_STATUS_FAILED:
			printf("Failed.\n");
			break;
		case CSMI_SAS_DRIVE_STATUS_DEGRADED:
			printf("Degraded.\n");
			break;
		default:
			printf("Undefined.\n");
			break;
	}
}
void PrintDriveUsage(unsigned char bDriveUsage)
{
	switch(bDriveUsage){
		case CSMI_SAS_DRIVE_CONFIG_NOT_USED:
			printf("Unconfigured.\n");
			break;
		case CSMI_SAS_DRIVE_CONFIG_MEMBER:
			printf("Part of this RAID set.\n");
			break;
		case CSMI_SAS_DRIVE_CONFIG_SPARE:
			printf("Hot swap spare of this RAID set.\n");
			break;
		default:
			printf("Undefined.\n");
			break;
	}
}
void PrintProtocol(unsigned char bProtocol)
{

	if(bProtocol & CSMI_SAS_PROTOCOL_SATA)
		printf("SATA Protocol ");

	if(bProtocol & CSMI_SAS_PROTOCOL_SMP)
		printf("SMP Protocol ");

	if(bProtocol & CSMI_SAS_PROTOCOL_STP)
		printf("STP Protocol ");

	if(bProtocol & CSMI_SAS_PROTOCOL_SSP)
		printf("SSP Protocol ");

	if(bProtocol == 0)
	    printf("Undefined ");

	printf("\n");

}
void PrintLinkRate(unsigned char bLinkRate)
{
	switch(bLinkRate){
		case CSMI_SAS_LINK_RATE_UNKNOWN:
		// Same value as above CSMI_SAS_PROGRAMMED_LINK_RATE_UNCHANGED:
			printf("Unchanged / Not Connected / Link Rate does not apply.\n");
			break;
		case CSMI_SAS_PHY_DISABLED:
			printf("Phy disabled.\n");
			break;
		case CSMI_SAS_LINK_RATE_FAILED:
			printf("Link Rate Failed.\n");
			break;
		case CSMI_SAS_SATA_SPINUP_HOLD:
			printf("Link detected, but in a wait state to release spin-up hold.\n");
			break;
		case CSMI_SAS_LINK_RATE_1_5_GBPS:
		// Same vale as above CSMI_SAS_PROGRAMMED_LINK_RATE_1_5_GBPS:
			printf("1.5 Gb/s.\n");
			break;
		case CSMI_SAS_LINK_RATE_3_0_GBPS:
		// Same value as above CSMI_SAS_PROGRAMMED_LINK_RATE_3_0_GBPS:
			printf("3.0 Gb/s.\n");
			break;
		default:
			printf("Undefined\n");
			break;
	}
}
void PrintAutoDiscoverState(unsigned char bAutoDiscover)
{
	switch(bAutoDiscover){
		case CSMI_SAS_DISCOVER_NOT_SUPPORTED:
			printf("Not Supported.\n");
			break;
		case CSMI_SAS_DISCOVER_NOT_STARTED:
			printf("Not Started.\n");
			break;
		case CSMI_SAS_DISCOVER_IN_PROGRESS:
			printf("In Progress.\n");
			break;
		case CSMI_SAS_DISCOVER_COMPLETE:
			printf("Complete.\n");
			break;
		case CSMI_SAS_DISCOVER_ERROR:
			printf("Error.\n");
			break;
		default:
			printf("Undefined\n");
			break;
	}
}
void DisplayIoctlHeader(IOCTL_HEADER Hdr)
{
	printf("\tIOCTL Header\n");
	printf("\t------------\n");
	printf("\tLength\t\t: %d \n", Hdr.Length);
	printf("\tTimeout\t\t: %d \n", Hdr.Timeout);
	printf("\tDirection\t: ");
	PrintDirection(Hdr.Direction);
	printf("\tReturn Code\t: ");
	PrintReturnCode(Hdr.ReturnCode);
	printf("\n");
}

void DisplayFirmwareStatus(short status)
{
	switch(status){
	case CSMI_SAS_FWD_SUCCESS:
		printf("CSMI_SAS_FWD_SUCCESS\n");
		break;
	case CSMI_SAS_FWD_FAILED:
		printf("CSMI_SAS_FWD_FAILED\n");
		break;
	case CSMI_SAS_FWD_USING_RROM:
		printf("CSMI_SAS_FWD_USING_RROM\n");
		break;
	case CSMI_SAS_FWD_REJECT:
		printf("CSMI_SAS_FWD_REJECT\n");
		break;
	case CSMI_SAS_FWD_DOWNREV:
		printf("CSMI_SAS_FWD_DOWNREV\n");
		break;
	default:
		printf("UNKNOWN\n");
		break;
	}
}

void DisplayFirmwareSeverity(short severity)
{
	switch(severity) {
	case CSMI_SAS_FWD_INFORMATION:
		printf("CSMI_SAS_FWD_INFORMATION\n");
		break;
	case CSMI_SAS_FWD_WARNING:
		printf("CSMI_SAS_FWD_WARNING\n");
		break;
	case CSMI_SAS_FWD_ERROR:
		printf("CSMI_SAS_FWD_ERROR\n");
		break;
	case CSMI_SAS_FWD_FATAL:
		printf("CSMI_SAS_FWD_FATAL\n");
		break;
	default:
		printf("UNKNOWN\n");
		break;
	}
}


void DisplayFirmwareDownload(CSMI_SAS_FIRMWARE_DOWNLOAD Info)
{
	printf("\tFirmwareDownload\n");
	printf("\t----------------\n");
	printf("\tBufferLength\t\t: %d \n", Info.uBufferLength);
//	printf("\tDownloadFlags\t\t: %d \n", Info.uDownloadFlags);
	printf("\tStatus\t\t\t: ");
	DisplayFirmwareStatus(Info.usStatus);
	printf("\tSeverity\t\t: ");
	DisplayFirmwareSeverity(Info.usSeverity);
	printf("\n");
}

void DisplayPassthruStatus(CSMI_SAS_SSP_PASSTHRU_STATUS Status )
{
	short * ResponseLength = (short *)Status.bResponseLength;
	
	printf("\tPassthru Status\n");
	printf("\t----------------\n");
	printf("\tConnectionStatus\t: %d \n", Status.bConnectionStatus);
	printf("\tDataPresent\t\t: %d \n", Status.bDataPresent);
	printf("\tResponseLength\t\t: %d \n", *ResponseLength);
	printf("\tStatus\t\t\t: %d \n", Status.bStatus);
	printf("\tDataBytes\t\t: %d \n", Status.uDataBytes);
	printf("\n");
}

void DisplayStpPassthruStatus(CSMI_SAS_STP_PASSTHRU_STATUS Status )
{
	int i;
		
	printf("\tPassthru Status\n");
	printf("\t----------------\n");
	printf("\tConnectionStatus\t: %d \n", Status.bConnectionStatus);
	printf("\tDataBytes\t\t: %d \n", Status.uDataBytes);
	printf("\tStatusFIS\t\t: ");
	for(i=0;i<20;i++) printf("%02x ",Status.bStatusFIS[i]);
	printf("\n");
	printf("\tSCR\t\t\t: ");
	for(i=0;i<16;i++) printf("%02x ",Status.uSCR[i]);
	printf("\n");
}

void issue_firmware_download(int fd, int IOControllerNumber, char* fname)
{
	int retval, i, offset;
	CSMI_SAS_CNTLR_CONFIG_BUFFER CtlrConfig;
	CSMI_SAS_FIRMWARE_DOWNLOAD_BUFFER FirmBuff;
	__u32 cntrlflags;
	int image;
	struct stat imageStat;
	void* mem;

	printf("\tIOCTL : CC_CSMI_SAS_FIRMWARE_DOWNLOAD \n");
	printf("\t~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ \n");
// get cntrl config first to see if firmware download is supported
	memset(&CtlrConfig, 0, sizeof(CSMI_SAS_CNTLR_CONFIG_BUFFER));

	CtlrConfig.IoctlHeader.Length = sizeof(CSMI_SAS_CNTLR_CONFIG_BUFFER);
	CtlrConfig.IoctlHeader.Timeout = CSMI_ALL_TIMEOUT;
	CtlrConfig.IoctlHeader.Direction = CSMI_SAS_DATA_READ;
	CtlrConfig.IoctlHeader.IOControllerNumber = IOControllerNumber;

	retval=ioctl(fd, CC_CSMI_SAS_GET_CNTLR_CONFIG, &CtlrConfig);

	if (retval != 0){
		printf("IOCTL FAILED. Could not get cntrl config.\n");
		printf("Errno=%d \n", errno);
		return;
	}

	cntrlflags = CtlrConfig.Configuration.uControllerFlags;

// DEBUG
//	printf("uControllerFlags: %08x\n", cntrlflags);

// both flags must be set
	if( (cntrlflags & CSMI_SAS_CNTLR_FWD_SUPPORT )
	    && (cntrlflags & CSMI_SAS_CNTLR_FWD_ONLINE) )
	{
// one of these flags must also be set
	  if( (cntrlflags & CSMI_SAS_CNTLR_FWD_SRESET)
	      || (cntrlflags & CSMI_SAS_CNTLR_FWD_HRESET) )
	  {

		if ((image = open(fname, O_RDONLY)) < 0)
		{
			printf("Couldn't open %s!\n",fname);
			perror("Error is");
			return ;
		}

		stat(fname, &imageStat);

		FirmBuff.IoctlHeader.Length = sizeof(CSMI_SAS_FIRMWARE_DOWNLOAD_BUFFER) + imageStat.st_size;
		FirmBuff.IoctlHeader.Timeout = CSMI_ALL_TIMEOUT;
		FirmBuff.IoctlHeader.Direction = CSMI_SAS_DATA_READ;
		FirmBuff.IoctlHeader.IOControllerNumber = IOControllerNumber;

		FirmBuff.Information.uBufferLength = imageStat.st_size;
		FirmBuff.Information.uDownloadFlags = CSMI_SAS_FWD_HARD_RESET;
		FirmBuff.Information.usStatus=0;
		FirmBuff.Information.usSeverity=0;
		for(i=0;i<32;i++) FirmBuff.Information.bReserved[i]=0;

		mem = malloc(sizeof(CSMI_SAS_FIRMWARE_DOWNLOAD_BUFFER)
		  + imageStat.st_size);

		if(mem == NULL)
		{
			printf("Couldn't allocate enough memory!\n");
	 		close(image);
			return ;
		}

		bcopy(&FirmBuff, mem, sizeof(CSMI_SAS_FIRMWARE_DOWNLOAD_BUFFER));
		offset = offsetof(CSMI_SAS_FIRMWARE_DOWNLOAD_BUFFER,bDataBuffer);
		read(image, mem+offset, imageStat.st_size);

		printf("\tPlease wait while image is being downloaded. Don't turn power off\n\n\n");
		retval=ioctl(fd, CC_CSMI_SAS_FIRMWARE_DOWNLOAD, mem);

		if (retval != 0){
			printf("IOCTL FAILED. Errno=%d \n", errno);
			free(mem);
			close(image);
			return;
		}

		bcopy(mem, &FirmBuff, sizeof(CSMI_SAS_FIRMWARE_DOWNLOAD_BUFFER));

		DisplayIoctlHeader(FirmBuff.IoctlHeader);
		printf("\tIOCTL Specific Data\n");
		printf("\t-------------------\n");
		DisplayFirmwareDownload(FirmBuff.Information);
		printf("_______________________________________________________\n");

		free(mem);
 		close(image);
	  }
	  else
	  { printf("Controller does not support online flash.\n\n"); return; }
	}
	else
	{ printf("Controller does not support online flash.\n\n"); return; }

}

void issue_get_driver_info(int fd,int IOControllerNumber)
{
	int retval;
	CSMI_SAS_DRIVER_INFO_BUFFER DrvInfo;

	printf("\tIOCTL : CC_CSMI_SAS_GET_DRIVER_INFO \n");
	printf("\t~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ \n");

	memset(&DrvInfo, 0, sizeof(CSMI_SAS_DRIVER_INFO_BUFFER));

	DrvInfo.IoctlHeader.Length = sizeof(CSMI_SAS_DRIVER_INFO_BUFFER);
	DrvInfo.IoctlHeader.Timeout = CSMI_ALL_TIMEOUT;
	DrvInfo.IoctlHeader.Direction = CSMI_SAS_DATA_READ;
	DrvInfo.IoctlHeader.IOControllerNumber = IOControllerNumber;

	retval=ioctl(fd, CC_CSMI_SAS_GET_DRIVER_INFO, &DrvInfo);

	if (retval != 0){
		printf("IOCTL FAILED. Errno=%d \n", errno);
		return;
	}

	DisplayIoctlHeader(DrvInfo.IoctlHeader);
	printf("\tIOCTL Specific Data\n");
	printf("\t-------------------\n");
	DisplayDriverInfo(DrvInfo.Information);
	printf("_______________________________________________________\n");
}

void issue_get_ctlr_config(int fd,int IOControllerNumber)
{
	int retval;
	CSMI_SAS_CNTLR_CONFIG_BUFFER CtlrConfig;

	printf("\tIOCTL : CC_CSMI_SAS_GET_CNTLR_CONFIG \n");
	printf("\t~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ \n");

	memset(&CtlrConfig, 0, sizeof(CSMI_SAS_CNTLR_CONFIG_BUFFER));

	CtlrConfig.IoctlHeader.Length = sizeof(CSMI_SAS_CNTLR_CONFIG_BUFFER);
	CtlrConfig.IoctlHeader.Timeout = CSMI_ALL_TIMEOUT;
	CtlrConfig.IoctlHeader.Direction = CSMI_SAS_DATA_READ;
	CtlrConfig.IoctlHeader.IOControllerNumber = IOControllerNumber;

	retval=ioctl(fd, CC_CSMI_SAS_GET_CNTLR_CONFIG, &CtlrConfig);

	if (retval != 0){
		printf("IOCTL FAILED. Errno=%d \n", errno);
		return;
	}

	DisplayIoctlHeader(CtlrConfig.IoctlHeader);
	printf("\tIOCTL Specific Data\n");
	printf("\t-------------------\n");
	DisplayCntlrConfig(CtlrConfig.Configuration);
	printf("_______________________________________________________\n");
}
void issue_get_ctlr_status(int fd, int IOControllerNumber)
{
	int retval;
	CSMI_SAS_CNTLR_STATUS_BUFFER CtlrStatus;

	printf("\tIOCTL : CC_CSMI_SAS_GET_CNTLR_STATUS \n");
	printf("\t~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ \n");

	memset(&CtlrStatus, 0, sizeof(CSMI_SAS_CNTLR_STATUS_BUFFER));

	CtlrStatus.IoctlHeader.Length = sizeof(CSMI_SAS_CNTLR_CONFIG_BUFFER);
	CtlrStatus.IoctlHeader.Timeout = CSMI_ALL_TIMEOUT;
	CtlrStatus.IoctlHeader.Direction = CSMI_SAS_DATA_READ;
	CtlrStatus.IoctlHeader.IOControllerNumber = IOControllerNumber;

	retval=ioctl(fd, CC_CSMI_SAS_GET_CNTLR_STATUS, &CtlrStatus);

	if (retval != 0){
		printf("IOCTL FAILED. Errno=%d \n", errno);
		return;
	}

	DisplayIoctlHeader(CtlrStatus.IoctlHeader);
	printf("\tIOCTL Specific Data\n");
	printf("\t-------------------\n");
	DisplayCntlrStatus(CtlrStatus.Status);
	printf("_______________________________________________________\n");
}
void issue_get_raid_info(int fd, int IOControllerNumber)
{
	int retval;
	CSMI_SAS_RAID_INFO_BUFFER RaidInfo;

	printf("\tIOCTL : CC_CSMI_SAS_GET_RAID_INFO \n");
	printf("\t~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ \n");

	memset(&RaidInfo, 0, sizeof(CSMI_SAS_RAID_INFO_BUFFER));

	RaidInfo.IoctlHeader.Length = sizeof(CSMI_SAS_RAID_INFO_BUFFER);
	RaidInfo.IoctlHeader.Timeout = CSMI_ALL_TIMEOUT;
	RaidInfo.IoctlHeader.Direction = CSMI_SAS_DATA_READ;
	RaidInfo.IoctlHeader.IOControllerNumber = IOControllerNumber;

	retval=ioctl(fd, CC_CSMI_SAS_GET_RAID_INFO, &RaidInfo);

	if (retval != 0){
		printf("IOCTL FAILED. Errno=%d \n", errno);
		return;
	}

	DisplayIoctlHeader(RaidInfo.IoctlHeader);
	printf("\tIOCTL Specific Data\n");
	printf("\t-------------------\n");
	DisplayRaidInfo(RaidInfo.Information);
	printf("_______________________________________________________\n");
}
void issue_get_raid_config(int fd, int IOControllerNumber)
{
	int retval,index;
	CSMI_SAS_RAID_CONFIG_BUFFER *pRaidConfig;
	CSMI_SAS_RAID_INFO_BUFFER RaidInfo;
	unsigned long TotalRaidSets, MaxDrivesPerSet;
	__u32 data_length;

	printf("\tIOCTL : CC_CSMI_SAS_GET_RAID_CONFIG \n");
	printf("\t~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ \n");

	// Issue Get Raid Info, to get the max number of drivers per set
	memset(&RaidInfo, 0, sizeof(CSMI_SAS_RAID_INFO_BUFFER));

	RaidInfo.IoctlHeader.Length = sizeof(CSMI_SAS_RAID_INFO_BUFFER);
	RaidInfo.IoctlHeader.Timeout = CSMI_ALL_TIMEOUT;
	RaidInfo.IoctlHeader.Direction = CSMI_SAS_DATA_READ;
	RaidInfo.IoctlHeader.IOControllerNumber = IOControllerNumber;
	retval=ioctl(fd, CC_CSMI_SAS_GET_RAID_INFO, &RaidInfo);

	if (retval != 0){
		printf("IOCTL FAILED. Errno=%d \n", errno);
		printf("_______________________________________________________\n");
		return;
	}
	MaxDrivesPerSet = RaidInfo.Information.uMaxDrivesPerSet;
	TotalRaidSets = RaidInfo.Information.uNumRaidSets;

	if(TotalRaidSets == 0)
		printf("\tNo RAID sets.\n");

	for(index = 0; index < TotalRaidSets; index++){
		data_length = 
			offsetof(CSMI_SAS_RAID_CONFIG,Drives) +
			(MaxDrivesPerSet*sizeof(CSMI_SAS_RAID_DRIVES));
		pRaidConfig=(CSMI_SAS_RAID_CONFIG_BUFFER *)malloc(data_length + sizeof(IOCTL_HEADER));
		if(pRaidConfig == NULL){
			printf("Allocation of <%d> bytes for CSMI_SAS_RAID_CONFIG failed\n",
				(data_length + sizeof(IOCTL_HEADER)));
			printf("_______________________________________________________\n");
			return;
		}

		memset(pRaidConfig, 0, data_length + sizeof(IOCTL_HEADER));
		pRaidConfig->IoctlHeader.Length = data_length;
		pRaidConfig->IoctlHeader.Timeout = CSMI_ALL_TIMEOUT;
		pRaidConfig->IoctlHeader.Direction = CSMI_SAS_DATA_READ;
		pRaidConfig->IoctlHeader.IOControllerNumber = IOControllerNumber;
		pRaidConfig->Configuration.uRaidSetIndex=index;
		retval=ioctl(fd, CC_CSMI_SAS_GET_RAID_CONFIG, pRaidConfig);

		if (retval != 0){
			printf("Get Raid Config IOCTL FAILED for Raid Index %d\n", index);
		}
		else{
			printf("Raid Index %d :\n", index);
			printf("---------------\n");
			DisplayIoctlHeader(pRaidConfig->IoctlHeader);
			printf("\tIOCTL Specific Data\n");
			printf("\t-------------------\n");
			DisplayRaidConfig(&pRaidConfig->Configuration);
		}
		free(pRaidConfig);
	}
	printf("_______________________________________________________\n");
}
void issue_get_phy_info(int fd, int menu, int IOControllerNumber)
{
	int retval, PhyIndex;
	CSMI_SAS_PHY_INFO_BUFFER PhyInfo;

	printf("\tIOCTL : CC_CSMI_SAS_GET_PHY_INFO \n");
	printf("\t~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ \n");

	memset(&PhyInfo, 0, sizeof(CSMI_SAS_PHY_INFO_BUFFER));

	PhyInfo.IoctlHeader.Length = sizeof(CSMI_SAS_PHY_INFO_BUFFER);
	PhyInfo.IoctlHeader.Timeout = CSMI_ALL_TIMEOUT;
	PhyInfo.IoctlHeader.Direction = CSMI_SAS_DATA_READ;
	PhyInfo.IoctlHeader.IOControllerNumber = IOControllerNumber;

	retval=ioctl(fd, CC_CSMI_SAS_GET_PHY_INFO, &PhyInfo);

	if (retval != 0){
		printf("IOCTL FAILED. Errno=%d \n", errno);
		return;
	}

	DisplayIoctlHeader(PhyInfo.IoctlHeader);
	printf("\tIOCTL Specific Data\n");
	printf("\t-------------------\n");
	DisplayPhyInfo(fd, PhyInfo.Information);

	if(menu == 0){
		for(PhyIndex = 0; PhyIndex < PhyInfo.Information.bNumberOfPhys; PhyIndex++){
			issue_get_link_error(fd, PhyInfo.Information.Phy[PhyIndex].Identify.bPhyIdentifier, IOControllerNumber);
			issue_get_sata_signature(fd, PhyInfo.Information.Phy[PhyIndex].Identify.bPhyIdentifier, IOControllerNumber);
		}
	}
	printf("_______________________________________________________\n");
}
void issue_get_link_error(int fd, unsigned char bPhyIdentifier, int IOControllerNumber)
{
	int retval;
	CSMI_SAS_LINK_ERRORS_BUFFER ErrorInfo;

	printf("\tIOCTL : CC_CSMI_SAS_GET_LINK_ERRORS \n");
	printf("\t~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ \n");

	memset(&ErrorInfo, 0, sizeof(CSMI_SAS_LINK_ERRORS_BUFFER));

	ErrorInfo.IoctlHeader.Length = sizeof(CSMI_SAS_LINK_ERRORS_BUFFER);
	ErrorInfo.IoctlHeader.Timeout = CSMI_ALL_TIMEOUT;
	ErrorInfo.IoctlHeader.Direction = CSMI_SAS_DATA_READ;
	ErrorInfo.Information.bPhyIdentifier = bPhyIdentifier;
	ErrorInfo.IoctlHeader.IOControllerNumber = IOControllerNumber;
//	ErrorInfo.Information.bResetCounts = CSMI_SAS_LINK_ERROR_DONT_RESET_COUNTS;
	ErrorInfo.Information.bResetCounts = CSMI_SAS_LINK_ERROR_RESET_COUNTS;

	retval=ioctl(fd, CC_CSMI_SAS_GET_LINK_ERRORS, &ErrorInfo);

	if (retval != 0){
		printf("IOCTL FAILED. Errno=%d \n", errno);
		return;
	}

	DisplayIoctlHeader(ErrorInfo.IoctlHeader);
	printf("\tIOCTL Specific Data\n");
	printf("\t-------------------\n");
	DisplayErrorInfo(ErrorInfo.Information);
	printf("_______________________________________________________\n");
}
void issue_get_sata_signature(int fd, unsigned char bPhyIdentifier, int IOControllerNumber)
{
	int retval;
	CSMI_SAS_SATA_SIGNATURE_BUFFER Signature;

	printf("\tIOCTL : CC_CSMI_SAS_GET_SATA_SIGNATURE \n");
	printf("\t~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ \n");

	memset(&Signature, 0, sizeof(CSMI_SAS_SATA_SIGNATURE_BUFFER));

	Signature.IoctlHeader.Length = sizeof(CSMI_SAS_SATA_SIGNATURE_BUFFER);
	Signature.IoctlHeader.Timeout = CSMI_ALL_TIMEOUT;
	Signature.IoctlHeader.Direction = CSMI_SAS_DATA_READ;
	Signature.IoctlHeader.IOControllerNumber = IOControllerNumber;
	Signature.Signature.bPhyIdentifier = bPhyIdentifier;

	retval=ioctl(fd, CC_CSMI_SAS_GET_SATA_SIGNATURE, &Signature);

	if (retval != 0){
		printf("IOCTL FAILED. Errno=%d \n", errno);
		return;
	}

	DisplayIoctlHeader(Signature.IoctlHeader);
	printf("\tIOCTL Specific Data\n");
	printf("\t-------------------\n");
	DisplaySignature(Signature.Signature);
	printf("_______________________________________________________\n");
}
void issue_get_scsi_address(int fd, int TargetId, int LunId, int IOControllerNumber)
{
	int retval;
	CSMI_SAS_GET_SCSI_ADDRESS_BUFFER ScsiAddr;
	CSMI_SAS_GET_DEVICE_ADDRESS_BUFFER DeviceAddress;

	printf("\tIOCTL : CSMI_SAS_GET_SCSI_ADDRESS \n");
	printf("\t~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ \n");

	memset(&ScsiAddr, 0, sizeof(CSMI_SAS_GET_SCSI_ADDRESS_BUFFER));

	ScsiAddr.IoctlHeader.Length = sizeof(CSMI_SAS_GET_SCSI_ADDRESS_BUFFER);
	ScsiAddr.IoctlHeader.Timeout = CSMI_ALL_TIMEOUT;
	ScsiAddr.IoctlHeader.Direction = CSMI_SAS_DATA_READ;
	ScsiAddr.IoctlHeader.Direction = CSMI_SAS_DATA_READ;
	ScsiAddr.IoctlHeader.IOControllerNumber = IOControllerNumber;

	/* Use the get_device_address IOCTL to get the SAS address & lun */
	memset(&DeviceAddress, 0, sizeof(CSMI_SAS_GET_DEVICE_ADDRESS_BUFFER));

	DeviceAddress.IoctlHeader.Length = sizeof(CSMI_SAS_GET_DEVICE_ADDRESS_BUFFER);
	DeviceAddress.IoctlHeader.Timeout = CSMI_ALL_TIMEOUT;
	DeviceAddress.IoctlHeader.Direction = CSMI_SAS_DATA_READ;
	DeviceAddress.IoctlHeader.IOControllerNumber = IOControllerNumber;
	DeviceAddress.bTargetId = TargetId;
	DeviceAddress.bLun = LunId;

	retval=ioctl(fd, CC_CSMI_SAS_GET_DEVICE_ADDRESS, &DeviceAddress);

	if (retval != 0){
		printf("IOCTL FAILED (could not get SASAddress & SASLun). Errno=%d \n", errno);
		return;
	}
	/* Should have SAS address and lun now */

//	strcpy(ScsiAddr.bSASAddress, DeviceAddress.bSASAddress);
//	strcpy(ScsiAddr.bSASLun, DeviceAddress.bSASLun);
	memcpy(ScsiAddr.bSASAddress, DeviceAddress.bSASAddress,8);
	memcpy(ScsiAddr.bSASLun, DeviceAddress.bSASLun,8);

	ScsiAddr.bHostIndex = 0;
	ScsiAddr.bPathId = 0;
	ScsiAddr.bTargetId = 0;
	ScsiAddr.bLun = 0;

	retval=ioctl(fd, CC_CSMI_SAS_GET_SCSI_ADDRESS, &ScsiAddr);

	if (retval != 0){
		printf("IOCTL FAILED. Errno=%d \n", errno);
		return;
	}

	DisplayIoctlHeader(ScsiAddr.IoctlHeader);
	printf("\tIOCTL Specific Data\n");
	printf("\t-------------------\n");
	DisplaySCSIAddr(ScsiAddr);
	printf("_______________________________________________________\n");
}
void issue_get_device_address(int fd, int HostIndex, int PathId, int TargetId, int LunId, int IOControllerNumber)
{
	int retval, i;
	CSMI_SAS_GET_DEVICE_ADDRESS_BUFFER DeviceAddress;

	printf("\tIOCTL : CSMI_SAS_GET_DEVICE_ADDRESS \n");
	printf("\t~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ \n");

	memset(&DeviceAddress, 0, sizeof(CSMI_SAS_GET_DEVICE_ADDRESS_BUFFER));

	DeviceAddress.IoctlHeader.Length = sizeof(CSMI_SAS_GET_DEVICE_ADDRESS_BUFFER);
	DeviceAddress.IoctlHeader.Timeout = CSMI_ALL_TIMEOUT;
	DeviceAddress.IoctlHeader.Direction = CSMI_SAS_DATA_READ;
	DeviceAddress.IoctlHeader.IOControllerNumber = IOControllerNumber;

	DeviceAddress.bHostIndex = HostIndex;
	DeviceAddress.bPathId = PathId;
	DeviceAddress.bTargetId = TargetId;
	DeviceAddress.bLun = LunId;
	for(i=0; i<8; i++){
		DeviceAddress.bSASAddress[i] = '0';
		DeviceAddress.bSASLun[i] = '0';
	}
	retval=ioctl(fd, CC_CSMI_SAS_GET_DEVICE_ADDRESS, &DeviceAddress);

	if (retval != 0){
		printf("IOCTL FAILED. Errno=%d \n", errno);
		return;
	}

	DisplayIoctlHeader(DeviceAddress.IoctlHeader);
	printf("\tIOCTL Specific Data\n");
	printf("\t-------------------\n");
	DisplayDeviceAddress(DeviceAddress);
	printf("_______________________________________________________\n");
}
void issue_get_location(int fd, int HostIndex, int PathId, int TargetId, int LunId, int IOControllerNumber)
{
	int retval, i;
	CSMI_SAS_GET_LOCATION_BUFFER GetLocation;

	printf("\tIOCTL : CSMI_SAS_GET_LOCATION \n");
	printf("\t~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ \n");

	memset(&GetLocation, 0, sizeof(CSMI_SAS_GET_LOCATION_BUFFER));

	GetLocation.IoctlHeader.Length = sizeof(CSMI_SAS_GET_LOCATION_BUFFER);
	GetLocation.IoctlHeader.Timeout = CSMI_ALL_TIMEOUT;
	GetLocation.IoctlHeader.Direction = CSMI_SAS_DATA_READ;
	GetLocation.IoctlHeader.IOControllerNumber = IOControllerNumber;

	GetLocation.bHostIndex = HostIndex;
	GetLocation.bPathId = PathId;
	GetLocation.bTargetId = TargetId;
	GetLocation.bLun = LunId;
	GetLocation.bLengthOfLocationIdentifier = sizeof(CSMI_SAS_LOCATION_IDENTIFIER);
	for(i=0; i<8; i++){
		GetLocation.Location[0].bSASAddress[i] = '0';
		GetLocation.Location[0].bSASLun[i] = '0';
	}
	retval=ioctl(fd, CC_CSMI_SAS_GET_LOCATION, &GetLocation);

	if (retval != 0){
		printf("IOCTL FAILED. Errno=%d \n", errno);
		return;
	}

	DisplayIoctlHeader(GetLocation.IoctlHeader);
	printf("\tIOCTL Specific Data\n");
	printf("\t-------------------\n");
	DisplayGetLocation(GetLocation);
	printf("_______________________________________________________\n");
}

void issue_set_phy_info(int fd, int PhyId, int NegLinkRate, int MinLinkRate, int MaxLinkRate,int IOControllerNumber)
{
	int retval;
	CSMI_SAS_SET_PHY_INFO_BUFFER PhyInfo;

	printf("\tIOCTL : CC_CSMI_SAS_SET_PHY_INFO \n");
	printf("\t~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ \n");

	memset(&PhyInfo, 0, sizeof(CSMI_SAS_SET_PHY_INFO_BUFFER));

	PhyInfo.IoctlHeader.Length = sizeof(CSMI_SAS_SET_PHY_INFO_BUFFER);
	PhyInfo.IoctlHeader.Timeout = CSMI_ALL_TIMEOUT;
	PhyInfo.IoctlHeader.Direction = CSMI_SAS_DATA_WRITE;
	PhyInfo.IoctlHeader.IOControllerNumber = IOControllerNumber;
	PhyInfo.Information.bPhyIdentifier = PhyId;
	PhyInfo.Information.bNegotiatedLinkRate = NegLinkRate;
	PhyInfo.Information.bProgrammedMinimumLinkRate = MinLinkRate;
	PhyInfo.Information.bProgrammedMaximumLinkRate = MaxLinkRate;

	retval=ioctl(fd, CC_CSMI_SAS_SET_PHY_INFO, &PhyInfo);

	if (retval != 0){
		printf("IOCTL FAILED. Errno=%d \n", errno);
		return;
	}

	DisplayIoctlHeader(PhyInfo.IoctlHeader);
	printf("\tIOCTL Specific Data\n");
	printf("\t-------------------\n");
	printf("\tPhy Identifier : %d \n", PhyInfo.Information.bPhyIdentifier);
	printf("\tNeg Link Rate  : ");
	switch(PhyInfo.Information.bNegotiatedLinkRate){
		case CSMI_SAS_LINK_RATE_NEGOTIATE:
			printf("Negotiate a new link rate \n");
			break;
		case CSMI_SAS_LINK_RATE_PHY_DISABLED:
			printf("Disable the Phy \n");
			break;
	}
	printf("Min Link Rate   : ");
	PrintLinkRate(PhyInfo.Information.bProgrammedMinimumLinkRate);
	printf("Max Link Rate   : ");
	PrintLinkRate(PhyInfo.Information.bProgrammedMaximumLinkRate);
	printf("_______________________________________________________\n");

}

void issue_phy_control(int fd, int IOControllerNumber, int PhyId)
{
	int retval;
	CSMI_SAS_PHY_CONTROL_BUFFER PhyControl;

	printf("\tIOCTL : CC_CSMI_SAS_PHY_CONTROL \n");
	printf("\t~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ \n");

	memset(&PhyControl, 0, sizeof(CSMI_SAS_PHY_CONTROL_BUFFER));

	PhyControl.IoctlHeader.Length = sizeof(CSMI_SAS_PHY_CONTROL_BUFFER);
	PhyControl.IoctlHeader.Timeout = CSMI_ALL_TIMEOUT;
	PhyControl.IoctlHeader.Direction = CSMI_SAS_DATA_READ;
	PhyControl.IoctlHeader.IOControllerNumber = IOControllerNumber;

	printf("\tIssuing Phy Link Reset\n");
	PhyControl.uFunction = CSMI_SAS_PC_LINK_RESET;
	PhyControl.bPhyIdentifier = PhyId;
	
	retval=ioctl(fd, CC_CSMI_SAS_PHY_CONTROL, &PhyControl);

	if (retval != 0){
		printf("IOCTL FAILED. Errno=%d \n", errno);
		return;
	}

	DisplayIoctlHeader(PhyControl.IoctlHeader);
	printf("\tIOCTL Specific Data\n");
	printf("\t-------------------\n");
	DisplayPhyControl(PhyControl);
	printf("_______________________________________________________\n");

	printf("\tIssuing Phy Hard Reset\n");
	PhyControl.uFunction = 	CSMI_SAS_PC_HARD_RESET;
	retval=ioctl(fd, CC_CSMI_SAS_PHY_CONTROL, &PhyControl);

	if (retval != 0){
		printf("IOCTL FAILED. Errno=%d \n", errno);
		return;
	}

	DisplayIoctlHeader(PhyControl.IoctlHeader);
	printf("\tIOCTL Specific Data\n");
	printf("\t-------------------\n");
	DisplayPhyControl(PhyControl);
	printf("_______________________________________________________\n");
}


//#define	DATA_TRAN_SIZE	96
#define	DATA_TRAN_SIZE	96
#define CDB_LEN		6
void issue_ssp_passthru(int fd, int IOControllerNumber)
{
	__u32 sas_addr_low32, sas_addr_hi32;
	PCSMI_SAS_SSP_PASSTHRU_BUFFER ssp_passthru;
	int sz;
	/* inquiry cdb */
	char cdb[] = {0x12, 0, 0, 0, DATA_TRAN_SIZE, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};

	printf("\tIOCTL : CC_CSMI_SAS_SSP_PASSTHRU \n");
	printf("\t~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ \n");

	/* issue Inquiry */
	sz = sizeof(CSMI_SAS_SSP_PASSTHRU_BUFFER) + DATA_TRAN_SIZE;
	ssp_passthru = (PCSMI_SAS_SSP_PASSTHRU_BUFFER)malloc(sz);
	if(ssp_passthru == NULL) {
		printf("%s: no memory\n",__FUNCTION__);
		return;
	}

	memset(ssp_passthru,0,sz);

	/* setup header */
	ssp_passthru->IoctlHeader.Length = sz;
	ssp_passthru->IoctlHeader.Timeout = CSMI_ALL_TIMEOUT;
	ssp_passthru->IoctlHeader.Direction = CSMI_SAS_DATA_READ;
	ssp_passthru->IoctlHeader.IOControllerNumber = IOControllerNumber;

	printf("Enter the high 32 bit SAS Address(in hex): ");
	scanf("%x", &sas_addr_hi32);

	printf("Enter the low 32 bit SAS Address(in hex): ");
	scanf("%x", &sas_addr_low32);

	/* setup CSMI_SAS_SSP_PASSTHRU */
	ssp_passthru->Parameters.bDestinationSASAddress[7]
	 = ((__u8*)&sas_addr_low32)[0];
	ssp_passthru->Parameters.bDestinationSASAddress[6]
	 = ((__u8*)&sas_addr_low32)[1];
	ssp_passthru->Parameters.bDestinationSASAddress[5]
	 = ((__u8*)&sas_addr_low32)[2];
	ssp_passthru->Parameters.bDestinationSASAddress[4]
	 = ((__u8*)&sas_addr_low32)[3];
	ssp_passthru->Parameters.bDestinationSASAddress[3]
	 = ((__u8*)&sas_addr_hi32)[0];
	ssp_passthru->Parameters.bDestinationSASAddress[2]
	 = ((__u8*)&sas_addr_hi32)[1];
	ssp_passthru->Parameters.bDestinationSASAddress[1]
	 = ((__u8*)&sas_addr_hi32)[2];
	ssp_passthru->Parameters.bDestinationSASAddress[0]
	 = ((__u8*)&sas_addr_hi32)[3];
	memcpy(&ssp_passthru->Parameters.bCDB,cdb,16);
	ssp_passthru->Parameters.bCDBLength = CDB_LEN;
	ssp_passthru->Parameters.uDataLength = DATA_TRAN_SIZE;
	ssp_passthru->Parameters.bPortIdentifier = CSMI_SAS_IGNORE_PORT;
	ssp_passthru->Parameters.uFlags =
	    CSMI_SAS_SSP_READ | CSMI_SAS_SSP_TASK_ATTRIBUTE_SIMPLE;

	if (ioctl(fd, CC_CSMI_SAS_SSP_PASSTHRU, ssp_passthru) != 0){
		printf("IOCTL FAILED. Errno=%d \n", errno);
		return;
	}
	
	DisplayIoctlHeader(ssp_passthru->IoctlHeader);
	printf("\tIOCTL Specific Data\n");
	printf("\t-------------------\n");
	DisplayPassthruStatus(ssp_passthru->Status);
	
	if(ssp_passthru->Status.uDataBytes) {
		ShowBuf("Data In:  ",
		ssp_passthru->bDataBuffer,
		ssp_passthru->Status.uDataBytes);
	}
	printf("_______________________________________________________\n");
	free(ssp_passthru);

}
void issue_smp_passthru(int fd, int IOControllerNumber)
{
	__u32 sas_addr_low32, sas_addr_hi32, port_id;
	PCSMI_SAS_SMP_PASSTHRU_BUFFER smp_passthru;
	int sz;
	/* report general */
	char cdb[] = {0x40, 0, 0, 0};

	printf("\tIOCTL : CC_CSMI_SAS_SMP_PASSTHRU \n");
	printf("\t~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ \n");

	sz = sizeof(CSMI_SAS_SMP_PASSTHRU_BUFFER);
	smp_passthru = (PCSMI_SAS_SMP_PASSTHRU_BUFFER)malloc(sz);
	if(smp_passthru == NULL) {
		printf("%s: no memory\n",__FUNCTION__);
		return;
	}

	memset(smp_passthru,0,sz);

	/* setup header */
	smp_passthru->IoctlHeader.Length = sz;
	smp_passthru->IoctlHeader.Timeout = CSMI_ALL_TIMEOUT;
	smp_passthru->IoctlHeader.Direction = CSMI_SAS_DATA_READ;
	smp_passthru->IoctlHeader.IOControllerNumber = IOControllerNumber;

	printf("Enter the port number: ");
	scanf("%x", &port_id);

	printf("Enter the high 32 bit SAS Address(in hex): ");
	scanf("%x", &sas_addr_hi32);

	printf("Enter the low 32 bit SAS Address(in hex): ");
	scanf("%x", &sas_addr_low32);

	/* setup CSMI_SAS_SSP_PASSTHRU */
	smp_passthru->Parameters.bDestinationSASAddress[7]
	 = ((__u8*)&sas_addr_low32)[0];
	smp_passthru->Parameters.bDestinationSASAddress[6]
	 = ((__u8*)&sas_addr_low32)[1];
	smp_passthru->Parameters.bDestinationSASAddress[5]
	 = ((__u8*)&sas_addr_low32)[2];
	smp_passthru->Parameters.bDestinationSASAddress[4]
	 = ((__u8*)&sas_addr_low32)[3];
	smp_passthru->Parameters.bDestinationSASAddress[3]
	 = ((__u8*)&sas_addr_hi32)[0];
	smp_passthru->Parameters.bDestinationSASAddress[2]
	 = ((__u8*)&sas_addr_hi32)[1];
	smp_passthru->Parameters.bDestinationSASAddress[1]
	 = ((__u8*)&sas_addr_hi32)[2];
	smp_passthru->Parameters.bDestinationSASAddress[0]
	 = ((__u8*)&sas_addr_hi32)[3];

	smp_passthru->Parameters.bPhyIdentifier=CSMI_SAS_USE_PORT_IDENTIFIER;
	smp_passthru->Parameters.bPortIdentifier = (__u8)port_id;
	smp_passthru->Parameters.bConnectionRate=0 /* use current speed */;
	smp_passthru->Parameters.uRequestLength=4;
	memcpy(&smp_passthru->Parameters.Request,cdb,4);

	if (ioctl(fd, CC_CSMI_SAS_SMP_PASSTHRU, smp_passthru) != 0){
		printf("IOCTL FAILED. Errno=%d \n", errno);
		return;
	}
	DisplayIoctlHeader(smp_passthru->IoctlHeader);
	printf("\tIOCTL Specific Data\n");
	printf("\t-------------------\n");
	printf("\tSMP Passthru\n");
	printf("\tConnectionStatus\t: %d \n",smp_passthru->Parameters.bConnectionStatus);
	printf("\tResponseBytes\t\t: %d \n", smp_passthru->Parameters.uResponseBytes);
	ShowBuf("Data In:  ", &smp_passthru->Parameters.Response,
	    smp_passthru->Parameters.uResponseBytes);
	printf("_______________________________________________________\n");

}
void issue_stp_passthru(int fd, int IOControllerNumber)
{
	__u32 sas_addr_low32, sas_addr_hi32;
	PCSMI_SAS_STP_PASSTHRU_BUFFER stp_passthru;
	int sz;
	/* Identify device */
	char bCommandFIS[] = {0x27,0x80,0xEC};

	printf("\tIOCTL : CC_CSMI_SAS_STP_PASSTHRU \n");
	printf("\t~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ \n");

	/* issue Inquiry */
	sz = sizeof(CSMI_SAS_STP_PASSTHRU_BUFFER) + 512 /* full ident data is 512 bytes */;
	stp_passthru = (PCSMI_SAS_STP_PASSTHRU_BUFFER)malloc(sz);
	if(stp_passthru == NULL) {
		printf("%s: no memory\n",__FUNCTION__);
		return;
	}

	memset(stp_passthru,0,sz);

	/* setup header */
	stp_passthru->IoctlHeader.Length = sz;
	stp_passthru->IoctlHeader.Timeout = CSMI_ALL_TIMEOUT;
	stp_passthru->IoctlHeader.Direction = CSMI_SAS_DATA_READ;
	stp_passthru->IoctlHeader.IOControllerNumber = IOControllerNumber;

	printf("Enter the high 32 bit SAS Address(in hex): ");
	scanf("%x", &sas_addr_hi32);

	printf("Enter the low 32 bit SAS Address(in hex): ");
	scanf("%x", &sas_addr_low32);

	/* setup CSMI_SAS_SSP_PASSTHRU */
	stp_passthru->Parameters.bDestinationSASAddress[7]
	 = ((__u8*)&sas_addr_low32)[0];
	stp_passthru->Parameters.bDestinationSASAddress[6]
	 = ((__u8*)&sas_addr_low32)[1];
	stp_passthru->Parameters.bDestinationSASAddress[5]
	 = ((__u8*)&sas_addr_low32)[2];
	stp_passthru->Parameters.bDestinationSASAddress[4]
	 = ((__u8*)&sas_addr_low32)[3];
	stp_passthru->Parameters.bDestinationSASAddress[3]
	 = ((__u8*)&sas_addr_hi32)[0];
	stp_passthru->Parameters.bDestinationSASAddress[2]
	 = ((__u8*)&sas_addr_hi32)[1];
	stp_passthru->Parameters.bDestinationSASAddress[1]
	 = ((__u8*)&sas_addr_hi32)[2];
	stp_passthru->Parameters.bDestinationSASAddress[0]
	 = ((__u8*)&sas_addr_hi32)[3];
	memcpy(&stp_passthru->Parameters.bCommandFIS,bCommandFIS,3);
	stp_passthru->Parameters.uDataLength = 512;
	stp_passthru->Parameters.bPortIdentifier = CSMI_SAS_IGNORE_PORT;
	stp_passthru->Parameters.bConnectionRate=0 /* use current speed */;
	stp_passthru->Parameters.uFlags =
	    CSMI_SAS_STP_READ | CSMI_SAS_STP_PIO;

	if (ioctl(fd, CC_CSMI_SAS_STP_PASSTHRU, stp_passthru) != 0){
		printf("IOCTL FAILED. Errno=%d \n", errno);
		return;
	}
	DisplayIoctlHeader(stp_passthru->IoctlHeader);
	printf("\tIOCTL Specific Data\n");
	printf("\t-------------------\n");
	DisplayStpPassthruStatus(stp_passthru->Status);
	if(stp_passthru->Status.uDataBytes) {
		ShowBuf("Data In:  ",
		stp_passthru->bDataBuffer,
		stp_passthru->Status.uDataBytes);
	}
	printf("_______________________________________________________\n");
	free(stp_passthru);
}


void issue_tm_passthru(int fd, int TargetId, int LunId, int IOControllerNumber)
{

	int retval;
	CSMI_SAS_SSP_TASK_IU_BUFFER tm_passthru;

	printf("\tIOCTL : CC_CSMI_SAS_TASK_MANAGEMENT \n");
	printf("\t~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ \n");

	memset(&tm_passthru, 0, sizeof(CSMI_SAS_SSP_TASK_IU_BUFFER));

	tm_passthru.IoctlHeader.Length = sizeof(CSMI_SAS_SSP_TASK_IU_BUFFER);
	tm_passthru.IoctlHeader.Timeout = CSMI_ALL_TIMEOUT;
	tm_passthru.IoctlHeader.Direction = CSMI_SAS_DATA_READ;
	tm_passthru.IoctlHeader.IOControllerNumber = IOControllerNumber;

	tm_passthru.Parameters.bTargetId = TargetId;
	tm_passthru.Parameters.bLun = LunId;
	tm_passthru.Parameters.bPathId = IOControllerNumber; /* this is probally incorrect ??*/
	tm_passthru.Parameters.uFlags = CSMI_SAS_TASK_IU;
	tm_passthru.Parameters.bTaskManagementFunction = CSMI_SAS_SSP_LOGICAL_UNIT_RESET;
	tm_passthru.Parameters.uInformation = CSMI_SAS_SSP_TEST;

	retval=ioctl(fd, CC_CSMI_SAS_TASK_MANAGEMENT, &tm_passthru);

	if (retval != 0){
		printf("IOCTL FAILED. Errno=%d \n", errno);
		return;
	}
	
	DisplayIoctlHeader(tm_passthru.IoctlHeader);
	printf("\tIOCTL Specific Data\n");
	printf("\t-------------------\n");
	DisplayPassthruStatus(tm_passthru.Status);
	printf("_______________________________________________________\n");
}

#define BYTES_PER_LINE   16
void ShowBuf(char *titleP, void *dataBufP, int count)
{
	int done;
	int lineDone = 0;
	int i;
	unsigned char *bufP = (unsigned char *) dataBufP;
	unsigned char textBuf [BYTES_PER_LINE + 1];
	unsigned char nextByte;
	unsigned char sepChar;

	printf ("%s\n", titleP);

//	if (count > 96)
//		count = 96;

	done = 0;
	while (done < count) {
		if (lineDone == 0)
			printf(" %.4X: ", done);

		if (lineDone == (BYTES_PER_LINE/2))
			sepChar = '-';
		else
			sepChar = ' ';

		nextByte = bufP[done];

		printf ("%c%.2X", sepChar, nextByte);
		if ( (nextByte >= 0x20) && (nextByte <= 0x7F))
			textBuf[lineDone] = nextByte;
		else
			textBuf[lineDone] = '.';

		lineDone++;
		done++;

		if ((lineDone == BYTES_PER_LINE) || (done == count)) {
			for (i=lineDone; i<BYTES_PER_LINE; i++)
				printf("   ");

			textBuf[lineDone]='\0';
			printf("  %s\n", textBuf);
			lineDone =0;
		}
	}
	return;
}
void DisplayDriverInfo(CSMI_SAS_DRIVER_INFO Info)
{
	printf("\tName\t\t: %s\n", Info.szName);
	printf("\tDescription\t: %s\n", Info.szDescription);
	printf("\tMajor Revision\t: %d\n", Info.usMajorRevision);
	printf("\tMinor Revision\t: %d\n", Info.usMinorRevision);
	printf("\tBuild Revision\t: %d\n", Info.usBuildRevision);
	printf("\tRelease Revision: %d\n", Info.usReleaseRevision);
	printf("\tCSMI Maj Rev\t: %d\n",Info.usCSMIMajorRevision);
	printf("\tCSMI Min Rev\t: %d\n",Info.usCSMIMinorRevision);
}
void DisplayCntlrConfig(CSMI_SAS_CNTLR_CONFIG Config)
{
	printf("\tBase I/O Address: 0x%x\n", Config.uBaseIoAddress);
	printf("\tBoard ID \t: 0x%x\n", Config.uBoardID);
	printf("\tSlot Number\t: %d\n", Config.usSlotNumber);
	printf("\tController Class: %d\n", Config.bControllerClass);
	printf("\tI/O Bus Type\t: %d\n", Config.bIoBusType);
	printf("\tSerial Number\t: %s\n", Config.szSerialNumber);
	printf("\tMajor Revision\t: %d\n", Config.usMajorRevision);
	printf("\tMinor Revision\t: %d\n", Config.usMinorRevision);
	printf("\tBuild Revision\t: %d\n", Config.usBuildRevision);
	printf("\tRelease Revision: %d\n", Config.usReleaseRevision);
	printf("\tBIOS Maj Rev\t: %d\n", Config.usBIOSMajorRevision);
	printf("\tBIOS Min Rev\t: %d\n", Config.usBIOSMinorRevision);
	printf("\tBIOS Build Rev\t: %d\n", Config.usBIOSBuildRevision);
	printf("\tBIOS Rel Revision: %d\n", Config.usBIOSReleaseRevision);
	printf("\tController Flags: 0x%x\n", Config.uControllerFlags);

	printf("\n\n\tRedundant Controller Info:\n");
	printf("\t--------------------------\n");
	printf("\tMajor Revision\t: %d\n", Config.usRromMajorRevision);
	printf("\tMinor Revision\t: %d\n", Config.usRromMinorRevision);
	printf("\tBuild Revision\t: %d\n", Config.usRromBuildRevision);
	printf("\tRelease Revision: %d\n", Config.usRromReleaseRevision);
	printf("\tBIOS Maj Rev\t: %d\n", Config.usRromBIOSMajorRevision);
	printf("\tBIOS Min Rev\t: %d\n", Config.usRromBIOSMinorRevision);
	printf("\tBIOS Build Rev\t: %d\n", Config.usRromBIOSBuildRevision);
	printf("\tBIOS Rel Revision: %d\n", Config.usRromBIOSReleaseRevision);
}
void DisplayCntlrStatus(CSMI_SAS_CNTLR_STATUS Status)
{

	printf("\tStatus\t\t: ");
 	switch(Status.uStatus){
		case CSMI_SAS_CNTLR_STATUS_GOOD:
			printf("Good\n");
			break;
		case CSMI_SAS_CNTLR_STATUS_FAILED:
			printf("Failed\n");
			break;
		case CSMI_SAS_CNTLR_STATUS_OFFLINE:
			printf("Offline\n");
			break;
		case CSMI_SAS_CNTLR_STATUS_POWEROFF:
			printf("Power OFF\n");
			break;
		default:
			printf("Unknown!! \n");
			break;
	}

	// Do not print reason for offline if controller is not offline
	// if (Status.uStatus != CSMI_SAS_CNTLR_STATUS_OFFLINE)
		// return;

	printf("\n\tReason for Offline\t: ");

	switch(Status.uOfflineReason){
		case CSMI_SAS_OFFLINE_REASON_NO_REASON:
			printf("No Reason\n");
			break;
		case CSMI_SAS_OFFLINE_REASON_INITIALIZING:
			printf("Initializing\n");
			break;
		case CSMI_SAS_OFFLINE_REASON_BACKSIDE_BUS_DEGRADED:
			printf("Backside Bus Degraded\n");
			break;
		case CSMI_SAS_OFFLINE_REASON_BACKSIDE_BUS_FAILURE:
			printf("Backside Bus Failure\n");
			break;
	}
}
void DisplayRaidInfo(CSMI_SAS_RAID_INFO Info)
{
	printf("\tNumber of Raid Sets\t: %d\n", Info.uNumRaidSets);
	printf("\tMax Drivers per Set\t: %d\n", Info.uMaxDrivesPerSet);
}
void DisplayRaidConfig(CSMI_SAS_RAID_CONFIG * Config)
{
	int i, DrvIndex;
	printf("\tCapacity\t: %dMB\n", Config->uCapacity);
	printf("\tStripe Size\t: %dKB\n", Config->uStripeSize);
	printf("\tRaid Type\t: ");
	PrintRaidType(Config->bRaidType);
	printf("\tStatus\t: ");
	PrintRaidStatus(Config->bStatus);
	PrintRaidStatusMoreInfo(Config->bStatus, Config->bInformation);
	printf("# of drives in Raid: %d\n\n", Config->bDriveCount);

	for(DrvIndex = 0; DrvIndex < Config->bDriveCount; DrvIndex++){
		printf("\t#%d drive info \n", DrvIndex);
		printf("\t++++++++++++++ \n");
		printf("\tModel\t: %s\n", Config->Drives[DrvIndex].bModel);
		printf("\tFirmware\t: %s\n", Config->Drives[DrvIndex].bFirmware);
		printf("\tSerial No.\t: %s\n", Config->Drives[DrvIndex].bSerialNumber);
		printf("\tSAS Addr.\t: ");
		for(i = 0; i < 8; i++)
			printf("%02x ", Config->Drives[DrvIndex].bSASAddress[i]);
		printf("\tSAS LUN\t: ");
		for(i = 0; i < 8; i++)
			printf("%02x ", Config->Drives[DrvIndex].bSASLun[i]);
		printf("\tStatus\t: ");
		PrintDriveStatus(Config->Drives[DrvIndex].bDriveStatus);
		printf("\tUsage\t: ");
		PrintDriveUsage(Config->Drives[DrvIndex].bDriveUsage);
	}
}
void DisplayPhyInfo(int fd, CSMI_SAS_PHY_INFO Info)
{
	int PhyIndex;

	printf("\tTotal number of Phy = %d \n\n", Info.bNumberOfPhys);

	for(PhyIndex = 0; PhyIndex < Info.bNumberOfPhys; PhyIndex++)
	{
		printf("\t-------------------------------------------------\n");
		printf("\tPort Identifier         : %d \n", Info.Phy[PhyIndex].bPortIdentifier);
		printf("\tNegotiated Link Rate    : ");
		PrintLinkRate(Info.Phy[PhyIndex].bNegotiatedLinkRate);
/* EDM - Link Rate Bit Definition
 * bit [3:0] Hardware Link Rate
 * bit [7:4] Programmed Link Rate
 */
		printf("\tMinimum Programmed Link Rate       : ");
		PrintLinkRate(Info.Phy[PhyIndex].bMinimumLinkRate >> 4);
		printf("\tMaximum Programmed Link Rate       : ");
		PrintLinkRate(Info.Phy[PhyIndex].bMaximumLinkRate >> 4);
		printf("\tMinimum Hardware Link Rate       : ");
		PrintLinkRate(Info.Phy[PhyIndex].bMinimumLinkRate & 0xF);
		printf("\tMaximum Hardware Link Rate       : ");
		PrintLinkRate(Info.Phy[PhyIndex].bMaximumLinkRate & 0xF);
		printf("\t# of BROADCASE(CHANGE) primitives: %d \n", Info.Phy[PhyIndex].bPhyChangeCount);
		printf("\tState of Auto Discover  : ");
		PrintAutoDiscoverState(Info.Phy[PhyIndex].bAutoDiscover);
		printf("\n");
		printf("\tInfo transfered to the attached device during a link reset:\n");
		DisplaySASIdentify(Info.Phy[PhyIndex].Identify);
		printf("\n");
		printf("\tInformation about the attached device: \n");
		DisplaySASIdentify(Info.Phy[PhyIndex].Attached);
		printf("\n");
	}
}
void DisplaySASIdentify(CSMI_SAS_IDENTIFY Identify)
{
	int i;
	printf("\tDevice Type             : ");
	switch(Identify.bDeviceType){
		case CSMI_SAS_PHY_UNUSED:
			printf("Phy Unused.\n");
			break;
		case CSMI_SAS_END_DEVICE:
			printf("SAS End Device.\n");
			break;
		case CSMI_SAS_EDGE_EXPANDER_DEVICE:
			printf("SAS Edge Expander.\n");
			break;
		case CSMI_SAS_FANOUT_EXPANDER_DEVICE:
			printf("SAS Fanout Expander.\n");
			break;
		default:
			printf("Undefined.\n");
			break;
	}
	printf("\tInitiator Port Protocol : ");
	PrintProtocol(Identify.bInitiatorPortProtocol);
	printf("\tTarget Port Protocol    : ");
	PrintProtocol(Identify.bTargetPortProtocol);
	printf("\tSAS Address             : ");
	for(i=0;i<8;i++)
		printf("%02x ", Identify.bSASAddress[i]);
	printf("\n");
	printf("\tPhy Identifier          : %d \n", Identify.bPhyIdentifier);
}
void DisplayErrorInfo(CSMI_SAS_LINK_ERRORS Info)
{
	printf("\tPhy Identifier\t\t: %d \n", Info.bPhyIdentifier);
	printf("\tReset Counts\t\t: %d \n", Info.bResetCounts);
	printf("\tInvalid dwords\t\t: %d \n", Info.uInvalidDwordCount);
	printf("\tDisparity Errors\t: %d \n", Info.uRunningDisparityErrorCount);
	printf("\tLoss of dword sync\t: %d \n", Info.uLossOfDwordSyncCount);
	printf("\tPhy Reset Problem\t: %d \n", Info.uPhyResetProblemCount);
}
void DisplaySignature(CSMI_SAS_SATA_SIGNATURE Signature)
{
	int i;

	printf("\tPhy Identifier\t: %d \n", Signature.bPhyIdentifier);
/*	printf("\tSignature\t: <%s> \n", Signature.bSignatureFIS);*/
	printf("\tSignature\t: ");
	for(i=0;i<20;i++)
		printf("%02x ", Signature.bSignatureFIS[i]);;
	printf("\n");
}
void DisplaySCSIAddr(CSMI_SAS_GET_SCSI_ADDRESS_BUFFER ScsiAddrBuff)
{
	int i;
	printf("\tSAS Address\t: ");
	for(i=0; i<8; i++)
	  printf("%02x ", ScsiAddrBuff.bSASAddress[i]);
	printf("\n\tSAS LUN\t\t: ");
	for(i=0; i<8; i++)
	  printf("%02x ", ScsiAddrBuff.bSASLun[i]);
	printf("\n\tHost Index\t: %d\n", ScsiAddrBuff.bHostIndex);
	printf("\tPath ID\t\t: %d\n", ScsiAddrBuff.bPathId);
	printf("\tTarget ID\t: %d\n", ScsiAddrBuff.bTargetId);
	printf("\tLUN\t\t: %d\n", ScsiAddrBuff.bLun);
}

void DisplayDeviceAddress(CSMI_SAS_GET_DEVICE_ADDRESS_BUFFER DeviceAddress)
{
	int i;
	printf("\tHost Index\t: %d\n", DeviceAddress.bHostIndex);
	printf("\tPath ID\t\t: %d\n", DeviceAddress.bPathId);
	printf("\tTarget ID\t: %d\n", DeviceAddress.bTargetId);
	printf("\tLun\t\t: %d\n", DeviceAddress.bLun);
	printf("\tSAS Address\t: ");
	for(i=0; i<8; i++)
	  printf("%02x ", DeviceAddress.bSASAddress[i]);
	printf("\n\tSAS LUN\t\t: ");
	for(i=0; i<8; i++)
	  printf("%02x ", DeviceAddress.bSASLun[i]);
	printf("\n");
}
void DisplayGetLocation(CSMI_SAS_GET_LOCATION_BUFFER GetLocation)
{
	int i;
	printf("\tHost Index\t: %d\n", GetLocation.bHostIndex);
	printf("\tPath ID\t\t: %d\n", GetLocation.bPathId);
	printf("\tTarget ID\t: %d\n", GetLocation.bTargetId);
	printf("\tLun\t\t: %d\n", GetLocation.bLun);
	printf("\tNumber Of Location Identifiers\t\t: %d\n", GetLocation.bNumberOfLocationIdentifiers);
	printf("\tLength Of Location Identifiers\t\t: %d\n", GetLocation.bLengthOfLocationIdentifier);
	printf("\tLocation Flags\t: %d\n", GetLocation.Location[0].bLocationFlags);
	printf("\tSAS Address\t: ");
	for(i=0; i<8; i++)
	  printf("%02x ", GetLocation.Location[0].bSASAddress[i]);
	printf("\n");
	printf("\tSAS LUN\t\t: ");
	for(i=0; i<8; i++)
	  printf("%02x ", GetLocation.Location[0].bSASLun[i]);
	printf("\n");
	printf("\tEnclosure Identifier\t: ");
	for(i=0; i<8; i++)
	  printf("%02x ", GetLocation.Location[0].bEnclosureIdentifier[i]);
	printf("\n");
	printf("\tEnclosure Name\t: %s\n", GetLocation.Location[0].bEnclosureName);
	printf("\tBay Prefix\t: %s\n", GetLocation.Location[0].bBayPrefix);
	printf("\tBay Identifier\t: %d\n", GetLocation.Location[0].bBayIdentifier);
	printf("\tLocation State\t: %d\n", GetLocation.Location[0].bLocationState);
}
void DisplayPhyControl(CSMI_SAS_PHY_CONTROL_BUFFER PhyControl)
{
	printf("\tFunction\t: %d\n", PhyControl.uFunction);
	printf("\tPhy Identifier\t: %d\n", PhyControl.bPhyIdentifier);
	printf("\tLength Of Control\t: %d\n", PhyControl.usLengthOfControl);
	printf("\tNumber Of Controls\t: %d\n", PhyControl.bNumberOfControls);
	printf("\tLink Flags\t: %d\n", PhyControl.uLinkFlags);
	printf("\tSpinup Rate\t: %d\n", PhyControl.bSpinupRate);
}

void menu_driven_testing(int fd)
{
	int option=1, menu=1, PhyId;
	int HostIndex, PathId, TargetId, LunId;
	int NegLinkRate, MinLinkRate, MaxLinkRate;
	int IOControllerNumber;
	char fname[32];

	while(option != 20){
		printf("                    CSMI IOCTL TESTING                \n");
		printf("_______________________________________________________\n");
		printf(" 1. CC_CSMI_SAS_GET_DRIVER_INFO \n");
		printf(" 2. CC_CSMI_SAS_GET_CNTLR_CONFIG \n");
		printf(" 3. CC_CSMI_SAS_GET_CNTLR_STATUS \n");
		printf(" 4. CC_CSMI_SAS_FIRMWARE_DOWNLOAD \n");
		printf(" 5. CC_CSMI_SAS_GET_RAID_INFO \n");
		printf(" 6. CC_CSMI_SAS_GET_RAID_CONFIG \n");
		printf(" 7. CC_CSMI_SAS_GET_PHY_INFO \n");
		printf(" 8. CC_CSMI_SAS_GET_LINK_ERRORS \n");
		printf(" 9. CC_CSMI_SAS_SMP_PASSTHROUGH \n");
		printf("10. CC_CSMI_SAS_SSP_PASSTHROUGH \n");
		printf("11. CC_CSMI_SAS_STP_PASSTHROUGH \n");
		printf("12. CC_CSMI_SAS_GET_SATA_SIGNATURE \n");
		printf("13. CC_CSMI_SAS_GET_SCSI_ADDRESS \n");
		printf("14. CC_CSMI_SAS_GET_DEVICE_ADDRESS \n");
		printf("15. CC_CSMI_SAS_SET_PHY_INFO \n");
		printf("16. CC_CSMI_SAS_TASK_MANAGEMENT \n");
		printf("17. CC_CSMI_SAS_GET_CONNECTOR_INFO \n");
		printf("18. CC_CSMI_SAS_GET_LOCATION \n");
		printf("19. CC_CSMI_SAS_PHY_CONTROL \n");
		printf("20. Quit \n");

		printf("\n\n\n\n");
		printf("Enter the IOCTL number: ");
		scanf("%d", &option);
		printf("Enter the IOC Number(decimal): ");
		scanf("%d", &IOControllerNumber);
		switch(option){
			case 1:
				issue_get_driver_info(fd,IOControllerNumber);
				break;
			case 2:
				issue_get_ctlr_config(fd,IOControllerNumber);
				break;
			case 3:
				issue_get_ctlr_status(fd,IOControllerNumber);
				break;
			case 4:
				printf("Enter the image name: ");
				scanf("%s", fname);
				issue_firmware_download(fd, IOControllerNumber, fname);
				break;
			case 5:
				issue_get_raid_info(fd,IOControllerNumber);
				break;
			case 6:
				issue_get_raid_config(fd,IOControllerNumber);
				break;
			case 7:
				issue_get_phy_info(fd, menu,IOControllerNumber);
				break;
			case 8:
				printf("Enter the Phy Identifier(decimal): ");
				scanf("%d", &PhyId);
				issue_get_link_error(fd, PhyId, IOControllerNumber);
				break;
			case 9:
				issue_smp_passthru(fd,IOControllerNumber);
				break;
			case 10:
				issue_ssp_passthru(fd,IOControllerNumber);
				break;
			case 11:
				issue_stp_passthru(fd,IOControllerNumber);
				break;
			case 12:
				printf("Enter the Phy Identifier(decimal): ");
				scanf("%d", &PhyId);
				issue_get_sata_signature(fd, PhyId, IOControllerNumber);
				break;
			case 13:
				printf("Enter the Target ID (decimal): ");
				scanf("%d", &TargetId);
				printf("Enter the LUN ID (decimal): ");
				scanf("%d", &LunId);
				issue_get_scsi_address(fd, TargetId, LunId, IOControllerNumber);
				break;
			case 14:
				printf("Enter the Host Index (decimal): ");
				scanf("%d", &HostIndex);
				printf("Enter the Path ID (decimal): ");
				scanf("%d", &PathId);
				printf("Enter the Target ID (decimal): ");
				scanf("%d", &TargetId);
				printf("Enter the LUN ID (decimal): ");
				scanf("%d", &LunId);
				issue_get_device_address(fd, HostIndex, PathId, TargetId, LunId, IOControllerNumber);
				break;
			case 15:
			case 17:
				printf("\n\nThis IOCTL is currently unsupported in the mpt fusion driver!!!\n\n");
				break;
				printf("Enter the Phy Identifier (decimal): ");
				scanf("%d", &PhyId);
				printf("Enter the Negotiated Link Rate \n");
				printf(" [Negotiate a new link rate (0) / Disable the Phy (1)] : \n");
				scanf("%d", &NegLinkRate);

				printf("Enter the Minimum Link Rate \n");
				printf(" [ Unchanged (0) / 1.5 Gb/s (8) / 3.0 Gb/s (9)] : \n");
				scanf("%d", &MinLinkRate);

				printf("Enter the Maximum Link Rate \n");
				printf(" [ Unchanged (0) / 1.5 Gb/s (8) / 3.0 Gb/s (9)] : \n");
				scanf("%d", &MaxLinkRate);

				issue_set_phy_info(fd, PhyId, NegLinkRate, MinLinkRate,
				    MaxLinkRate, IOControllerNumber);
				break;
			case 16:
				printf("Enter the Target ID (decimal): ");
				scanf("%d", &TargetId);
				printf("Enter the LUN ID (decimal): ");
				scanf("%d", &LunId);
				issue_tm_passthru(fd, TargetId, LunId, IOControllerNumber);
				break;
			case 18:
				printf("Enter the Host Index (decimal): ");
				scanf("%d", &HostIndex);
				printf("Enter the Path ID (decimal): ");
				scanf("%d", &PathId);
				printf("Enter the Target ID (decimal): ");
				scanf("%d", &TargetId);
				printf("Enter the LUN ID (decimal): ");
				scanf("%d", &LunId);
				issue_get_location(fd, HostIndex, PathId, TargetId, LunId, IOControllerNumber);
				break;
			case 19:
				printf("Enter the Phy Identifier(decimal): ");
				scanf("%d", &PhyId);
				issue_phy_control(fd, IOControllerNumber, PhyId);
				break;
			case 20:
				exit(EXIT_SUCCESS);
			default:
				printf("Invalid Option\n");
				break;
		}
	}
}
