/* Include files
 */
#include "apps.h"


/* Global Variables
 */
extern int	g_iocnum;
extern int	g_fd;

/* Extern Declarations
 */
extern uchar OptionsMenuHandler (CmdOptions *menuPtr);
extern CmdOptions eventTypes[];

/*****************************************************************
 * General information IOCTLS. None of these IOCTLs require
 * an interrupt to be generated. Data is taken from host driver
 * memory or from system memory.     
 *
 *****************************************************************/
/*****************************************************************
 * GetIocInfo
 *
 * The driver will fill in the contents of the structure: 
 *  adapterType - 909, 929, ...
 *  PCI Device Id - from PCI configuration space
 *  PCI Hardware Revision - from PCI configuration space
 *  PCI Subsystem Id
 *  PCI Subsystem Vendor Id
 *  number of targets
 *  FW version -  firmware version 
 *  BIOS Version -  32 bit integer
 *  driver version - (string format) as found in mptbase.h
 *
 * Data returned in this structure is stored statically
 * within the driver.
 *****************************************************************/
void GetIocInfo(void)
{
	struct mpt_ioctl_iocinfo  thisBuf;
	int sz = sizeof(struct mpt_ioctl_iocinfo);

	memset (&thisBuf, 0, sz);

	/* Complete the header.
	 */
	thisBuf.hdr.iocnum = g_iocnum;
	thisBuf.hdr.port = 0;
	thisBuf.hdr.maxDataSize = sz;
	ShowBuf ("MPTIOCINFO Start:", &thisBuf, sz, 0);

	if (ioctl(g_fd, (unsigned long) MPTIOCINFO, (char *) &thisBuf) != 0)
		perror ("MPTIOCINFO ioctl failed");
	else {
		ShowBuf ("MPTIOCINFO Data:", &thisBuf, sz, 0);

#ifndef APP_MEM_TEST
		printf("IOC Information:\n");
		printf("====================================\n");
		printf("Adapter Type             = %d\n", thisBuf.adapterType);
		printf("PCI Device ID            = 0x%04x\n", thisBuf.pciId);
		printf("PCI HW Revision          = %d\n", thisBuf.hwRev);
		printf("PCI sub-system Device ID = 0x%04x\n", thisBuf.subSystemDevice);
		printf("PCI sub-system Vendor ID = 0x%04x\n", thisBuf.subSystemVendor);
		printf("Number of Targets        = 0x%x\n", thisBuf.numDevices);
		printf("FW version               = 0x%x\n", thisBuf.FWVersion);
		printf("BIOS version             = 0x%x\n", thisBuf.BIOSVersion);
		printf("Driver version           = %s\n", thisBuf.driverVersion);
		printf("bus Change Event         = 0x%x\n", thisBuf.busChangeEvent);
		printf("Host ID                  = 0x%x\n", thisBuf.hostId);
		printf("PCI Bus                  = 0x%x\n", thisBuf.pciInfo.u.bits.busNumber);
		printf("PCI Device               = 0x%x\n", thisBuf.pciInfo.u.bits.deviceNumber);
		printf("PCI Function             = 0x%x\n", thisBuf.pciInfo.u.bits.functionNumber);
#endif
	}
	return;
}

/*****************************************************************
 * GetOldIocInfo - REV 0 data structure
 *
 * The driver will fill in the contents of the structure: 
 *  adapterType - 909, 929, ...
 *  PCI Device Id - from PCI configuration space
 *  PCI Hardware Revision - from PCI configuration space
 *  PCI Subsystem Id
 *  PCI Subsystem Vendor Id
 *  number of targets
 *  FW version -  firmware version 
 *  BIOS Version -  32 bit integer
 *  driver version - (string format) as found in mptbase.h
 *
 * Data returned in this structure is stored statically
 * within the driver.
 *****************************************************************/
void GetOldIocInfo(void)
{
	struct mpt_ioctl_iocinfo_rev0  thisBuf;
	int sz = sizeof(struct mpt_ioctl_iocinfo_rev0);

	memset (&thisBuf, 0, sz);

	/* Complete the header.
	 */
	thisBuf.hdr.iocnum = g_iocnum;
	thisBuf.hdr.port = 0;
	thisBuf.hdr.maxDataSize = sz;

	ShowBuf ("MPTIOCINFO Start:", &thisBuf, sz, 0);
	if (ioctl(g_fd, (unsigned long) MPTIOCINFO1, (char *) &thisBuf) != 0)
		perror ("MPTIOCINFO ioctl failed");
	else {
		ShowBuf ("MPTIOCINFO Data:", &thisBuf, sz, 0);
		printf("IOC Information (rev 0):\n");
		printf("====================================\n");
		printf("Adapter Type             = %d\n", thisBuf.adapterType);
		printf("PCI Device ID            = 0x%04x\n", thisBuf.pciId);
		printf("PCI HW Revision          = %d\n", thisBuf.hwRev);
		printf("PCI sub-system Device ID = 0x%x\n", thisBuf.subSystemDevice);
		printf("PCI sub-system Vendor ID = 0x%x\n", thisBuf.subSystemVendor);
		printf("Number of Targets        = 0x%x\n", thisBuf.numDevices);
		printf("FW version               = 0x%x\n", thisBuf.FWVersion);
		printf("BIOS version             = 0x%x\n", thisBuf.BIOSVersion);
		printf("Driver version           = %s\n", thisBuf.driverVersion);
		printf("bus Change Event         = 0x%x\n", thisBuf.busChangeEvent);
		printf("Host ID                  = 0x%x\n", thisBuf.hostId);
	}
	return;
}

/*****************************************************************
 * GetTargetInfo
 *
 *
 * The driver will fill in the contents of the structure: 
 *  numDevices - total number of devices on this device
 *  targetInfo -  array of device information.
 *      LUN << 16 | bus << 8 | SCSI ID;
 *
 * Data returned in this structure is stored statically
 * within the driver.
 *****************************************************************/
void GetTargetInfo(void)
{
	struct mpt_ioctl_targetinfo  *pthisBuf;
	char *pmem;
#ifndef APP_MEM_TEST
	int  *pinfo;
	int  jj;
#endif
	int  sz, max;

#ifndef APP_MEM_TEST
	max = AskValue(1, 10, "Enter the maximum number of targets to report");
#else
	max = 10;
#endif
	sz = sizeof(struct mpt_ioctl_targetinfo) + (max - 1) * sizeof(int);

	if ((pmem = (char *) malloc (sz)) == NULL) {
		printf("GetTargetInfo: Error! No memory.\n");
		return;
	}

	pthisBuf = (struct mpt_ioctl_targetinfo *)pmem;
	memset(pthisBuf, 0, sz);

	/* Complete the header.
	 */
	pthisBuf->hdr.iocnum = g_iocnum;
	pthisBuf->hdr.port = 0;
	pthisBuf->hdr.maxDataSize = sz;

	ShowBuf("MPTTARGETINFO Start:", pthisBuf, sz, 0);
	if (ioctl(g_fd, (unsigned long) MPTTARGETINFO, (char *) pthisBuf) != 0)
		perror("MPTTARGETINFO ioctl failed");
	else {
		ShowBuf("MPTTARGETINFO Data:", pthisBuf, sz, 0);
#ifndef APP_MEM_TEST
		printf("Target Information:\n");
		printf("=============================\n");
		printf("numDevices %d\n", pthisBuf->numDevices);
		pinfo = pthisBuf->targetInfo;
		for (jj = 0; jj < pthisBuf->numDevices; jj++) {
			printf("Device Number %d:  Info: 0x%x\n", jj, *pinfo);
			pinfo++;
		}
#endif

	}
	free(pmem);

	return;
}
/*****************************************************************
 * Event Reporting data structures.  The data returned is
 * dependent on the defines in mpi_ioc.h.
 *****************************************************************/
/*****************************************************************
 * EventQuery
 *
 * The driver will fill in the contents of the structure. 
 *
 * Data returned in this structure is stored statically
 * within the driver.
 *****************************************************************/
void EventQuery(void)
{
	struct mpt_ioctl_eventquery  thisBuf;
	int  sz = sizeof (struct mpt_ioctl_eventquery);

	memset(&thisBuf, 0, sz);

	/* Complete the header.
	 */
	thisBuf.hdr.iocnum = g_iocnum;
	thisBuf.hdr.port = 0;
	thisBuf.hdr.maxDataSize = sz;

	ShowBuf("MPTEVENTQUERY Start:", &thisBuf, sz, 0);

	if (ioctl(g_fd, (unsigned long) MPTEVENTQUERY, (char *) &thisBuf) != 0)
		perror("MPTEVENTQUERY ioctl failed");
	else {
		ShowBuf("MPTEVENTQUERY Data:", &thisBuf, sz, 0);

		printf("Event Information:\n");
		printf("==========================\n");
		printf("eventEntries =  %d\n", thisBuf.eventEntries);
		printf("eventTypes   =  0x%x\n", thisBuf.eventTypes);
	}

	return;
}

/*****************************************************************
 * EventReport
 *
 *
 * The driver will fill in the contents of the structure: 
 *  numDevices - total number of devices on this device
 *  targetInfo -  array of device information.
 *      LUN << 16 | bus << 8 | SCSI ID;
 *
 * Data returned in this structure is stored statically
 * within the driver.
 *****************************************************************/
void EventReport(void)
{
	struct mpt_ioctl_eventreport  *pthisBuf;
	char *pmem;
	MPT_IOCTL_EVENTS  *pinfo;
	int  jj, sz = 256;
	int  max;

	max = AskValue(1, 10, "Enter the maximum number of event reports");
	sz = max * sizeof(MPT_IOCTL_EVENTS) + sizeof(mpt_ioctl_header);

	if ((pmem = (char *) malloc (sz)) == NULL) {
		printf("GetTargetInfo: Error! No memory.\n");
		return;
	}

	pthisBuf = (struct mpt_ioctl_eventreport *)pmem;
	memset(pthisBuf, 0, sz); 

	/* Complete the header.
	 */
	pthisBuf->hdr.iocnum = g_iocnum;
	pthisBuf->hdr.port = 0;
	pthisBuf->hdr.maxDataSize = sz;

	ShowBuf("MPTEVENTREPORT Start:", pthisBuf, sz, 0);

	if (ioctl(g_fd, (unsigned long) MPTEVENTREPORT, (char *) pthisBuf) != 0)
		perror("MPTEVENTREPORT ioctl failed");
	else {
		ShowBuf("MPTEVENTREPORT Data:", pthisBuf, sz, 0);
		printf("Event Report:\n");
		printf("==========================\n");
		pinfo = pthisBuf->eventData;
		for (jj = 0; jj < max; jj++) {
			printf("Event        = 0x%x \n", pinfo->event);
			printf("EventContext = 0x%x \n", pinfo->eventContext);
			ShowBuf("Event Data  = ", pinfo->data, 8, 1);
			pinfo++;
		}
	}

	free(pmem);
	return;
}

/*****************************************************************
 * EventEnable
 *
 * Instruct host driver to cache specified event types.
 *
 * No Data returned.
 *****************************************************************/
void EventEnable(void)
{
	struct mpt_ioctl_eventenable  thisBuf;
	int  sz = sizeof (struct mpt_ioctl_eventenable);
	u8 action;

	memset(&thisBuf, 0, sz);

	/*
	 * Complete the header.
	 */
	thisBuf.hdr.iocnum = g_iocnum;
	thisBuf.hdr.port = 0;
	thisBuf.hdr.maxDataSize = sz;

	action = 0;
	thisBuf.eventTypes =  0;
	while (action != 0xFF) {
		action =  OptionsMenuHandler (eventTypes);
	if (action != 0xFF)
		thisBuf.eventTypes |=  (1 << action);
	}

	ShowBuf("MPTEVENTENABLE Start:", &thisBuf, sz, 0);

	if (ioctl(g_fd, (unsigned long) MPTEVENTENABLE, (char *) &thisBuf) != 0)
		perror("MPTEVENTENABLE ioctl failed");
	else {
		ShowBuf("MPTEVENTENABLE Data:", &thisBuf, sz, 0);
		printf("Event Enable:\n");
		printf("==========================\n");
		printf("eventTypes 0x%x\n", thisBuf.eventTypes);
	}

	return;
}

/*****************************************************************
 * DiagnosticReset IOCTL
 *
 * Reset the adapter hardware and F/W. Risky!
 *****************************************************************/
void DiagnosticReset(void)
{
	struct mpt_ioctl_diag_reset  thisBuf;
	int sz = sizeof(struct mpt_ioctl_diag_reset);

	memset(&thisBuf, 0, sz);

	/* Complete the header.
	 */
	thisBuf.hdr.iocnum = g_iocnum;

	ShowBuf("MPTHARDRESET Start:", &thisBuf, sz, 0);
	printf ("Hard Reset Begin .... ");

	if (ioctl(g_fd, (unsigned long) MPTHARDRESET, (char *) &thisBuf) != 0)
		perror ("MPTHARDRESET ioctl failed");
	else
		printf ("Ok.\n");

	return;
}

/*****************************************************************
 * TestIoctl
 *****************************************************************/
void TestIoctl(void)
{
	struct mpt_ioctl_test  thisBuf;
	int sz = sizeof(struct mpt_ioctl_test);

	memset(&thisBuf, 0, sz);

	/* Complete the header.
	 */
	thisBuf.hdr.iocnum = g_iocnum;
	thisBuf.hdr.port = 0;
	thisBuf.hdr.maxDataSize = sz;

	ShowBuf("MPTTEST Start:", &thisBuf, sz, 0);

	if (ioctl(g_fd, (unsigned long) MPTTEST, (char *) &thisBuf) != 0)
		perror ("MPTTEST ioctl failed");
	else {
		ShowBuf ("MPTTEST Data:", &thisBuf, sz, 0);
		printf("IOCTL Test:\n");
		printf("==========================\n");
		printf ("name        =  %s\n", thisBuf.name);
		printf ("chip number =  %x\n", thisBuf.chip_type);
		printf ("product     =  %s\n", thisBuf.product);
	}

	return;
}


/*****************************************************************
 * ReplaceFW
 *****************************************************************/
void ReplaceFW(void)
{
	printf("Please use the mptfdu utility in the utils/ directory.\n");
	return;
}


/*****************************************************************
 * DiagRegesterIoctl
 *****************************************************************/
void DiagRegisterIoctl(void)
{
	mpt_diag_register_t  thisBuf;
	int sz = sizeof(thisBuf);

	memset(&thisBuf, 0, sz);

	/* Complete the header.
	 */
	thisBuf.hdr.iocnum = g_iocnum;
	thisBuf.hdr.port = 0;
	thisBuf.hdr.maxDataSize = sz;

	thisBuf.data.UniqueId = 
		AskValue(0, 0xffffffff, "Enter UniqueId to Register");

	thisBuf.data.BufferType = thisBuf.data.UniqueId & 0x000000ff;

	if (thisBuf.data.BufferType == MPI_DIAG_BUF_TYPE_EXTENDED ) {
		thisBuf.data.ExtendedType = 
			AskValue(0, 0xff, "Enter ExtendedType to Register");
	}

	thisBuf.data.TraceLevel = 0;
	thisBuf.data.ProductSpecific[0] = 
		AskValue(0, 0x400, "Enter ProductSpecific value");

	thisBuf.data.ProductSpecific[1] = (u32)0;
	thisBuf.data.ProductSpecific[2] = (u32)0;
	thisBuf.data.ProductSpecific[3] = (u32)0;

	thisBuf.data.RequestedBufferSize = AskValue(0x4000, 1000000, "Enter BufferSize to Register");

	ShowBuf("DIAG REGISTER BUFFER Start:", &thisBuf, sz, 0);

	if (ioctl(g_fd, (unsigned long) MPTDIAGREGISTER, (char *) &thisBuf) != 0)
		perror ("Diag Register Buffer ioctl failed");
	else {
		ShowBuf ("Diag Register Buffer Data:", &thisBuf, sz, 0);
	}

	return;
}


/*****************************************************************
 * DiagReleaseIoctl
 *****************************************************************/
void DiagReleaseIoctl(void)
{
	mpt_diag_release_t  thisBuf;
	int sz = sizeof(thisBuf);

	memset(&thisBuf, 0, sz);

	/* Complete the header.
	 */
	thisBuf.hdr.iocnum = g_iocnum;
	thisBuf.hdr.port = 0;
	thisBuf.hdr.maxDataSize = sz;

	thisBuf.data.UniqueId = 
		AskValue(0, 0xffffffff, "Enter UniqueId to Release");

	ShowBuf("DIAG RELEASE Start:", &thisBuf, sz, 0);

	if (ioctl(g_fd, (unsigned long) MPTDIAGRELEASE, (char *) &thisBuf) != 0)
		perror ("Diag Release ioctl failed");
	else {
		ShowBuf ("Diag Release Data:", &thisBuf, sz, 0);
	}

	return;
}


/*****************************************************************
 * DiagUnregisterIoctl
 *****************************************************************/
void DiagUnregisterIoctl(void)
{
	mpt_diag_unregister_t  thisBuf;
	int sz = sizeof(thisBuf);

	memset(&thisBuf, 0, sz);

	/* Complete the header.
	 */
	thisBuf.hdr.iocnum = g_iocnum;
	thisBuf.hdr.port = 0;
	thisBuf.hdr.maxDataSize = sz;

	thisBuf.data.UniqueId = 
		AskValue(0, 0xffffffff, "Enter UniqueId to Unregister");

	ShowBuf("DIAG UNREGISTER Start:", &thisBuf, sz, 0);

	if (ioctl(g_fd, (unsigned long) MPTDIAGUNREGISTER, (char *) &thisBuf) != 0)
		perror ("Diag Unregister ioctl failed");
	else {
		ShowBuf ("Diag Unregister Data:", &thisBuf, sz, 0);
	}

	return;
}


/*****************************************************************
 * DiagQueryIoctl
 *****************************************************************/
void DiagQueryIoctl(void)
{
	mpt_diag_query_t  thisBuf;
	int sz = sizeof(thisBuf);

	memset(&thisBuf, 0, sz);

	/* Complete the header.
	 */
	thisBuf.hdr.iocnum = g_iocnum;
	thisBuf.hdr.port = 0;
	thisBuf.hdr.maxDataSize = sz;

	thisBuf.data.UniqueId = 
		AskValue(0, 0xffffffff, "Enter UniqueId to Query");

	thisBuf.data.BufferType = thisBuf.data.UniqueId & 0x000000ff;

	if ( thisBuf.data.BufferType == MPI_DIAG_BUF_TYPE_EXTENDED ) {
		thisBuf.data.ExtendedType = 
			AskValue(0, 0xff, "Enter ExtendedType to Query");
	}

	ShowBuf("DIAG QUERY Start:", &thisBuf, sz, 0);

	if (ioctl(g_fd, (unsigned long) MPTDIAGQUERY, (char *) &thisBuf) != 0)
		perror ("Diag Query ioctl failed");
	else {
		ShowBuf ("Diag Query Data:", &thisBuf, sz, 0);
	}

	return;
}


/*****************************************************************
 * DiagReadBufferIoctl
 *****************************************************************/
void DiagReadBufferIoctl(void)
{
	mpt_diag_read_buffer_t  *pthisBuf;
	char *pmem;
	int  sz_request = sizeof(mpt_diag_read_buffer_t);
	int  sz, max, reregister;

#ifndef APP_MEM_TEST
	max = AskValue(1, 1000000, "Enter Number of Bytes to Read");
#else
	max = 40;
#endif
	sz = sz_request + (max - 4); /* one u32 is already in mpt_diag_read_buffer_t */

	if ((pmem = (char *) malloc (sz)) == NULL) {
		printf("DiagReadBufferIoctl: Error! No memory.\n");
		return;
	}

	pthisBuf = (mpt_diag_read_buffer_t *)pmem;
	memset(pthisBuf, 0, sz);

	/* Complete the header.
	 */
	pthisBuf->hdr.iocnum = g_iocnum;
	pthisBuf->hdr.port = 0;
	pthisBuf->hdr.maxDataSize = sz;

	pthisBuf->data.UniqueId = 
		AskValue(0, 0xffffffff, "Enter UniqueId to Read");

	pthisBuf->data.BytesToRead = max;

	pthisBuf->data.StartingOffset = AskValue(0, 1000000, "Enter StartingOffset to Read");

	reregister = AskValue(0, 1, "Reregister?");

	if (reregister)
		pthisBuf->data.Flags |= MPI_FW_DIAG_FLAG_REREGISTER;

	ShowBuf("DIAG READ BUFFER Start:", pthisBuf, sz_request, 0);

	if (ioctl(g_fd, (unsigned long) MPTDIAGREADBUFFER, (char *) pthisBuf) != 0)
		perror ("Diag Read Buffer ioctl failed");
	else {
		ShowBuf ("Diag Read Buffer Data:", pthisBuf, sz, 0);
	}

	free(pmem);
	return;
}

