/*****************************************************************
 *                                                               *
 * Copyright 2000 LSI Logic. All rights reserved.                *
 *                                                               *
 * This file is confidential and a trade secret of LSI Logic.    *
 * The receipt of or possession of this file does not convey any *
 * rights to reproduce or disclose its contents or to            *
 * manufacture, use, or sell anything it may describe, in whole, *
 * or in part, without the specific written consent of LSI Logic.*
 *                                                               *
 *****************************************************************/

/*****************************************************************
 *      Name:       mptio.c
 *      Title:      Utility for UnixWare MPT driver
 *      Programmer:  Pamela Delaney
 *      Creation Date: October 4, 2000
 *
 *  Description:
 *      Handles the non-io mpt command construction and ioctl
 *      issuing. Must be linked with apps.c
 *
 *  Version History
 *  ---------------
 *
 *  Date    Who Description
 *  ---------------------------
 *  4/18/01 PAD Major changes. Redefined mpiBlkPtr_t. 
 *              Added Linux support.
 *
 *
 *****************************************************************/


/*
 * System Include Files
 */
#include "apps.h"

#define SCSI            0

/*
 * Defines. IO_ALIGN used for memory allocation, second is used
 * limint the number of bytes in the RF that are to be displayed.  
 */
#define IO_ALIGN 4

#define REPLY_FRAME_DISPLAY     0x30
#define DATA_DIR_NONE           0
#define DATA_DIR_IN             1
#define DATA_DIR_OUT            2
#define DATAIN_LENGTH           20
#define MAX_FILE_NAME_LENGTH	32

/*
 * Internal Function Prototypes - setup for each of the specific
 * mpt commands that can be issued. 
 */
int mptio_GetIocInfo(void);
void mptio_GetTargetInfo(void);
void mptConfigIoctl (u8 pagetype, u8 bus, u8 id);
int IssueScsiCommand(u8 bus, u8 id, u8 lun, u8 type);

/*
 * Utility functions - these handle the allocation and freeing
 * of memory as well, user input, and menu display.
 */
void IssueCommand ( u8 pagetype, u8 bus, u8 id );
int IssueScsiMptCommand ( u8 bus, u8 id, u8 lun, u8 type );


/*
 * Internal Declarations. Global. Points to the mpt Ioctl block
 * that is to be passed to the host driver.  Only one command at
 * a time may be issued.....so a single global variable works just fine. 
 */
extern mpiIoctlBlk_t    *mpiBlkPtr;
char InqData[INQ_SIZE];
int g_adapterType=0;
u8 RAID_PhysDiskID;
u8 RAID_NumPhysDisks;
u8 RAID_PhysDiskNum[6];
u8 HOTSPARE_NumHotSpares;
u8 HOTSPARE_PhysDiskNum[6];
u8 HOTSPARE_Flags[6];


/*
 * External Declarations.
 */

extern int g_fd;
extern int g_iocnum;


/*****************************************************************
 *
 *
 *      Non-IO MPT Ioctls
 *
 *
 *****************************************************************/

/*****************************************************************
 * mptio_GetIocInfo
 *
 * The driver will fill in the contents of the structure: 
 *  adapterType - 909, 929, ...
 *  PCI Device Id - from PCI configuration space
 *  PCI Hardware Revision - from PCI configuration space
 *  PCI Subsystem Id
 *  PCI Subsystem Vendor Id
 *  number of targets
 *  FW version -  firmware version 
 *  BIOS Version -  32 bit integer
 *  driver version - (string format) as found in mptbase.h
 *
 * Data returned in this structure is stored statically
 * within the driver.
 *****************************************************************/
int mptio_GetIocInfo(void)
{
    typedef struct Chip_chars
    {
        char chip_name[5];             /* chip string */
        unsigned char    MPT_Type;     /* 1=SCSI, 2=FC, 3=SAS */
        unsigned short   devid;        /* Device Id */
        char chip_type[5];             /* chip type string */
    } Chip_chars;

    struct Chip_chars Chips[]={
	{"1030", 1, 0x0030, "SCSI"},
	{"1035", 1, 0x0032, "SCSI"},
	{"929X", 2, 0x0626, "FIBRE"},
	{"919X", 2, 0x0628, "FIBRE"},
	{"929",  2, 0x0622, "FIBRE"},
	{"919",  2, 0x0624, "FIBRE"},
	{"909",  2, 0x0621, "FIBRE"},
	{"1064", 3, 0x0050, "SAS"},
    };
    unsigned char numChips=sizeof(Chips)/sizeof(Chip_chars);

    struct mpt_ioctl_iocinfo_rev0  thisBuf;
    int sz = sizeof(struct mpt_ioctl_iocinfo_rev0);
    ushort devid;          /* PCI Device ID */
    char   *ChipName;
    char   *ChipType;
    u8  ii;

    memset(&thisBuf, 0, sz); 

    /*
     * Complete the header.
     */
    thisBuf.hdr.iocnum = g_iocnum;
    thisBuf.hdr.port = 0;
    thisBuf.hdr.maxDataSize = sz;

    if ( ioctl(g_fd, MPTIOCINFO1, (char *)&thisBuf) == 0)
    {
        devid = thisBuf.pciId;
        for ( ii = 0; ii < numChips; ii++)
        {
	    if ( devid == Chips[ii].devid )
	    {
		ChipName = Chips[ii].chip_name;
		ChipType = Chips[ii].chip_type;
		g_adapterType = thisBuf.adapterType;
		goto found_devid;
	    }
        }
        ChipName = "UNKWN";
        ChipType = "UNKWN";

found_devid:
        printf("\n%s %s-%d ha=%d id=%d BIOS: %x.%02x.%02x.%02x FW: %x.%02x.%02x.%02x Driver: %s\n", 
               ChipType, ChipName, thisBuf.hwRev, g_iocnum, thisBuf.hostId, 
               (u8)(thisBuf.BIOSVersion>>24),
               (u8)(thisBuf.BIOSVersion>>16),
               (u8)(thisBuf.BIOSVersion>>8),
               (u8)thisBuf.BIOSVersion,
               ((u8)(thisBuf.FWVersion>>24)),
               ((u8)(thisBuf.FWVersion>>16)),
               ((u8)(thisBuf.FWVersion>>8)),
               ((u8)thisBuf.FWVersion),
               thisBuf.driverVersion);

        return 1;
    }
    return 0;
}


/*****************************************************************
 * mptio_GetTargetInfo
 *
 *
 * The driver will fill in the contents of the structure: 
 *  numDevices - total number of devices on this device
 *  targetInfo -  array of device information.
 *	LUN << 16 | bus << 8 | SCSI ID;
 *
 * Data returned in this structure is stored statically
 * within the driver.
 *****************************************************************/
void mptio_GetTargetInfo(void)
{
    struct mpt_ioctl_targetinfo  *pdataBuf;
    char *pmem;
    int  *pinfo;
    volatile int  ii, jj, sz, max;
    volatile u8 pagetype, bus, id, lun, type;
    
    max = 16;
    sz = sizeof(struct mpt_ioctl_targetinfo) + (max) * sizeof(int);

    if ((pmem = (char *) malloc (sz)) == NULL) {
	    printf("GetTargetInfo: No memory for targetinfo.\n");
	    return;
    }

    pdataBuf = (struct mpt_ioctl_targetinfo *)pmem;
    memset(pdataBuf, 0, sz); 

    /*
     * Complete the header.
     */
    pdataBuf->hdr.iocnum = g_iocnum;
    pdataBuf->hdr.port = 0;
    pdataBuf->hdr.maxDataSize = sz;

    if ( ioctl(g_fd, MPTTARGETINFO, (char *)pdataBuf) == 0)
    {
//	printf ("MPTTARGETINFO completed on ioc=%d numDevices=%d\n", g_iocnum, pdataBuf->numDevices);
	if (pdataBuf->numDevices) 
        {
	    pinfo = pdataBuf->targetInfo;
	    for (jj = 0; jj < pdataBuf->numDevices; jj++) 
            {
                id = (u8)(*pinfo);
                bus = (u8)(*pinfo>>8);
		lun = (u8)(*pinfo>>16);
		type = (u8)(*pinfo>>24);
//		printf ("device id=%d bus=%d lun=%d type=%x found\n", id, bus, lun, type);
                if ( type == 0xC0 )
                {
                        printf("\nRaid Volume: ");
                        printf("on bus=%d id=%d lun=%d  Inactive\n", bus, id, lun);
                }
                else
                {
		    if ( IssueScsiCommand(bus, id, lun, type) )
                    {
                        InqData[INQ_SIZE] = 0x00;
                        if ( type & 0x80 )
                        {
                            printf("\nRaid Volume: %s  ", &InqData[8]);
                            printf("on bus=%d id=%d lun=%d\n", bus, id, lun);
                            pagetype = MPI_CONFIG_PAGETYPE_RAID_VOLUME;
                            mptConfigIoctl (pagetype, bus, id);

                            for ( ii = 0; ii < RAID_NumPhysDisks; ii++ )
                            {
                                pagetype = MPI_CONFIG_PAGETYPE_RAID_PHYSDISK;
                                id = RAID_PhysDiskNum[ii];
                                printf("\nRaid Device %d:", id);
                                mptConfigIoctl (pagetype, bus, id);
                                pagetype = MPI_CONFIG_PAGETYPE_SCSI_DEVICE;
                                mptConfigIoctl (pagetype, bus, RAID_PhysDiskID);
                            }

                            pagetype = MPI_CONFIG_PAGETYPE_IOC;
                            mptConfigIoctl (pagetype, bus, id);

                            if ( HOTSPARE_NumHotSpares )
                            {
                                for ( ii = 0; ii < HOTSPARE_NumHotSpares; ii++ )
                                {
                                    pagetype = MPI_CONFIG_PAGETYPE_RAID_PHYSDISK;
                                    id = HOTSPARE_PhysDiskNum[ii];
                                    if ( HOTSPARE_Flags[ii] )
                                        printf("\nHot Spare %d:", id);
                                    else
                                        printf("\nInactive Hot Spare %d:", id);
                                    mptConfigIoctl (pagetype, bus, id);
                                    pagetype = MPI_CONFIG_PAGETYPE_SCSI_DEVICE;
                                    mptConfigIoctl (pagetype, bus, RAID_PhysDiskID);
                                }
                            }
                            else
                                    printf("\nNo Hot Spare Device on Raid Volume\n");
                        }
                        else
                        {
                            printf("\nDevice: %s  ", &InqData[8]);
                            printf("\non bus=%d id=%d lun=%d\n", bus, id, lun);
                            if ( g_adapterType  == SCSI )
                                pagetype = MPI_CONFIG_PAGETYPE_SCSI_DEVICE;
                            else
                                pagetype = MPI_CONFIG_PAGETYPE_FC_PORT;
                            mptConfigIoctl (pagetype, bus, id);
                        }
                    }
                    else
                    {
                        printf("\nDevice: not present  ");
                        printf("on bus=%d id=%d lun=%d\n", bus, id, lun);
                    }
                }
		pinfo++;
	    }
	}
        else
        {
            printf ("No devices attached.\n");
        }
    }
    else
        printf ("\n GetTargetInfo: Ioctl Failed\n");
    free(pmem);
}

/*****************************************************************
 * mptConfigIoctl
 *
 * Get the IOC Facts and report to the user. Memory freed in
 * function call IssueCommand.
 *****************************************************************/
void mptConfigIoctl (u8 pagetype, u8 bus, u8 id)
{
    uint numBytes = (sizeof(Config_t) - sizeof(SGE_IO_UNION)) 
                                        + sizeof (SGESimple64_t);
    Config_t *ConfigRequest;
    uint pageaddress = 0;
    u8 action, pagenumber, pageversion, pagelength;

//    printf ("mptConfigIoctl: calling allocIoctlBlk for numBytes=%d\n", numBytes );
    if ( (mpiBlkPtr = allocIoctlBlk(numBytes)) == NULL)
    {
        printf ("\n mptConfigIoctl: allocIoctlBlk failed for pagetype=%x\n", pagetype );
        return;
    }

    ConfigRequest = (Config_t *)&mpiBlkPtr->MF;
    mpiBlkPtr->dataSgeOffset = (sizeof (Config_t) - sizeof(SGE_IO_UNION))/4;

    /*
     * Set the page number.
     */
    pagenumber = 0;

    /*
     * Set the action and the page type
     */
    action = MPI_CONFIG_ACTION_PAGE_READ_CURRENT;
    if ( pagetype == MPI_CONFIG_PAGETYPE_SCSI_DEVICE )
        pagelength = 3;
    else if ( pagetype == MPI_CONFIG_PAGETYPE_RAID_PHYSDISK )
        pagelength = 30;
    else if ( pagetype == MPI_CONFIG_PAGETYPE_FC_PORT )
        pagelength = 19;
    else if (  pagetype == MPI_CONFIG_PAGETYPE_RAID_VOLUME )
        pagelength = 25;
    else if (  pagetype == MPI_CONFIG_PAGETYPE_IOC )
    {
        pagenumber = 5;
        pagelength = 5;
    }
    else
    {
       printf ("mptConfigIoctl: Unknown pagetype=%x\n", pagetype);
       return;
    }

    /*
     * For the selected action, page, and page number, set the
     * page version and page length.
     */
    mpiBlkPtr->dataOutSize = 0;

    pageversion = 0;
        
    mpiBlkPtr->dataInSize = pagelength * 4;

    if ( allocDataFrame( DATA_DIR_IN ) )
    {
        printf ("Config: Unable to allocate data buffer.");
        freeAllocMem ();
        return;
    }

    /*
     * For selection action, page, and page number, page version 
     * and page length, set the page address.
     */
    pageaddress = (bus << 8) | id;

    /* Remark:  SCSI_DEVICE page has some funky pageaddress 
     * formats when handling hidden devices in IM.
     */

    /*
     * Populate the Config Request
     */

    ConfigRequest->Action = action;
    ConfigRequest->Function     = MPI_FUNCTION_CONFIG;
    ConfigRequest->MsgContext   = -1;
    ConfigRequest->Header.PageVersion = pageversion;
    ConfigRequest->Header.PageLength = pagelength;
    ConfigRequest->Header.PageNumber = pagenumber;
    ConfigRequest->Header.PageType = pagetype;
    ConfigRequest->PageAddress = pageaddress;

    IssueCommand( pagetype, bus, id );
}

/*****************************************************************
 * IssueCommand
 *
 * Generic command to issue the MPT command using the special
 * MPTCOMMAND Ioctl Function.
 *****************************************************************/
void IssueCommand ( u8 pagetype, u8 bus, u8 id )
{
        char *speed;
        u8 ii;
        uint tmp;
	uint negoParms;
        SCSIDevicePage0_t *pSDP0;
        RaidVolumePage0_t *pVOL0;
        RaidPhysDiskPage0_t *pRAIDPhysDisk0;
        IOCPage5_t *pIOCPage5;
        FCPortPage0_t *pFCP0;

	/* Set the IOC number prior to issuing this command.
	 */
	mpiBlkPtr->hdr.iocnum = g_iocnum;
	mpiBlkPtr->hdr.port = 0;

    	if (ioctl(g_fd, (unsigned long) MPTCOMMAND, (char *) mpiBlkPtr) == 0)
        {
                if (mpiBlkPtr->dataInBufPtr)
                {
                    if ( pagetype == MPI_CONFIG_PAGETYPE_SCSI_DEVICE )
                    {
                        pSDP0 = (SCSIDevicePage0_t *) mpiBlkPtr->dataInBufPtr;
			negoParms = le32_to_cpu(pSDP0->NegotiatedParameters);
                        printf("%s", negoParms & MPI_SCSIDEVPAGE0_NP_WIDE ? "Wide" : "Narrow");

	                if (negoParms & MPI_SCSIDEVPAGE0_NP_NEG_SYNC_OFFSET_MASK) 
                        {
	                    tmp = (negoParms & MPI_SCSIDEVPAGE0_NP_NEG_SYNC_OFFSET_MASK) >> 16;
                            printf("\t\tOffset: 0x%x", tmp);
	                    tmp = (negoParms & MPI_SCSIDEVPAGE0_NP_NEG_SYNC_PERIOD_MASK) >> 8;
		            if (tmp == 0x08)
			        speed = "Ultra320";
		            else if (tmp <= 0x09)
			        speed = "Ultra160";
		            else if (tmp <= 0x0A)
		  	        speed = "Ultra2";
		            else if (tmp <= 0x0C)
			        speed = "Ultra";
		            else
			        speed = "Fast or Slower";

                            printf("\t\tSync Factor: 0x%x (%s)\n", tmp, speed);
                            if ( tmp < 0x0A )
                            {
	                        tmp = negoParms & MPI_SCSIDEVPAGE0_NP_QAS;
		                printf("QAS %s", tmp ? "enabled" : "disabled");
	                        tmp = negoParms & MPI_SCSIDEVPAGE0_NP_DT;
		                printf("\t\tDT %s", tmp ? "enabled" : "disabled");
	                        tmp = negoParms & MPI_SCSIDEVPAGE0_NP_IU;
		                printf("\t\tIU %s\n", tmp ? "enabled" : "disabled");
                            }
                        } else 
                            printf("\t\tAsynchronous\n");
                    }
                    else if ( pagetype == MPI_CONFIG_PAGETYPE_RAID_PHYSDISK )
                    {
                        pRAIDPhysDisk0 = ( RaidPhysDiskPage0_t *) mpiBlkPtr->dataInBufPtr;
                        printf(" %s", pRAIDPhysDisk0->InquiryData.VendorID);
                        RAID_PhysDiskID = pRAIDPhysDisk0->PhysDiskID;
                        printf("\non bus=%d id=%d", bus, RAID_PhysDiskID);

                        if ( pRAIDPhysDisk0->PhysDiskStatus.State == MPI_PHYSDISK0_STATUS_ONLINE )
                            printf ("  Online\n");
                        else if ( pRAIDPhysDisk0->PhysDiskStatus.State == MPI_PHYSDISK0_STATUS_MISSING )
                            printf ("  Missing\n");
                        else if ( pRAIDPhysDisk0->PhysDiskStatus.State == MPI_PHYSDISK0_STATUS_NOT_COMPATIBLE )
                            printf ("  Not Compatible\n");
                        else if ( pRAIDPhysDisk0->PhysDiskStatus.State == MPI_PHYSDISK0_STATUS_INITIALIZING )
                            printf ("  Initializing\n");
                        else if ( (pRAIDPhysDisk0->PhysDiskStatus.State == MPI_PHYSDISK0_STATUS_FAILED)  ||
                                  (pRAIDPhysDisk0->PhysDiskStatus.State == MPI_PHYSDISK0_STATUS_FAILED_REQUESTED)  )
                            printf ("  Failed\n");
                        else if ( (pRAIDPhysDisk0->PhysDiskStatus.State == MPI_PHYSDISK0_STATUS_OFFLINE_REQUESTED)  ||
                                  (pRAIDPhysDisk0->PhysDiskStatus.State == MPI_PHYSDISK0_STATUS_OTHER_OFFLINE)  )
                            printf ("  Offline\n");
                        else
                            printf ("\n");
                    }
                    else if ( pagetype == MPI_CONFIG_PAGETYPE_FC_PORT )
                    {
                        pFCP0 = (FCPortPage0_t *) mpiBlkPtr->dataInBufPtr;
                        printf("Wide");
                        if ( pFCP0->CurrentSpeed & MPI_FCPORTPAGE0_CURRENT_SPEED_2GBIT )
                             printf ("\tSpeed: 2 Gb/s\n");
                        else if ( pFCP0->CurrentSpeed & MPI_FCPORTPAGE0_CURRENT_SPEED_10GBIT )
                             printf ("\tSpeed: 10 Gb/s\n");
                        else
                             printf ("\tSpeed: 1 Gb/s\n");
                    }
                    else if ( pagetype == MPI_CONFIG_PAGETYPE_RAID_VOLUME )
                    {
                        pVOL0 = (RaidVolumePage0_t *) mpiBlkPtr->dataInBufPtr;
                        if ( pVOL0->VolumeStatus.State == MPI_RAIDVOL0_STATUS_STATE_OPTIMAL )
                            printf ("Optimal");
                        else if ( pVOL0->VolumeStatus.State == MPI_RAIDVOL0_STATUS_STATE_DEGRADED )
                            printf ("Degraded");
                        else
                            printf ("Failed");

                        if ( pVOL0->VolumeStatus.Flags & MPI_RAIDVOL0_STATUS_FLAG_ENABLED )
                            printf ("\t\tEnabled");
                        else
                            printf ("\t\tDisabled");

                        if ( pVOL0->VolumeStatus.Flags & MPI_RAIDVOL0_STATUS_FLAG_QUIESCED )
                            printf ("\t\tQuiesced");

                        if ( pVOL0->VolumeStatus.Flags & MPI_RAIDVOL0_STATUS_FLAG_RESYNC_IN_PROGRESS )
                            printf ("\t\tResyncing");

                        if ( pVOL0->VolumeStatus.Flags & MPI_RAIDVOL0_STATUS_FLAG_VOLUME_INACTIVE )
                            printf ("\t\tInactive");
                        tmp = ( pVOL0->MaxLBA >> 11 );
                        printf ("\t\tSize: %d MB\n", tmp);
                        RAID_NumPhysDisks =  pVOL0->NumPhysDisks;
//  printf ("RAID_NumPhysDisks=%d\n", RAID_NumPhysDisks);
                        for ( ii = 0; ii < RAID_NumPhysDisks; ii++ )
                        {
                                RAID_PhysDiskNum[ii] = (u8)pVOL0->PhysDisk[ii].PhysDiskNum;
//  printf ( "RAID_PhysDiskNum[%d]=%d\n", ii, RAID_PhysDiskNum[ii]);
                        }
                    }
                    else if ( pagetype == MPI_CONFIG_PAGETYPE_IOC )
                    {
                        pIOCPage5 = (IOCPage5_t *) mpiBlkPtr->dataInBufPtr;
                        HOTSPARE_NumHotSpares =  pIOCPage5->NumHotSpares;
//  printf ("HOTSPARE_NumHotSpares=%d\n", HOTSPARE_NumHotSpares);
                        for ( ii = 0; ii < HOTSPARE_NumHotSpares; ii++ )
                        {
                                HOTSPARE_PhysDiskNum[ii] = (u8)pIOCPage5->HotSpare[ii].PhysDiskNum;
                                HOTSPARE_Flags[ii] = (u8)pIOCPage5->HotSpare[ii].Flags;
//  printf ( "HOTSPARE_PhysDiskNum[%d]=%d\n", ii, HOTSPARE_PhysDiskNum[ii]);
                        }
                    }
                    else
                        printf ("IssueCommand: Unknown pagetype=%x\n", pagetype);
                }
        }
        else
            printf ("\n IssueCommand: Ioctl Failed\n");
        freeAllocMem ();
}

/*****************************************************************
 *      GenerateScsiMptBlock - using the inputs, construct
 *      an MPT SCSI I/O block.
 *
 *      dataSize - amount (in Bytes) of data to transfer
 *      dataDir - DATA_DIR_IN or DATA_DIR_OUT or DATA_DIR_NONE
 *      cdb - pointer to 16 byte cdb
 *      cmdSize - SCSI IO Cmd Size
 * 
 *
 *      Remark: Host driver will set the sense buffer physical
 *      address, complete the SGE and set the messge context.
 *      Remark: Host driver will set the task attribute flag on
 *      the ioRequest Control field.
 *****************************************************************/

int IssueScsiCommand(u8 bus, u8 id, u8 lun, u8 type)
{
    uint numBytes = sizeof (SCSIIORequest_t);
    SCSIIORequest_t *ioRequest;
    int ii;
	u8 inqCmd[] = 
	{
	 0x12,
	 0x0,
	 0x0,
	 0x0,
	 INQ_SIZE,
	 0x0,
	 0x0,
	 0x0,
	 0x0,
	 0x0,
	 0x0,
	 0x0,
	 0x0,
	 0x0,
	 0x0,
	 0x0,
	};

    if ( mpiBlkPtr == NULL ) {
    	if ( (mpiBlkPtr = allocIoctlBlk(numBytes)) == NULL)
    	{
        	printf ("IssueScsiCommand: Unable to allocate mpiBlkPtr.");
        	return 0;
    	}
//	printf ("IssueScsiCommand: mpiBlkPtr=%p allocated for numBytes=%d\n", mpiBlkPtr, numBytes);
    }
    /*
     * Populate the MPI Block 
     */
    mpiBlkPtr->dataOutSize = 0;
    mpiBlkPtr->dataInSize = INQ_SIZE+1;

    if ( allocDataFrame( DATA_DIR_IN ) )
    {
        printf ("SCSI IO: Unable to allocate data buffer.");
        freeAllocMem ();
        return 0 ;
    }
    mpiBlkPtr->dataInSize = INQ_SIZE;

    if ( allocSenseBuffer())
    {
        printf ("SCSI IO: Unable to allocate sense buffer.");
        freeAllocMem ();
        return 0;
    }

    mpiBlkPtr->dataSgeOffset = (sizeof(SCSIIORequest_t) - sizeof(SGE_IO_UNION))/ 4;
    ioRequest = (SCSIIORequest_t *)&mpiBlkPtr->MF;

    /* mpiBlk is complete except for the MF contents.
     */
    ioRequest->Function = MPI_FUNCTION_SCSI_IO_REQUEST;
    ioRequest->Bus = bus;
    ioRequest->TargetID = id;
    ioRequest->LUN[1] = lun;

    ioRequest->CDBLength = 6;
    ioRequest->SenseBufferLength = 0x18;
    ioRequest->MsgContext = -1;	/* reset by host driver */

    ioRequest->Control = MPI_SCSIIO_CONTROL_READ;

    for (ii = 0; ii < 16; ii++) {
        ioRequest->CDB[ii] = inqCmd[ii];
    }
    ioRequest->DataLength = INQ_SIZE;

    /* Set by host driver:
     * ioRequest->SenseBufferLowAddr
     * ioRequest->SGE
     * update the Control field.
     */

    /*
     * Issue the MPT Command
     */
    return ( IssueScsiMptCommand( bus, id, lun, type) );
}

/*****************************************************************
 * IssueScsiMptCommand
 *
 * Generic command to issue the MPT command using the special
 * MPTCOMMAND Ioctl Function.
 *****************************************************************/
int IssueScsiMptCommand ( u8 bus, u8 id, u8 lun, u8 type )
{
    int rc;

	/* Set the IOC number prior to issuing this command.
	 */
	mpiBlkPtr->hdr.iocnum = g_iocnum;
	mpiBlkPtr->hdr.port = 0;

    if ( (rc = ioctl(g_fd, (unsigned long) MPTCOMMAND, (char *) mpiBlkPtr)) == 0)
    {
        if (mpiBlkPtr->dataInBufPtr)
        {
	    for ( rc = 0; rc < INQ_SIZE; rc++ ) {
	    	InqData[rc] = *(char *)(&mpiBlkPtr->dataInBufPtr[rc]);
	    }
            rc = 1;
        }
        else
	{
            printf ("id=%d type=%x MPTCOMMAND dataInBufPtr is NULL", id, type);
            rc = 0;
        }
    }
    else
        printf ("id=%d type=%x MPTCOMMAND ioctl failed, rc=%x", id, type, rc);
    freeAllocMem ();
    return rc;
}

