
#include "apps.h"

/* Utility function prototypes
 */
void DoScsiMptBlock(int dataSize, int dataDir, char *cdb, char cmdSize);
void DisplayConfigData(void);
void DisplayIOUnit(u8 number);
void DisplayIOC(u8 number);
void DisplayBIOS(u8 number);
void DisplaySCSIPort(u8 number);
void DisplaySCSIDevice(u8 number);
void DisplayFCPort(u8 number);
void DisplayFCDevice(u8 number);
void DisplayLan(u8 number);
void DisplayRaidVolume(u8 number);
void DisplayManufacturing(u8 number);
void DisplayRaidPhysDisk(u8 number);

/* External Declarations.
 */
extern mpiIoctlBlk_t 	*mpiBlkPtr;
extern int 		g_passthrough;

extern uchar OptionsMenuHandler (CmdOptions *menuPtr);
extern CmdOptions configAction[];
extern CmdOptions configPageType[];
extern char g_NumPages[];
extern CmdOptions raidVolumeAction[];
extern u8     g_buffer[GBL_BUF_SIZE];

#ifdef APP_MEM_TEST
extern int g_target;
#endif

char *who_init[] = {
	"No One",
	"System BIOS",
	"ROM BIOS",
	"PCI Peer",
	"Host Driver",
	"Manufacturer"
};

char *product[] = {
	"Unknown",
	"Initiator",
	"Target-Initiator",
	"Target",
	"Mirroring",
	"Striping",
	"CTX"
};

char *phys_interface[] = {
	"Unknown",
	"HVD",
	"SE",
	"LVD"
};


char *raid_types [] = {
	"Integrated Striping",
	"Integrated Mirroring Enhanced",
	"Integrated Mirroring",
};


/*****************************************************************
 *
 *
 *      MPT Ioctls (Non-SCSI I/O)
 *
 *
 *****************************************************************/
/*****************************************************************
 * InitIoctl
 *
 * Host driver has already initalized this IOC. We don't want a user
 * to issue a command that would mess up the IOC firmware.
 *****************************************************************/
void InitIoctl (void)
{
	printf("IOC Initialization IOCTL not supported.");
	return;
}

/*****************************************************************
 * FactsIoctl
 *
 * Get the IOC Facts and report to the user. Memory freed in
 * function call IssueMptCommand.
 *****************************************************************/
void FactsIoctl (void)
{
	IOCFacts_t *FactsRequest;
	uint numBytes = sizeof (IOCFacts_t) + sizeof (SGESimple64_t);
	int tmp;

	if ((mpiBlkPtr = allocIoctlBlk(numBytes)) == NULL)
		return;

	/* Populate the Facts MPI Message
	 */
	FactsRequest = (IOCFacts_t *) mpiBlkPtr->MF;
	mpiBlkPtr->dataInSize = 0;
	mpiBlkPtr->dataOutSize = 0;
	mpiBlkPtr->dataSgeOffset = sizeof (IOCFacts_t)/ 4;

	FactsRequest->Reserved[0]  = 0;
	FactsRequest->Reserved[1]  = 0;
	FactsRequest->ChainOffset  = 0;
	FactsRequest->Function     = MPI_FUNCTION_IOC_FACTS;
	FactsRequest->Reserved1[0] = 0;
	FactsRequest->Reserved1[1] = 0;
	FactsRequest->Reserved1[2] = 0;
	FactsRequest->MsgFlags     = 0;
	FactsRequest->MsgContext   = -1;

	/* Issue the MPT Command
	 */
	if (IssueMptCommand(MPT_FLAGS_KEEP_MEM) == 0) {
		IOCFactsReply_t *rr = (IOCFactsReply_t *)mpiBlkPtr->replyFrameBufPtr;
		printf("IOC Facts Data:\n");
		printf("==========================================================================\n");
		printf("Msg Version    = 0x%04x       Msg Length = 0x%02x     Function    = 0x%02x\n",
				le16_to_cpu(rr->MsgVersion), rr->MsgLength, rr->Function);
		printf("Header Version = 0x%04x       IOC Number = 0x%02x     MsgFlags    = 0x%02x\n",
				le16_to_cpu(rr->HeaderVersion), rr->IOCNumber, rr->MsgFlags);
		printf("IOC Exceptions = 0x%04x       IOC Status = 0x%04x \n",
				le16_to_cpu(rr->IOCExceptions), le16_to_cpu(rr->IOCStatus));
		printf("IOC LogInfo    = 0x%08x \n", le32_to_cpu(rr->IOCLogInfo));
		printf("Max Chains     = 0x%02x         WhoInit    = 0x%02x (%s)\n",
				rr->MaxChainDepth, rr->WhoInit, who_init[rr->WhoInit]);
		printf("Block Size     = 0x%02x         Flags      = 0x%02x\n",
				rr->BlockSize, rr->Flags);
		printf("Reply Q Depth  = 0x%04x       Request Sz = 0x%04x(u32s)\n",
				le16_to_cpu(rr->ReplyQueueDepth), le16_to_cpu(rr->RequestFrameSize));
		printf("Old FW Version = 0x%04x       ProductID  = 0x%04x (",
				le16_to_cpu(rr->Reserved_0101_FWVersion), le16_to_cpu(rr->ProductID));

		tmp = le16_to_cpu(rr->ProductID) & MPI_FW_HEADER_PID_TYPE_MASK;
		if (tmp == MPI_FW_HEADER_PID_TYPE_SCSI)
			printf("SCSI ");
		else if (tmp == MPI_FW_HEADER_PID_TYPE_FC)
			printf("FC ");
		else
			printf("Unknown ");
		tmp = (le16_to_cpu(rr->ProductID) & MPI_FW_HEADER_PID_PROD_MASK) >> 8;
		if (tmp <= 6)
			printf("%s)\n", product[tmp]);
		else
			printf("Error in PROD!\n");

		printf("Host MFA High  = 0x%08x \n", le32_to_cpu(rr->CurrentHostMfaHighAddr));
		printf("Global Credits = 0x%04x       Num Ports  = 0x%02x     Event State = 0x%02x (%s)\n",
				le16_to_cpu(rr->GlobalCredits), rr->NumberOfPorts,
				rr->EventState, rr->EventState ? "On" : "Off");
		printf("Sense High     = 0x%08x \n", le32_to_cpu(rr->CurrentSenseBufferHighAddr));
		printf("Reply Frame Sz = 0x%04x(u8s)  Max Devices = 0x%02x    Max Bus     = 0x%02x \n",
				le16_to_cpu(rr->CurReplyFrameSize), rr->MaxDevices, rr->MaxBuses);
		printf("FW Image Size  = 0x%08x(u8s) \n", le32_to_cpu(rr->FWImageSize));
		printf("FW Version     = 0x%08x \n", le32_to_cpu(rr->FWVersion.Word));
	}
	freeAllocMem();
	return;
}

/*****************************************************************
 * ConfigIoctl
 *
 * Get the IOC Facts and report to the user. Memory freed in
 * function call IssueMptCommand.
 *****************************************************************/
void ConfigIoctl (void)
{
	Config_t *ConfigRequest;
	uint numBytes = (sizeof(Config_t) - sizeof(SGE_IO_UNION))
						+ sizeof (SGESimple64_t);
	uint pageaddress = 0;
	int read_header_only = 0;
	uchar action, pagenumber, pagetype, pageversion, pagelength;
	uchar bus = 0;
	uchar target;

	if ((mpiBlkPtr = allocIoctlBlk(numBytes)) == NULL)
		return;

	ConfigRequest = (Config_t *) mpiBlkPtr->MF;
	mpiBlkPtr->dataSgeOffset = (sizeof (Config_t) - sizeof(SGE_IO_UNION))/4;

	/* Set the action and the page type
	 */
	action =  OptionsMenuHandler (configAction);
	pagetype =  OptionsMenuHandler (configPageType);

	/* Set the page number.
	 */
	if (g_NumPages[pagetype] == 0) {
		freeAllocMem();
		return;
	} else if (g_NumPages[pagetype] > 1)
		pagenumber = AskValue(0, g_NumPages[pagetype] - 1 , "Enter the page number");
	else
		pagenumber = 0;

	/* For selection page type, set the page address.
	 */
	switch (pagetype) {
	case MPI_CONFIG_PAGETYPE_SCSI_DEVICE:
		target = AskValue(0, MAX_SCSI_ID , "Enter the target ID number");
		pageaddress = (bus << 8) | target;

		/* Remark:  SCSI_DEVICE page has some funky pageaddress
		 * formats when handling hidden devices in IM.
		 */
		break;

	case MPI_CONFIG_PAGETYPE_FC_DEVICE:
		target = AskValue(0, MAX_FC_ID, "Enter the target ID number");
		pageaddress = MPI_FC_DEVICE_PAGE0_PGAD_FORM_BUS_TID | (bus << 8) | target;

		/* Remark:  FC_DEVICE page has DID vs TID formats
		 * for data.
		 */
		break;

	case MPI_CONFIG_PAGETYPE_RAID_VOLUME:
		target = AskValue(0, MAX_SCSI_ID , "Enter the target ID number");
		pageaddress = (bus << 8) | target;
		break;

	case MPI_CONFIG_PAGETYPE_FC_PORT:
		pageaddress = 0;

		/* Remark:  FC_PORT page has funky page addressing for
		 * reading pages whose length > 1kb.
		 * Not implemented yet.
		 */
		break;

	case MPI_CONFIG_PAGETYPE_RAID_PHYSDISK:
		pageaddress = AskValue(0, MAX_SCSI_ID , "Enter the Phys Disk number");
		break;

		/* pageaddress reserved cases:
			MPI_CONFIG_PAGETYPE_IO_UNIT:
			MPI_CONFIG_PAGETYPE_IOC:
			MPI_CONFIG_PAGETYPE_BIOS:
			MPI_CONFIG_PAGETYPE_MANUFACTURING:
			address is 0 since Port is 0 cases:
			MPI_CONFIG_PAGETYPE_SCSI_PORT:
			MPI_CONFIG_PAGETYPE_LAN:
		*/
	default:
		pageaddress = 0;
	}

	/* For the selected action, page, and page number, set the
	 * page version and page length.
	 */
	mpiBlkPtr->dataInSize = mpiBlkPtr->dataOutSize = 0;

	if (action == (uchar) MPI_CONFIG_ACTION_PAGE_HEADER)
		read_header_only = 1;

	/* Populate the Config Request & read the page header
	 */
	ConfigRequest->Action       = (uchar) MPI_CONFIG_ACTION_PAGE_HEADER;
	ConfigRequest->Function     = MPI_FUNCTION_CONFIG;
	ConfigRequest->MsgContext   = -1;
	ConfigRequest->Header.PageNumber = pagenumber;
	ConfigRequest->Header.PageType = pagetype;
	ConfigRequest->PageAddress = cpu_to_le32(pageaddress);

	if (IssueMptCommand(MPT_FLAGS_KEEP_MEM) == 0) {
		ConfigReply_t *rr = (ConfigReply_t *)mpiBlkPtr->replyFrameBufPtr;

		pageversion = rr->Header.PageVersion;
		pagelength = rr->Header.PageLength;
		if (read_header_only) {
			printf("Version = 0x%x, Length = 0x%x, Number = 0x%x, Type = 0x%x\n",
					pageversion, pagelength, pagenumber, pagetype);
			freeAllocMem();
			return;
		}
	} else {
		freeAllocMem();
		return;
	}

	if ((action == (uchar) MPI_CONFIG_ACTION_PAGE_HEADER) ||
		(action == (uchar) MPI_CONFIG_ACTION_PAGE_DEFAULT )) {
		/* PAGE_DEFAULT is a write from memory
		 * No data transfer occurs.
		 */
		;
	} else if ((action == (uchar) MPI_CONFIG_ACTION_PAGE_READ_CURRENT)
			|| (action == (uchar) MPI_CONFIG_ACTION_PAGE_READ_DEFAULT)
			|| (action == (uchar) MPI_CONFIG_ACTION_PAGE_READ_NVRAM)) {

		mpiBlkPtr->dataInSize = pagelength * 4;
		if (allocDataFrame(DATA_DIR_IN)) {
			printf ("Config: Unable to allocate data buffer.");
			freeAllocMem();
			return;
		}
	} else if ((action == (uchar) MPI_CONFIG_ACTION_PAGE_WRITE_CURRENT)
			|| (action == (uchar) MPI_CONFIG_ACTION_PAGE_WRITE_NVRAM)) {

		mpiBlkPtr->dataOutSize = pagelength * 4;
		if (allocDataFrame(DATA_DIR_OUT)) {
			printf("Config: Unable to allocate data buffer.");
			freeAllocMem();
			return;
		}
	} else {
		printf("Unsupported action (%d)\n", action);
		freeAllocMem();
		return;
	}

	/* Update the Config Request
	 */
	ConfigRequest->Action       = action;
	ConfigRequest->Header.PageVersion = pageversion;
	ConfigRequest->Header.PageLength = pagelength;

	if (IssueMptCommand(MPT_FLAGS_KEEP_MEM) == 0) {
		if (mpiBlkPtr->dataInSize > 0)
			DisplayConfigData();
	}
	freeAllocMem();
	return;
}

/*****************************************************************
 * PortFactsIoctl
 *
 * Get the Port Facts and report to the user. Memory freed in
 * function call IssueMptCommand.
 *****************************************************************/
void PortFactsIoctl (void)
{
	PortFacts_t *FactsRequest;
	uint numBytes = sizeof (PortFacts_t) + sizeof (SGESimple64_t);

	if ((mpiBlkPtr = allocIoctlBlk(numBytes)) == NULL)
		return;

	FactsRequest = (PortFacts_t *) mpiBlkPtr->MF;

	mpiBlkPtr->dataSgeOffset = sizeof (PortFacts_t)/ 4;

	FactsRequest->Function     = MPI_FUNCTION_PORT_FACTS;

	/* PortNumber refers to the firmware port number,
	 * not the UnixWare pci slot number.
	 * The driver requires each pci function to be mapped to a separate
	 * port, the maximum port number is 0.
	 */
	FactsRequest->MsgContext   = -1;

	if (IssueMptCommand(MPT_FLAGS_KEEP_MEM) == 0) {
		PortFactsReply_t *rr = (PortFactsReply_t *)mpiBlkPtr->replyFrameBufPtr;
		printf("Port Facts Data:\n");
		printf("================================================================\n");
		printf("Msg Length         = 0x%02x         Function            = 0x%02x\n",
								rr->MsgLength, rr->Function);
		printf("Port Number        = 0x%02x         Msg Flags           = 0x%02x\n",
								rr->PortNumber, rr->MsgFlags);
		printf("IOC Status         = 0x%04x \n", le16_to_cpu(rr->IOCStatus));
		printf("IOC LogInfo        = 0x%08x \n", le32_to_cpu(rr->IOCLogInfo));
		printf("Max Devices        = 0x%04x       Port SCSI ID        = 0x%04x\n",
				le16_to_cpu(rr->IOCLogInfo), le16_to_cpu(rr->PortSCSIID));
		printf("Protocol Flags     = 0x%04x       MaxPostedCmdBuffers = 0x%04x\n",
				le16_to_cpu(rr->ProtocolFlags), le16_to_cpu(rr->MaxPostedCmdBuffers));
		printf("Max Persistent IDs = 0x%04x       Max LAN Buckets     = 0x%04x\n",
				le16_to_cpu(rr->MaxPersistentIDs), le16_to_cpu(rr->MaxLanBuckets));
	}
	freeAllocMem();
	return;
}

/*****************************************************************
 * PortEnableIoctl
 *
 * Host driver has already enabled the port.  We don't want a user
 * to be able to mess up the running firmware.
 *****************************************************************/
void PortEnableIoctl (void)
{
	printf("Port Enable IOCTL not supported.");
	return;
}

/*****************************************************************
 * EventNotificationIoctl
 *
 * Enable or disable event notification in the driver.
 *****************************************************************/
void EventNotificationIoctl (void)
{
	uint numBytes = sizeof (EventNotification_t) + sizeof (SGESimple64_t);
	EventNotification_t *EventRequest;
	uchar onOff;

	if ((mpiBlkPtr = allocIoctlBlk(numBytes)) == NULL)
		return;

	EventRequest = (EventNotification_t *) mpiBlkPtr->MF;

	mpiBlkPtr->dataInSize = 0;
	mpiBlkPtr->dataOutSize = 0;
	mpiBlkPtr->dataSgeOffset = sizeof (EventNotification_t)/ 4;

	onOff = AskBits(1, "new event state (1-on/0-off)" );

	EventRequest->Switch = onOff;
	EventRequest->Function     = MPI_FUNCTION_EVENT_NOTIFICATION;
	EventRequest->MsgContext   = -1;

	IssueMptCommand(MPT_FLAGS_STANDARD);

	return;
}


/*****************************************************************
 * EventAckIoctl
 *
 * Issued only in response to an Event - this message must be
 * requested by the Firmware.
 *****************************************************************/
void EventAckIoctl (void)
{
    printf("EventAckIoctl IOCTL not supported.");
    return;
}

/*****************************************************************
 * Firmware Upload Ioctl
 *****************************************************************/
void FirmWareUploadIoctl (void)
{
	printf("\nPlease use the application mptfdu that is found\n"
		"in the utils directory.  Thank You!\n");
	return;
}


/*****************************************************************
 * Firmware Download Ioctl
 *****************************************************************/
void FirmWareDownloadIoctl (void)
{
	printf("\nPlease use the application mptfdu that is found\n"
		"in the utils directory.  Thank You!\n");
	return;
}

/*****************************************************************
 *  RAID Volume Ioctl
 *****************************************************************/
void RaidVolumeIoctl (void)
{
	MpiRaidActionRequest_t *pRequest;
	RaidPhysDiskPage0_t	*pRaidPDP0;
	RaidVolumePage0_t	*pRaidVP0;
	uint 			numBytes;
	uint			actionData;
	uchar			action;
	uchar			physDiskNum;
	uchar 			bus = 0;
	uchar target = AskValue(0, MAX_FC_ID, "Enter the Volume ID number");

	numBytes = (sizeof(MpiRaidActionRequest_t) - sizeof(SGE_IO_UNION))
						+ sizeof (SGESimple64_t);

	if ((mpiBlkPtr = allocIoctlBlk(numBytes)) == NULL)
		return;

	pRequest = (MpiRaidActionRequest_t *) mpiBlkPtr->MF;
	mpiBlkPtr->dataSgeOffset = (sizeof (MpiRaidActionRequest_t) - sizeof(SGE_IO_UNION))/4;

	/* Initialize data in/data out sizes: Change below if need to */
	mpiBlkPtr->dataInSize = mpiBlkPtr->dataOutSize = 0;

	/* Set the action, specify the physical disk number and
	 * select the action data.
	 */
	action =  OptionsMenuHandler(raidVolumeAction);
	physDiskNum = AskValue(0, 0xFF, "Enter the Physical Disk Number ");

	if (action == MPI_RAID_ACTION_CHANGE_VOLUME_SETTINGS)
		actionData = AskValue(0, 0xFFFFFFFF, "Enter the Volume Settings ");
	else if (action == MPI_RAID_ACTION_CHANGE_PHYSDISK_SETTINGS)
		actionData = AskValue(0, 0xFFFFFFFF, "Enter the Physical Disk Settings ");
	else if (action == MPI_RAID_ACTION_REPLACE_PHYSDISK)
		actionData = AskValue(0, 0xFFFFFFFF, "Enter the PhysDiskNum ");
	else if (action == MPI_RAID_ACTION_DELETE_VOLUME)  {
		uchar deleteConfig = AskValue(0, 1, "Delete associated Config Pages? (1=yes) ");
		if (deleteConfig)
			actionData = MPI_RAID_ACTION_ADATA_DEL_PHYS_DISKS;
		else
			actionData = MPI_RAID_ACTION_ADATA_KEEP_PHYS_DISKS;
	} else if (action == MPI_RAID_ACTION_CREATE_PHYSDISK) {
		actionData = 0;
		mpiBlkPtr->dataOutSize = sizeof(RaidPhysDiskPage0_t);

		if (allocDataFrame(DATA_DIR_OUT)) {
			printf("Config: Unable to allocate data buffer.");
			freeAllocMem();
			return;
		}
		pRaidPDP0 = (pRaidPhysDiskPage0_t) mpiBlkPtr->dataOutBufPtr;
		pRaidPDP0->PhysDiskID = AskValue(0, 0xFF, "Enter Disk ID for Creation ");
		pRaidPDP0->PhysDiskBus = bus;
		pRaidPDP0->PhysDiskSettings.HotSparePool = AskValue (0, 0xFF, "Enter the Hot Spare Pool (0=none) ");
	} else if (action == MPI_RAID_ACTION_CREATE_VOLUME) {
		actionData = 0;
		/* add to sizeof(RVP0) because default RAIDVolumePage0_t array has
		 *  MPI_RAID_VOL_PAGE_0_PHYSDISK_MAX defined to 1, and we want to specify two disks
		 *  thus add one more sizeof RAID_VOL0_PHYS_DISK
		 */
		mpiBlkPtr->dataOutSize = sizeof(RaidVolumePage0_t) + sizeof(RAID_VOL0_PHYS_DISK);

		if (allocDataFrame(DATA_DIR_OUT)) {
			printf("Config: Unable to allocate data buffer.");
			freeAllocMem();
			return;
		}
		pRaidVP0 = (pRaidVolumePage0_t) mpiBlkPtr->dataOutBufPtr;
		pRaidVP0->VolumeID = AskValue(0, 0xF, "Enter the Target ID you want the IM Device to have ");
		pRaidVP0->VolumeBus = bus;
		pRaidVP0->VolumeType = MPI_RAID_VOL_TYPE_IM;
		pRaidVP0->MaxLBA = AskValue(0, 0xFFFF, "Enter size of Virtual Disk (MB) ");
		pRaidVP0->MaxLBA *= 2048; /* So its Megabytes */
		pRaidVP0->NumPhysDisks = 2;

		pRaidVP0->PhysDisk[0].PhysDiskMap = MPI_RAIDVOL0_PHYSDISK_PRIMARY;
		pRaidVP0->PhysDisk[0].PhysDiskNum = AskValue(0, 0xFF, "Enter the PhysDiskNum of Primary Disk ");
		pRaidVP0->PhysDisk[1].PhysDiskMap = MPI_RAIDVOL0_PHYSDISK_SECONDARY;
		pRaidVP0->PhysDisk[1].PhysDiskNum = AskValue(0, 0xFF, "Enter the PhysDiskNum of Secondary Disk ");
	} else
		actionData = 0;

	/* Populate the RAID Volume Request
	 */
	pRequest->Action       = action;
	pRequest->Function     = MPI_FUNCTION_RAID_ACTION;
	pRequest->VolumeID     = target;
	pRequest->VolumeBus    = bus;
	pRequest->PhysDiskNum  = physDiskNum;
	pRequest->MsgContext   = -1;
	pRequest->ActionDataWord  = cpu_to_le32(actionData);

	IssueMptCommand(MPT_FLAGS_STANDARD);

	return;
}

/*****************************************************************
 * BusResetIoctl
 *
 * Issue a Task Mgmt command (Bus Reset).
 *****************************************************************/
void BusResetIoctl(void)
{
	SCSITaskMgmt_t *tmRequest;
	uint numBytes = sizeof (SCSITaskMgmt_t) + sizeof (SGESimple64_t);

	if ((mpiBlkPtr = allocIoctlBlk(numBytes)) == NULL)
		return;

	/* Populate the Facts MPI Message
	 */
	tmRequest = (SCSITaskMgmt_t *) mpiBlkPtr->MF;
	mpiBlkPtr->dataSgeOffset = sizeof (SCSITaskMgmt_t)/ 4;

	tmRequest->TargetID      = AskValue (0, MAX_FC_ID , "Enter the target ID number");
	tmRequest->Function      = MPI_FUNCTION_SCSI_TASK_MGMT;
	tmRequest->TaskType      = MPI_SCSITASKMGMT_TASKTYPE_RESET_BUS;
	tmRequest->MsgContext    = -1;


	/* Issue the MPT Command
	 */
	IssueMptCommand(MPT_FLAGS_FREE_MEM);

	return;
}

/*****************************************************************
 * SCSI Enclosure Processor
 *
 * 
 * Function calls IssueMptCommand.
 *****************************************************************/
void EnclosureProcessor (void)
{
	SEPRequest_t *Request;
	uint numBytes = sizeof (SEPRequest_t) + sizeof (SGESimple64_t);
	int dataDir, flag = 0;

	if ((mpiBlkPtr = allocIoctlBlk(numBytes)) == NULL)
		return;
	Request = (SEPRequest_t *) mpiBlkPtr->MF;

	dataDir = DATA_DIR_IN; 
	if (AskValue(0, 1 , "Enter direction (read = 1; write = 0): ") == 0)
		dataDir = DATA_DIR_OUT;

	if (dataDir == DATA_DIR_IN) {
		mpiBlkPtr->dataInSize = 0;
	} else {
	 	flag = AskValue(0, 1 , "Use data in global buffer (1=yes)? ");
		mpiBlkPtr->dataOutSize = 0;
		dataDir = DATA_DIR_OUT; 
	}
	mpiBlkPtr->dataSgeOffset = sizeof (SEPRequest_t)/ 4;

	if (allocDataFrame(dataDir)) {
		printf ("SCSI IO: Unable to allocate data buffer.");
		freeAllocMem();
		return;
	}

	if (flag) {
		memcpy(mpiBlkPtr->dataOutBufPtr, g_buffer, 16);
	}

	Request->TargetID     = AskValue(0, MAX_FC_ID , "Enter the target ID number");
	Request->Function     = MPI_FUNCTION_SCSI_ENCLOSURE_PROCESSOR;
	Request->Action       = (dataDir == DATA_DIR_IN) ?  MPI_SEP_REQ_ACTION_READ_STATUS:
					MPI_SEP_REQ_ACTION_WRITE_STATUS;
	Request->MsgContext   = -1;

	if (IssueMptCommand(MPT_FLAGS_KEEP_MEM) == 0) {
		SEPReply_t *rr = (SEPReply_t *)mpiBlkPtr->replyFrameBufPtr;
		u32 ss = le32_to_cpu(rr->SlotStatus);

		printf("Enclosure Processor Data:\n");
		printf("================================================================\n");
		printf("TargetID           = 0x%02x         Bus                 = 0x%02x\n",
								rr->TargetID, rr->Bus);
		printf("Msg Length         = 0x%02x         Function            = 0x%02x\n",
								rr->MsgLength, rr->Function);
		printf("Action             = 0x%02x(%s)      Msg Flags           = 0x%02x\n",
				rr->Action, (rr->Action == 1) ? "r": "w", rr->MsgFlags);
		printf("IOC Status         = 0x%04x \n", le16_to_cpu(rr->IOCStatus));
		printf("IOC LogInfo        = 0x%08x \n", le32_to_cpu(rr->IOCLogInfo));
		printf("SlotStatus         = 0x%08x", ss);

		{
			char error[21 * 24];
			char *perror = error;
			int  did = 0;
			int cnt = 0;
			
			if (ss & MPI_SEP_REPLY_SLOTSTATUS_NO_ERROR) {
				if (did)
					cnt += sprintf(perror + cnt, "; ");
				cnt += sprintf(perror + cnt, "no error");
				did++;
			}
			if (ss & MPI_SEP_REPLY_SLOTSTATUS_DEV_FAULTY) {
				if (did)
					cnt += sprintf(perror + cnt, "; ");
				cnt += sprintf(perror, "faulty");
				did++;
			}
			if (ss & MPI_SEP_REPLY_SLOTSTATUS_DEV_REBUILDING) {
				if (did)
					cnt += sprintf(perror + cnt, "; ");
				cnt += sprintf(perror + cnt, "rebuilding");
				did++;
			}
			if (ss & MPI_SEP_REPLY_SLOTSTATUS_IN_FAILED_ARRAY) {
				if (did)
					cnt += sprintf(perror + cnt, "; ");
				cnt += sprintf(perror + cnt, "in failed array");
				did++;
			}
			if (ss & MPI_SEP_REPLY_SLOTSTATUS_IN_CRITICAL_ARRAY) {
				if (did)
					cnt += sprintf(perror + cnt, "; ");
				cnt += sprintf(perror + cnt, "in critical array");
				did++;
			}
			if (ss & MPI_SEP_REPLY_SLOTSTATUS_PARITY_CHECK) {
				if (did)
					cnt += sprintf(perror + cnt, "; ");
				cnt += sprintf(perror + cnt, "parity check");
				did++;
			}
			if (ss & MPI_SEP_REPLY_SLOTSTATUS_PREDICTED_FAULT) {
				if (did)
					cnt += sprintf(perror + cnt, "; ");
				cnt += sprintf(perror + cnt, "predicted fault");
				did++;
			}
			if (ss & MPI_SEP_REPLY_SLOTSTATUS_UNCONFIGURED) {
				if (did)
					cnt += sprintf(perror + cnt, "; ");
				cnt += sprintf(perror + cnt, "unconfigured");
				did++;
			}
			if (ss & MPI_SEP_REPLY_SLOTSTATUS_HOT_SPARE) {
				if (did)
					cnt += sprintf(perror + cnt, "; ");
				cnt += sprintf(perror + cnt, "hot spare");
				did++;
			}
			if (ss & MPI_SEP_REPLY_SLOTSTATUS_REBUILD_STOPPED) {
				if (did)
					cnt += sprintf(perror + cnt, "; ");
				cnt += sprintf(perror + cnt, "rebuild stopped");
				did++;
			}
			if (ss & MPI_SEP_REPLY_SLOTSTATUS_REPORT) {
				if (did)
					cnt += sprintf(perror + cnt, "; ");
				cnt += sprintf(perror + cnt, "report");
				did++;
			}
			if (ss & MPI_SEP_REPLY_SLOTSTATUS_IDENTIFY_REQUEST) {
				if (did)
					cnt += sprintf(perror + cnt, "; ");
				cnt += sprintf(perror + cnt, "identify request");
				did++;
			}
			if (ss & MPI_SEP_REPLY_SLOTSTATUS_REMOVE_READY) {
				if (did)
					cnt += sprintf(perror + cnt, "; ");
				cnt += sprintf(perror + cnt, "remove ready");
				did++;
			}
			if (ss & MPI_SEP_REPLY_SLOTSTATUS_INSERT_READY) {
				if (did)
					cnt += sprintf(perror + cnt, "; ");
				cnt += sprintf(perror + cnt, "insert ready");
				did++;
			}
			if (ss & MPI_SEP_REPLY_SLOTSTATUS_DO_NOT_REMOVE) {
				if (did)
					cnt += sprintf(perror + cnt, "; ");
				cnt += sprintf(perror + cnt, "do not remove");
				did++;
			}
			if (ss & MPI_SEP_REPLY_SLOTSTATUS_B_BYPASS_ENABLED) {
				if (did)
					cnt += sprintf(perror + cnt, "; ");
				cnt += sprintf(perror + cnt, "B byass enabled");
				did++;
			}
			if (ss & MPI_SEP_REPLY_SLOTSTATUS_A_BYPASS_ENABLED) {
				if (did)
					cnt += sprintf(perror + cnt, "; ");
				cnt += sprintf(perror + cnt, "A byass enabled");
				did++;
			}
			if (ss & MPI_SEP_REPLY_SLOTSTATUS_B_ENABLE_BYPASS) {
				if (did)
					cnt += sprintf(perror + cnt, "; ");
				cnt += sprintf(perror + cnt, "B enable byass");
				did++;
			}
			if (ss & MPI_SEP_REPLY_SLOTSTATUS_A_ENABLE_BYPASS) {
				if (did)
					cnt += sprintf(perror + cnt, "; ");
				cnt += sprintf(perror + cnt, "A enable byass");
				did++;
			}
			if (ss & MPI_SEP_REPLY_SLOTSTATUS_DEV_OFF) {
				if (did)
					cnt += sprintf(perror + cnt, "; ");
				cnt += sprintf(perror + cnt, "dev off");
				did++;
			}
			if (ss & MPI_SEP_REPLY_SLOTSTATUS_FAULT_SENSED) {
				if (did)
					cnt += sprintf(perror + cnt, "; ");
				cnt += sprintf(perror + cnt, "fault sensed");
				did++;
			}
			if (ss & MPI_SEP_REPLY_SLOTSTATUS_SWAPPED) {
				if (did)
					cnt += sprintf(perror + cnt, "; ");
				cnt += sprintf(perror + cnt, "swapped");
			}

			printf(" (%s)", perror);
		}
	}
	freeAllocMem();
	return;
}

/*****************************************************************
 *
 *
 *      LINUX SCSI I/O in MPT Format
 *      Limited to small I/O's .... single SGE
 *
 *
 *****************************************************************/
/*****************************************************************
 * Inquiry Cmd
 *
 * Issue an Inquiry command in MPT format.
 *****************************************************************/
void InquiryCmd(void)
{
    int dataLength = INQ_SIZE;
    char cdb[] = {0x12, 0, 0, 0, INQ_SIZE, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
    char dir = DATA_DIR_IN;

    DoScsiMptBlock(dataLength, dir, cdb, CMD_LENGTH_6);
}

void TurCmd(void)
{
    int dataLength = 0;
    char cdb[] = {0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
    char dir = DATA_DIR_NONE;

    DoScsiMptBlock(dataLength, dir, cdb, CMD_LENGTH_6);
}

void ReadCapacityCmd(void)
{
    int dataLength = 8;
    char cdb[] = {0x25, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
    char dir = DATA_DIR_IN;

    DoScsiMptBlock(dataLength, dir, cdb, CMD_LENGTH_10);
}

void ModeSenseCmd(void)
{
    int dataLength = 0xC;
    char cdb[] = {0x1A, 0, 0, 0, 0x0C, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
    char dir = DATA_DIR_IN;

    DoScsiMptBlock(dataLength, dir, cdb, CMD_LENGTH_6);
}

void ModeSelectCmd(void)
{
    int dataLength = 0x0C;
    char cdb[] = {0x15, 0, 0, 0, 0x0C, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
    char dir = DATA_DIR_OUT;

    DoScsiMptBlock(dataLength, dir, cdb, CMD_LENGTH_6);
}

void Read10Cmd(void)
{
    int dataLength = 8*512;
    char cdb[] = {0x28, 0, 0, 0, 0, 0, 0, 0, 0x08, 0, 0, 0, 0, 0, 0, 0};
    char dir = DATA_DIR_IN;

    DoScsiMptBlock(dataLength, dir, cdb, CMD_LENGTH_10);
}


/*****************************************************************
 *      GenerateScsiMptBlock - using the inputs, construct
 *      an MPT SCSI I/O block.
 *
 *      dataSize - amount (in Bytes) of data to transfer
 *      dataDir - DATA_DIR_IN or DATA_DIR_OUT or DATA_DIR_NONE
 *      cdb - pointer to 16 byte cdb
 *      cmdSize - SCSI IO Cmd Size
 *
 *
 *      Remark: Host driver will set the sense buffer physical
 *      address, complete the SGE and set the messge context.
 *      Remark: Host driver will set the task attribute flag on
 *      the ioRequest Control field.
 *****************************************************************/
void DoScsiMptBlock(int dataSize, int dataDir, char *cdb, char cmdSize)
{
	uint numBytes = sizeof (SCSIIORequest_t);
	SCSIIORequest_t *ioRequest;
	int ii, flag = 0;
	uchar target, bus, lun;
	uchar physDiskNum;

	if ((mpiBlkPtr = allocIoctlBlk(numBytes)) == NULL)
		return;
#ifndef APP_MEM_TEST
	printf ("SCSI I/O: Cmd Op code %x : mpiBlkPtr %p, numBytes = %d\n", *cdb, mpiBlkPtr, numBytes);
#endif
	/* Populate the MPI Block
	 */
	mpiBlkPtr->dataInSize = 0;
	mpiBlkPtr->dataOutSize = 0;
	if (dataDir == DATA_DIR_OUT) {
		mpiBlkPtr->dataOutSize = dataSize;
	 	flag = AskValue(0, 1 , "Use data in global buffer (1=yes)? ");
	} else if (dataDir == DATA_DIR_IN)
		mpiBlkPtr->dataInSize = dataSize;

	if (allocDataFrame(dataDir)) {
		printf ("SCSI IO: Unable to allocate data buffer.");
		freeAllocMem();
		return;
	}

	if (flag) {
		memcpy(mpiBlkPtr->dataOutBufPtr, g_buffer,
			(dataSize < GBL_BUF_SIZE ? dataSize: GBL_BUF_SIZE));
	}

	if (allocSenseBuffer()) {
		printf("SCSI IO: Unable to allocate sense buffer.");
		freeAllocMem();
		return;
	}

	mpiBlkPtr->dataSgeOffset = (sizeof(SCSIIORequest_t) - sizeof(SGE_IO_UNION))/ 4;
	ioRequest = (SCSIIORequest_t *) mpiBlkPtr->MF;

#ifndef APP_MEM_TEST
	/* Get from user for Linux.
	 */
	target = AskValue(0, MAX_FC_ID , "Enter the target ID number");
#else
	target = (u8) g_target;
#endif
	bus = 0;
	lun = 0;

	/* mpiBlk is complete except for the MF contents.
	 */
	if (g_passthrough) {
		physDiskNum = AskValue(0, 0xFF, "Enter the Physical Disk Number ");
		ioRequest->TargetID = physDiskNum;
		ioRequest->Function = MPI_FUNCTION_RAID_SCSI_IO_PASSTHROUGH;
	} else {
		ioRequest->TargetID = target;
		ioRequest->Bus = bus;
		ioRequest->Function = MPI_FUNCTION_SCSI_IO_REQUEST;
	}

	ioRequest->CDBLength = cmdSize;
	ioRequest->SenseBufferLength = mpiBlkPtr->maxSenseBytes;
	ioRequest->MsgContext = -1;

	ioRequest->LUN[1] = lun;

	for (ii = 0; ii < 16; ii++)
		ioRequest->CDB[ii] = cdb[ii];

	/* Set by host driver:
	 * ioRequest->SenseBufferLowAddr
	 * ioRequest->SGE
	 * Control field.
	 * data length field
	 */

	/* Issue the MPT Command
	 */
	IssueMptCommand(MPT_FLAGS_STANDARD);

	return;
}
/*=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=*/

/* Routine to display the data in English format, or if
 * translation not yet implemented, in byte format.
 * Extract from request, page number and type.
 */
void DisplayConfigData(void)
{
	ConfigReply_t *rr = (ConfigReply_t *)mpiBlkPtr->replyFrameBufPtr;
	u8 type = rr->Header.PageType;
	u8 num = rr->Header.PageNumber;
	u8 ver = rr->Header.PageVersion;
	u8 len = rr->Header.PageLength;

	printf("\n\n\n");
	printf("Version = 0x%02x Length = 0x%02x Number = 0x%02x Type = 0x%02x\n",
			ver, len, num, type);
	printf("\n");

	if (len == 0)
		return;

	if (num >= g_NumPages[type])
		return;

	switch(type)
	{
	case MPI_CONFIG_PAGETYPE_IO_UNIT:
		DisplayIOUnit(num);
		break;

	case MPI_CONFIG_PAGETYPE_IOC:
		DisplayIOC(num);
		break;

	case MPI_CONFIG_PAGETYPE_BIOS:
		DisplayBIOS(num);
		break;

	case MPI_CONFIG_PAGETYPE_SCSI_PORT:
		DisplaySCSIPort(num);
		break;

	case MPI_CONFIG_PAGETYPE_SCSI_DEVICE:
		DisplaySCSIDevice(num);
		break;

	case MPI_CONFIG_PAGETYPE_FC_PORT:
		DisplayFCPort(num);
		break;

	case MPI_CONFIG_PAGETYPE_FC_DEVICE:
		DisplayFCDevice(num);
		break;

	case MPI_CONFIG_PAGETYPE_LAN:
		DisplayLan(num);
		break;

	case MPI_CONFIG_PAGETYPE_RAID_VOLUME:
		DisplayRaidVolume(num);
		break;

	case MPI_CONFIG_PAGETYPE_MANUFACTURING:
		DisplayManufacturing(num);
		break;

	case MPI_CONFIG_PAGETYPE_RAID_PHYSDISK:
		DisplayRaidPhysDisk(num);
		break;

	default:
		break;
	}

	return;

}

void DisplayIOUnit(u8 number)
{
	int ii;

	switch (number) {
	case 0:
		{
			IOUnitPage0_t *dd = (IOUnitPage0_t *) mpiBlkPtr->dataInBufPtr;
			printf("IO Unit Page 0 Data:\n");
			printf("============================================\n");
			ii = le32_to_cpu(dd->UniqueValue.High);
			printf("UniqueValue = 0x%08x", ii);
			ii = le32_to_cpu(dd->UniqueValue.Low);
			printf("%08x", ii);
		}
		break;

	case 1:
		{
			IOUnitPage1_t *dd = (IOUnitPage1_t *) mpiBlkPtr->dataInBufPtr;
			printf("IO Unit Page 1 Data:\n");
			printf("=============================\n");
			printf("Flags = 0x%08x\n", le32_to_cpu(dd->Flags));
		}
		break;

	case 2:
		{
			IOUnitPage2_t *dd = (IOUnitPage2_t *) mpiBlkPtr->dataInBufPtr;
			printf("IO Unit Page 2 Data:\n");
			printf("=======================================\n");
			printf("Flags        = 0x%08x\n", le32_to_cpu(dd->Flags));
			printf("Bios Version = 0x%08x\n", le32_to_cpu(dd->BiosVersion));
			printf("Adapter Num  = Bus  DevFn Flags\n");
			for (ii = 0; ii < 4; ii++) {
				printf("Adapter %d    = 0x%02x 0x%02x 0x%04x\n",
				       ii,
				       dd->AdapterOrder[ii].PciBusNumber,
				       dd->AdapterOrder[ii].PciDeviceAndFunctionNumber,
				       le16_to_cpu(dd->AdapterOrder[ii].AdapterFlags));
			}
		}
		break;

	case 3:
		{
			IOUnitPage3_t *dd = (IOUnitPage3_t *) mpiBlkPtr->dataInBufPtr;
			printf("IO Unit Page 3 Data:\n");
			printf("=============================\n");
			printf("GPIO Count = 0x%02x\n", dd->GPIOCount);
			for (ii = 0; ii < dd->GPIOCount; ii++)
				printf("GPIO Val   = 0x%04x\n", le16_to_cpu(dd->GPIOVal[ii]));
		}
		break;
	}
	return;
}

void DisplayIOC(u8 number)
{
	int ii;

	switch (number) {
	case 0:
		{
			IOCPage0_t *dd = (IOCPage0_t *) mpiBlkPtr->dataInBufPtr;
			printf("IOC Page 0 Data:\n");
			printf("==================================\n");
			printf("Total NVStore  = 0x%08x\n", le32_to_cpu(dd->TotalNVStore));
			printf("Free NVStore   = 0x%08x\n", le32_to_cpu(dd->FreeNVStore));
			printf("Device ID      = 0x%04x\n", le16_to_cpu(dd->VendorID));
			printf("Vendor ID      = 0x%04x\n", le16_to_cpu(dd->DeviceID));
			printf("Revision ID    = 0x%02x\n", dd->DeviceID);
			printf("Class Code     = 0x%08x\n", le32_to_cpu(dd->ClassCode));
			printf("SubSystem ID   = 0x%04x\n", le16_to_cpu(dd->SubsystemID));
			printf("SubSystem Ven  = 0x%04x\n", le16_to_cpu(dd->SubsystemVendorID));
		}
		break;
	case 1:
		{
			IOCPage1_t *dd = (IOCPage1_t *) mpiBlkPtr->dataInBufPtr;
			printf("IOC Page 1 Data:\n");
			printf("=================================================\n");
			printf("Flags              = 0x%08x (1 = enabled)\n", le32_to_cpu(dd->Flags));
			printf("Coalescing Timeout = 0x%08x\n", le32_to_cpu(dd->CoalescingTimeout));
			printf("Coalescing Depth   = 0x%02x\n", dd->CoalescingDepth);
			printf("PCI Slot Number    = 0x%02x\n", dd->PCISlotNum);
		}
		break;
	case 2:
		{
			IOCPage2_t *dd = (IOCPage2_t *) mpiBlkPtr->dataInBufPtr;
			printf("IOC Page 2 Data:\n");
			printf("===========================================================\n");
			printf("Capabilities       = 0x%08x\n", le32_to_cpu(dd->CapabilitiesFlags));
			printf("Total Volumes      = 0x%08x\n", dd->NumActiveVolumes);
			printf("Max Volumes        = 0x%02x\n", dd->MaxVolumes);
			printf("Num Active P Disks = 0x%02x\n", dd->NumActivePhysDisks);
			printf("Max Phys Disks     = 0x%02x\n", dd->MaxPhysDisks);

			for (ii = 0; ii < dd->MaxVolumes; ii++) {
				printf("ID = 0x%02x  ", dd->RaidVolume[ii].VolumeID);
				printf("Bus = 0x%02x  ", dd->RaidVolume[ii].VolumeBus);
				printf("IOC = 0x%02x  ", dd->RaidVolume[ii].VolumeIOC);
				printf("Pag = 0x%02x  ", dd->RaidVolume[ii].VolumePageNumber);
			        printf("Type = 0x%02x  ", dd->RaidVolume[ii].VolumeType);
				printf("Flag = 0x%02x\n", dd->RaidVolume[ii].Flags);
			}
		}
		break;
	case 3:
		{
			IOCPage3_t *dd = (IOCPage3_t *) mpiBlkPtr->dataInBufPtr;
			printf("IOC Page 3 Data:\n");
			printf("===========================================================\n");
			printf("Num Phys Disks  = 0x%02x\n", dd->NumPhysDisks);
			for (ii = 0; ii < dd->NumPhysDisks; ii++) {
				printf("ID = 0x%02x  ", dd->PhysDisk[ii].PhysDiskID);
				printf("Bus = 0x%02x  ", dd->PhysDisk[ii].PhysDiskBus);
				printf("IOC = 0x%02x  ", dd->PhysDisk[ii].PhysDiskIOC);
				printf("Num = 0x%02x  \n", dd->PhysDisk[ii].PhysDiskNum);
			}
		}
		break;
	case 4:
		{
			IOCPage4_t *dd = (IOCPage4_t *) mpiBlkPtr->dataInBufPtr;
			printf("IOC Page 4 Data:\n");
			printf("=============================\n");
			printf("Num Active SEP  = 0x%02x\n", dd->ActiveSEP);
			printf("Maximum SEP     = 0x%02x\n", dd->MaxSEP);
			for (ii = 0; ii < dd->MaxSEP; ii++) {
				printf("ID = 0x%02x  ", dd->SEP[ii].SEPTargetID);
				printf("Bus = 0x%02x\n", dd->SEP[ii].SEPBus);
			}
		}
		break;
	case 5:
		{
			IOCPage5_t *dd = (IOCPage5_t *) mpiBlkPtr->dataInBufPtr;
			printf("IOC Page 5 Data:\n");
			printf("=================================================\n");
			printf("Num HotSpares = 0x%02x\n", dd->NumHotSpares);
			for (ii = 0; ii < dd->NumHotSpares; ii++) {
				printf("ID = 0x%02x  ", dd->HotSpare[ii].PhysDiskNum);
				printf("Pool = 0x%02x  ", dd->HotSpare[ii].HotSparePool);
				printf("Flags = 0x%02x\n", dd->HotSpare[ii].Flags);
			}
		}
		break;

	}
	return;
}
void DisplayBIOS(u8 number)
{
	ShowBuf("Data In:  ", mpiBlkPtr->dataInBufPtr, mpiBlkPtr->dataInSize, 1);
	return;
}

void DisplaySCSIPort(u8 number)
{
	int ii;
	switch (number) {
	case 0:
		{
			SCSIPortPage0_t *dd = (SCSIPortPage0_t *) mpiBlkPtr->dataInBufPtr;
			printf("SCSI Port Page 0 Data:\n");
			printf("==================================\n");
			ii = le32_to_cpu(dd->Capabilities);
			printf("Capabilities       = 0x%08x ", le32_to_cpu(dd->Capabilities));
			printf("(%s%s%s%s factor = 0x%02x @ offset = 0x%02x)\n",
				ii & MPI_SCSIPORTPAGE0_CAP_WIDE ? "wide " : "",
				ii & MPI_SCSIPORTPAGE0_CAP_IU ? "IU " : "",
				ii & MPI_SCSIPORTPAGE0_CAP_DT ? "DT " : "",
				ii & MPI_SCSIPORTPAGE0_CAP_QAS ? "QAS " : "",
				((ii >> 8) & 0xFF), ((ii >> 16) & 0xFF));

			ii = le32_to_cpu(dd->PhysicalInterface);
			printf("Physical Interface = 0x%08x (%s)\n", ii, phys_interface[(ii & 0x3)]);
		}
		break;
	case 1:
		{
			SCSIPortPage1_t *dd = (SCSIPortPage1_t *) mpiBlkPtr->dataInBufPtr;
			printf("SCSI Port Page 1 Data:\n");
			printf("==================================\n");
			printf("Configuration      = 0x%08x\n", le32_to_cpu(dd->Configuration));
			printf("On Bus Timer Value = 0x%08x\n", le32_to_cpu(dd->OnBusTimerValue));
		}
		break;
	case 2:
		{
			SCSIPortPage2_t *dd = (SCSIPortPage2_t *) mpiBlkPtr->dataInBufPtr;
			printf("SCSI Port Page 2 Data:\n");
			printf("==================================\n");
			printf("Port Flags    = 0x%08x\n", le32_to_cpu(dd->PortFlags));
			printf("Port Settings = 0x%08x\n", le32_to_cpu(dd->PortSettings));
			for (ii=0; ii < 16; ii++) {
				printf("id=%d timeout=%d factor=0x%02x flags=0x%04x\n",
					ii, dd->DeviceSettings[ii].Timeout,
					dd->DeviceSettings[ii].SyncFactor,
					le16_to_cpu(dd->DeviceSettings[ii].DeviceFlags));
			}
		}
		break;

	}
	return;
}
void DisplaySCSIDevice(u8 number)
{
	int ii;
	switch (number) {
	case 0:
		{
			SCSIDevicePage0_t *dd = (SCSIDevicePage0_t *) mpiBlkPtr->dataInBufPtr;
			printf("SCSI Device Page 0 Data:\n");
			printf("==================================\n");
			ii = le32_to_cpu(dd->NegotiatedParameters);
			printf("Negotiated   = 0x%08x \n\t( %s factor = 0x%02x @ offset = 0x%02x "
			       "%s%s%s%s%s%s%s%s)\n",
			       ii,
			       ii & MPI_SCSIDEVPAGE0_NP_WIDE ? "Wide ": "Narrow",
				((ii >> 8) & 0xFF),
				((ii >> 16) & 0xFF),
			       ii & MPI_SCSIDEVPAGE0_NP_IU ? "IU ": "",
			       ii & MPI_SCSIDEVPAGE0_NP_DT ? "DT ": "",
			       ii & MPI_SCSIDEVPAGE0_NP_QAS ? "QAS ": "",
			       ii & MPI_SCSIDEVPAGE0_NP_HOLD_MCS ? "HOLD_MCS ": "",
			       ii & MPI_SCSIDEVPAGE0_NP_WR_FLOW ? "WR_FLOW ": "",
			       ii & MPI_SCSIDEVPAGE0_NP_RD_STRM ? "RD_STRM ": "",
			       ii & MPI_SCSIDEVPAGE0_NP_RTI ? "RTI ": "",
			       ii & MPI_SCSIDEVPAGE0_NP_PCOMP_EN ? "PCOMP_EN ": "");

			printf("Information  = 0x%08x\n", le32_to_cpu(dd->Information));
		}
		break;
	case 1:
		{
			SCSIDevicePage1_t *dd = (SCSIDevicePage1_t *) mpiBlkPtr->dataInBufPtr;
			printf("SCSI Device Page 1 Data:\n");
			printf("==================================\n");
			ii = le32_to_cpu(dd->RequestedParameters);
			printf("Requested      = 0x%08x ( %s factor = 0x%02x @ offset = 0x%02x "
			       "%s%s%s%s%s%s%s%s)\n",
			       ii,
			       ii & MPI_SCSIDEVPAGE0_NP_WIDE ? "Wide ": "",
				((ii >> 8) & 0xFF),
				((ii >> 16) & 0xFF),
			       ii & MPI_SCSIDEVPAGE0_NP_IU ? "IU ": "",
			       ii & MPI_SCSIDEVPAGE0_NP_DT ? "DT ": "",
			       ii & MPI_SCSIDEVPAGE0_NP_QAS ? "QAS ": "",
			       ii & MPI_SCSIDEVPAGE0_NP_HOLD_MCS ? "HOLD_MCS ": "",
			       ii & MPI_SCSIDEVPAGE0_NP_WR_FLOW ? "WR_FLOW ": "",
			       ii & MPI_SCSIDEVPAGE0_NP_RD_STRM ? "RD_STRM ": "",
			       ii & MPI_SCSIDEVPAGE0_NP_RTI ? "RTI ": "",
			       ii & MPI_SCSIDEVPAGE0_NP_PCOMP_EN ? "PCOMP_EN ": "");

			printf("Configuration  = 0x%08x ( 0x02 = no SDTR, 0x01 = no WDTR )\n",
						le32_to_cpu(dd->Configuration));
		}
		break;
	case 2:
		{
			SCSIDevicePage2_t *dd = (SCSIDevicePage2_t *) mpiBlkPtr->dataInBufPtr;
			printf("SCSI Device Page 2 Data:\n");
			printf("==================================\n");
			printf("DV          = 0x%08x \n", le32_to_cpu(dd->DomainValidation));
			printf("Parity Pipe = 0x%08x \n", le32_to_cpu(dd->ParityPipeSelect));
			printf("Data Pipe   = 0x%08x \n", le32_to_cpu(dd->DataPipeSelect));
		}
		break;
	case 3:
		{
			SCSIDevicePage3_t *dd = (SCSIDevicePage3_t *) mpiBlkPtr->dataInBufPtr;
			printf("SCSI Device Page 3 Data:\n");
			printf("==================================\n");
			printf("Msg Reject Count   = 0x%08x \n", le16_to_cpu(dd->MsgRejectCount));
			printf("Phase Error Count  = 0x%08x \n", le16_to_cpu(dd->PhaseErrorCount));
			printf("Parity Error Count = 0x%08x \n", le16_to_cpu(dd->ParityErrorCount));
		}
		break;
	}
	return;
}
void DisplayFCPort(u8 number)
{
	ShowBuf("Data In:  ", mpiBlkPtr->dataInBufPtr, mpiBlkPtr->dataInSize, 1);
	return;
}
void DisplayFCDevice(u8 number)
{
	ShowBuf("Data In:  ", mpiBlkPtr->dataInBufPtr, mpiBlkPtr->dataInSize, 1);
	return;
}
void DisplayLan(u8 number)
{
	ShowBuf("Data In:  ", mpiBlkPtr->dataInBufPtr, mpiBlkPtr->dataInSize, 1);
	return;
}
void DisplayRaidVolume(u8 number)
{
	int ii;

	switch (number) {
	case 0:
		{
			RaidVolumePage0_t *dd = (RaidVolumePage0_t *) mpiBlkPtr->dataInBufPtr;
			printf("Raid Volume Page 0 Data:\n");
			printf("============================================\n");
			printf("ID          = 0x%02x\n", dd->VolumeID);
			printf("Bus         = 0x%02x\n", dd->VolumeBus);
			printf("IOC         = 0x%02x\n", dd->VolumeIOC);
			printf("Type        = 0x%02x ( %s )\n", 
					dd->VolumeType, raid_types[dd->VolumeType]);
			ii = dd->VolumeStatus.Flags;
			printf("Flags       = 0x%02x ( %s%s%s%s)\n",
			       ii,
			       ii & MPI_RAIDVOL0_STATUS_FLAG_ENABLED ? "enabled " : "",
			       ii & MPI_RAIDVOL0_STATUS_FLAG_QUIESCED ? "queisced " : "",
			       ii & MPI_RAIDVOL0_STATUS_FLAG_RESYNC_IN_PROGRESS ? "resyncing " : "",
			       ii & MPI_RAIDVOL0_STATUS_FLAG_VOLUME_INACTIVE ? "inactive " : "");
			ii = dd->VolumeStatus.State;
			printf("State       = 0x%02x ( %s%s%s)\n",
			       ii,
			       ii & MPI_RAIDVOL0_STATUS_STATE_OPTIMAL ? "optimal " : "",
			       ii & MPI_RAIDVOL0_STATUS_STATE_DEGRADED ? "degraded " : "",
			       ii & MPI_RAIDVOL0_STATUS_STATE_FAILED ? "failed " : "");

			printf("Settings    = 0x%04x\n", le16_to_cpu(dd->VolumeSettings.Settings));
			printf("Hot Spare   = 0x%02x\n", dd->VolumeSettings.HotSparePool);
			printf("Max LBA     = 0x%08x\n", le32_to_cpu(dd->MaxLBA));
			printf("StripeSize  = 0x%08x\n", le32_to_cpu(dd->StripeSize));
			for (ii = 0; ii < MPI_RAID_VOL_PAGE_0_PHYSDISK_MAX; ii++) {
				printf("Map = 0x%02x ", dd->PhysDisk[ii].PhysDiskMap);
				printf("Num = 0x%02x\n", dd->PhysDisk[ii].PhysDiskNum);
			}
		}
		break;
	}

	return;
}
void DisplayManufacturing(u8 number)
{
	int ii;

	switch (number) {
	case 0:
		{
			ManufacturingPage0_t *dd = (ManufacturingPage0_t *) mpiBlkPtr->dataInBufPtr;
			printf("Manufacturing Page 0 Data:\n");
			printf("============================================\n");
			printf("Chip Name      = %s\n", dd->ChipName);
			printf("Chip Revision  = %s\n", dd->ChipRevision);
			printf("Board Name     = %s\n", dd->BoardName);
			printf("Board Assembly = %s\n", dd->BoardAssembly);
			printf("Board Tracer   = %s\n", dd->BoardTracerNumber);
		}
		break;
	case 1:
		{
			printf("Manufacturing Page 1 Data:\n");
			printf("============================================\n");
			ShowBuf("", mpiBlkPtr->dataInBufPtr + 4, mpiBlkPtr->dataInSize, 1);
		}
		break;
	case 2:
		{
			ManufacturingPage2_t *dd = (ManufacturingPage2_t *) mpiBlkPtr->dataInBufPtr;
			printf("Manufacturing Page 2 Data:\n");
			printf("============================================\n");
			printf("Chip ID  = 0x%04x\n", le16_to_cpu(dd->ChipId.DeviceID));
			printf("Revision = 0x%04x\n", dd->ChipId.PCIRevisionID);
			printf("HW Settings:\n");
			for (ii = 0; ii < MPI_MAN_PAGE_2_HW_SETTINGS_WORDS; ii++)
				printf("0x%08x\n", le32_to_cpu(dd->HwSettings[ii]));
		}
		break;
	case 3:
		{
			ManufacturingPage3_t *dd = (ManufacturingPage3_t *) mpiBlkPtr->dataInBufPtr;
			printf("Manufacturing Page 3 Data:\n");
			printf("============================================\n");
			printf("Chip ID  = 0x%04x\n", le16_to_cpu(dd->ChipId.DeviceID));
			printf("Revision = 0x%04x\n", dd->ChipId.PCIRevisionID);
			printf("Info Settings:\n");
			for (ii = 0; ii < MPI_MAN_PAGE_3_INFO_WORDS; ii++)
				printf("0x%08x ", le32_to_cpu(dd->Info[ii]));
		}
		break;
	case 4:
		{
			ManufacturingPage4_t *dd = (ManufacturingPage4_t *) mpiBlkPtr->dataInBufPtr;
			printf("Manufacturing Page 4 Data:\n");
			printf("============================================\n");
			printf("Info Offset (0) = 0x%04x\n", dd->InfoOffset0);
			printf("Info Size   (0) = 0x%04x\n", dd->InfoSize0);
			printf("Info Offset (1) = 0x%04x\n", dd->InfoOffset1);
			printf("Info Size   (1) = 0x%04x\n", dd->InfoSize1);
			printf("Inquiry Size    = 0x%04x\n", dd->InquirySize);
			ShowBuf("Inquiry Data:  ", dd->InquiryData, dd->InquirySize, 1);

			ii = le32_to_cpu(dd->ISVolumeSettings);
			if (ii & MPI_RAIDVOL0_SETTING_USE_DEFAULTS) {
				printf("IS Settings     = 0x%08x (Hot Spare =0x%02x, Default)\n",
							ii, (ii >> 16) & 0xFF);
			} else {
				printf("IS Settings     = 0x%08x Hot Spare = 0x%02x"
				       "%s %s %s %s %s\n",
					ii,
					((ii >> 16) & 0xFF),
					ii & MPI_RAIDVOL0_SETTING_USE_PRODUCT_ID_SUFFIX ?
						"\n\tUse Suffix":"",
					ii & MPI_RAIDVOL0_SETTING_PRIORITY_RESYNC ?
						"\n\tPriority Resync":"",
					ii & MPI_RAIDVOL0_SETTING_AUTO_CONFIGURE ?
						"\n\tAuto Config":"",
					ii & MPI_RAIDVOL0_SETTING_OFFLINE_ON_SMART ?
						"\n\tOffline if SMART error":"",
					ii & MPI_RAIDVOL0_SETTING_WRITE_CACHING_ENABLE ?
						"\n\tWrite Caching Enabled":"");
			}

			ii = le32_to_cpu(dd->IMEVolumeSettings);
			if (ii & MPI_RAIDVOL0_SETTING_USE_DEFAULTS) {
				printf("IME Settings    = 0x%08x (Hot Spare =0x%02x, Default)\n",
							ii, (ii >> 16) & 0xFF);
			} else {
				printf("IME Settings    = 0x%08x Hot Spare = 0x%02x"
				       "%s %s %s %s %s\n",
					ii,
					((ii >> 16) & 0xFF),
					ii & MPI_RAIDVOL0_SETTING_USE_PRODUCT_ID_SUFFIX ?
						"\n\tUse Suffix":"",
					ii & MPI_RAIDVOL0_SETTING_PRIORITY_RESYNC ?
						"\n\tPriority Resync":"",
					ii & MPI_RAIDVOL0_SETTING_AUTO_CONFIGURE ?
						"\n\tAuto Config":"",
					ii & MPI_RAIDVOL0_SETTING_OFFLINE_ON_SMART ?
						"\n\tOffline if SMART error":"",
					ii & MPI_RAIDVOL0_SETTING_WRITE_CACHING_ENABLE ?
						"\n\tWrite Caching Enabled":"");
			}

			ii = le32_to_cpu(dd->IMVolumeSettings);
			if (ii & MPI_RAIDVOL0_SETTING_USE_DEFAULTS) {
				printf("IM Settings     = 0x%08x (Hot Spare =0x%02x, Default)\n",
							ii, (ii >> 16) & 0xFF);
			} else {
				printf("IM Settings     = 0x%08x Hot Spare = 0x%02x"
				       "%s %s %s %s %s\n",
					ii,
					((ii >> 16) & 0xFF),
					ii & MPI_RAIDVOL0_SETTING_USE_PRODUCT_ID_SUFFIX ?
						"\n\tUse Suffix":"",
					ii & MPI_RAIDVOL0_SETTING_PRIORITY_RESYNC ?
						"\n\tPriority Resync":"",
					ii & MPI_RAIDVOL0_SETTING_AUTO_CONFIGURE ?
						"\n\tAuto Config":"",
					ii & MPI_RAIDVOL0_SETTING_OFFLINE_ON_SMART ?
						"\n\tOffline if SMART error":"",
					ii & MPI_RAIDVOL0_SETTING_WRITE_CACHING_ENABLE ?
						"\n\tWrite Caching Enabled":"");
			}
		}
		break;
	}

	return;
}

void DisplayRaidPhysDisk(u8 number)
{
	switch (number) {
	case 0:
		{
			RaidPhysDiskPage0_t *dd = (RaidPhysDiskPage0_t *) mpiBlkPtr->dataInBufPtr;
			printf("Raid Phys Disk Page 0 Data:\n");
			printf("============================================\n");
			printf("ID          = 0x%02x\n", dd->PhysDiskID);
			printf("Bus         = 0x%02x\n", dd->PhysDiskBus);
			printf("IOC         = 0x%02x\n", dd->PhysDiskIOC);
			printf("Num         = 0x%02x\n", dd->PhysDiskNum);
			printf("SEP ID      = 0x%02x\n", dd->PhysDiskSettings.SepID);
			printf("SEP Bus     = 0x%02x\n", dd->PhysDiskSettings.SepBus);
			printf("Hot Spare   = 0x%02x\n", dd->PhysDiskSettings.HotSparePool);
			printf("PD Settings = 0x%02x\n", dd->PhysDiskSettings.PhysDiskSettings);
			printf("Identifier  = %s\n", dd->DiskIdentifier);
			printf("Inq Vendor  = %s\n", dd->InquiryData.VendorID);
			printf("Inq Prod    = %s\n", dd->InquiryData.ProductID);
			printf("Inq Rev     = %s\n", dd->InquiryData.ProductRevLevel);
			printf("Inq Info    = %s\n", dd->InquiryData.Info);
			printf("Flags       = 0x%02x\n", dd->PhysDiskStatus.Flags);
			printf("State       = 0x%02x\n", dd->PhysDiskStatus.State);
			printf("Max LBA     = 0x%08x\n", le32_to_cpu(dd->MaxLBA));
			printf("Error CDB   = 0x%02x\n", dd->ErrorData.ErrorCdbByte);
			printf("SKey        = 0x%02x\n", dd->ErrorData.ErrorSenseKey);
			printf("Total Cnt   = 0x%02x\n", le16_to_cpu(dd->ErrorData.ErrorCount));
			printf("ASC         = 0x%02x\n", dd->ErrorData.ErrorASC);
			printf("ASCQ        = 0x%02x\n", dd->ErrorData.ErrorASCQ);
			printf("Smart Cnt   = 0x%02x\n", le16_to_cpu(dd->ErrorData.SmartCount));
			printf("Smart ASC   = 0x%02x\n", dd->ErrorData.SmartASC);
			printf("Smart ASCQ  = 0x%02x\n", dd->ErrorData.SmartASCQ);
		}
		break;

	}
	return;
}

