/*****************************************************************
 *                                                               *
 * Copyright 2000-2002 LSI Logic. All rights reserved.           *
 *                                                               *
 * This fiel is confidential and a trade secret of LSI Logic.    *
 * The receipt of or possession of this file does not convey any *
 * rights to reproduce or disclose its contents or to            *
 * manufacture, use, or sell anything it may describe, in whole, *
 * or in part, without the specific written consent of LSI Logic.*
 *                                                               *
 *****************************************************************/

/* System Include Files
 */
#include "apps.h"

/* Function Prototypes
 */
void GetVolumeInfo (int *vol_ids);
void GetHotSpareInfo (void);
int GetVolumeIds (int *vol_ids);
void GetPhysDiskInfo(RaidVol0PhysDisk_t *pDisk, int count);
void GetResyncPercentage(RaidVol0PhysDisk_t *pDisk, u8 * pVol, int count);

/* Globals
 */
extern mpiIoctlBlk_t    *mpiBlkPtr;
extern int              g_iocnum;
extern int              g_bigEndian;

char *raid_types [] = {
	"Integrated Striping",
	"Integrated Mirroring Enhanced",
	"Integrated Mirroring",
};

char *raid_status [] = {
	"Optimal",
	"Degraded",
	"Failed",
};

/*****************************************************************
 *
 *
 *  MAIN
 *
 *
 *****************************************************************/
int main (int argc, char *argv[])
{
	FILE *pfile;
	char *name = "mptctl";
	char dataIn[PROC_READ_LINE];
	int  ids[MAX_SCSI_ID+1];
	int  *pVolIds;
	int  c;
	int ii;
	test_endianess_t test;

	pVolIds = &ids[0];
	for (ii=0; ii <= MAX_SCSI_ID; ii++)
		ids[ii] = -1;

	/* Initialize (force a buffer flush)
	 */
	setvbuf(stdout,NULL,_IONBF,0);

	/* Get the system endianness....MPI request frames must have
	 * Little Endianness. If bigEndian, byte swap on data coming from FW
	 * and all request frame fields.
	 */
	test.u.foo = 0x01020304;
	g_bigEndian = 0;
	if (test.u.bar[0] != (test.u.foo & 0xFF))
		g_bigEndian = 1;

	/* Open /proc/modules to determine if
	 * driver loaded.
	 */
	snprintf(dataIn, PROC_READ_LINE, "/proc/modules");
	if ((pfile = fopen(dataIn, "r")) == NULL ) {
		printf("Open of /proc/modules failed!\n");
		DoExit();
	}

	c = 0;
	while (1) {
		if (fgets(dataIn, PROC_READ_LINE, pfile) == NULL)
			break;

		if (strstr(dataIn, name) != NULL) {
			c = 1;
			break;
		}
	}
	fclose(pfile);

	if (c == 0) {
		system("insmod mptctl");
	}

	/* Usage
	 */
	if (argc < 2) {
		fprintf (stderr, "usage: %s iocnum [target]\n", argv[0]);
		return (0);
	}

	/* Get and open scsi node.
	 */
	if (OpenDevice (argv[1]))
		DoExit();
	
	ii = -1;
	if (argc == 3) {
		if (CheckNumericalInput(argv[2]) != APP_TRUE) {
			printf("not a digit\n");
		} else {
			ii = atoi(argv[2]);
			if ((ii < 0) || (ii <= MAX_SCSI_ID))
				ii = -1;
		}
	}

	printf("\n\nCompiled with Fusion-MPT %s Driver Header Files\n\n", DRIVER_VERSION);

	if (ii != -1) {
		ids[0] = ii;
		GetVolumeInfo(pVolIds);
		GetHotSpareInfo();
	} else if (GetVolumeIds(pVolIds) > 0) {
		GetVolumeInfo (pVolIds);
		GetHotSpareInfo();
	} else {
		printf("Logical Volumes Not Supported or None Found.\n");
	}

	DoExit();
	return 0;
}

/*****************************************************************
 * GetVolumeIds
 *
 *****************************************************************/
int GetVolumeIds (int *vol_ids)
{
	Config_t *ConfigRequest;
	ConfigReply_t *pReply = NULL;
	IOCPage2_t *pPg2 = NULL;
	uint numBytes;
	uint  num_volumes = 0;
	int  ii, status;

	numBytes = (sizeof(Config_t) - sizeof(SGE_IO_UNION)) + sizeof (SGESimple64_t);
	if ((mpiBlkPtr = allocIoctlBlk(numBytes)) == NULL)
		return num_volumes;

	ConfigRequest = (Config_t *) mpiBlkPtr->MF;
	mpiBlkPtr->dataInSize = mpiBlkPtr->dataOutSize = 0;
	mpiBlkPtr->dataInBufPtr = mpiBlkPtr->dataOutBufPtr = NULL;
	mpiBlkPtr->dataSgeOffset = (sizeof (Config_t) - sizeof(SGE_IO_UNION))/4;

	pReply = (ConfigReply_t *)mpiBlkPtr->replyFrameBufPtr;

	/* Populate the Config Request
	 */
	ConfigRequest->Action       = MPI_CONFIG_ACTION_PAGE_HEADER;
	ConfigRequest->Function     = MPI_FUNCTION_CONFIG;
	ConfigRequest->MsgContext   = -1;
	ConfigRequest->Header.PageNumber = 2;
	ConfigRequest->Header.PageType = MPI_CONFIG_PAGETYPE_IOC;

	status = IssueMptCommand(MPT_FLAGS_KEEP_MEM);

	if ((status != 0) || (pReply->Header.PageLength == 0)) {
		freeAllocMem();
		return num_volumes;
	}

	mpiBlkPtr->dataInSize = pReply->Header.PageLength * 4;
	if (allocDataFrame(DATA_DIR_IN)) {
		printf ("Config: Unable to allocate data buffer.");
		freeAllocMem();
		return num_volumes;
	}

	ConfigRequest->Action = MPI_CONFIG_ACTION_PAGE_READ_CURRENT;
	ConfigRequest->Header.PageVersion = pReply->Header.PageVersion;
	ConfigRequest->Header.PageLength = pReply->Header.PageLength;
	status = IssueMptCommand(MPT_FLAGS_KEEP_MEM);

	pPg2 = (IOCPage2_t *) mpiBlkPtr->dataInBufPtr;
	if (status == 0) {
		num_volumes = pPg2->NumActiveVolumes;
		if (num_volumes > 0){
			for (ii=0; ii < num_volumes; ii++)
				vol_ids[ii] = pPg2->RaidVolume[ii].VolumeID;
		}
	}
	freeAllocMem();
	return num_volumes;
}

/*****************************************************************
 * GetVolumeInfo
 *
 *****************************************************************/
void GetVolumeInfo (int *vol_ids)
{
	Config_t 		*ConfigRequest;
	ConfigReply_t 		*pReply = NULL;
	RaidVolumePage0_t 	*pRVP0 = NULL;
	RaidVol0PhysDisk_t	disk_num[16];
	uchar			pdisk_vol[16];
	uint numBytes;
	int  status;
	int  ii, idx, id;
	uint vol_count = 0;
	int  pdisk_cnt = 0;
	int resync_on = 0;
	uchar bus = 0;
	u8 tmp;

	numBytes = (sizeof(Config_t) - sizeof(SGE_IO_UNION)) + sizeof (SGESimple64_t);
	if ((mpiBlkPtr = allocIoctlBlk(numBytes)) == NULL)
		return;

	ConfigRequest = (Config_t *) mpiBlkPtr->MF;
	mpiBlkPtr->dataInSize = mpiBlkPtr->dataOutSize = 0;
	mpiBlkPtr->dataInBufPtr = mpiBlkPtr->dataOutBufPtr = NULL;
	mpiBlkPtr->dataSgeOffset = (sizeof (Config_t) - sizeof(SGE_IO_UNION))/4;

	pReply = (ConfigReply_t *)mpiBlkPtr->replyFrameBufPtr;

	/* Populate the Config Request
	 */
	ConfigRequest->Action       = MPI_CONFIG_ACTION_PAGE_HEADER;
	ConfigRequest->Function     = MPI_FUNCTION_CONFIG;
	ConfigRequest->MsgContext   = -1;
	ConfigRequest->Header.PageType = MPI_CONFIG_PAGETYPE_RAID_VOLUME;
	ConfigRequest->PageAddress = cpu_to_le32((bus << 8) | 0);

	status = IssueMptCommand(MPT_FLAGS_KEEP_MEM);

	printf("Logical Volume Information:\n");
	if (status != 0) {
		printf("\tUnsupported.\n");
		freeAllocMem();
		return;
	} else if (pReply->Header.PageLength == 0) {
		printf("\tNone.\n");
		freeAllocMem();
		return;
	}

	mpiBlkPtr->dataInSize = pReply->Header.PageLength * 4;
	if (allocDataFrame(DATA_DIR_IN)) {
		printf ("Config: Unable to allocate data buffer.");
		freeAllocMem();
		return;
	}

	ConfigRequest->Action = MPI_CONFIG_ACTION_PAGE_READ_CURRENT;
	ConfigRequest->Header.PageVersion = pReply->Header.PageVersion;
	ConfigRequest->Header.PageLength = pReply->Header.PageLength;
	idx = 0;
	while ((vol_ids[idx] >= 0) && (idx <= MAX_SCSI_ID)) {
		id = vol_ids[idx];
		ConfigRequest->PageAddress = cpu_to_le32((bus << 8) | id);
		status = IssueMptCommand(MPT_FLAGS_KEEP_MEM);

		/* Kludge - IO FW 1.1.60 returns IOCSTATUS of GOOD to
		 * this call resulting in bad information.
		 * IM FW returns a status of 0x0022 - INVALID PAGE
		 */
		pRVP0 = (RaidVolumePage0_t *) mpiBlkPtr->dataInBufPtr;
		if ((status == 0) && (pRVP0->NumPhysDisks > 0)){
			vol_count++;
			printf("\tID=0x%x, Bus=0x%x, IOC=0x%x, %s(0x%x)",
				pRVP0->VolumeID,pRVP0->VolumeBus, pRVP0->VolumeIOC,
				raid_types[pRVP0->VolumeType], pRVP0->VolumeType);
			if (pRVP0->VolumeType == MPI_RAID_VOL_TYPE_IS)
				printf(", Stripe Size=0x%x", pRVP0->StripeSize);
			printf(", %d Physical Disks", pRVP0->NumPhysDisks);
			printf("\n");

			printf("\tStatus: ");
			tmp =  pRVP0->VolumeStatus.Flags;
			printf("%s", tmp & MPI_RAIDVOL0_STATUS_FLAG_VOLUME_INACTIVE ? "Not Active" : "Active");

			printf(", %s", tmp & MPI_RAIDVOL0_STATUS_FLAG_ENABLED ? "Enabled" : "Disabled");
			if (tmp & MPI_RAIDVOL0_STATUS_FLAG_QUIESCED)
				printf(", Quiesced Phys Disk IOs");

			if (tmp & MPI_RAIDVOL0_STATUS_FLAG_RESYNC_IN_PROGRESS) {
				printf(", Resync In Progress");
				resync_on = 1;
			}
			printf("\n");

			tmp =  (pRVP0->VolumeStatus.State) & 0x03;
			printf("\tState: %s\n", raid_status[tmp]);

			for (ii = pdisk_cnt; ii < pdisk_cnt + pRVP0->NumPhysDisks; ii++) {
				disk_num[ii].PhysDiskNum = pRVP0->PhysDisk[ii].PhysDiskNum;
				disk_num[ii].PhysDiskMap = pRVP0->PhysDisk[ii].PhysDiskMap;
				pdisk_vol[ii] = pRVP0->VolumeID;
			}
			pdisk_cnt += pRVP0->NumPhysDisks;
		}
		idx++;
	}
	freeAllocMem();

	if (pdisk_cnt > 0)
		GetPhysDiskInfo((RaidVol0PhysDisk_t *) &disk_num, pdisk_cnt);

	if (resync_on)
		GetResyncPercentage((RaidVol0PhysDisk_t *) &disk_num, 
						(uchar *) &pdisk_vol, pdisk_cnt);


	if (vol_count == 0)
		printf("\tNo Logical Volumes Found.\n");

	return;
}

/*****************************************************************
 * GetHotSpare
 *
 *****************************************************************/
void GetHotSpareInfo (void)
{
	Config_t *ConfigRequest;
	ConfigReply_t *pReply = NULL;
	IOCPage5_t *pPg5 = NULL;
	uint numBytes;
	int  status;
	uint num_spares = 0;
	u8 tmp;

	numBytes = (sizeof(Config_t) - sizeof(SGE_IO_UNION)) + sizeof (SGESimple64_t);
	if ((mpiBlkPtr = allocIoctlBlk(numBytes)) == NULL)
		return;

	ConfigRequest = (Config_t *) mpiBlkPtr->MF;
	mpiBlkPtr->dataInSize = mpiBlkPtr->dataOutSize = 0;
	mpiBlkPtr->dataInBufPtr = mpiBlkPtr->dataOutBufPtr = NULL;
	mpiBlkPtr->dataSgeOffset = (sizeof (Config_t) - sizeof(SGE_IO_UNION))/4;

	pReply = (ConfigReply_t *)mpiBlkPtr->replyFrameBufPtr;

	/* Populate the Config Request
	 */
	ConfigRequest->Action       = MPI_CONFIG_ACTION_PAGE_HEADER;
	ConfigRequest->Function     = MPI_FUNCTION_CONFIG;
	ConfigRequest->MsgContext   = -1;
	ConfigRequest->Header.PageNumber = 5;
	ConfigRequest->Header.PageType = MPI_CONFIG_PAGETYPE_IOC;
	ConfigRequest->PageAddress = cpu_to_le32(0);

	status = IssueMptCommand(MPT_FLAGS_KEEP_MEM);

	printf("Hot Spare Information:\n");
	if (status != 0) {
		printf("\tUnsupported.\n");
		freeAllocMem();
		return;
	} else if (pReply->Header.PageLength == 0 ) {
		printf("\tNone.\n");
		freeAllocMem();
		return;
	}

	mpiBlkPtr->dataInSize = pReply->Header.PageLength * 4;
	if (allocDataFrame(DATA_DIR_IN)) {
		printf ("Config: Unable to allocate data buffer.");
		freeAllocMem();
		return;
	}

	ConfigRequest->Action = MPI_CONFIG_ACTION_PAGE_READ_CURRENT;
	ConfigRequest->Header.PageVersion = pReply->Header.PageVersion;
	ConfigRequest->Header.PageLength = pReply->Header.PageLength;
	status = IssueMptCommand(MPT_FLAGS_KEEP_MEM);
	pPg5 = (IOCPage5_t *) mpiBlkPtr->dataInBufPtr;

	if ((status == 0) && (pPg5->NumHotSpares > 0)){
		num_spares = pPg5->NumHotSpares;
			
		/* Get Phys Disk Information
		 */
		{
			Ioc5HotSpare_t	disk_num[num_spares];
			int	ii;

			for (ii = 0; ii < num_spares; ii++) {
				disk_num[ii].PhysDiskNum = pPg5->HotSpare[ii].PhysDiskNum;
				disk_num[ii].HotSparePool = pPg5->HotSpare[ii].HotSparePool;
			}

			freeAllocMem();	/* Do not reference pPg5 any more */

			numBytes = (sizeof(Config_t) - sizeof(SGE_IO_UNION)) + sizeof (SGESimple64_t);
			if ((mpiBlkPtr = allocIoctlBlk(numBytes)) == NULL)
				return;

			ConfigRequest = (Config_t *) mpiBlkPtr->MF;
			mpiBlkPtr->dataInSize = mpiBlkPtr->dataOutSize = 0;
			mpiBlkPtr->dataInBufPtr = mpiBlkPtr->dataOutBufPtr = NULL;
			mpiBlkPtr->dataSgeOffset = (sizeof (Config_t) - sizeof(SGE_IO_UNION))/4;

			pReply = (ConfigReply_t *)mpiBlkPtr->replyFrameBufPtr;

			/* Populate the Config Request
			 */
			ConfigRequest->Action = MPI_CONFIG_ACTION_PAGE_HEADER;
			ConfigRequest->Function     = MPI_FUNCTION_CONFIG;
			ConfigRequest->MsgContext   = -1;
			ConfigRequest->Header.PageType = MPI_CONFIG_PAGETYPE_RAID_PHYSDISK;
			ConfigRequest->PageAddress = cpu_to_le32((uint)disk_num[0].PhysDiskNum);

			status = IssueMptCommand(MPT_FLAGS_KEEP_MEM);
			if ((status == 0) && (pReply->Header.PageLength > 0)) {
				mpiBlkPtr->dataInSize = pReply->Header.PageLength * 4;
				if (allocDataFrame(DATA_DIR_IN)) {
					printf ("Config: Unable to allocate data buffer.");
					freeAllocMem();
					return;
				}
			}

			ConfigRequest->Action = MPI_CONFIG_ACTION_PAGE_READ_CURRENT;
			ConfigRequest->Header.PageVersion = pReply->Header.PageVersion;
			ConfigRequest->Header.PageLength = pReply->Header.PageLength;
			for (ii = 0; ii < num_spares; ii++){
				ConfigRequest->PageAddress = cpu_to_le32((uint)disk_num[ii].PhysDiskNum);

				status = IssueMptCommand(MPT_FLAGS_KEEP_MEM);
				if (status == 0) {
					RaidPhysDiskPage0_t *pRPD0 = (RaidPhysDiskPage0_t *)
										mpiBlkPtr->dataInBufPtr;
			
					printf("\tNum=%d ID=%d", pRPD0->PhysDiskNum, pRPD0->PhysDiskID);
					tmp=pRPD0->PhysDiskStatus.Flags;
					if (tmp & MPI_PHYSDISK0_STATUS_FLAG_OUT_OF_SYNC)
						printf(", Out of Sync");
					if (tmp & MPI_PHYSDISK0_STATUS_FLAG_QUIESCED)
						printf(", Quiesced");
					tmp=pRPD0->PhysDiskStatus.State;
					if (tmp & MPI_PHYSDISK0_STATUS_ONLINE)
						printf(", Online");
					if (tmp & MPI_PHYSDISK0_STATUS_MISSING)
						printf(", Missing");
					if (tmp & MPI_PHYSDISK0_STATUS_NOT_COMPATIBLE)
						printf(", Not Compatible");
					if (tmp & MPI_PHYSDISK0_STATUS_FAILED)
						printf(", Failed");
					if (tmp & MPI_PHYSDISK0_STATUS_INITIALIZING)
						printf(", Initializing");
					if (tmp & MPI_PHYSDISK0_STATUS_FAILED_REQUESTED)
						printf(", Failed Requested");
					printf("\n");
				} else
					printf("\t\tNot Available.\n");
			}
		}
	} else if (status == 0)
		printf("Not Available.\n");
	else
		printf("No Hot Spare Disks.\n");

	freeAllocMem();

	return;
}

void GetPhysDiskInfo(RaidVol0PhysDisk_t *pDisk, int count)
{
	Config_t 		*ConfigRequest;
	ConfigReply_t 		*pReply = NULL;
	uint numBytes;
	int  status;
	int  ii;
	u8 tmp;

	printf("\tPhysical Disk Information:\n");

	numBytes = (sizeof(Config_t) - sizeof(SGE_IO_UNION)) + sizeof (SGESimple64_t);
	if ((mpiBlkPtr = allocIoctlBlk(numBytes)) == NULL)
		return;

	ConfigRequest = (Config_t *) mpiBlkPtr->MF;
	mpiBlkPtr->dataInSize = mpiBlkPtr->dataOutSize = 0;
	mpiBlkPtr->dataInBufPtr = mpiBlkPtr->dataOutBufPtr = NULL;
	mpiBlkPtr->dataSgeOffset = (sizeof (Config_t) - sizeof(SGE_IO_UNION))/4;

	pReply = (ConfigReply_t *)mpiBlkPtr->replyFrameBufPtr;

	/* Populate the Config Request
	 */
	ConfigRequest->Action       = MPI_CONFIG_ACTION_PAGE_HEADER;
	ConfigRequest->Function     = MPI_FUNCTION_CONFIG;
	ConfigRequest->MsgContext   = -1;
	ConfigRequest->Header.PageType = MPI_CONFIG_PAGETYPE_RAID_PHYSDISK;
	ConfigRequest->PageAddress = cpu_to_le32((uint)pDisk[0].PhysDiskNum);

	status = IssueMptCommand(MPT_FLAGS_KEEP_MEM);
	if ((status == 0) && (pReply->Header.PageLength > 0)) {
		mpiBlkPtr->dataInSize = pReply->Header.PageLength * 4;
		if (allocDataFrame(DATA_DIR_IN)) {
			printf ("Config: Unable to allocate data buffer.");
			freeAllocMem();
			return;
		}
	}

	ConfigRequest->Action = MPI_CONFIG_ACTION_PAGE_READ_CURRENT;
	ConfigRequest->Header.PageVersion = pReply->Header.PageVersion;
	ConfigRequest->Header.PageLength = pReply->Header.PageLength;
	for (ii = 0; ii < count; ii++){
		ConfigRequest->PageAddress = cpu_to_le32((uint)pDisk[ii].PhysDiskNum);

		status = IssueMptCommand(MPT_FLAGS_KEEP_MEM);
		if (status == 0) {
			RaidPhysDiskPage0_t *pRPD0 = (RaidPhysDiskPage0_t *)
								mpiBlkPtr->dataInBufPtr;
				
			printf("\t\tPhys Disk Num=%d at ID=%d", 
						pRPD0->PhysDiskNum, pRPD0->PhysDiskID);
			tmp=pRPD0->PhysDiskStatus.Flags;
			if (tmp & MPI_PHYSDISK0_STATUS_FLAG_OUT_OF_SYNC)
				printf(", Out of Sync");
			if (tmp & MPI_PHYSDISK0_STATUS_FLAG_QUIESCED)
				printf(", Quiesced");
			tmp=pRPD0->PhysDiskStatus.State;
			if (tmp & MPI_PHYSDISK0_STATUS_ONLINE)
				printf(", Online");
			if (tmp & MPI_PHYSDISK0_STATUS_MISSING)
				printf(", Missing");
			if (tmp & MPI_PHYSDISK0_STATUS_NOT_COMPATIBLE)
				printf(", Not Compatible");
			if (tmp & MPI_PHYSDISK0_STATUS_FAILED)
				printf(", Failed");
			if (tmp & MPI_PHYSDISK0_STATUS_INITIALIZING)
				printf(", Initializing");
			if (tmp & MPI_PHYSDISK0_STATUS_FAILED_REQUESTED)
				printf(", Failed Requested");
			printf("\n");
		} else
			printf("\t\tNot Available.\n");
	}

	freeAllocMem();
	return;
}

void GetResyncPercentage(RaidVol0PhysDisk_t *pDisk, uchar *pVol,int count)
{
	MpiRaidActionRequest_t	*pRequest;
	uint			blks_done;
	uint 			numBytes;
	int			ii;
	uint			tot_blks, blks_left;
	int  			status;

	numBytes = (sizeof(MpiRaidActionRequest_t) - sizeof(SGE_IO_UNION))
						+ sizeof (SGESimple64_t);

	if ((mpiBlkPtr = allocIoctlBlk(numBytes)) == NULL)
		return;

	pRequest = (MpiRaidActionRequest_t *) mpiBlkPtr->MF;
	mpiBlkPtr->dataSgeOffset = (sizeof (MpiRaidActionRequest_t) - sizeof(SGE_IO_UNION))/4;

	/* Initialize data in/data out sizes: Change below if need to */
	mpiBlkPtr->dataInSize = mpiBlkPtr->dataOutSize = 0;

	/* Populate the RAID Volume Request
	 */
	pRequest->Action       = MPI_RAID_ACTION_INDICATOR_STRUCT;
	pRequest->Function     = MPI_FUNCTION_RAID_ACTION;
	pRequest->MsgContext   = -1;
	pRequest->ActionDataWord  = cpu_to_le32(0); /* action data is 0 */

	printf("Resync Information:\n");
	for (ii = 0; ii < count; ii++ )
	{
		pRequest->VolumeID     = (u8) pVol[ii];
		pRequest->PhysDiskNum  = pDisk[ii].PhysDiskNum;

		status = IssueMptCommand(MPT_FLAGS_KEEP_MEM);
		if (status == 0) {
			uint *pdata = (uint *) mpiBlkPtr->replyFrameBufPtr;
			pdata += 6;

			tot_blks = cpu_to_le32(*pdata);
			pdata++;
			pdata++;
			blks_left = cpu_to_le32(*pdata);
			pdata++;

			blks_done = tot_blks - blks_left;

			printf("\tPhys Disk %d on Volume %d:\n\t\t0x%x of 0x%x Remaining,",
				pDisk[ii].PhysDiskNum, pVol[ii], 
				blks_left, tot_blks );

			printf(" Done 0x%x, (%d%%) Complete \n",
				blks_done, ((blks_done >> 6) * 100) / (tot_blks >> 6));
		}
	}

	freeAllocMem();
	return;
}
