#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <linux/compiler.h>
#include <errno.h>
#include <stdint.h>
#include <stdlib.h>
#include <signal.h>
#include <string.h>
#include <getopt.h>

#include <endian.h>
#include <linux/types.h>
#if __BYTE_ORDER == __BIG_ENDIAN
#include <linux/byteorder/big_endian.h>
#endif
#if __BYTE_ORDER == __LITTLE_ENDIAN
#include <linux/byteorder/little_endian.h>
#endif

#include "lsi/mpi_type.h"
#include "lsi/mpi.h"
#include "lsi/mpi_ioc.h"
#include "lsi/mpi_cnfg.h"
#include "lsi/mpi_init.h"
#include "lsi/mpi_fc.h"
#include "lsi/mpi_sas.h"
#include "lsi/mpi_raid.h"
#include "lsi/mpi_tool.h"
typedef U8 u8;
typedef U16 u16;
typedef U32 u32;
#include "mptctl.h"

#define MPTCTL_EVENT_LOG_SIZE         (0x00000032)

static int		driver_hndl;
static int		clear_event_log=1; /* event log clean ?? : 0 = false, 1=true */
static u32		event_context=0;

#define DEVICE_NODE	"/dev/mptctl"

static void aen_handler(int signal);
static void event_enable(void);
static void event_report(void);
static void get_event_context(void);
static void EventDescriptionStr(u8 event, u32 evData0, char *evStr);

int
main(int argc, char **argv)
{
	int	oflags;

	printf("open %s\n",DEVICE_NODE);
	if ((driver_hndl = open(DEVICE_NODE, O_RDONLY)) == -1) {
		perror(DEVICE_NODE);
		return errno;
	}
	signal(SIGIO, &aen_handler);
	if(fcntl(driver_hndl, F_SETOWN, getpid()) == -1){
		perror("AEN: registration, F_SETOWN");
		goto out;
	}
	oflags = fcntl(driver_hndl, F_GETFL);
	if(fcntl(driver_hndl, F_SETFL, oflags | FASYNC) == -1 ) {
		perror("AEN: registration, F_SETFL");
		goto out;
	}

	event_enable(); /*enable event monitoring*/
	get_event_context(); /* figure out what the current event context id is */

	printf("event_context=%x\n\n",event_context);
	printf("CNTR-C to exit\n");
	while(1) sleep(10);
out:
	close(driver_hndl);

	return 0;
}

static void
aen_handler(int signal)
{
	event_report(); /*dump event data*/
}

static void
event_enable(void)
{
	struct mpt_ioctl_eventenable  thisBuf;
	int  sz = sizeof (struct mpt_ioctl_eventenable);
	u8 action=0;

	memset(&thisBuf, 0, sz);

	/*
	 * Complete the header.
	 */
	thisBuf.hdr.iocnum = 0; /* hard coded to IOC Number = 0 */
	thisBuf.hdr.port = 0;
	thisBuf.hdr.maxDataSize = sz;

	action = 0;
	thisBuf.eventTypes =  0;

	/* set events to monitor */
	thisBuf.eventTypes |=  (1 << MPI_EVENT_LOG_DATA);
	thisBuf.eventTypes |=  (1 << MPI_EVENT_STATE_CHANGE);
	thisBuf.eventTypes |=  (1 << MPI_EVENT_UNIT_ATTENTION);
	thisBuf.eventTypes |=  (1 << MPI_EVENT_IOC_BUS_RESET);
	thisBuf.eventTypes |=  (1 << MPI_EVENT_EXT_BUS_RESET);
	thisBuf.eventTypes |=  (1 << MPI_EVENT_RESCAN);
	thisBuf.eventTypes |=  (1 << MPI_EVENT_LINK_STATUS_CHANGE);
	thisBuf.eventTypes |=  (1 << MPI_EVENT_LOOP_STATE_CHANGE);
	thisBuf.eventTypes |=  (1 << MPI_EVENT_LOGOUT);
	thisBuf.eventTypes |=  (1 << MPI_EVENT_EVENT_CHANGE);
	thisBuf.eventTypes |=  (1 << MPI_EVENT_INTEGRATED_RAID);
	thisBuf.eventTypes |=  (1 << MPI_EVENT_SCSI_DEVICE_STATUS_CHANGE);
	thisBuf.eventTypes |=  (1 << MPI_EVENT_SAS_DEVICE_STATUS_CHANGE);
	thisBuf.eventTypes |=  (1 << MPI_EVENT_ON_BUS_TIMER_EXPIRED);
	thisBuf.eventTypes |=  (1 << MPI_EVENT_QUEUE_FULL);
	thisBuf.eventTypes |=  (1 << MPI_EVENT_SAS_DEVICE_STATUS_CHANGE);
	thisBuf.eventTypes |=  (1 << MPI_EVENT_SAS_SES);
	thisBuf.eventTypes |=  (1 << MPI_EVENT_PERSISTENT_TABLE_FULL);
	thisBuf.eventTypes |=  (1 << MPI_EVENT_SAS_PHY_LINK_STATUS);
	thisBuf.eventTypes |=  (1 << MPI_EVENT_SAS_DISCOVERY_ERROR);

	if (ioctl(driver_hndl, (unsigned long) MPTEVENTENABLE,
		(char *) &thisBuf) != 0)
		perror("MPTEVENTENABLE ioctl failed");
	else {
		printf("Event Enable:\n");
		printf("==========================\n");
		printf("eventTypes %08x\n", thisBuf.eventTypes);
	}

	return;
}

static void
event_report(void)
{
	struct mpt_ioctl_eventreport  *pthisBuf;
	char *pmem;
	MPT_IOCTL_EVENTS  *pinfo;
	int  jj, sz;
	char evStr[100];

	sz = MPTCTL_EVENT_LOG_SIZE * sizeof(MPT_IOCTL_EVENTS) +
	       	sizeof(mpt_ioctl_header);
	if ((pmem = (char *) malloc (sz)) == NULL) {
		printf("EventReport: Error! No memory.\n");
		return;
	}

	pthisBuf = (struct mpt_ioctl_eventreport *)pmem;
	memset(pthisBuf, 0, sz);

	/* Complete the header.
	 */
	pthisBuf->hdr.iocnum = 0; /* hard coded to IOC Number = 0 */
	pthisBuf->hdr.port = 0;
	pthisBuf->hdr.maxDataSize = sz;

	if (ioctl(driver_hndl, (unsigned long) MPTEVENTREPORT,
	       	(char *) pthisBuf) != 0)
		perror("MPTEVENTREPORT ioctl failed");
	else {
		pinfo = pthisBuf->eventData;
		for (jj = 0; jj < MPTCTL_EVENT_LOG_SIZE; jj++, pinfo++) {
			if((!pinfo->event) &&
			   (!pinfo->eventContext) &&
			   (!pinfo->data[0]) &&
			   (!pinfo->data[1]))
			   continue;
			if(clear_event_log==1) {
				clear_event_log=0;
			}else if ( pinfo->eventContext != event_context) {
				continue;
			}
			EventDescriptionStr(pinfo->event,
				pinfo->data[0], evStr);
			printf("MPT event (%s=%02Xh) detected!\n",
				evStr, pinfo->event);
			printf("event context = %08x \n",
				pinfo->eventContext);
			printf("event data    = %08x\t%08x\n\n",
			    __le32_to_cpu(pinfo->data[0]),
			    __le32_to_cpu(pinfo->data[1]));

			event_context++; 
			/* TODO - 
			 * (1) handle wrapping buffer
			 * (2) handle event_context when it reach's u32
			 */
		}
	}
	free(pmem);
	printf("\n");
	return;
}

static void
get_event_context(void)
{
	struct mpt_ioctl_eventreport  *pthisBuf;
	char *pmem;
	MPT_IOCTL_EVENTS  *pinfo;
	int  jj, sz;

	clear_event_log=0;

	sz = MPTCTL_EVENT_LOG_SIZE * sizeof(MPT_IOCTL_EVENTS) +
	       	sizeof(mpt_ioctl_header);
	if ((pmem = (char *) malloc (sz)) == NULL) {
		printf("EventReport: Error! No memory.\n");
		return;
	}

	pthisBuf = (struct mpt_ioctl_eventreport *)pmem;
	memset(pthisBuf, 0, sz);

	/* Complete the header.
	 */
	pthisBuf->hdr.iocnum = 0; /* hard coded to IOC Number = 0 */
	pthisBuf->hdr.port = 0;
	pthisBuf->hdr.maxDataSize = sz;

	if (ioctl(driver_hndl, (unsigned long) MPTEVENTREPORT,
	       	(char *) pthisBuf) != 0)
		perror("MPTEVENTREPORT ioctl failed");
	else {
		pinfo = pthisBuf->eventData;
		for (jj = 0; jj < MPTCTL_EVENT_LOG_SIZE; jj++, pinfo++) {
			if((!pinfo->event) &&
			   (!pinfo->eventContext) &&
			   (!pinfo->data[0]) &&
			   (!pinfo->data[1]))
			   continue;
			clear_event_log=0;
			event_context = (pinfo->eventContext > event_context) ?
				pinfo->eventContext : event_context;
		}
	}
	free(pmem);
	return;
}

static void
EventDescriptionStr(u8 event, u32 evData0, char *evStr)
{
	char *ds;

	switch(event) {
	case MPI_EVENT_NONE:
		ds = "None";
		break;
	case MPI_EVENT_LOG_DATA:
		ds = "Log Data";
		break;
	case MPI_EVENT_STATE_CHANGE:
		ds = "State Change";
		break;
	case MPI_EVENT_UNIT_ATTENTION:
		ds = "Unit Attention";
		break;
	case MPI_EVENT_IOC_BUS_RESET:
		ds = "IOC Bus Reset";
		break;
	case MPI_EVENT_EXT_BUS_RESET:
		ds = "External Bus Reset";
		break;
	case MPI_EVENT_RESCAN:
		ds = "Bus Rescan Event";
		/* Ok, do we need to do anything here? As far as
		   I can tell, this is when a new device gets added
		   to the loop. */
		break;
	case MPI_EVENT_LINK_STATUS_CHANGE:
		if (evData0 == MPI_EVENT_LINK_STATUS_FAILURE)
			ds = "Link Status(FAILURE) Change";
		else
			ds = "Link Status(ACTIVE) Change";
		break;
	case MPI_EVENT_LOOP_STATE_CHANGE:
		if (evData0 == MPI_EVENT_LOOP_STATE_CHANGE_LIP)
			ds = "Loop State(LIP) Change";
		else if (evData0 == MPI_EVENT_LOOP_STATE_CHANGE_LPE)
			ds = "Loop State(LPE) Change";			/* ??? */
		else
			ds = "Loop State(LPB) Change";			/* ??? */
		break;
	case MPI_EVENT_LOGOUT:
		ds = "Logout";
		break;
	case MPI_EVENT_EVENT_CHANGE:
		if (evData0)
			ds = "Events(ON) Change";
		else
			ds = "Events(OFF) Change";
		break;
	case MPI_EVENT_INTEGRATED_RAID:
	{
		u8 ReasonCode = (u8)(evData0 >> 16);
		switch (ReasonCode) {
		case MPI_EVENT_RAID_RC_VOLUME_CREATED :
			ds = "Integrated Raid: Volume Created";
			break;
		case MPI_EVENT_RAID_RC_VOLUME_DELETED :
			ds = "Integrated Raid: Volume Deleted";
			break;
		case MPI_EVENT_RAID_RC_VOLUME_SETTINGS_CHANGED :
			ds = "Integrated Raid: Volume Settings Changed";
			break;
		case MPI_EVENT_RAID_RC_VOLUME_STATUS_CHANGED :
			ds = "Integrated Raid: Volume Status Changed";
			break;
		case MPI_EVENT_RAID_RC_VOLUME_PHYSDISK_CHANGED :
			ds = "Integrated Raid: Volume Physdisk Changed";
			break;
		case MPI_EVENT_RAID_RC_PHYSDISK_CREATED :
			ds = "Integrated Raid: Physdisk Created";
			break;
		case MPI_EVENT_RAID_RC_PHYSDISK_DELETED :
			ds = "Integrated Raid: Physdisk Deleted";
			break;
		case MPI_EVENT_RAID_RC_PHYSDISK_SETTINGS_CHANGED :
			ds = "Integrated Raid: Physdisk Settings Changed";
			break;
		case MPI_EVENT_RAID_RC_PHYSDISK_STATUS_CHANGED :
			ds = "Integrated Raid: Physdisk Status Changed";
			break;
		case MPI_EVENT_RAID_RC_DOMAIN_VAL_NEEDED :
			ds = "Integrated Raid: Domain Validation Needed";
			break;
		case MPI_EVENT_RAID_RC_SMART_DATA :
			ds = "Integrated Raid; Smart Data";
			break;
		case MPI_EVENT_RAID_RC_REPLACE_ACTION_STARTED :
			ds = "Integrated Raid: Replace Action Started";
			break;
		default:
			ds = "Integrated Raid";
		break;
		}
		break;
	}
	case MPI_EVENT_SCSI_DEVICE_STATUS_CHANGE:
		ds = "SCSI Device Status Change";
		break;
	case MPI_EVENT_SAS_DEVICE_STATUS_CHANGE:
	{
		u8 ReasonCode = (u8)(evData0 >> 16);
		switch (ReasonCode) {
		case MPI_EVENT_SAS_DEV_STAT_RC_ADDED:
			ds = "SAS Device Status Change: Added";
			break;
		case MPI_EVENT_SAS_DEV_STAT_RC_NOT_RESPONDING:
			ds = "SAS Device Status Change: Deleted";
			break;
		case MPI_EVENT_SAS_DEV_STAT_RC_SMART_DATA:
			ds = "SAS Device Status Change: SMART Data";
			break;
		case MPI_EVENT_SAS_DEV_STAT_RC_NO_PERSIST_ADDED:
			ds = "SAS Device Status Change: No Persistancy Added";
			break;
		default:
			ds = "SAS Device Status Change: Unknown";
		break;
		}
		break;
	}
	case MPI_EVENT_ON_BUS_TIMER_EXPIRED:
		ds = "Bus Timer Expired";
		break;
	case MPI_EVENT_QUEUE_FULL:
		ds = "Queue Full";
		break;
	case MPI_EVENT_SAS_SES:
		ds = "SAS SES Event";
		break;
	case MPI_EVENT_PERSISTENT_TABLE_FULL:
		ds = "Persistent Table Full";
		break;
	case MPI_EVENT_SAS_PHY_LINK_STATUS:
		ds = "SAS PHY Link Status";
		break;
	case MPI_EVENT_SAS_DISCOVERY_ERROR:
		ds = "SAS Discovery Error";
		break;

	/*
	 *  MPT base "custom" events may be added here...
	 */
	default:
		ds = "Unknown";
		break;
	}
	strcpy(evStr,ds);
}
