/*
    i801.c - Part of lm_sensors, Linux kernel modules for hardware
              monitoring
    Copyright (c) 1998 - 2002  Frodo Looijaard <frodol@dds.nl>,
    Philip Edelbrock <phil@netroedge.com>, and Mark D. Studebaker
    <mdsxyz123@yahoo.com>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

/*
    SUPPORTED DEVICES	PCI ID
    82801AA		2413           
    82801AB		2423           
    82801BA		2443           
    82801CA/CAM		2483           
    82801DB		24C3   (HW PEC supported, 32 byte buffer not supported)
    82801EB		24D3   (HW PEC supported, 32 byte buffer not supported)
    6300ESB		25A4
    ICH6		266A
    ICH7		27DA
    ESB2		269B
    This driver supports several versions of Intel's I/O Controller Hubs (ICH).
    For SMBus support, they are similar to the PIIX4 and are part
    of Intel's '810' and other chipsets.
    See the doc/busses/i2c-i801 file for details.
    I2C Block Read and Process Call are not supported.
*/

/* Note: we assume there can only be one I801, with one SMBus interface */

/* Another note: This interface is extremely sensitive to timing and
   failure handling.  If you don't wait at least one jiffie after
   starting the transaction before checking things, you will screw it
   up.  If you don't wait a jiffie after the final check, you will
   screw it up.  If you screw it up by these manners or by abandoning
   an operation in progress, the I2C bus is likely stuck and won't
   work any more.  Gotta love this hardware. */

#include <linux/module.h>
#include <linux/pci.h>
#include <linux/kernel.h>
#include <linux/stddef.h>
#include <linux/delay.h>
#include <linux/sched.h>
#include <linux/ioport.h>
#include <linux/init.h>
#include <linux/i2c.h>
#include <asm/io.h>

#ifdef I2C_FUNC_SMBUS_BLOCK_DATA_PEC
#define HAVE_PEC
#endif

/* I801 SMBus address offsets */
#define SMBHSTSTS	(0 + i801_smba)
#define SMBHSTCNT	(2 + i801_smba)
#define SMBHSTCMD	(3 + i801_smba)
#define SMBHSTADD	(4 + i801_smba)
#define SMBHSTDAT0	(5 + i801_smba)
#define SMBHSTDAT1	(6 + i801_smba)
#define SMBBLKDAT	(7 + i801_smba)
#define SMBPEC		(8 + i801_smba)	/* ICH4 only */
#define SMBAUXSTS	(12 + i801_smba)	/* ICH4 only */
#define SMBAUXCTL	(13 + i801_smba)	/* ICH4 only */

/* PCI Address Constants */
#define SMBBA		0x020
#define SMBHSTCFG	0x040
#define SMBREV		0x008

/* Host configuration bits for SMBHSTCFG */
#define SMBHSTCFG_HST_EN	1
#define SMBHSTCFG_SMB_SMI_EN	2
#define SMBHSTCFG_I2C_EN	4

/* Other settings */
#define MAX_TIMEOUT_US		100000
#define RETRY_TIME_US		500 /* Retry minimum is 500us */
#define ENABLE_INT9		0	/* set to 0x01 to enable - untested */

/* I801 command constants */
#define I801_QUICK		0x00
#define I801_BYTE		0x04
#define I801_BYTE_DATA		0x08
#define I801_WORD_DATA		0x0C
#define I801_PROC_CALL		0x10	/* later chips only, unimplemented */
#define I801_BLOCK_DATA		0x14
#define I801_I2C_BLOCK_DATA	0x18	/* unimplemented */
#define I801_BLOCK_LAST		0x34
#define I801_I2C_BLOCK_LAST	0x38	/* unimplemented */
#define I801_START		0x40
#define I801_PEC_EN		0x80	/* ICH4 only */

/* insmod parameters */

/* If force_addr is set to anything different from 0, we forcibly enable
   the I801 at the given address. VERY DANGEROUS! */
static u16 force_addr;
module_param(force_addr, ushort, 0);
MODULE_PARM_DESC(force_addr,
		 "Forcibly enable the I801 at the given address. "
		 "EXTREMELY DANGEROUS!");

static unsigned short i801_smba;
static struct pci_dev *I801_dev;
static int isich4;

struct i801_i2c_data
{
	int i;
	int len;
	unsigned char hostc;
	int block;
	int hwpec;
	int xact;
	int hststs;
	int wait_intr;
	int finished;
};
struct i801_i2c_data i801_data;

static int i801_setup(struct pci_dev *dev)
{
	int error_return = 0;
	unsigned char temp;

	/* Note: we keep on searching until we have found 'function 3' */
	if (PCI_FUNC(dev->devfn) != 3)
		return -ENODEV;

	I801_dev = dev;
	if ((dev->device == PCI_DEVICE_ID_INTEL_82801DB_3) ||
	    (dev->device == PCI_DEVICE_ID_INTEL_82801EB_3) ||
	    (dev->device == PCI_DEVICE_ID_INTEL_ESB_4))
		isich4 = 1;
	else
		isich4 = 0;

	/* Determine the address of the SMBus areas */
	if (force_addr) {
		i801_smba = force_addr & 0xfff0;
	} else {
		pci_read_config_word(I801_dev, SMBBA, &i801_smba);
		i801_smba &= 0xfff0;
		if (i801_smba == 0) {
			dev_err(&dev->dev, "SMB base address uninitialized "
				"- upgrade BIOS or use force_addr=0xaddr\n");
			return -ENODEV;
		}
	}

	if (!request_region(i801_smba, (isich4 ? 16 : 8), "i801-smbus")) {
		dev_err(&dev->dev, "I801_smb region 0x%x already in use!\n",
			i801_smba);
		error_return = -EBUSY;
		goto END;
	}

	pci_read_config_byte(I801_dev, SMBHSTCFG, &temp);
	temp &= ~SMBHSTCFG_I2C_EN;	/* SMBus timing */
	pci_write_config_byte(I801_dev, SMBHSTCFG, temp);

	/* If force_addr is set, we program the new address here. Just to make
	   sure, we disable the device first. */
	if (force_addr) {
		pci_write_config_byte(I801_dev, SMBHSTCFG, temp & 0xfe);
		pci_write_config_word(I801_dev, SMBBA, i801_smba);
		pci_write_config_byte(I801_dev, SMBHSTCFG, temp | 0x01);
		dev_warn(&dev->dev, "WARNING: I801 SMBus interface set to "
			"new address %04x!\n", i801_smba);
	} else if ((temp & 1) == 0) {
		pci_write_config_byte(I801_dev, SMBHSTCFG, temp | 1);
		dev_warn(&dev->dev, "enabling SMBus device\n");
	}

	if (temp & 0x02)
		dev_dbg(&dev->dev, "I801 using Interrupt SMI# for SMBus.\n");
	else
		dev_dbg(&dev->dev, "I801 using PCI Interrupt for SMBus.\n");

	pci_read_config_byte(I801_dev, SMBREV, &temp);
	dev_dbg(&dev->dev, "SMBREV = 0x%X\n", temp);
	dev_dbg(&dev->dev, "I801_smba = 0x%X\n", i801_smba);

END:
	return error_return;
}

static void i801_check_hststs(struct i2c_adapter *adap,
			      struct i2c_op_q_entry *entry,
			      struct i801_i2c_data *d)
{
	if (d->hststs & 0x10) {
		entry->result = -EIO;
		dev_dbg(&I801_dev->dev,
			"Error: Failed bus transaction\n");
	} else if (d->hststs & 0x08) {
		entry->result = -EIO;
		dev_err(&I801_dev->dev, "Bus collision!\n");
		/* Clock stops and slave is stuck in mid-transmission */
	} else if (d->hststs & 0x04) {
		entry->result = -EIO;
		dev_dbg(&I801_dev->dev, "Error: no response!\n");
	}
}

static void i801_finish(struct i2c_adapter *adap,
			struct i2c_op_q_entry *entry,
			struct i801_i2c_data *d)
{
	d->finished = 1;

#ifdef HAVE_PEC
	if (isich4 && d->hwpec) {
		if (entry->smbus.size != I2C_SMBUS_QUICK &&
		   entry->smbus.size != I2C_SMBUS_I2C_BLOCK_DATA)
			outb_p(0, SMBAUXCTL);
	}
#endif

	if (d->block || (entry->result < 0) ||
	   ((entry->smbus.read_write == I2C_SMBUS_WRITE)
	    || (d->xact == I801_QUICK)))
		return;

	switch (d->xact & 0x7f) {
	case I801_BYTE:	/* Result put in SMBHSTDAT0 */
	case I801_BYTE_DATA:
		entry->smbus.data->byte = inb_p(SMBHSTDAT0);
		break;
	case I801_WORD_DATA:
		entry->smbus.data->word = inb_p(SMBHSTDAT0)
		    + (inb_p(SMBHSTDAT1) << 8);
		break;
	}
}

static void i801_transaction_final_check(struct i2c_adapter *adap,
					 struct i2c_op_q_entry *entry,
					 struct i801_i2c_data *d)
{
	i801_check_hststs(adap, entry, d);

	if ((inb_p(SMBHSTSTS) & 0x1f) != 0x00)
		outb_p(inb(SMBHSTSTS), SMBHSTSTS);

	if ((d->hststs = (0x1f & inb_p(SMBHSTSTS))) != 0x00) {
		dev_dbg(&I801_dev->dev, "Failed reset at end of transaction "
			"(%02x)\n", d->hststs);
	}
	dev_dbg(&I801_dev->dev, "Transaction (post): CNT=%02x, CMD=%02x, "
		"ADD=%02x, DAT0=%02x, DAT1=%02x\n", inb_p(SMBHSTCNT),
		inb_p(SMBHSTCMD), inb_p(SMBHSTADD), inb_p(SMBHSTDAT0),
		inb_p(SMBHSTDAT1));
}

static void i801_transaction_poll(struct i2c_adapter *adap,
				  struct i2c_op_q_entry *entry,
				  struct i801_i2c_data *d)
{
	d->hststs = inb_p(SMBHSTSTS);
	if (!(d->hststs & 0x01)) {
		i801_transaction_final_check(adap, entry, d);
		i801_finish(adap, entry, d);
	} else if (entry->time_left <= 0) {
		dev_dbg(&I801_dev->dev, "SMBus Timeout!\n");
		entry->result = -EIO;
		i801_transaction_final_check(adap, entry, d);
	}
}

static void i801_transaction_start(struct i2c_adapter *adap,
				   struct i2c_op_q_entry *entry,
				   struct i801_i2c_data *d)
{
	dev_dbg(&I801_dev->dev, "Transaction (pre): CNT=%02x, CMD=%02x, "
		"ADD=%02x, DAT0=%02x, DAT1=%02x\n", inb_p(SMBHSTCNT),
		inb_p(SMBHSTCMD), inb_p(SMBHSTADD), inb_p(SMBHSTDAT0),
		inb_p(SMBHSTDAT1));

	/* Make sure the SMBus host is ready to start transmitting */
	/* 0x1f = Failed, Bus_Err, Dev_Err, Intr, Host_Busy */
	if ((d->hststs = (0x1f & inb_p(SMBHSTSTS))) != 0x00) {
		dev_dbg(&I801_dev->dev, "SMBus busy (%02x). Resetting...\n",
			d->hststs);
		outb_p(d->hststs, SMBHSTSTS);
		if ((d->hststs = (0x1f & inb_p(SMBHSTSTS))) != 0x00) {
			dev_dbg(&I801_dev->dev, "Failed! (%02x)\n", d->hststs);
			entry->result = -EIO;
			return;
		} else {
			dev_dbg(&I801_dev->dev, "Successfull!\n");
		}
	}

	outb_p(inb(SMBHSTCNT) | I801_START, SMBHSTCNT);
}

static void i801_block_finish(struct i2c_adapter *adap,
			      struct i2c_op_q_entry *entry,
			      struct i801_i2c_data *d)
{
	if (entry->smbus.size == I2C_SMBUS_I2C_BLOCK_DATA) {
		/* restore saved configuration register value */
		pci_write_config_byte(I801_dev, SMBHSTCFG, d->hostc);
	}

	i801_finish(adap, entry, d);
}

static void i801_block_poll_wait_intr(struct i2c_adapter *adap,
				      struct i2c_op_q_entry *entry,
				      struct i801_i2c_data *d)
{
#ifdef HAVE_PEC
	if (entry->result >= 0 &&
	   isich4 &&
	   entry->smbus.size == I2C_SMBUS_BLOCK_DATA_PEC)
	{
		/* wait for INTR bit as advised by Intel */
		d->hststs = inb_p(SMBHSTSTS);
		if (d->hststs & 0x02) {
			outb_p(d->hststs, SMBHSTSTS); 
			i801_block_finish(adap, entry, d);
		} else if (entry->time_left <= 0) {
			/* Timed out */
			outb_p(d->hststs, SMBHSTSTS); 
			entry->result = -EIO;
			dev_dbg(&I801_dev->dev, "PEC Timeout!\n");
		}
	} else
#endif
		i801_block_finish(adap, entry, d);
}

static void i801_block_next_byte(struct i2c_adapter *adap,
				 struct i2c_op_q_entry *entry,
				 struct i801_i2c_data *d)
{
	int smbcmd;
	unsigned char errmask;

	if (d->i > d->len) {
		d->wait_intr = 1;
		entry->time_left = MAX_TIMEOUT_US;
		i801_block_poll_wait_intr(adap, entry, d);
		return;
	}

	if (d->i == d->len && entry->smbus.read_write == I2C_SMBUS_READ)
		smbcmd = I801_BLOCK_LAST;
	else
		smbcmd = I801_BLOCK_DATA;
	outb_p(smbcmd | ENABLE_INT9, SMBHSTCNT);

	dev_dbg(&I801_dev->dev, "Block (pre %d): CNT=%02x, CMD=%02x, "
		"ADD=%02x, DAT0=%02x, BLKDAT=%02x\n", d->i,
		inb_p(SMBHSTCNT), inb_p(SMBHSTCMD), inb_p(SMBHSTADD),
		inb_p(SMBHSTDAT0), inb_p(SMBBLKDAT));

	/* Make sure the SMBus host is ready to start transmitting */
	d->hststs = inb_p(SMBHSTSTS);
	if (d->i == 1) {
		/* Erronenous conditions before transaction: 
		 * Byte_Done, Failed, Bus_Err, Dev_Err, Intr, Host_Busy */
		errmask=0x9f; 
	} else {
		/* Erronenous conditions during transaction: 
		 * Failed, Bus_Err, Dev_Err, Intr */
		errmask=0x1e; 
	}
	if (d->hststs & errmask) {
		dev_dbg(&I801_dev->dev, "SMBus busy (%02x). "
			"Resetting...\n", d->hststs);
		outb_p(d->hststs, SMBHSTSTS);
		if (((d->hststs = inb_p(SMBHSTSTS)) & errmask) != 0x00) {
			dev_err(&I801_dev->dev,
				"Reset failed! (%02x)\n", d->hststs);
			entry->result = -EIO;
			return;
		}
		if (d->i != 1) {
			/* if die in middle of block transaction, fail */
			entry->result = -EIO;
			return;
		}
	}

	if (d->i == 1)
		outb_p(inb(SMBHSTCNT) | I801_START, SMBHSTCNT);
}

/* Called on timer ticks.  This checks the result of the
   transaction. */
static void i801_block_poll(struct i2c_adapter *adap,
			    struct i2c_op_q_entry *entry,
			    struct i801_i2c_data *d)
{
	d->hststs = inb_p(SMBHSTSTS);
	if (!(d->hststs & 0x80)) {
		/* Not ready yet */
		if (entry->time_left <= 0) {
			dev_dbg(&I801_dev->dev, "SMBus Timeout!\n");
			entry->result = -EIO;
		}
		return;
	}

	i801_check_hststs(adap, entry, d);
	if (d->i == 1 && entry->smbus.read_write == I2C_SMBUS_READ) {
		d->len = inb_p(SMBHSTDAT0);
		if (d->len < 1)
			d->len = 1;
		if (d->len > 32)
			d->len = 32;
		entry->smbus.data->block[0] = d->len;
	}

	/* Retrieve/store value in SMBBLKDAT */
	if (entry->smbus.read_write == I2C_SMBUS_READ)
		entry->smbus.data->block[d->i] = inb_p(SMBBLKDAT);
	if (entry->smbus.read_write == I2C_SMBUS_WRITE && d->i+1 <= d->len)
		outb_p(entry->smbus.data->block[d->i+1], SMBBLKDAT);
	if ((d->hststs & 0x9e) != 0x00)
		outb_p(d->hststs, SMBHSTSTS);  /* signals SMBBLKDAT ready */
	
	if ((d->hststs = (0x1e & inb_p(SMBHSTSTS))) != 0x00) {
		dev_dbg(&I801_dev->dev,
			"Bad status (%02x) at end of transaction\n",
			d->hststs);
	}
	dev_dbg(&I801_dev->dev, "Block (post %d): CNT=%02x, CMD=%02x, "
		"ADD=%02x, DAT0=%02x, BLKDAT=%02x\n", d->i,
		inb_p(SMBHSTCNT), inb_p(SMBHSTCMD), inb_p(SMBHSTADD),
		inb_p(SMBHSTDAT0), inb_p(SMBBLKDAT));

	if (entry->result)
		return;

	(d->i)++;
	i801_block_next_byte(adap, entry, d);
}

static void i801_block_start(struct i2c_adapter *adap,
			     struct i2c_op_q_entry *entry,
			     struct i801_i2c_data *d)
{
	if (entry->smbus.read_write == I2C_SMBUS_WRITE) {
		d->len = entry->smbus.data->block[0];
		if (d->len < 1)
			d->len = 1;
		if (d->len > 32)
			d->len = 32;
		outb_p(d->len, SMBHSTDAT0);
		outb_p(entry->smbus.data->block[1], SMBBLKDAT);
	} else {
		d->len = 32;	/* max for reads */
	}

	if(isich4 && entry->smbus.size != I2C_SMBUS_I2C_BLOCK_DATA) {
		/* set 32 byte buffer */
	}

	d->i = 1;
	i801_block_next_byte(adap, entry, d);
}

/* General poll routine.  Called periodically by the i2c code. */
static void i801_poll(struct i2c_adapter *adap,
		      struct i2c_op_q_entry *entry,
		      unsigned int us_since_last_poll)
{
	struct i801_i2c_data *d = entry->data;

	dev_dbg(&I801_dev->dev, "Poll call for %p %p at %ld\n", adap, entry,
		jiffies);

	if (!d)
		/* The entry hasn't been started yet. */
		return;

	if (d->finished) {
		/* We delay an extra poll to keep the hardware happy.
		   Otherwise the hardware is not ready when we start
		   the next operation. */
		i2c_op_done(adap, entry);
		return;
	}

	/* Decrement timeout */
	entry->time_left -= us_since_last_poll;

	/* Wait a jiffie normally. */
	entry->call_again_us = RETRY_TIME_US;

	if (d->block) {
		if (d->wait_intr) {
			i801_block_poll_wait_intr(adap, entry, d);
		} else {
			i801_block_poll(adap, entry, d);
		}
		if (entry->result < 0)
			/* Error, finish the transaction */
			i801_block_finish(adap, entry, d);
	} else {
		i801_transaction_poll(adap, entry, d);
		if (entry->result < 0)
			/* Error, finish the transaction */
			i801_finish(adap, entry, d);
	}
}

/* Start a general SMBUS transaction on the i801.  Figure out what
   kind of transaction it is, set it up, and start it. */
static void i801_start(struct i2c_adapter *adap,
		       struct i2c_op_q_entry *entry)
{
	struct i801_i2c_data *d = adap->algo_data;

	dev_dbg(&I801_dev->dev, "start call for %p %p at %ld\n", adap, entry,
		jiffies);

	if (entry->smbus.size == I2C_SMBUS_I2C_BLOCK_DATA) {
		if (entry->smbus.read_write == I2C_SMBUS_WRITE) {
			/* set I2C_EN bit in configuration register */
			pci_read_config_byte(I801_dev, SMBHSTCFG, &d->hostc);
			pci_write_config_byte(I801_dev, SMBHSTCFG,
					      d->hostc | SMBHSTCFG_I2C_EN);
		} else {
			dev_err(&I801_dev->dev,
				"I2C_SMBUS_I2C_BLOCK_READ not DB!\n");
			entry->result = -EINVAL;
			return;
		}
	}

	d->block = 0;
	d->hwpec = 0;
	d->xact = 0;
	d->wait_intr = 0;
	d->finished = 0;

#ifdef HAVE_PEC
	if (isich4)
		d->hwpec = (entry->smbus.flags & I2C_CLIENT_PEC) != 0;
#endif

	switch (entry->smbus.size) {
	case I2C_SMBUS_QUICK:
		outb_p(((entry->smbus.addr & 0x7f) << 1)
		       | (entry->smbus.read_write & 0x01),
		       SMBHSTADD);
		d->xact = I801_QUICK;
		break;
	case I2C_SMBUS_BYTE:
		outb_p(((entry->smbus.addr & 0x7f) << 1)
		       | (entry->smbus.read_write & 0x01),
		       SMBHSTADD);
		if (entry->smbus.read_write == I2C_SMBUS_WRITE)
			outb_p(entry->smbus.command, SMBHSTCMD);
		d->xact = I801_BYTE;
		break;
	case I2C_SMBUS_BYTE_DATA:
		outb_p(((entry->smbus.addr & 0x7f) << 1)
		       | (entry->smbus.read_write & 0x01),
		       SMBHSTADD);
		outb_p(entry->smbus.command, SMBHSTCMD);
		if (entry->smbus.read_write == I2C_SMBUS_WRITE)
			outb_p(entry->smbus.data->byte, SMBHSTDAT0);
		d->xact = I801_BYTE_DATA;
		break;
	case I2C_SMBUS_WORD_DATA:
		outb_p(((entry->smbus.addr & 0x7f) << 1)
		       | (entry->smbus.read_write & 0x01),
		       SMBHSTADD);
		outb_p(entry->smbus.command, SMBHSTCMD);
		if (entry->smbus.read_write == I2C_SMBUS_WRITE) {
			outb_p(entry->smbus.data->word & 0xff, SMBHSTDAT0);
			outb_p((entry->smbus.data->word & 0xff00) >> 8,
			       SMBHSTDAT1);
		}
		d->xact = I801_WORD_DATA;
		break;
	case I2C_SMBUS_BLOCK_DATA:
	case I2C_SMBUS_I2C_BLOCK_DATA:
#ifdef HAVE_PEC
	case I2C_SMBUS_BLOCK_DATA_PEC:
		if (d->hwpec && entry->smbus.size == I2C_SMBUS_BLOCK_DATA)
			entry->smbus.size = I2C_SMBUS_BLOCK_DATA_PEC;
#endif
		outb_p(((entry->smbus.addr & 0x7f) << 1)
		       | (entry->smbus.read_write & 0x01),
		       SMBHSTADD);
		outb_p(entry->smbus.command, SMBHSTCMD);
		d->block = 1;
		break;
	case I2C_SMBUS_PROC_CALL:
	default:
		dev_err(&I801_dev->dev, "Unsupported transaction %d\n",
			entry->smbus.size);
		entry->result = -EINVAL;
		return;
	}

#ifdef HAVE_PEC
	if (isich4 && d->hwpec) {
		if (entry->smbus.size != I2C_SMBUS_QUICK &&
		   entry->smbus.size != I2C_SMBUS_I2C_BLOCK_DATA)
			outb_p(1, SMBAUXCTL);	/* enable HW PEC */
	}
#endif
	if (d->block) {
		i801_block_start(adap, entry, d);
		if (entry->result < 0)
			/* Error, finish the transaction */
			i801_block_finish(adap, entry, d);
	} else {
		outb_p(d->xact | ENABLE_INT9, SMBHSTCNT);
		i801_transaction_start(adap, entry, d);
		if (entry->result < 0)
			/* Error, finish the transaction */
			i801_finish(adap, entry, d);
	}

	/* Wait extra long here, we want at least 2 ticks to guarantee
	   we wait >= 1 tick.  But we want to wait at least 100us no
	   matter what. */
#if (1000000 / HZ) < 100
	entry->call_again_us = 200;
#else
	entry->call_again_us = (1000000 / HZ) * 2;
#endif
	entry->time_left = MAX_TIMEOUT_US;

	if (d->finished)
		i2c_op_done(adap, entry);
	else
		entry->data = d;
}


static u32 i801_func(struct i2c_adapter *adapter)
{
	return I2C_FUNC_SMBUS_QUICK | I2C_FUNC_SMBUS_BYTE |
	    I2C_FUNC_SMBUS_BYTE_DATA | I2C_FUNC_SMBUS_WORD_DATA |
	    I2C_FUNC_SMBUS_BLOCK_DATA | I2C_FUNC_SMBUS_WRITE_I2C_BLOCK
#ifdef HAVE_PEC
	     | (isich4 ? I2C_FUNC_SMBUS_BLOCK_DATA_PEC |
	                 I2C_FUNC_SMBUS_HWPEC_CALC
	               : 0)
#endif
	    ;
}

static struct i2c_algorithm smbus_algorithm = {
	.smbus_start    = i801_start,
	.poll           = i801_poll,
	.functionality	= i801_func,
};

static struct i2c_adapter i801_adapter = {
	.owner		= THIS_MODULE,
	.class		= I2C_CLASS_HWMON,
	.algo		= &smbus_algorithm,
	.algo_data      = &i801_data,
	.name		= "unset",
};

static struct pci_device_id i801_ids[] = {
	{ PCI_DEVICE(PCI_VENDOR_ID_INTEL, PCI_DEVICE_ID_INTEL_82801AA_3) },
	{ PCI_DEVICE(PCI_VENDOR_ID_INTEL, PCI_DEVICE_ID_INTEL_82801AB_3) },
	{ PCI_DEVICE(PCI_VENDOR_ID_INTEL, PCI_DEVICE_ID_INTEL_82801BA_2) },
	{ PCI_DEVICE(PCI_VENDOR_ID_INTEL, PCI_DEVICE_ID_INTEL_82801CA_3) },
	{ PCI_DEVICE(PCI_VENDOR_ID_INTEL, PCI_DEVICE_ID_INTEL_82801DB_3) },
	{ PCI_DEVICE(PCI_VENDOR_ID_INTEL, PCI_DEVICE_ID_INTEL_82801EB_3) },
	{ PCI_DEVICE(PCI_VENDOR_ID_INTEL, PCI_DEVICE_ID_INTEL_ESB_4) },
	{ PCI_DEVICE(PCI_VENDOR_ID_INTEL, PCI_DEVICE_ID_INTEL_ICH6_16) },
	{ PCI_DEVICE(PCI_VENDOR_ID_INTEL, PCI_DEVICE_ID_INTEL_ICH7_17) },
	{ PCI_DEVICE(PCI_VENDOR_ID_INTEL, PCI_DEVICE_ID_INTEL_ESB2_17) },
	{ 0, }
};

MODULE_DEVICE_TABLE (pci, i801_ids);

static int __devinit i801_probe(struct pci_dev *dev,
				const struct pci_device_id *id)
{

	if (i801_setup(dev)) {
		dev_warn(&dev->dev,
			"I801 not detected, module not inserted.\n");
		return -ENODEV;
	}

	/* set up the driverfs linkage to our parent device */
	i801_adapter.dev.parent = &dev->dev;

	snprintf(i801_adapter.name, I2C_NAME_SIZE,
		"SMBus I801 adapter at %04x", i801_smba);
	return i2c_add_adapter(&i801_adapter);
}

static void __devexit i801_remove(struct pci_dev *dev)
{
	i2c_del_adapter(&i801_adapter);
	release_region(i801_smba, (isich4 ? 16 : 8));
}

static struct pci_driver i801_driver = {
	.name		= "i801_smbus",
	.id_table	= i801_ids,
	.probe		= i801_probe,
	.remove		= __devexit_p(i801_remove),
};

static int __init i2c_i801_init(void)
{
	return pci_register_driver(&i801_driver);
}

static void __exit i2c_i801_exit(void)
{
	pci_unregister_driver(&i801_driver);
}

MODULE_AUTHOR ("Frodo Looijaard <frodol@dds.nl>, "
		"Philip Edelbrock <phil@netroedge.com>, "
		"and Mark D. Studebaker <mdsxyz123@yahoo.com>");
MODULE_DESCRIPTION("I801 SMBus driver");
MODULE_LICENSE("GPL");

module_init(i2c_i801_init);
module_exit(i2c_i801_exit);
