
/****************************************************************************
 * testPKHA.c - Public key known-answer test for SEC2 device driver
 ****************************************************************************
 * Copyright (c) 2004-2005 Freescale Semiconductor
 * All Rights Reserved. 
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version
 * 2 of the License, or (at your option) any later version.
 ***************************************************************************/

/* Revision History:
 * 1.1.0 Dec 05,2004 sec - prep for linux-compatible driver release
 * 1.2   02-Feb-2005 sec - type fixes, buffer frees fixed
 * 1.3   Aug 14,2005
 */


#include "sec2drvTest.h"
#include "Sec2.h"  /* driver interfaces for this security core */


#define TESTPKHA_ADD
#define TESTPKHA_SUBTRACT
#define TESTPKHA_MULT
#define TESTPKHA_F2MMULT
#define TESTPKHA_RRMODP
#define TESTPKHA_EXPO
#define TESTPKHA_ECC_MULTK



int testEccMultkPtoQf2mAff(int fd);
int testEccFpPtMultkPtoQAff(int fd);
int testEccF2mPtMultkPtoQAff2(int fd);
int testEccFpPtMultkPtoQProj(int fd);
int testEccMultkPtoQf2mProj(int fd);
int pkha_exp_ss(int fd);


int pkhaTestStatus; /* overall test status */


static const unsigned char add_p0data[] = {
    0x00, 0x00, 0x00, 0x00, 0x45, 0x65, 0x45, 0x65,
    0x45, 0x65, 0x45, 0x65, 0x45, 0x65, 0x45, 0x65,
    0x45, 0x65, 0x45, 0x65, 0x45, 0x65, 0x45, 0x65,
    0x45, 0x65, 0x45, 0x65, 0x45, 0x65, 0x45, 0x65
};

static const unsigned char add_p1data[] = {
    0x00, 0x00, 0x00, 0x00, 0x22, 0x22, 0x22, 0x22,
    0x22, 0x22, 0x22, 0x22, 0x22, 0x22, 0x22, 0x22,
    0x22, 0x22, 0x22, 0x22, 0x22, 0x22, 0x22, 0x22,
    0x22, 0x22, 0x22, 0x22, 0x22, 0x22, 0x22, 0x22
};

static const unsigned char add_p3data[] = {
    0x00, 0x00, 0x00, 0x00, 0x99, 0x99, 0x99, 0x99,
    0x99, 0x99, 0x99, 0x99, 0x99, 0x99, 0x99, 0x99,
    0x99, 0x99, 0x99, 0x99, 0x99, 0x99, 0x99, 0x99,
    0x99, 0x99, 0x99, 0x99, 0x99, 0x99, 0x99, 0x99
};

static const unsigned char add_exp_p4[] =  {
    0x00, 0x00, 0x00, 0x00, 0x67, 0x87, 0x67, 0x87,
    0x67, 0x87, 0x67, 0x87, 0x67, 0x87, 0x67, 0x87,
    0x67, 0x87, 0x67, 0x87, 0x67, 0x87, 0x67, 0x87,
    0x67, 0x87, 0x67, 0x87, 0x67, 0x87, 0x67, 0x87
};


int pkha_add(int fd)
{
    int           status, tstat;
    MOD_2OP_REQ   addReq;
    unsigned char p4data[32];

    
    memset(p4data,  0, 32);
    memset(&addReq, 0, sizeof(addReq));


    /* ADD test */
    addReq.opId       = DPD_MM_LDCTX_ADD_ULCTX;
    addReq.bDataBytes = 32;
    addReq.aDataBytes = 32;
    addReq.modBytes   = 32;
    addReq.outBytes   = 32;
    addReq.bData      = (unsigned char *)add_p0data;
    addReq.aData      = (unsigned char *)add_p1data;
    addReq.modData    = (unsigned char *)add_p3data;
    addReq.outData    = p4data;

    status = putKmem(fd, (void *)add_p0data, (void **)&addReq.bData, addReq.bDataBytes);
    if (status)
        return status;
    
    status = putKmem(fd, (void *)add_p1data, (void **)&addReq.aData,  addReq.aDataBytes);
    if (status) {
        freeKmem(fd, (void **)&addReq.bData);
        return status;
    }
    
    status = putKmem(fd, (void *)add_p3data, (void **)&addReq.modData, addReq.modBytes);
    if (status) {
        freeKmem(fd, (void **)&addReq.bData);
        freeKmem(fd, (void **)&addReq.aData);
        return status;
    }

    status = putKmem(fd, NULL, (void **)&addReq.outData, addReq.outBytes);
    if (status) {
        freeKmem(fd, (void **)&addReq.bData);
        freeKmem(fd, (void **)&addReq.aData);
        freeKmem(fd, (void **)&addReq.modData);
        return status;
    }

    armCompletion(&addReq);
    status = ioctl(fd, IOCTL_PROC_REQ, (int)&addReq);

    if ((status = waitCompletion("testPKHA(): modular add test", status, &addReq)))
    {
        freeKmem(fd, (void **)&addReq.bData);
        freeKmem(fd, (void **)&addReq.aData);
        freeKmem(fd, (void **)&addReq.modData);
        freeKmem(fd, (void **)&addReq.outData);
        return status;
    }
    getKmem(fd, p4data, (void **)&addReq.outData, addReq.outBytes);

    freeKmem(fd, (void **)&addReq.bData);
    freeKmem(fd, (void **)&addReq.aData);
    freeKmem(fd, (void **)&addReq.modData);
    freeKmem(fd, (void **)&addReq.outData);


    if (memcmp((unsigned char *)p4data,
               (unsigned char *)add_exp_p4,
               32) != 0)
    {
      dumpm(p4data, 32);
      tstat = -1;
    }
    else
    {
      printf("Results from ADD as expected\n");
      tstat = 0;
    }

    return(tstat);
}







static const unsigned char sub_p1data[] = {
    0x00, 0x00, 0x00, 0x00, 0x45, 0x65, 0x45, 0x65,
    0x45, 0x65, 0x45, 0x65, 0x45, 0x65, 0x45, 0x65,
    0x45, 0x65, 0x45, 0x65, 0x45, 0x65, 0x45, 0x65,
    0x45, 0x65, 0x45, 0x65, 0x45, 0x65, 0x45, 0x65
};

static const unsigned char sub_p0data[] = {
    0x00, 0x00, 0x00, 0x00, 0x22, 0x22, 0x22, 0x22,
    0x22, 0x22, 0x22, 0x22, 0x22, 0x22, 0x22, 0x22,
    0x22, 0x22, 0x22, 0x22, 0x22, 0x22, 0x22, 0x22,
    0x22, 0x22, 0x22, 0x22, 0x22, 0x22, 0x22, 0x22
};

static const unsigned char sub_p3data[] = {
    0x00, 0x00, 0x00, 0x00, 0x99, 0x99, 0x99, 0x99,
    0x99, 0x99, 0x99, 0x99, 0x99, 0x99, 0x99, 0x99,
    0x99, 0x99, 0x99, 0x99, 0x99, 0x99, 0x99, 0x99,
    0x99, 0x99, 0x99, 0x99, 0x99, 0x99, 0x99, 0x99
};
    
static const unsigned char sub_exp_p4[] = {
    0x00, 0x00, 0x00, 0x00, 0x23, 0x43, 0x23, 0x43,
    0x23, 0x43, 0x23, 0x43, 0x23, 0x43, 0x23, 0x43,
    0x23, 0x43, 0x23, 0x43, 0x23, 0x43, 0x23, 0x43,
    0x23, 0x43, 0x23, 0x43, 0x23, 0x43, 0x23, 0x43
};



int pkha_subtract(int fd)
{
    int status, tstat;
    MOD_2OP_REQ   subReq;
    unsigned char p4data[256];

    memset (p4data,  0, 256);
    memset (&subReq, 0, sizeof(subReq));

    /* SUBTRACT test */
    subReq.opId       = DPD_MM_LDCTX_SUB_ULCTX;
    subReq.bDataBytes = 28;
    subReq.aDataBytes = 28;
    subReq.modBytes   = 28;
    subReq.outBytes   = 28;
    subReq.bData      = (unsigned char *)sub_p0data;
    subReq.aData      = (unsigned char *)sub_p1data;
    subReq.modData    = (unsigned char *)sub_p3data;
    subReq.outData    = p4data;

    status = putKmem(fd, (void *)sub_p0data, (void **)&subReq.bData, subReq.bDataBytes);
    if (status)
        return status;
    
    status = putKmem(fd, (void *)sub_p1data, (void **)&subReq.aData,  subReq.aDataBytes);
    if (status) {
        freeKmem(fd, (void **)&subReq.bData);
        return status;
    }
    
    status = putKmem(fd, (void *)sub_p3data, (void **)&subReq.modData, subReq.modBytes);
    if (status) {
        freeKmem(fd, (void **)&subReq.bData);
        freeKmem(fd, (void **)&subReq.aData);
        return status;
    }

    status = putKmem(fd, NULL, (void **)&subReq.outData, subReq.outBytes);
    if (status) {
        freeKmem(fd, (void **)&subReq.bData);
        freeKmem(fd, (void **)&subReq.aData);
        freeKmem(fd, (void **)&subReq.modData);
        return status;
    }

    
    armCompletion(&subReq);
    status = ioctl(fd, IOCTL_PROC_REQ, (int)&subReq);

    if ((status = waitCompletion("testPKHA(): modular subtraction test", status, &subReq)))
    {
        freeKmem(fd, (void **)&subReq.bData);
        freeKmem(fd, (void **)&subReq.aData);
        freeKmem(fd, (void **)&subReq.modData);
        freeKmem(fd, (void **)&subReq.outData);
        return status;
    }
    
    getKmem(fd, p4data, (void **)&subReq.outData, subReq.outBytes);

    freeKmem(fd, (void **)&subReq.bData);
    freeKmem(fd, (void **)&subReq.aData);
    freeKmem(fd, (void **)&subReq.modData);
    freeKmem(fd, (void **)&subReq.outData);

    if (memcmp((unsigned char *)p4data,
               (unsigned char *)sub_exp_p4,
               28) != 0)
    {
      dumpm(p4data, 28);
      tstat = -1;
    }
    else
    {
      printf ("Results from SUBTRACT as expected\n");
      tstat = 0;
    }

    return(tstat);
}






#define MUL2_DATASIZE (128)

static const unsigned char mul2_p0data[] =
{
    0x5f, 0xed, 0x51, 0x23, 0x5f, 0xed, 0x51, 0x23,
    0x5f, 0xed, 0x51, 0x23, 0x5f, 0xed, 0x51, 0x23,
    0x5f, 0xed, 0x51, 0x23, 0x5f, 0xed, 0x51, 0x23,
    0x5f, 0xed, 0x51, 0x23, 0x5f, 0xed, 0x51, 0x23,
    0x5f, 0xed, 0x51, 0x23, 0x5f, 0xed, 0x51, 0x23,
    0x5f, 0xed, 0x51, 0x23, 0x5f, 0xed, 0x51, 0x23,
    0x5f, 0xed, 0x51, 0x23, 0x5f, 0xed, 0x51, 0x23,
    0x5f, 0xed, 0x51, 0x23, 0x5f, 0xed, 0x51, 0x23,
    0x5f, 0xed, 0x51, 0x23, 0x5f, 0xed, 0x51, 0x23,
    0x5f, 0xed, 0x51, 0x23, 0x5f, 0xed, 0x51, 0x23,
    0x5f, 0xed, 0x51, 0x23, 0x5f, 0xed, 0x51, 0x23,
    0x5f, 0xed, 0x51, 0x23, 0x5f, 0xed, 0x51, 0x23,
    0x5f, 0xed, 0x51, 0x23, 0x5f, 0xed, 0x51, 0x23,
    0x5f, 0xed, 0x51, 0x23, 0x5f, 0xed, 0x51, 0x23,
    0x5f, 0xed, 0x51, 0x23, 0x5f, 0xed, 0x51, 0x23,
    0x5f, 0xed, 0x51, 0x23, 0x5f, 0xed, 0x51, 0x23
};

static const unsigned char mul2_p1data[] =
{
    0x12, 0x34, 0x56, 0x78, 0x90, 0xab, 0xcd, 0xef,
    0x12, 0x34, 0x56, 0x78, 0x90, 0xab, 0xcd, 0xef,
    0x12, 0x34, 0x56, 0x78, 0x90, 0xab, 0xcd, 0xef,
    0x12, 0x34, 0x56, 0x78, 0x90, 0xab, 0xcd, 0xef,
    0x12, 0x34, 0x56, 0x78, 0x90, 0xab, 0xcd, 0xef,
    0x12, 0x34, 0x56, 0x78, 0x90, 0xab, 0xcd, 0xef,
    0x12, 0x34, 0x56, 0x78, 0x90, 0xab, 0xcd, 0xef,
    0x12, 0x34, 0x56, 0x78, 0x90, 0xab, 0xcd, 0xef,
    0x12, 0x34, 0x56, 0x78, 0x90, 0xab, 0xcd, 0xef,
    0x12, 0x34, 0x56, 0x78, 0x90, 0xab, 0xcd, 0xef,
    0x12, 0x34, 0x56, 0x78, 0x90, 0xab, 0xcd, 0xef,
    0x12, 0x34, 0x56, 0x78, 0x90, 0xab, 0xcd, 0xef,
    0x12, 0x34, 0x56, 0x78, 0x90, 0xab, 0xcd, 0xef,
    0x12, 0x34, 0x56, 0x78, 0x90, 0xab, 0xcd, 0xef,
    0x12, 0x34, 0x56, 0x78, 0x90, 0xab, 0xcd, 0xef,
    0x12, 0x34, 0x56, 0x78, 0x90, 0xab, 0xcd, 0xef
};

static const unsigned char mul2_p3data[] = 
{
    0xa5, 0xa5, 0xa5, 0xa5, 0xa5, 0xa5, 0xa5, 0xa5,
    0xa5, 0xa5, 0xa5, 0xa5, 0xa5, 0xa5, 0xa5, 0xa5,
    0xa5, 0xa5, 0xa5, 0xa5, 0xa5, 0xa5, 0xa5, 0xa5,
    0xa5, 0xa5, 0xa5, 0xa5, 0xa5, 0xa5, 0xa5, 0xa5,
    0xa5, 0xa5, 0xa5, 0xa5, 0xa5, 0xa5, 0xa5, 0xa5,
    0xa5, 0xa5, 0xa5, 0xa5, 0xa5, 0xa5, 0xa5, 0xa5,
    0xa5, 0xa5, 0xa5, 0xa5, 0xa5, 0xa5, 0xa5, 0xa5,
    0xa5, 0xa5, 0xa5, 0xa5, 0xa5, 0xa5, 0xa5, 0xa5,
    0xa5, 0xa5, 0xa5, 0xa5, 0xa5, 0xa5, 0xa5, 0xa5,
    0xa5, 0xa5, 0xa5, 0xa5, 0xa5, 0xa5, 0xa5, 0xa5,
    0xa5, 0xa5, 0xa5, 0xa5, 0xa5, 0xa5, 0xa5, 0xa5,
    0xa5, 0xa5, 0xa5, 0xa5, 0xa5, 0xa5, 0xa5, 0xa5,
    0xa5, 0xa5, 0xa5, 0xa5, 0xa5, 0xa5, 0xa5, 0xa5,
    0xa5, 0xa5, 0xa5, 0xa5, 0xa5, 0xa5, 0xa5, 0xa5,
    0xa5, 0xa5, 0xa5, 0xa5, 0xa5, 0xa5, 0xa5, 0xa5,
    0xa5, 0xa5, 0xa5, 0xa5, 0xa5, 0xa5, 0xa5, 0xa5
};
    
static const unsigned char mul2_exp_p4[] =
{
    0x02, 0x35, 0x24, 0xef, 0x02, 0x35, 0x24, 0xef,
    0x02, 0x35, 0x24, 0xef, 0x02, 0x35, 0x24, 0xef,
    0x02, 0x35, 0x24, 0xef, 0x02, 0x35, 0x24, 0xef,
    0x02, 0x35, 0x24, 0xef, 0x02, 0x35, 0x24, 0xef,
    0x02, 0x35, 0x24, 0xef, 0x02, 0x35, 0x24, 0xef,
    0x02, 0x35, 0x24, 0xef, 0x02, 0x35, 0x24, 0xef,
    0x02, 0x35, 0x24, 0xef, 0x02, 0x35, 0x24, 0xef,
    0x02, 0x35, 0x24, 0xef, 0x02, 0x35, 0x24, 0xef,
    0x02, 0x35, 0x24, 0xef, 0x02, 0x35, 0x24, 0xef,
    0x02, 0x35, 0x24, 0xef, 0x02, 0x35, 0x24, 0xef,
    0x02, 0x35, 0x24, 0xef, 0x02, 0x35, 0x24, 0xef,
    0x02, 0x35, 0x24, 0xef, 0x02, 0x35, 0x24, 0xef,
    0x02, 0x35, 0x24, 0xef, 0x02, 0x35, 0x24, 0xef,
    0x02, 0x35, 0x24, 0xef, 0x02, 0x35, 0x24, 0xef,
    0x02, 0x35, 0x24, 0xef, 0x02, 0x35, 0x24, 0xef,
    0x02, 0x35, 0x24, 0xef, 0x02, 0x35, 0x24, 0xef
};


int pkha_mul2(int fd)
{
    int         status, tstat;
    MOD_2OP_REQ mult2Req;

    unsigned char p4data[MUL2_DATASIZE];

    memset(p4data,    0, MUL2_DATASIZE);
    memset(&mult2Req, 0, sizeof(mult2Req));

    /* MULT2 test */
    mult2Req.opId       = DPD_MM_LDCTX_MUL2_ULCTX;
    mult2Req.bDataBytes = MUL2_DATASIZE;
    mult2Req.aDataBytes = MUL2_DATASIZE;
    mult2Req.modBytes   = MUL2_DATASIZE;
    mult2Req.outBytes   = MUL2_DATASIZE;
    mult2Req.bData      = (unsigned char *)mul2_p0data;
    mult2Req.aData      = (unsigned char *)mul2_p1data;
    mult2Req.modData    = (unsigned char *)mul2_p3data;
    mult2Req.outData    = p4data;

    status = putKmem(fd, (void *)mul2_p0data, (void **)&mult2Req.bData, mult2Req.bDataBytes);
    if (status)
        return status;

    status = putKmem(fd, (void *)mul2_p1data, (void **)&mult2Req.aData,  mult2Req.aDataBytes);
    if (status) {
        freeKmem(fd, (void **)&mult2Req.bData);
        return status;
    }
    
    status = putKmem(fd, (void *)mul2_p3data, (void **)&mult2Req.modData, mult2Req.modBytes);
    if (status) {
        freeKmem(fd, (void **)&mult2Req.bData);
        freeKmem(fd, (void **)&mult2Req.aData);
        return status;
    }

    status = putKmem(fd, NULL, (void **)&mult2Req.outData, mult2Req.outBytes);
    if (status) {
        freeKmem(fd, (void **)&mult2Req.bData);
        freeKmem(fd, (void **)&mult2Req.aData);
        freeKmem(fd, (void **)&mult2Req.modData);
        return status;
    }

    armCompletion(&mult2Req);
    status = ioctl(fd, IOCTL_PROC_REQ, (int)&mult2Req);

    if ((status = waitCompletion("testPKHA(): modular mult test", status, &mult2Req)))
    {
        freeKmem(fd, (void **)&mult2Req.bData);
        freeKmem(fd, (void **)&mult2Req.aData);
        freeKmem(fd, (void **)&mult2Req.modData);
        freeKmem(fd, (void **)&mult2Req.outData);
        return status;
    }
    
    getKmem(fd, p4data, (void **)&mult2Req.outData, mult2Req.outBytes);

    freeKmem(fd, (void **)&mult2Req.bData);
    freeKmem(fd, (void **)&mult2Req.aData);
    freeKmem(fd, (void **)&mult2Req.modData);
    freeKmem(fd, (void **)&mult2Req.outData);


    if (memcmp((unsigned char *)p4data,
               (unsigned char *)mul2_exp_p4,
               MUL2_DATASIZE) != 0)
    {
        dumpm(p4data, MUL2_DATASIZE);
        tstat = -1;
    }
    else
    {
        printf("pkha_mul2(): Results from MULT2 as expected\n");
        tstat = 0;
    }

    return(tstat);
}






static const unsigned char f2mmul2_p0data[] = {
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
    0x00, 0x00, 0x00, 0x00, 0xb5, 0xdb, 0xef, 0x4d,
    0x54, 0x20, 0xcc, 0x1a, 0x86, 0xc9, 0x89, 0x67         
    };

static const unsigned char f2mmul2_p1data[] = {
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x12, 0x34, 0x56, 0x78,
    0x90, 0xab, 0xcd, 0xef, 0x01, 0xb2, 0x37, 0x3c,
    0x16, 0xfb, 0x0b, 0xa9, 0x1b, 0x80, 0xa8, 0x10
};

static const unsigned char f2mmul2_p3data[] =
{
    0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 
    0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x92, 0xfe,
    0x77, 0xe7, 0x0c, 0x12, 0xa4, 0x23, 0x4c, 0x33
};

static const unsigned char f2mmul2_exp_p4[] =
{
    0x00, 0x00, 0x00, 0x01, 0x6b, 0x9e, 0x7b, 0x1e,
    0x6d, 0x4b, 0x7b, 0x64, 0xe4, 0x2a, 0xe5, 0xc4,
    0x1d, 0x1b, 0x0e, 0xe5, 0x2f, 0xb1, 0xd1, 0xdf
};



int pkha_f2m_mult2(int fd)
{
    int         status, tstat;
    MOD_2OP_REQ f2m_mult2Req;

    unsigned char p4data[256];

    memset (p4data, 0, 256);
    memset (&f2m_mult2Req, 0, sizeof(f2m_mult2Req));


    /* F2M MULT2 test */
    f2m_mult2Req.opId       = DPD_POLY_LDCTX_A0_B0_MUL2_ULCTX;
    f2m_mult2Req.bDataBytes = 256;
    f2m_mult2Req.aDataBytes = 256;
    f2m_mult2Req.modBytes   = 24;
    f2m_mult2Req.outBytes   = 24;
    f2m_mult2Req.bData      = (unsigned char *)f2mmul2_p0data;
    f2m_mult2Req.aData      = (unsigned char *)f2mmul2_p1data;
    f2m_mult2Req.modData    = (unsigned char *)f2mmul2_p3data;
    f2m_mult2Req.outData    = p4data;

    status = putKmem(fd, (void *)f2mmul2_p0data, (void **)&f2m_mult2Req.bData, f2m_mult2Req.bDataBytes);
    if (status)
        return status;
    
    status = putKmem(fd, (void *)f2mmul2_p1data, (void **)&f2m_mult2Req.aData,  f2m_mult2Req.aDataBytes);
    if (status) {
        freeKmem(fd, (void **)&f2m_mult2Req.bData);
        return status;
    }
    
    status = putKmem(fd, (void *)f2mmul2_p3data, (void **)&f2m_mult2Req.modData, f2m_mult2Req.modBytes);
    if (status) {
        freeKmem(fd, (void **)&f2m_mult2Req.bData);
        freeKmem(fd, (void **)&f2m_mult2Req.aData);
        return status;
    }

    status = putKmem(fd, NULL, (void **)&f2m_mult2Req.outData, f2m_mult2Req.outBytes);
    if (status) {
        freeKmem(fd, (void **)&f2m_mult2Req.bData);
        freeKmem(fd, (void **)&f2m_mult2Req.aData);
        freeKmem(fd, (void **)&f2m_mult2Req.modData);
        return status;
    }

    
    armCompletion(&f2m_mult2Req);
    status = ioctl(fd, IOCTL_PROC_REQ, (int)&f2m_mult2Req);

    if ((status = waitCompletion("testPKHA(): modular mult test", status, &f2m_mult2Req)))
    {
        freeKmem(fd, (void **)&f2m_mult2Req.bData);
        freeKmem(fd, (void **)&f2m_mult2Req.aData);
        freeKmem(fd, (void **)&f2m_mult2Req.modData);
        freeKmem(fd, (void **)&f2m_mult2Req.outData);
        return status;
    }
    
    getKmem(fd, p4data, (void **)&f2m_mult2Req.outData, f2m_mult2Req.outBytes);

    freeKmem(fd, (void **)&f2m_mult2Req.bData);
    freeKmem(fd, (void **)&f2m_mult2Req.aData);
    freeKmem(fd, (void **)&f2m_mult2Req.modData);
    freeKmem(fd, (void **)&f2m_mult2Req.outData);


    if (memcmp((unsigned char *)p4data,
               (unsigned char *)f2mmul2_exp_p4,
               24) != 0)
    {
      dumpm(p4data, 24);
      tstat = -1;
    }
    else
    {
      printf ("Results from F2M_MULT2 as expected\n");
      tstat = 0;
    }

    return(tstat);
}





static const unsigned char modp_p3data[] =
{
    0xf3, 0xc4, 0x4d, 0x17, 0x3c, 0x01, 0x2e, 0x0a, 
    0xec, 0x59, 0x15, 0xab, 0xe4, 0xc8, 0x9b, 0x89, 
    0xaa, 0x08, 0xac, 0x54, 0x5a, 0x0a, 0x0f, 0xb5, 
    0xaa, 0x08, 0xac, 0x54, 0x5a, 0x0a, 0x0f, 0xb5,  
    0xaa, 0x08, 0xac, 0x54, 0x5a, 0x0a, 0x0f, 0xb5  
};
    
static const unsigned char modp_exp_p4[] =
{
    0x37, 0x8e, 0x17, 0x36, 0x7d, 0x63, 0xf8, 0xec,
    0xc7, 0x19, 0x9d, 0xbd, 0xa1, 0x16, 0xea, 0x99,
    0xb8, 0x60, 0x69, 0x84, 0xa0, 0x35, 0x31, 0xa4,
    0xd5, 0xd1, 0x0d, 0x59, 0x06, 0x45, 0x36, 0x8b,
    0x18, 0x74, 0xe0, 0xa3, 0xbb, 0xe2, 0x51, 0xeb
};
    

int pkha_rrmodp(int fd)
{
    int            status, tstat;
    MOD_RRMODP_REQ rrmodpReq;


    unsigned char p4data[256];


    memset (p4data, 0, 256);
    memset(&rrmodpReq, 0, sizeof(rrmodpReq));


    /* RRMODP test */
    rrmodpReq.opId     = DPD_MM_LDCTX_RRMODP_ULCTX;
    rrmodpReq.nBytes   = 64;
    rrmodpReq.pBytes   = 40;
    rrmodpReq.outBytes = 40;
    rrmodpReq.pData    = (unsigned char *)modp_p3data;
    rrmodpReq.outData  = p4data;

    status = putKmem(fd, (void *)modp_p3data, (void **)&rrmodpReq.pData, rrmodpReq.pBytes);
    if (status)
        return status;
    
    status = putKmem(fd, NULL, (void **)&rrmodpReq.outData,  rrmodpReq.outBytes);
    if (status) {
        freeKmem(fd, (void **)&rrmodpReq.outData);
        return status;
    }
    
    armCompletion(&rrmodpReq);
    status = ioctl(fd, IOCTL_PROC_REQ, (int)&rrmodpReq);

    if ((status = waitCompletion("testPKHA(): RRMODP test", status, &rrmodpReq)))
    {
        freeKmem(fd, (void **)&rrmodpReq.pData);
        freeKmem(fd, (void **)&rrmodpReq.outData);
        return status;
    }
    
    getKmem(fd, p4data, (void **)&rrmodpReq.outData, rrmodpReq.outBytes);

    freeKmem(fd, (void **)&rrmodpReq.pData);
    freeKmem(fd, (void **)&rrmodpReq.outData);

    if (memcmp((unsigned char *)p4data,
               (unsigned char *)modp_exp_p4,
               40) != 0)
    {
      printf("RRMODP result miscompare\n");
      printf("actual result:\n");
      dumpm(p4data, 40);
      printf("expected result:\n");
      dumpm((unsigned char *)modp_exp_p4, 40);
      tstat = -1;
    }
    else
    {
      printf("Results from RRMODP as expected\n");
      tstat = 0;
    }

    return(tstat);
}




/* Basis of main Public Key feature test */
int testPKHA(int fd)
{
  printf ("testPKHA(): Public Key Hardware Accelerator test started\n");
  pkhaTestStatus = 0;
  
#ifdef TESTPKHA_ADD
  if (pkha_add(fd))
    pkhaTestStatus++;
#endif

#ifdef TESTPKHA_SUBTRACT
  if (pkha_subtract(fd))
    pkhaTestStatus++;
#endif

#ifdef TESTPKHA_MULT
  if (pkha_mul2(fd))
    pkhaTestStatus++;
#endif

#ifdef TESTPKHA_F2MMULT
  if (pkha_f2m_mult2(fd))
    pkhaTestStatus++;
#endif

#ifdef TESTPKHA_EXPO
  if (pkha_exp_ss(fd))
    pkhaTestStatus++;
#endif

#ifdef TESTPKHA_RRMODP
  if (pkha_rrmodp(fd))
    pkhaTestStatus++;
#endif

#ifdef TESTPKHA_ECC_MULTK
  if (testEccFpPtMultkPtoQAff(fd))
    pkhaTestStatus++;
  if (testEccMultkPtoQf2mAff(fd))
    pkhaTestStatus++;
  if (testEccF2mPtMultkPtoQAff2(fd))
    pkhaTestStatus++;
  if (testEccFpPtMultkPtoQProj(fd))
    pkhaTestStatus++;
  if (testEccMultkPtoQf2mProj(fd))
    pkhaTestStatus++;
#endif


  if (!pkhaTestStatus)
    printf("testPKHA(): All Tests Passed\n");
  else
    printf("testPKHA(): %d Tests Failed\n", pkhaTestStatus);
    
  return(pkhaTestStatus);
}
