
/****************************************************************************
 * morepkhaTest.c - Elliptic curve known-answer testsrt for SEC2 driver
 ****************************************************************************
 * Copyright (c) 2004-2005 Freescale Semiconductor
 * All Rights Reserved. 
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version
 * 2 of the License, or (at your option) any later version.
 ***************************************************************************/

/* Revision History:
 *  01a,  May 02,2003 dgs 
 *        Jun 14,2004 sec - convert for compounded ECC operations for SEC2 
 *  1.1.0 Dec 05,2004 sec - prep for Linux-compatible driver release
 *  1.2   02-Feb-2005 sec - convert test data to const types
 *  1.3   Aug 11,2005
 */



#include "sec2drvTest.h"
#include "Sec2.h"




/* New MULTK P to Q F2M Affine test */
/* This one builds the field from raw data */

/* input a0 = 19 bytes */
static const unsigned char pqf2mAff_a0[] = 
{
    0x04, 0x2d, 0x9c, 0x6c, 0xf4, 0x20, 0xd2, 0x8f,
    0x5d, 0xf5, 0xef, 0x21, 0xa3, 0x29, 0xf3, 0xc2,
    0x0a, 0xb0, 0xa5
};

/* input a1 = 19 bytes */
static const unsigned char pqf2mAff_a1[] =
{
    0x04, 0xed, 0x25, 0x64, 0x5c, 0x5e, 0x0a, 0xa3,
    0x54, 0xb8, 0x19, 0x3e, 0xab, 0xcd, 0x09, 0x75,
    0x00, 0x2d, 0x31
};

/* input a2 = 19 bytes */
static const unsigned char pqf2mAff_a2[] =
{
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x01
};
  
/* input a3 = 19 bytes */
static const unsigned char pqf2mAff_a3[] =
{
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00
};
  
/* input b1 = 19 bytes */
static const unsigned char pqf2mAff_b0[] =
{
    0x03, 0x06, 0x2e, 0x4f, 0x2d, 0xb1, 0x7b, 0xfc,
    0x75, 0x1f, 0xa4, 0xab, 0xb5, 0x22, 0xf3, 0xe7,
    0xb7, 0x97, 0xb2
};
  
/* input b2 = 19 bytes */
static const unsigned char pqf2mAff_b1[] =
{
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
    0x00, 0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
    0x00, 0x00, 0x00
};

/* expected output from array build operation - 384 bytes */
static const unsigned char pqf2mAff_exp_build[] =
{
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0x2d, 0x9c,
    0x6c, 0xf4, 0x20, 0xd2, 0x8f, 0x5d, 0xf5, 0xef,
    0x21, 0xa3, 0x29, 0xf3, 0xc2, 0x0a, 0xb0, 0xa5,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x04, 0xed, 0x25,
    0x64, 0x5c, 0x5e, 0x0a, 0xa3, 0x54, 0xb8, 0x19,
    0x3e, 0xab, 0xcd, 0x09, 0x75, 0x00, 0x2d, 0x31,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x06, 0x2e,
    0x4f, 0x2d, 0xb1, 0x7b, 0xfc, 0x75, 0x1f, 0xa4,
    0xab, 0xb5, 0x22, 0xf3, 0xe7, 0xb7, 0x97, 0xb2,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00
};

static const unsigned char pqf2mAff_n[] = 
{
    0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08,
    0x00, 0x00, 0x01
};
  
static const unsigned char pqf2mAff_e[] =
{
    0x03
};

static const unsigned char pqf2mAff_exp_b1[] = 
{
    0x09, 0xa5, 0x55, 0x55, 0x9b, 0xf2, 0x2e, 0xf0,
    0xa1, 0xbf, 0x61, 0xa9, 0xe8, 0x5d, 0x47, 0xda,
    0x95, 0x25, 0xfb
};
  
static const unsigned char pqf2mAff_exp_b2[] = 
{
    0x00, 0x33, 0x9d, 0xc8, 0x47, 0xed, 0xd9, 0xe2,
    0x99, 0x11, 0x14, 0x27, 0x73, 0xff, 0xff, 0xdb,
    0xcb, 0xb0, 0xe4
};



int testEccMultkPtoQf2mAff(int fd)
{
    int              stat, tstat;
    ECC_SPKBUILD_REQ ecc_buildReq;
    ECC_POINT_REQ    eccPtRq;
    unsigned char    buildOut[384], b1out[19], b2out[19];

    tstat  = 0;
    memset(buildOut, 0, 384);
    memset(&ecc_buildReq, 0, sizeof(ecc_buildReq));
    memset(&eccPtRq, 0, sizeof(ECC_POINT_REQ));
  
    printf("ECC F2M Point Multiply Affine Test\n");
    

    ecc_buildReq.opId            = DPD_EC_SPKBUILD;

    ecc_buildReq.a0Data          = (unsigned char *)pqf2mAff_a0;
    ecc_buildReq.a0DataBytes     = 19;

    ecc_buildReq.a1Data          = (unsigned char *)pqf2mAff_a1;
    ecc_buildReq.a1DataBytes     = 19;

    ecc_buildReq.a2Data          = (unsigned char *)pqf2mAff_a2;
    ecc_buildReq.a2DataBytes     = 19;

    ecc_buildReq.a3Data          = (unsigned char *)pqf2mAff_a3;
    ecc_buildReq.a3DataBytes     = 19;

    ecc_buildReq.b0Data          = (unsigned char *)pqf2mAff_b0;
    ecc_buildReq.b0DataBytes     = 19;

    ecc_buildReq.b1Data          = (unsigned char *)pqf2mAff_b1;
    ecc_buildReq.b1DataBytes     = 19;

    ecc_buildReq.buildData       = buildOut;
    ecc_buildReq.buildDataBytes  = 384;
  
    stat = putKmem(fd, (void *)pqf2mAff_a0, (void **)&ecc_buildReq.a0Data, ecc_buildReq.a0DataBytes);
    if (stat)
        return -1;

    stat = putKmem(fd, (void *)pqf2mAff_a1, (void **)&ecc_buildReq.a1Data, ecc_buildReq.a1DataBytes);
    if (stat)
    {
        freeKmem(fd, (void **)&ecc_buildReq.a0Data);
        return -1;
    }

    stat = putKmem(fd, (void *)pqf2mAff_a2, (void **)&ecc_buildReq.a2Data, ecc_buildReq.a2DataBytes);
    if (stat)
    {
        freeKmem(fd, (void **)&ecc_buildReq.a0Data);
        freeKmem(fd, (void **)&ecc_buildReq.a1Data);
        return -1;
    }

    stat = putKmem(fd, (void *)pqf2mAff_a3, (void **)&ecc_buildReq.a3Data, ecc_buildReq.a3DataBytes);
    if (stat)
    {
        freeKmem(fd, (void **)&ecc_buildReq.a0Data);
        freeKmem(fd, (void **)&ecc_buildReq.a1Data);
        freeKmem(fd, (void **)&ecc_buildReq.a2Data);
        return -1;
    }

    stat = putKmem(fd, (void *)pqf2mAff_b0, (void **)&ecc_buildReq.b0Data, ecc_buildReq.b0DataBytes);
    if (stat)
    {
        freeKmem(fd, (void **)&ecc_buildReq.a0Data);
        freeKmem(fd, (void **)&ecc_buildReq.a1Data);
        freeKmem(fd, (void **)&ecc_buildReq.a2Data);
        freeKmem(fd, (void **)&ecc_buildReq.a3Data);
        return -1;
    }

    stat = putKmem(fd, (void *)pqf2mAff_b1, (void **)&ecc_buildReq.b1Data, ecc_buildReq.b1DataBytes);
    if (stat)
    {
        freeKmem(fd, (void **)&ecc_buildReq.a0Data);
        freeKmem(fd, (void **)&ecc_buildReq.a1Data);
        freeKmem(fd, (void **)&ecc_buildReq.a2Data);
        freeKmem(fd, (void **)&ecc_buildReq.a3Data);
        freeKmem(fd, (void **)&ecc_buildReq.b0Data);
        return -1;
    }

    stat = putKmem(fd, NULL, (void **)&ecc_buildReq.buildData, ecc_buildReq.buildDataBytes);
    if (stat)
    {
        freeKmem(fd, (void **)&ecc_buildReq.a0Data);
        freeKmem(fd, (void **)&ecc_buildReq.a1Data);
        freeKmem(fd, (void **)&ecc_buildReq.a2Data);
        freeKmem(fd, (void **)&ecc_buildReq.a3Data);
        freeKmem(fd, (void **)&ecc_buildReq.b0Data);
        freeKmem(fd, (void **)&ecc_buildReq.b1Data);
        return -1;
    }


    armCompletion(&ecc_buildReq);
    stat = ioctl(fd, IOCTL_PROC_REQ, (int)&ecc_buildReq);

    if ((stat = waitCompletion("testPKHA(): elliptic curve set build test", stat, &ecc_buildReq)))
    {
        freeKmem(fd, (void **)&ecc_buildReq.a0Data);
        freeKmem(fd, (void **)&ecc_buildReq.a1Data);
        freeKmem(fd, (void **)&ecc_buildReq.a2Data);
        freeKmem(fd, (void **)&ecc_buildReq.a3Data);
        freeKmem(fd, (void **)&ecc_buildReq.b0Data);
        freeKmem(fd, (void **)&ecc_buildReq.b1Data);
        freeKmem(fd, (void **)&ecc_buildReq.buildData);
    }
    
    getKmem(fd, buildOut, (void **)&ecc_buildReq.buildData, ecc_buildReq.buildDataBytes);

    freeKmem(fd, (void **)&ecc_buildReq.a0Data);
    freeKmem(fd, (void **)&ecc_buildReq.a1Data);
    freeKmem(fd, (void **)&ecc_buildReq.a2Data);
    freeKmem(fd, (void **)&ecc_buildReq.a3Data);
    freeKmem(fd, (void **)&ecc_buildReq.b0Data);
    freeKmem(fd, (void **)&ecc_buildReq.b1Data);
    freeKmem(fd, (void **)&ecc_buildReq.buildData);

    if (memcmp(buildOut, pqf2mAff_exp_build, 384) != 0)
    {
        printf("ECC Point Multiply Affine Test SPKbuild output data miscompare\n");
        dumpm(buildOut, 384);
        tstat = -1;
    }






    /* Now that the build is done, go do the multiply */
    eccPtRq.opId            = DPD_EC_F2M_AFF_LDCTX_kP_ULCTX;

    eccPtRq.nDataBytes      = 19;
    eccPtRq.nData           = (unsigned char *)pqf2mAff_n;

    eccPtRq.eDataBytes      = 1;
    eccPtRq.eData           = (unsigned char *)pqf2mAff_e;

    eccPtRq.buildDataBytes  = 384;
    eccPtRq.buildData       = buildOut;

    eccPtRq.b1DataBytes     = 19;
    eccPtRq.b1Data          = b1out;

    eccPtRq.b2DataBytes     = 19;
    eccPtRq.b2Data          = b2out;


    stat = putKmem(fd, (void *)pqf2mAff_n, (void **)&eccPtRq.nData, eccPtRq.nDataBytes);
    if (stat)
        return -1;

    stat = putKmem(fd, (void *)pqf2mAff_e, (void **)&eccPtRq.eData, eccPtRq.eDataBytes);
    if (stat)
    {
        freeKmem(fd, (void **)&eccPtRq.nData);
        return -1;
    }

    stat = putKmem(fd, (void *)buildOut, (void **)&eccPtRq.buildData, eccPtRq.buildDataBytes);
    if (stat)
    {
        freeKmem(fd, (void **)&eccPtRq.nData);
        freeKmem(fd, (void **)&eccPtRq.eData);
        return -1;
    }

    stat = putKmem(fd, NULL, (void **)&eccPtRq.b1Data, eccPtRq.b1DataBytes);
    if (stat)
    {
        freeKmem(fd, (void **)&eccPtRq.nData);
        freeKmem(fd, (void **)&eccPtRq.eData);
        freeKmem(fd, (void **)&eccPtRq.buildData);
        return -1;
    }

    stat = putKmem(fd, NULL, (void **)&eccPtRq.b2Data, eccPtRq.b2DataBytes);
    if (stat)
    {
        freeKmem(fd, (void **)&eccPtRq.nData);
        freeKmem(fd, (void **)&eccPtRq.eData);
        freeKmem(fd, (void **)&eccPtRq.buildData);
        freeKmem(fd, (void **)&eccPtRq.b1Data);
        return -1;
    }

    armCompletion(&eccPtRq);
    stat = ioctl(fd, IOCTL_PROC_REQ, (int)&eccPtRq);

    if ((stat = waitCompletion("testPKHA(): ECC P->Q F2M Affine test", stat, &eccPtRq)))
    {
        freeKmem(fd, (void **)&eccPtRq.nData);
        freeKmem(fd, (void **)&eccPtRq.eData);
        freeKmem(fd, (void **)&eccPtRq.buildData);
        freeKmem(fd, (void **)&eccPtRq.b1Data);
        freeKmem(fd, (void **)&eccPtRq.b2Data);
    }
    
    getKmem(fd, (void *)b1out, (void **)&eccPtRq.b1Data, eccPtRq.b1DataBytes);
    getKmem(fd, (void *)b2out, (void **)&eccPtRq.b2Data, eccPtRq.b1DataBytes);

    freeKmem(fd, (void **)&eccPtRq.nData);
    freeKmem(fd, (void **)&eccPtRq.eData);
    freeKmem(fd, (void **)&eccPtRq.buildData);
    freeKmem(fd, (void **)&eccPtRq.b1Data);
    freeKmem(fd, (void **)&eccPtRq.b2Data);



  if (memcmp(b1out, pqf2mAff_exp_b1, 19) != 0)
  {
    printf("ECC Point Multiply Affine Test output B1 data miscompare\n");
    dumpm(b1out, 19);
    tstat = -1;
  }
  if (memcmp(b2out, pqf2mAff_exp_b2, 19) != 0)
  {
    printf("ECC Point Multiply Affine Test output B2 data miscompare\n");
    dumpm(b2out, 19);
    tstat = -1;
  }

  if (!tstat)
    printf("ECC Point Multiply Affine Test test passed\n");
        
  return(tstat);

}




static const unsigned char ptpqAff_MODULUS[] =  /* 20 bytes */
{
    0xed, 0xbe, 0x63, 0xac,
    0x07, 0xf2, 0x60, 0x10,
    0xfb, 0x91, 0xc1, 0x5f,
    0x9e, 0x5d, 0x75, 0xca,
    0x39, 0x3d, 0x73, 0x6b
};
  
static const unsigned char ptpqAff_e[] =  /* 20 bytes */
{
    0x02, 0x72, 0x27, 0x96,
    0x6b, 0x6b, 0x9f, 0x53,
    0x84, 0x1f, 0x7b, 0x03,
    0x2b, 0x2f, 0xa9, 0x30,
    0x35, 0x28, 0xb2, 0xf2
};
  
  
static const unsigned char ptpqAff_builddata[] = /* 384 bytes */
{
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0xe1, 0x34, 0x60, 0x37,
    0x7d, 0x80, 0xa4, 0x91, 0xae, 0x4a, 0x06, 0x9f,
    0xcf, 0x1c, 0xcf, 0xf3, 0x7a, 0xe0, 0x72, 0x33,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0xea, 0x0a, 0xd5, 0xd3,
    0xde, 0x97, 0x7a, 0x76, 0xdd, 0xc1, 0x7b, 0xb6,
    0xea, 0xb0, 0xc8, 0x2b, 0x7f, 0x10, 0x8e, 0xf4,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0xed, 0xbe, 0x63, 0xac,
    0x07, 0xf2, 0x60, 0x10, 0xfb, 0x91, 0xc1, 0x5f,
    0x9e, 0x5d, 0x75, 0xca, 0x39, 0x3d, 0x73, 0x68,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x01, 0x4f, 0x3c, 0x88,
    0xbf, 0xe1, 0xcc, 0xc6, 0xa2, 0x2a, 0xfe, 0xc0,
    0x4a, 0x20, 0x53, 0xf3, 0x14, 0x57, 0x6c, 0x9f,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x2b, 0x10, 0xce, 0x96,
    0x4c, 0xaf, 0xf6, 0xaa, 0x50, 0xde, 0xa0, 0x75,
    0x0d, 0x84, 0x5d, 0x62, 0x5c, 0xa7, 0x66, 0x11,
};
  
static const unsigned char ptpqAff_exp_b3[] = /* 20 bytes */
{
    0x68, 0x60, 0xae, 0x45,
    0x94, 0x34, 0x7c, 0xe1,
    0x5c, 0x9f, 0xdc, 0xd7,
    0xda, 0xb6, 0xad, 0x86,
    0xde, 0x0d, 0x09, 0xd7
};
  
static const unsigned char ptpqAff_exp_b2[] = /* 20 bytes */
{
    0x4f, 0xa2, 0x44, 0xb0,
    0x73, 0x22, 0xb3, 0xc8,
    0xa5, 0x6b, 0xb7, 0xde,
    0xb0, 0x17, 0xd9, 0xc4,
    0x0e, 0x1f, 0xbd, 0x8d
};
  
static const unsigned char ptpqAff_exp_b1[] = /* 20 bytes */
{
    0x31, 0x14, 0x71, 0x2d,
    0xf3, 0xa2, 0x38, 0xb4,
    0xe3, 0x20, 0x5d, 0x08,
    0x45, 0xb5, 0xc5, 0x0c,
    0xe3, 0xf5, 0xe3, 0xd1
};



int testEccFpPtMultkPtoQAff(int fd)
{
    unsigned char  b1data[20], b2data[20], b3data[20];
    int            stat, tstat;
    ECC_POINT_REQ  ecc_mulkptoqReq;

    tstat  = 0;

    memset(b1data, 0, 20);
    memset(b2data, 0, 20);
    memset(b3data, 0, 20);
    memset (&ecc_mulkptoqReq, 0, sizeof(ecc_mulkptoqReq));

  
    printf("ECC Point MulK P->Q Affine Test\n");
    
    ecc_mulkptoqReq.opId             = DPD_EC_FP_AFF_LDCTX_kP_ULCTX;

    ecc_mulkptoqReq.nDataBytes       = 20;
    ecc_mulkptoqReq.nData            = (unsigned char *)ptpqAff_MODULUS;

    ecc_mulkptoqReq.eDataBytes       = 20;
    ecc_mulkptoqReq.eData            = (unsigned char *)ptpqAff_e;

    ecc_mulkptoqReq.buildDataBytes   = 384;
    ecc_mulkptoqReq.buildData        = (unsigned char *)ptpqAff_builddata;

    ecc_mulkptoqReq.b1DataBytes      = 20; 
    ecc_mulkptoqReq.b1Data           = (unsigned char *)b1data;

    ecc_mulkptoqReq.b2DataBytes      = 20;
    ecc_mulkptoqReq.b2Data           = (unsigned char *)b2data;

    ecc_mulkptoqReq.b3DataBytes      = 20;
    ecc_mulkptoqReq.b3Data           = (unsigned char *)b3data;


    stat = putKmem(fd, (void *)ptpqAff_MODULUS, (void **)&ecc_mulkptoqReq.nData, ecc_mulkptoqReq.nDataBytes);
    if (stat)
        return -1;

    stat = putKmem(fd, (void *)ptpqAff_e, (void **)&ecc_mulkptoqReq.eData, ecc_mulkptoqReq.eDataBytes);
    if (stat)
    {
        freeKmem(fd, (void **)&ecc_mulkptoqReq.nData);
        return -1;
    }

    stat = putKmem(fd, (void *)ptpqAff_builddata, (void **)&ecc_mulkptoqReq.buildData, ecc_mulkptoqReq.buildDataBytes);
    if (stat)
    {
        freeKmem(fd, (void **)&ecc_mulkptoqReq.nData);
        freeKmem(fd, (void **)&ecc_mulkptoqReq.eData);
        return -1;
    }

    stat = putKmem(fd, NULL, (void **)&ecc_mulkptoqReq.b1Data, ecc_mulkptoqReq.b1DataBytes);
    if (stat)
    {
        freeKmem(fd, (void **)&ecc_mulkptoqReq.nData);
        freeKmem(fd, (void **)&ecc_mulkptoqReq.eData);
        freeKmem(fd, (void **)&ecc_mulkptoqReq.buildData);
        return -1;
    }

    stat = putKmem(fd, NULL, (void **)&ecc_mulkptoqReq.b2Data, ecc_mulkptoqReq.b2DataBytes);
    if (stat)
    {
        freeKmem(fd, (void **)&ecc_mulkptoqReq.nData);
        freeKmem(fd, (void **)&ecc_mulkptoqReq.eData);
        freeKmem(fd, (void **)&ecc_mulkptoqReq.buildData);
        freeKmem(fd, (void **)&ecc_mulkptoqReq.b1Data);
        return -1;
    }

    stat = putKmem(fd, NULL, (void **)&ecc_mulkptoqReq.b3Data, ecc_mulkptoqReq.b3DataBytes);
    if (stat)
    {
        freeKmem(fd, (void **)&ecc_mulkptoqReq.nData);
        freeKmem(fd, (void **)&ecc_mulkptoqReq.eData);
        freeKmem(fd, (void **)&ecc_mulkptoqReq.buildData);
        freeKmem(fd, (void **)&ecc_mulkptoqReq.b1Data);
        freeKmem(fd, (void **)&ecc_mulkptoqReq.b2Data);
        return -1;
    }

    armCompletion(&ecc_mulkptoqReq);
    stat = ioctl(fd, IOCTL_PROC_REQ, (int)&ecc_mulkptoqReq);

    if ((stat = waitCompletion("testPKHA(): ECC Point Multiply Affine test", stat, &ecc_mulkptoqReq)))
    {
        freeKmem(fd, (void **)&ecc_mulkptoqReq.nData);
        freeKmem(fd, (void **)&ecc_mulkptoqReq.eData);
        freeKmem(fd, (void **)&ecc_mulkptoqReq.buildData);
        freeKmem(fd, (void **)&ecc_mulkptoqReq.b1Data);
        freeKmem(fd, (void **)&ecc_mulkptoqReq.b2Data);
        freeKmem(fd, (void **)&ecc_mulkptoqReq.b3Data);
    }
    
    getKmem(fd, b1data, (void **)&ecc_mulkptoqReq.b1Data, ecc_mulkptoqReq.b1DataBytes);
    getKmem(fd, b2data, (void **)&ecc_mulkptoqReq.b2Data, ecc_mulkptoqReq.b2DataBytes);
    getKmem(fd, b3data, (void **)&ecc_mulkptoqReq.b3Data, ecc_mulkptoqReq.b3DataBytes);

    freeKmem(fd, (void **)&ecc_mulkptoqReq.nData);
    freeKmem(fd, (void **)&ecc_mulkptoqReq.eData);
    freeKmem(fd, (void **)&ecc_mulkptoqReq.buildData);
    freeKmem(fd, (void **)&ecc_mulkptoqReq.b1Data);
    freeKmem(fd, (void **)&ecc_mulkptoqReq.b2Data);
    freeKmem(fd, (void **)&ecc_mulkptoqReq.b3Data);



    if (memcmp(b1data, ptpqAff_exp_b1, 20) != 0)
    {
        printf("ECC Point Multiply Affine Test - b1 output data mismatch:\n");
        dumpm(b1data, 20);
        tstat = -1;
    }

    if (memcmp(b2data, ptpqAff_exp_b2, 20) != 0)
    {
        printf("ECC Point Multiply Affine Test - b2 output data mismatch:\n");
        dumpm(b2data, 20);
        tstat = -1;
    }

    if (memcmp(b3data, ptpqAff_exp_b3, 20) != 0)
    {
        printf("ECC Point Multiply Affine Test - b3 output data mismatch:\n");
        dumpm(b3data, 20);
        tstat = -1;
    }

  return(tstat);
}



static const unsigned char ptpqAff2_MODULUS[] =
{
    0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x08, 0x01
};

static const unsigned char ptpqAff2_e[] =
{
    0x4b, 0x26, 0xda, 0x96, 0x9e, 0xc0, 0xeb, 0xf7,
    0xa1, 0xea, 0x4f, 0x75, 0xaa, 0x1c, 0x96, 0xf0,
    0xf6, 0x7a, 0x37, 0x7f, 0x1e, 0x9f, 0x39, 0xd4,
    0x6b, 0xa5, 0x2d
};


static const unsigned char ptpqAff2_builddata[] =
{
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x01, 0xb2, 0x37, 0x3c,
    0x16, 0xfb, 0x0b, 0xa9, 0x1b, 0x7c, 0xa8, 0x39,
    0x49, 0x22, 0xfe, 0xd6, 0x1d, 0x07, 0x88, 0x43,
    0x57, 0x58, 0x0a, 0x81, 0x0c, 0xaa, 0xa5, 0x53,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0xc4, 0x42, 0x21,
    0x2f, 0xb2, 0x31, 0x69, 0x64, 0x14, 0x33, 0x0a,
    0xb5, 0x7d, 0xbe, 0xf4, 0xd5, 0x42, 0x0c, 0xc1,
    0xa8, 0x6c, 0x98, 0x96, 0x77, 0x50, 0xe6, 0x5f,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x03, 0x11, 0xd3, 0xc3,
    0xb6, 0xe7, 0x67, 0xb7, 0x15, 0xb5, 0x3c, 0xad,
    0xda, 0x99, 0xa9, 0x80, 0xa4, 0xfd, 0x71, 0x45,
    0xd3, 0xf4, 0xe7, 0x3b, 0x26, 0x44, 0x78, 0xb2,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x10, 0x00
};

static const unsigned char ptpqAff2_exp_b1[] =
{
    0x02, 0xB3, 0x30, 0x45,
    0xA0, 0x97, 0x67, 0x90,
    0x3D, 0xA9, 0xA0, 0x78,
    0x90, 0xA8, 0x7E, 0xA8,
    0x65, 0xDE, 0x77, 0x41,
    0x5D, 0x8A, 0xB9, 0xBE,
    0x3E, 0x8C, 0x94, 0x69
};

static const unsigned char ptpqAff2_exp_b2[] =
{
    0x01, 0x72, 0x3E, 0xA9,
    0xA8, 0xBC, 0x14, 0x00,
    0x7C, 0x80, 0x09, 0xAD,
    0xCB, 0x36, 0x11, 0x31,
    0xD5, 0x42, 0x02, 0x36,
    0x06, 0xFE, 0xA9, 0xB9,
    0x2B, 0xAE, 0x97, 0x89
};




int testEccF2mPtMultkPtoQAff2(int fd)
{ 
    unsigned char b1data[32], b2data[32], b3data[32];
    int           stat, tstat;
    ECC_POINT_REQ ecc_f2mmulkptoqReq;

    tstat  = 0;
  
    memset(b1data, 0, 32);
    memset(b2data, 0, 32);
    memset(b3data, 0, 32);
    memset (&ecc_f2mmulkptoqReq, 0, sizeof(ecc_f2mmulkptoqReq));

    printf("ECC F2M Point Multiply Affine Test #2\n");
    

    ecc_f2mmulkptoqReq.opId            = DPD_EC_F2M_AFF_LDCTX_kP_ULCTX;

    ecc_f2mmulkptoqReq.nDataBytes      = 28;
    ecc_f2mmulkptoqReq.nData           = (unsigned char *)ptpqAff2_MODULUS;

    ecc_f2mmulkptoqReq.eDataBytes      = 27;
    ecc_f2mmulkptoqReq.eData           = (unsigned char *)ptpqAff2_e;

    ecc_f2mmulkptoqReq.buildDataBytes  = 384;
    ecc_f2mmulkptoqReq.buildData       = (unsigned char *)ptpqAff2_builddata;

    ecc_f2mmulkptoqReq.b1DataBytes     = 32;
    ecc_f2mmulkptoqReq.b1Data          = b1data;

    ecc_f2mmulkptoqReq.b2DataBytes     = 32;
    ecc_f2mmulkptoqReq.b2Data          = b2data;

    ecc_f2mmulkptoqReq.b3DataBytes     = 32;
    ecc_f2mmulkptoqReq.b3Data          = b3data;

    stat = putKmem(fd, (void *)ptpqAff2_MODULUS, (void **)&ecc_f2mmulkptoqReq.nData, ecc_f2mmulkptoqReq.nDataBytes);
    if (stat)
        return -1;

    stat = putKmem(fd, (void *)ptpqAff2_e, (void **)&ecc_f2mmulkptoqReq.eData, ecc_f2mmulkptoqReq.eDataBytes);
    if (stat)
    {
        freeKmem(fd, (void **)&ecc_f2mmulkptoqReq.nData);
        return -1;
    }

    stat = putKmem(fd, (void *)ptpqAff2_builddata, (void **)&ecc_f2mmulkptoqReq.buildData, ecc_f2mmulkptoqReq.buildDataBytes);
    if (stat)
    {
        freeKmem(fd, (void **)&ecc_f2mmulkptoqReq.nData);
        freeKmem(fd, (void **)&ecc_f2mmulkptoqReq.eData);
        return -1;
    }

    stat = putKmem(fd, NULL, (void **)&ecc_f2mmulkptoqReq.b1Data, ecc_f2mmulkptoqReq.b1DataBytes);
    if (stat)
    {
        freeKmem(fd, (void **)&ecc_f2mmulkptoqReq.nData);
        freeKmem(fd, (void **)&ecc_f2mmulkptoqReq.eData);
        freeKmem(fd, (void **)&ecc_f2mmulkptoqReq.buildData);
        return -1;
    }

    stat = putKmem(fd, NULL, (void **)&ecc_f2mmulkptoqReq.b2Data, ecc_f2mmulkptoqReq.b2DataBytes);
    if (stat)
    {
        freeKmem(fd, (void **)&ecc_f2mmulkptoqReq.nData);
        freeKmem(fd, (void **)&ecc_f2mmulkptoqReq.eData);
        freeKmem(fd, (void **)&ecc_f2mmulkptoqReq.buildData);
        freeKmem(fd, (void **)&ecc_f2mmulkptoqReq.b1Data);
        return -1;
    }

    stat = putKmem(fd, NULL, (void **)&ecc_f2mmulkptoqReq.b3Data, ecc_f2mmulkptoqReq.b3DataBytes);
    if (stat)
    {
        freeKmem(fd, (void **)&ecc_f2mmulkptoqReq.nData);
        freeKmem(fd, (void **)&ecc_f2mmulkptoqReq.eData);
        freeKmem(fd, (void **)&ecc_f2mmulkptoqReq.buildData);
        freeKmem(fd, (void **)&ecc_f2mmulkptoqReq.b1Data);
        freeKmem(fd, (void **)&ecc_f2mmulkptoqReq.b2Data);
        return -1;
    }

    armCompletion(&ecc_f2mmulkptoqReq);
    stat = ioctl(fd, IOCTL_PROC_REQ, (int)&ecc_f2mmulkptoqReq);

    if ((stat = waitCompletion("testPKHA(): ECC F2M Point Multiply Affine Test #2", stat, &ecc_f2mmulkptoqReq)))
    {
        freeKmem(fd, (void **)&ecc_f2mmulkptoqReq.nData);
        freeKmem(fd, (void **)&ecc_f2mmulkptoqReq.eData);
        freeKmem(fd, (void **)&ecc_f2mmulkptoqReq.buildData);
        freeKmem(fd, (void **)&ecc_f2mmulkptoqReq.b1Data);
        freeKmem(fd, (void **)&ecc_f2mmulkptoqReq.b2Data);
        freeKmem(fd, (void **)&ecc_f2mmulkptoqReq.b3Data);
    }
    
    getKmem(fd, b1data, (void **)&ecc_f2mmulkptoqReq.b1Data, ecc_f2mmulkptoqReq.b1DataBytes);
    getKmem(fd, b2data, (void **)&ecc_f2mmulkptoqReq.b2Data, ecc_f2mmulkptoqReq.b2DataBytes);
    getKmem(fd, b3data, (void **)&ecc_f2mmulkptoqReq.b3Data, ecc_f2mmulkptoqReq.b3DataBytes);

    freeKmem(fd, (void **)&ecc_f2mmulkptoqReq.nData);
    freeKmem(fd, (void **)&ecc_f2mmulkptoqReq.eData);
    freeKmem(fd, (void **)&ecc_f2mmulkptoqReq.buildData);
    freeKmem(fd, (void **)&ecc_f2mmulkptoqReq.b1Data);
    freeKmem(fd, (void **)&ecc_f2mmulkptoqReq.b2Data);
    freeKmem(fd, (void **)&ecc_f2mmulkptoqReq.b3Data);





    if (memcmp(b1data, ptpqAff2_exp_b1, 28) != 0)
    {
      printf("ECC F2M Point Multiply Affine Test #2 - b1 output data mismatch:\n");
      dumpm(b1data, 28);
      tstat = -1;
    }

    if (memcmp(b2data, ptpqAff2_exp_b2, 28) != 0)
    {
      printf("ECC F2M Point Multiply Affine Test #2 - b2 output data mismatch:\n");
      dumpm(b2data, 28);
      tstat = -1;
    }

    return(tstat);
}



static const unsigned char fppqProj_MODULUS[] =
{
    0xed, 0xbe, 0x63, 0xac,
    0x07, 0xf2, 0x60, 0x10,
    0xfb, 0x91, 0xc1, 0x5f,
    0x9e, 0x5d, 0x75, 0xca,
    0x39, 0x3d, 0x73, 0x6b
};
  
static const unsigned char fppqProj_e[] =
{
    0x02, 0x72, 0x27, 0x96,
    0x6b, 0x6b, 0x9f, 0x53,
    0x84, 0x1f, 0x7b, 0x03,
    0x2b, 0x2f, 0xa9, 0x30,
    0x35, 0x28, 0xb2, 0xf2
};
  
  
static const unsigned char fppqProj_builddata[] =
{
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0xe1, 0x34, 0x60, 0x37,
    0x7d, 0x80, 0xa4, 0x91, 0xae, 0x4a, 0x06, 0x9f,
    0xcf, 0x1c, 0xcf, 0xf3, 0x7a, 0xe0, 0x72, 0x33,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0xea, 0x0a, 0xd5, 0xd3,
    0xde, 0x97, 0x7a, 0x76, 0xdd, 0xc1, 0x7b, 0xb6,
    0xea, 0xb0, 0xc8, 0x2b, 0x7f, 0x10, 0x8e, 0xf4,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0xed, 0xbe, 0x63, 0xac,
    0x07, 0xf2, 0x60, 0x10, 0xfb, 0x91, 0xc1, 0x5f,
    0x9e, 0x5d, 0x75, 0xca, 0x39, 0x3d, 0x73, 0x68,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x01, 0x4f, 0x3c, 0x88,
    0xbf, 0xe1, 0xcc, 0xc6, 0xa2, 0x2a, 0xfe, 0xc0,
    0x4a, 0x20, 0x53, 0xf3, 0x14, 0x57, 0x6c, 0x9f,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x2b, 0x10, 0xce, 0x96,
    0x4c, 0xaf, 0xf6, 0xaa, 0x50, 0xde, 0xa0, 0x75,
    0x0d, 0x84, 0x5d, 0x62, 0x5c, 0xa7, 0x66, 0x11
};

static const unsigned char fppqProj_exp_b3[] =
{
    0x2c, 0xb6, 0x98, 0x6e,
    0x6b, 0xd7, 0x18, 0x23,
    0x9d, 0x0a, 0x50, 0xc8,
    0x4a, 0xa4, 0xdd, 0xdb,
    0x79, 0x7a, 0x42, 0x6c
};
  
static const unsigned char fppqProj_exp_b2[] =
{
    0xa9, 0x73, 0x41, 0xb8,
    0x2c, 0x75, 0x73, 0x51,
    0x99, 0xc6, 0xff, 0x74,
    0x34, 0x97, 0xfe, 0x29,
    0xfe, 0x2b, 0x33, 0x47
};
  
static const unsigned char fppqProj_exp_b1[] =
{
    0x33, 0x3e, 0x1c, 0xb8,
    0x18, 0xc0, 0x19, 0xdf,
    0xbd, 0x30, 0x80, 0xa2,
    0xb3, 0x5b, 0x47, 0x66,
    0x39, 0x8c, 0xbb, 0xae
};


int testEccFpPtMultkPtoQProj(int fd)
{
    unsigned char  b1data[32], b2data[32], b3data[32];
    int            stat, tstat;
    ECC_POINT_REQ  ecc_mulkptoqReq;

    tstat  = 0;

    memset(b1data, 0, 32);
    memset(b2data, 0, 32);
    memset(b3data, 0, 32);
    memset (&ecc_mulkptoqReq, 0, sizeof(ecc_mulkptoqReq));
  
    printf("ECC Point Multiply Projective Test\n");
    

    ecc_mulkptoqReq.opId             = DPD_EC_FP_PROJ_LDCTX_kP_ULCTX;

    ecc_mulkptoqReq.nDataBytes       = 20;
    ecc_mulkptoqReq.nData            = (unsigned char *)fppqProj_MODULUS;

    ecc_mulkptoqReq.eDataBytes       = 20;
    ecc_mulkptoqReq.eData            = (unsigned char *)fppqProj_e;

    ecc_mulkptoqReq.buildDataBytes   = 384;
    ecc_mulkptoqReq.buildData        = (unsigned char *)fppqProj_builddata;

    ecc_mulkptoqReq.b1DataBytes      = 32;
    ecc_mulkptoqReq.b1Data           = b1data;

    ecc_mulkptoqReq.b2DataBytes      = 32;
    ecc_mulkptoqReq.b2Data           = b2data;

    ecc_mulkptoqReq.b3DataBytes      = 32;
    ecc_mulkptoqReq.b3Data           = b3data;


    stat = putKmem(fd, (void *)fppqProj_MODULUS, (void **)&ecc_mulkptoqReq.nData, ecc_mulkptoqReq.nDataBytes);
    if (stat)
        return -1;

    stat = putKmem(fd, (void *)fppqProj_e, (void **)&ecc_mulkptoqReq.eData, ecc_mulkptoqReq.eDataBytes);
    if (stat)
    {
        freeKmem(fd, (void **)&ecc_mulkptoqReq.nData);
        return -1;
    }

    stat = putKmem(fd, (void *)fppqProj_builddata, (void **)&ecc_mulkptoqReq.buildData, ecc_mulkptoqReq.buildDataBytes);
    if (stat)
    {
        freeKmem(fd, (void **)&ecc_mulkptoqReq.nData);
        freeKmem(fd, (void **)&ecc_mulkptoqReq.eData);
        return -1;
    }

    stat = putKmem(fd, NULL, (void **)&ecc_mulkptoqReq.b1Data, ecc_mulkptoqReq.b1DataBytes);
    if (stat)
    {
        freeKmem(fd, (void **)&ecc_mulkptoqReq.nData);
        freeKmem(fd, (void **)&ecc_mulkptoqReq.eData);
        freeKmem(fd, (void **)&ecc_mulkptoqReq.buildData);
        return -1;
    }

    stat = putKmem(fd, NULL, (void **)&ecc_mulkptoqReq.b2Data, ecc_mulkptoqReq.b2DataBytes);
    if (stat)
    {
        freeKmem(fd, (void **)&ecc_mulkptoqReq.nData);
        freeKmem(fd, (void **)&ecc_mulkptoqReq.eData);
        freeKmem(fd, (void **)&ecc_mulkptoqReq.buildData);
        freeKmem(fd, (void **)&ecc_mulkptoqReq.b1Data);
        return -1;
    }

    stat = putKmem(fd, NULL, (void **)&ecc_mulkptoqReq.b3Data, ecc_mulkptoqReq.b3DataBytes);
    if (stat)
    {
        freeKmem(fd, (void **)&ecc_mulkptoqReq.nData);
        freeKmem(fd, (void **)&ecc_mulkptoqReq.eData);
        freeKmem(fd, (void **)&ecc_mulkptoqReq.buildData);
        freeKmem(fd, (void **)&ecc_mulkptoqReq.b1Data);
        freeKmem(fd, (void **)&ecc_mulkptoqReq.b2Data);
        return -1;
    }

    armCompletion(&ecc_mulkptoqReq);
    stat = ioctl(fd, IOCTL_PROC_REQ, (int)&ecc_mulkptoqReq);

    if ((stat = waitCompletion("testPKHA(): ECC F2M Point Multiply Affine Test #2", stat, &ecc_mulkptoqReq)))
    {
        freeKmem(fd, (void **)&ecc_mulkptoqReq.nData);
        freeKmem(fd, (void **)&ecc_mulkptoqReq.eData);
        freeKmem(fd, (void **)&ecc_mulkptoqReq.buildData);
        freeKmem(fd, (void **)&ecc_mulkptoqReq.b1Data);
        freeKmem(fd, (void **)&ecc_mulkptoqReq.b2Data);
        freeKmem(fd, (void **)&ecc_mulkptoqReq.b3Data);
    }
    
    getKmem(fd, b1data, (void **)&ecc_mulkptoqReq.b1Data, ecc_mulkptoqReq.b1DataBytes);
    getKmem(fd, b2data, (void **)&ecc_mulkptoqReq.b2Data, ecc_mulkptoqReq.b2DataBytes);
    getKmem(fd, b3data, (void **)&ecc_mulkptoqReq.b3Data, ecc_mulkptoqReq.b3DataBytes);

    freeKmem(fd, (void **)&ecc_mulkptoqReq.nData);
    freeKmem(fd, (void **)&ecc_mulkptoqReq.eData);
    freeKmem(fd, (void **)&ecc_mulkptoqReq.buildData);
    freeKmem(fd, (void **)&ecc_mulkptoqReq.b1Data);
    freeKmem(fd, (void **)&ecc_mulkptoqReq.b2Data);
    freeKmem(fd, (void **)&ecc_mulkptoqReq.b3Data);


    if (memcmp(b1data, fppqProj_exp_b1, 20) != 0)
    {
      printf("ECC Point Multiply Projective Test - b1 output data mismatch:\n");
      dumpm(b1data, 20);
      tstat = -1;
    }

    if (memcmp(b2data, fppqProj_exp_b2, 20) != 0)
    {
      printf("ECC Point Multiply Projective Test - b2 output data mismatch:\n");
      dumpm(b2data, 20);
      tstat = -1;
    }

    if (memcmp(b3data, fppqProj_exp_b3, 20) != 0)
    {
      printf("ECC Point Multiply Projective Test - b3 output data mismatch:\n");
      dumpm(b3data, 20);
      tstat = -1;
    }


    return(tstat);
}






static const unsigned char p2qf2mProj_MODULUS[] =
{
    0x04, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x08, 0x01
};

static const unsigned char p2qf2mProj_e[] =
{
    0x4b, 0x26, 0xda, 0x96, 0x9e, 0xc0, 0xeb, 0xf7,
    0xa1, 0xea, 0x4f, 0x75, 0xaa, 0x1c, 0x96, 0xf0,
    0xf6, 0x7a, 0x37, 0x7f, 0x1e, 0x9f, 0x39, 0xd4,
    0x6b, 0xa5, 0x2d
};
  
static const unsigned char p2qf2mProj_builddata[] =
{
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  /* a0 */
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x01, 0xb2, 0x37, 0x3c,
    0x16, 0xfb, 0x0b, 0xa9, 0x1b, 0x7c, 0xa8, 0x39,
    0x49, 0x22, 0xfe, 0xd6, 0x1d, 0x07, 0x88, 0x43,
    0x57, 0x58, 0x0a, 0x81, 0x0c, 0xaa, 0xa5, 0x53,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,  /* a1 */
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0xc4, 0x42, 0x21,
    0x2f, 0xb2, 0x31, 0x69, 0x64, 0x14, 0x33, 0x0a,
    0xb5, 0x7d, 0xbe, 0xf4, 0xd5, 0x42, 0x0c, 0xc1,
    0xa8, 0x6c, 0x98, 0x96, 0x77, 0x50, 0xe6, 0x5f,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* a2 */
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* a3 */
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* b0 */
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x03, 0x11, 0xd3, 0xc3,
    0xb6, 0xe7, 0x67, 0xb7, 0x15, 0xb5, 0x3c, 0xad,
    0xda, 0x99, 0xa9, 0x80, 0xa4, 0xfd, 0x71, 0x45,
    0xd3, 0xf4, 0xe7, 0x3b, 0x26, 0x44, 0x78, 0xb2,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, /* b1 */
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x10, 0x00
};


static const unsigned char p2qf2mProj_exp_b1[] =
{
    0x02, 0x75, 0x58, 0x7c,
    0x9e, 0x6f, 0xf5, 0xc1,
    0x36, 0x71, 0xa5, 0x8b,
    0x2f, 0x50, 0x1e, 0x7e,
    0xe5, 0xf1, 0x98, 0xd9,
    0x44, 0x00, 0xb2, 0x6d,
    0x47, 0xbd, 0x1a, 0x1f
};
  
static const unsigned char p2qf2mProj_exp_b2[] =
{
    0x03, 0x05, 0x4d, 0xb9,
    0x7a, 0x24, 0x72, 0x2d,
    0x32, 0xbe, 0x9e, 0xeb,
    0x28, 0xca, 0x77, 0xa7,
    0xbb, 0x01, 0xee, 0x69,
    0xdc, 0xa6, 0x19, 0x5c,
    0xfb, 0xc0, 0xce, 0xe4
};

static const unsigned char p2qf2mProj_exp_b3[] =
{
    0x00, 0x54, 0x20, 0x08,
    0x2e, 0x28, 0xe3, 0x6f,
    0xb4, 0xef, 0x7c, 0x3b,
    0xb8, 0x21, 0x0f, 0x97,
    0x15, 0xc2, 0xd6, 0xc6,
    0x49, 0x06, 0x22, 0x41,
    0x57, 0xc8, 0xf4, 0xf2
};


int testEccMultkPtoQf2mProj(int fd)
{
    unsigned char  b1data[32], b2data[32], b3data[32];
    int            stat, tstat;
    ECC_POINT_REQ  ecc_f2mmulkptoqReq;

    tstat  = 0;

    memset(b1data, 0, 32);
    memset(b2data, 0, 32);
    memset(b3data, 0, 32);
    memset(&ecc_f2mmulkptoqReq, 0, sizeof(ecc_f2mmulkptoqReq));

  
    printf("ECC F2M Point Multiply Projective Test\n");
    
    ecc_f2mmulkptoqReq.opId             = DPD_EC_F2M_PROJ_LDCTX_kP_ULCTX;

    ecc_f2mmulkptoqReq.nDataBytes       = 28;
    ecc_f2mmulkptoqReq.nData            = (unsigned char *)p2qf2mProj_MODULUS;

    ecc_f2mmulkptoqReq.eDataBytes       = 27;
    ecc_f2mmulkptoqReq.eData            = (unsigned char *)p2qf2mProj_e;

    ecc_f2mmulkptoqReq.buildDataBytes   = 384;
    ecc_f2mmulkptoqReq.buildData        = (unsigned char *)p2qf2mProj_builddata;

    ecc_f2mmulkptoqReq.b1DataBytes      = 32;
    ecc_f2mmulkptoqReq.b1Data           = b1data;

    ecc_f2mmulkptoqReq.b2DataBytes      = 32;
    ecc_f2mmulkptoqReq.b2Data           = b2data;

    ecc_f2mmulkptoqReq.b3DataBytes      = 32;
    ecc_f2mmulkptoqReq.b3Data           = b3data;

    stat = putKmem(fd, (void *)p2qf2mProj_MODULUS, (void **)&ecc_f2mmulkptoqReq.nData, ecc_f2mmulkptoqReq.nDataBytes);
    if (stat)
        return -1;

    stat = putKmem(fd, (void *)p2qf2mProj_e, (void **)&ecc_f2mmulkptoqReq.eData, ecc_f2mmulkptoqReq.eDataBytes);
    if (stat)
    {
        freeKmem(fd, (void **)&ecc_f2mmulkptoqReq.nData);
        return -1;
    }

    stat = putKmem(fd, (void *)p2qf2mProj_builddata, (void **)&ecc_f2mmulkptoqReq.buildData, ecc_f2mmulkptoqReq.buildDataBytes);
    if (stat)
    {
        freeKmem(fd, (void **)&ecc_f2mmulkptoqReq.nData);
        freeKmem(fd, (void **)&ecc_f2mmulkptoqReq.eData);
        return -1;
    }

    stat = putKmem(fd, NULL, (void **)&ecc_f2mmulkptoqReq.b1Data, ecc_f2mmulkptoqReq.b1DataBytes);
    if (stat)
    {
        freeKmem(fd, (void **)&ecc_f2mmulkptoqReq.nData);
        freeKmem(fd, (void **)&ecc_f2mmulkptoqReq.eData);
        freeKmem(fd, (void **)&ecc_f2mmulkptoqReq.buildData);
        return -1;
    }

    stat = putKmem(fd, NULL, (void **)&ecc_f2mmulkptoqReq.b2Data, ecc_f2mmulkptoqReq.b2DataBytes);
    if (stat)
    {
        freeKmem(fd, (void **)&ecc_f2mmulkptoqReq.nData);
        freeKmem(fd, (void **)&ecc_f2mmulkptoqReq.eData);
        freeKmem(fd, (void **)&ecc_f2mmulkptoqReq.buildData);
        freeKmem(fd, (void **)&ecc_f2mmulkptoqReq.b1Data);
        return -1;
    }

    stat = putKmem(fd, NULL, (void **)&ecc_f2mmulkptoqReq.b3Data, ecc_f2mmulkptoqReq.b3DataBytes);
    if (stat)
    {
        freeKmem(fd, (void **)&ecc_f2mmulkptoqReq.nData);
        freeKmem(fd, (void **)&ecc_f2mmulkptoqReq.eData);
        freeKmem(fd, (void **)&ecc_f2mmulkptoqReq.buildData);
        freeKmem(fd, (void **)&ecc_f2mmulkptoqReq.b1Data);
        freeKmem(fd, (void **)&ecc_f2mmulkptoqReq.b2Data);
        return -1;
    }

    armCompletion(&ecc_f2mmulkptoqReq);
    stat = ioctl(fd, IOCTL_PROC_REQ, (int)&ecc_f2mmulkptoqReq);

    if ((stat = waitCompletion("testPKHA(): ECC F2M Point Multiply Projective Test", stat, &ecc_f2mmulkptoqReq)))
    {
        freeKmem(fd, (void **)&ecc_f2mmulkptoqReq.nData);
        freeKmem(fd, (void **)&ecc_f2mmulkptoqReq.eData);
        freeKmem(fd, (void **)&ecc_f2mmulkptoqReq.buildData);
        freeKmem(fd, (void **)&ecc_f2mmulkptoqReq.b1Data);
        freeKmem(fd, (void **)&ecc_f2mmulkptoqReq.b2Data);
        freeKmem(fd, (void **)&ecc_f2mmulkptoqReq.b3Data);
    }
    
    getKmem(fd, b1data, (void **)&ecc_f2mmulkptoqReq.b1Data, ecc_f2mmulkptoqReq.b1DataBytes);
    getKmem(fd, b2data, (void **)&ecc_f2mmulkptoqReq.b2Data, ecc_f2mmulkptoqReq.b2DataBytes);
    getKmem(fd, b3data, (void **)&ecc_f2mmulkptoqReq.b3Data, ecc_f2mmulkptoqReq.b3DataBytes);

    freeKmem(fd, (void **)&ecc_f2mmulkptoqReq.nData);
    freeKmem(fd, (void **)&ecc_f2mmulkptoqReq.eData);
    freeKmem(fd, (void **)&ecc_f2mmulkptoqReq.buildData);
    freeKmem(fd, (void **)&ecc_f2mmulkptoqReq.b1Data);
    freeKmem(fd, (void **)&ecc_f2mmulkptoqReq.b2Data);
    freeKmem(fd, (void **)&ecc_f2mmulkptoqReq.b3Data);


    if (memcmp(b1data, p2qf2mProj_exp_b1, 28) != 0)
    {
      printf("ECC F2M Point Multiply Projective Test - b1 output data mismatch:\n");
      dumpm(b1data, 28);
      tstat = -1;
    }

    if (memcmp(b2data, p2qf2mProj_exp_b2, 28) != 0)
    {
      printf("ECC F2M Point Multiply Projective Test - b2 output data mismatch:\n");
      dumpm(b2data, 28);
      tstat = -1;
    }

    if (memcmp(b3data, p2qf2mProj_exp_b3, 28) != 0)
    {
      printf("ECC F2M Point Multiply Projective Test - b3 output data mismatch:\n");
      dumpm(b3data, 28);
      tstat = -1;
    }

    return(tstat);
}


