

/****************************************************************************
 * ccmp.c - Simulated CCMP packet test for SEC2 driver testing
 ****************************************************************************
 * Copyright (c) 2004-2005 Freescale Semiconductor
 * All Rights Reserved. 
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version
 * 2 of the License, or (at your option) any later version.
 ***************************************************************************/

/* Revision History:
 * 1.2 02-Feb-2005 sec - initial distribution
 * 1.3 Aug 11,2005
 */


#include "Sec2.h"
#include "sec2drvTest.h"

/* 16 bytes */
static const unsigned char key[] =
{
    0xDE, 0x2E, 0x1E, 0x0C, 0x14, 0x85, 0x73, 0xC1,
    0x50, 0xE6, 0x8A, 0x63, 0xA0, 0x6E, 0x4B, 0x57
};



/* 56 bytes */
static const unsigned char context[] = 
{
    /* IV */
    0x59, 0x9C, 0xBF, 0xF5, 0x8F, 0x6A, 0xE2, 0x31,
    0xE2, 0x35, 0x2D, 0xA6, 0xD3, 0xC3, 0x03, 0xcc,    
    
    /* hole */ 
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,

    /* CCMP counter */
    0x01, 0x9C, 0xBF, 0xF5, 0x8F, 0x6A, 0xE2, 0x31,
    0xE2, 0x35, 0x2D, 0xA6, 0xD3, 0xC3, 0x00, 0x00,
    
    /* counter modulus exponent */
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10
};


/* 32 bytes */
static unsigned const char AAheader[] =
{
    0x00, 0x1d, 0x77, 0x6E, 0x39, 0x2F, 0x8B, 0x17,
    0x9B, 0x8F, 0x64, 0x0A, 0xE4, 0xC0, 0x78, 0x24,
    0x0F, 0x0E, 0xCC, 0x55, 0x7D, 0x7B, 0x04, 0xDB,
    0x58, 0x5B, 0xBD, 0xE6, 0x36, 0x54, 0x37, 0x00
};

/* 976 bytes */
static unsigned const char plaintext[] =
{
    0x82, 0x48, 0xF2, 0xB3, 0x71, 0x16, 0x45, 0xF8,
    0xCA, 0xCC, 0xD2, 0xA5, 0x5E, 0x86, 0x9C, 0xED,
    0xDA, 0xC9, 0x65, 0x3D, 0x67, 0xF2, 0x3F, 0xB6,
    0xCC, 0xF0, 0x5C, 0x39, 0x0F, 0x50, 0x7E, 0x91,
    0x94, 0x73, 0x76, 0xD6, 0x56, 0x02, 0x8B, 0x16,
    0xC3, 0x53, 0x44, 0xE0, 0xA6, 0xAB, 0xF3, 0xDF,
    0xC6, 0xAC, 0x24, 0x2D, 0xC3, 0x24, 0xC3, 0x9B,
    0x3F, 0x46, 0xCD, 0xEF, 0x66, 0x27, 0x3F, 0xA9,
    0x37, 0xC1, 0x72, 0x3F, 0x9F, 0x7C, 0xB6, 0x78,
    0x19, 0x52, 0xCB, 0x7B, 0x1F, 0x8B, 0x9A, 0xC3,
    0x44, 0xA4, 0x96, 0x47, 0x97, 0x96, 0x24, 0x0F,
    0x16, 0xA5, 0x85, 0x97, 0x29, 0x28, 0x4D, 0xDD,
    0xE9, 0x82, 0xAB, 0x41, 0xD3, 0x65, 0xD2, 0x42,
    0x76, 0x88, 0x58, 0x18, 0x9B, 0x85, 0xC9, 0x4A,
    0x8C, 0xEE, 0x1C, 0xEA, 0x24, 0x24, 0xD8, 0x03,
    0x05, 0x98, 0x82, 0xD0, 0x06, 0x90, 0xAE, 0x5B,
    0x6F, 0x91, 0x3C, 0xDB, 0x0B, 0x65, 0x32, 0x11,
    0x2C, 0x0C, 0x96, 0x32, 0x95, 0x9F, 0x62, 0x40,
    0x20, 0xEF, 0xAF, 0xA0, 0x52, 0x13, 0x10, 0xFE,
    0xED, 0x22, 0x1A, 0xB1, 0x21, 0x10, 0xCF, 0x0A,
    0x16, 0x40, 0xEC, 0x82, 0xCE, 0x7E, 0xB3, 0x09,
    0xE8, 0xAF, 0x97, 0x9E, 0x0E, 0x18, 0x90, 0x67,
    0x38, 0xC5, 0xBE, 0xAB, 0x60, 0x3F, 0x83, 0xA2,
    0x45, 0xB0, 0xFC, 0xDA, 0x3A, 0xD3, 0xB5, 0x07,
    0xE3, 0x87, 0xD5, 0x13, 0x15, 0x36, 0xD8, 0xD6,
    0xDB, 0xA9, 0xF7, 0x85, 0x50, 0xF6, 0x22, 0x89,
    0x55, 0x26, 0x33, 0xB3, 0x36, 0xCD, 0x30, 0x85,
    0x1B, 0x45, 0x8D, 0x41, 0x76, 0x7D, 0x0E, 0x3A,
    0x23, 0x9A, 0xDD, 0x23, 0x96, 0x91, 0x6B, 0x8C,
    0xE1, 0xA8, 0xB7, 0xDC, 0x6D, 0xA9, 0x73, 0x37,
    0x91, 0x20, 0xBD, 0xFA, 0xF5, 0x9A, 0xFE, 0x69,
    0xD6, 0x4D, 0xAE, 0x96, 0x73, 0x27, 0xBF, 0xBF,
    0xAB, 0x61, 0x00, 0x8F, 0x5D, 0x26, 0xC1, 0xC8,
    0x2D, 0xF6, 0x81, 0xDD, 0xAF, 0xF8, 0x24, 0x8D,
    0xBC, 0x49, 0x16, 0x19, 0x84, 0xEA, 0xCE, 0xC5,
    0x86, 0xCB, 0xA3, 0xD0, 0x77, 0x75, 0x37, 0x6E,
    0xE8, 0x3E, 0x32, 0xCD, 0xDD, 0xE5, 0xF5, 0xAD,
    0x6A, 0x30, 0x69, 0x6C, 0x09, 0x60, 0x8F, 0xED,
    0xCF, 0xCD, 0xCD, 0x51, 0x24, 0x8B, 0x67, 0x7B,
    0xB0, 0x3C, 0x4D, 0x8E, 0x20, 0xB1, 0xB9, 0x75,
    0xAE, 0xFB, 0x71, 0xBA, 0x12, 0x78, 0x4A, 0xE8,
    0xEB, 0xC0, 0xF9, 0xD3, 0x03, 0xD8, 0x75, 0x50,
    0x5C, 0x14, 0xDC, 0x6C, 0xBC, 0xC1, 0xE8, 0xD1,
    0x46, 0x73, 0x00, 0xD8, 0x94, 0x8E, 0x95, 0xA9,
    0x4B, 0xEA, 0x90, 0xDB, 0xC1, 0xD0, 0x9D, 0xAE,
    0x6A, 0x35, 0x44, 0xE7, 0x71, 0xC0, 0x1F, 0x59,
    0x8A, 0x7A, 0xA2, 0x6A, 0x8C, 0x78, 0xE8, 0xB8,
    0xBC, 0x2C, 0x9C, 0x3C, 0xA1, 0xF6, 0xFE, 0x4A,
    0xB4, 0x05, 0xCB, 0x1D, 0xD4, 0x95, 0x6F, 0x6A,
    0x2B, 0xE6, 0xF2, 0xE2, 0xDB, 0xFE, 0x4F, 0xF3,
    0x0C, 0x65, 0x28, 0x06, 0x4C, 0xC6, 0xB8, 0xA9,
    0x12, 0x3A, 0x56, 0xD3, 0x5E, 0xDD, 0x44, 0x0C,
    0x45, 0x9C, 0xC8, 0xF6, 0xE2, 0xB7, 0x8E, 0x60,
    0xC9, 0x64, 0x09, 0xC8, 0x05, 0xF0, 0x4F, 0xBE,
    0xDA, 0xF3, 0xDF, 0x9F, 0x6D, 0x88, 0xCE, 0x54,
    0xE0, 0x19, 0x3E, 0x3A, 0x67, 0x2D, 0x9A, 0x3F,
    0xD4, 0x6B, 0xB3, 0x52, 0xF4, 0xB0, 0x6B, 0x3D,
    0xA6, 0x15, 0x20, 0xB3, 0x87, 0x9E, 0xC0, 0x87,
    0xB6, 0xEA, 0xBF, 0xAF, 0x96, 0x11, 0x6A, 0x41,
    0x04, 0xBD, 0x5A, 0x7A, 0x03, 0xB9, 0x6F, 0x99,
    0xBE, 0xE2, 0x50, 0xB7, 0xAF, 0xB0, 0xE4, 0xF5,
    0x7E, 0x9B, 0x5C, 0x97, 0x35, 0x57, 0x2B, 0x0E,
    0xA7, 0x38, 0x28, 0x89, 0x37, 0x2E, 0x24, 0x5F,
    0x05, 0xA2, 0xC9, 0x12, 0xE0, 0x9B, 0xC9, 0x58,
    0xE7, 0xEB, 0xDC, 0xF3, 0xE0, 0x30, 0x6B, 0x64,
    0x73, 0x3C, 0x47, 0xC0, 0x87, 0x00, 0x22, 0x1E,
    0x50, 0x71, 0x24, 0xC3, 0x08, 0x05, 0x2B, 0xE9,
    0xE2, 0xDB, 0xA0, 0x38, 0x24, 0x76, 0x34, 0x52,
    0xAD, 0xC8, 0xFB, 0x89, 0x69, 0x31, 0xEB, 0x8F,
    0x52, 0x98, 0xE3, 0xF0, 0x60, 0x28, 0x0E, 0x49,
    0xE8, 0xD5, 0x32, 0xB1, 0x69, 0x82, 0x26, 0xCC,
    0x22, 0xB4, 0xFF, 0x6D, 0x8D, 0x86, 0xB6, 0xC6,
    0xD5, 0x53, 0xEA, 0x79, 0x18, 0x4C, 0xA9, 0xF2,
    0x6B, 0xF7, 0xEA, 0x93, 0x4A, 0x87, 0x82, 0x9A,
    0xD2, 0xE8, 0x3A, 0xFB, 0x13, 0xFA, 0x35, 0x63,
    0xC4, 0x51, 0xFB, 0xA6, 0x11, 0x6C, 0xDC, 0x59,
    0xDA, 0xEE, 0x6D, 0x16, 0x90, 0x70, 0xA0, 0xF0,
    0x6B, 0x29, 0x6F, 0x6B, 0x07, 0xAB, 0x05, 0x83,
    0x65, 0xC8, 0xBF, 0xB3, 0x64, 0x09, 0xF0, 0x9C,
    0x2E, 0x1D, 0x84, 0x94, 0xAD, 0x44, 0x9D, 0x69,
    0x94, 0x61, 0x91, 0x6E, 0xED, 0x47, 0x32, 0x17,
    0x85, 0x45, 0xDD, 0xC5, 0x20, 0x7B, 0x2A, 0x2D,
    0x24, 0x0C, 0x97, 0xB1, 0x25, 0x14, 0x95, 0xD1,
    0x61, 0x0D, 0xEF, 0x4E, 0x48, 0xDF, 0x75, 0xAE,
    0x5D, 0x20, 0x39, 0xD6, 0xA3, 0xC2, 0x5C, 0x69,
    0x73, 0x1B, 0x66, 0x72, 0x89, 0x45, 0x4B, 0xEC,
    0x26, 0xC0, 0x43, 0x17, 0xA7, 0xF8, 0xCD, 0x01,
    0xF3, 0x9B, 0x0C, 0x53, 0xB0, 0x2A, 0x8A, 0x65,
    0x0D, 0x03, 0x83, 0x6C, 0xDE, 0x89, 0x61, 0xA5,
    0x98, 0xD3, 0xA4, 0xC0, 0x9A, 0x11, 0x28, 0xC9,
    0x1D, 0x9A, 0xDC, 0x9E, 0xA0, 0xDA, 0xA7, 0x64,
    0xD3, 0xBD, 0x30, 0xB9, 0x2B, 0xF6, 0x3A, 0x55,
    0x0E, 0x5D, 0x24, 0x65, 0x01, 0x7D, 0x3D, 0xFD,
    0x90, 0x3C, 0xAA, 0xA0, 0x45, 0x3C, 0x0B, 0x56,
    0x12, 0x89, 0xA3, 0x67, 0x71, 0x87, 0xC9, 0xEC,
    0x77, 0xEC, 0x3B, 0x12, 0x36, 0x7E, 0xA9, 0xCB,
    0x27, 0xE6, 0x67, 0xED, 0x3A, 0x46, 0x33, 0xDE,
    0xAD, 0x8F, 0x12, 0x4E, 0xF6, 0xAE, 0xA6, 0x8E,
    0xB6, 0x52, 0xAF, 0x2C, 0xF2, 0x52, 0x65, 0xF0,
    0xE0, 0x2E, 0x72, 0x72, 0x18, 0x36, 0x56, 0x6E,
    0xD4, 0x22, 0xE0, 0x4A, 0x1B, 0xD6, 0x5E, 0x8C,
    0x9A, 0x5A, 0xE0, 0x8D, 0xBC, 0xD6, 0x52, 0xF4,
    0x12, 0x02, 0x62, 0x2E, 0x63, 0x19, 0x11, 0x70,
    0x93, 0xED, 0xC4, 0x6B, 0x8E, 0x77, 0x45, 0x8C,
    0xDB, 0xE8, 0xDE, 0x8A, 0x92, 0xD6, 0xF2, 0xA4,
    0x9B, 0x07, 0xBC, 0xC8, 0xD4, 0x99, 0xE0, 0x80,
    0xE6, 0x20, 0xD3, 0x05, 0x9C, 0x00, 0x1C, 0xB4,
    0x90, 0x3E, 0x68, 0x02, 0x9F, 0xD4, 0xF5, 0xEB,
    0xA7, 0x7C, 0x03, 0x02, 0x41, 0x35, 0xEB, 0x49,
    0x6A, 0x43, 0xAB, 0xB4, 0x4E, 0x57, 0x8B, 0xC6,
    0x2C, 0xE9, 0x53, 0xE4, 0xDC, 0xF5, 0xFF, 0xAD,
    0x5D, 0x6C, 0x4B, 0xE8, 0x09, 0x89, 0x7B, 0xDA,
    0x0F, 0xC7, 0x9C, 0xCE, 0x35, 0x7C, 0xBB, 0x19,
    0x4A, 0x38, 0x45, 0xBA, 0x84, 0x07, 0x03, 0xB0,
    0x85, 0xD2, 0xEB, 0x93, 0xEC, 0x2C, 0xE8, 0xA4,
    0xCA, 0x9C, 0xB6, 0x13, 0x01, 0x17, 0x43, 0xFF,
    0x6F, 0x1B, 0x4F, 0x34, 0x01, 0xCD, 0x2F, 0x36,
    0xF6, 0xB5, 0xBE, 0xAE, 0x69, 0x17, 0xA2, 0xD0,
    0x1A, 0x0A, 0xFB, 0x33, 0x2C, 0x61, 0x7F, 0x87,
    0x8E, 0x38, 0x85, 0x13, 0x35, 0xDA, 0x75, 0x98,
    0x9D, 0x51, 0x46, 0x31, 0xA3, 0x90, 0xB9, 0x4D,
    0xB1, 0xE1, 0x71, 0xCB, 0x00, 0x00, 0x00, 0x00
};



/* 976 bytes */
static const unsigned char expected_out[] = 
{
    0xfc, 0x01, 0x40, 0xe4, 0xc7, 0xba, 0x7f, 0x61,
    0x7c, 0xed, 0x64, 0x9f, 0x19, 0x3a, 0xf6, 0xa8,
    0x90, 0xc9, 0xe3, 0x0d, 0xbe, 0xb2, 0x49, 0x6c,
    0xc7, 0x7f, 0xc8, 0xac, 0x94, 0x0e, 0x26, 0xb4,
    0x60, 0xb2, 0x2a, 0xf7, 0x72, 0x51, 0xbb, 0xe5,
    0xc0, 0x0f, 0xd8, 0x9a, 0x57, 0xfb, 0xa7, 0x75,
    0x4c, 0x57, 0x6b, 0x1d, 0xe4, 0xbd, 0x4b, 0x1a,
    0xb6, 0x5f, 0x69, 0xb6, 0x77, 0x11, 0xe4, 0x31,
    0x57, 0xb6, 0x29, 0x9b, 0x18, 0x8d, 0xcd, 0x2b,
    0x81, 0xa3, 0x78, 0xe1, 0x2a, 0xe5, 0xa6, 0x09,
    0xfc, 0xe3, 0x41, 0xcc, 0xca, 0xa0, 0xce, 0x88,
    0xb4, 0x4c, 0x09, 0xcb, 0x5e, 0xf4, 0xce, 0x27,
    0x9a, 0x9e, 0x13, 0x45, 0xe8, 0x24, 0x86, 0x88,
    0xfc, 0xd0, 0x43, 0xd9, 0x96, 0xd5, 0x73, 0x27,
    0x8c, 0x01, 0x11, 0x6a, 0x85, 0x77, 0xa2, 0xc9,
    0x5f, 0x69, 0xe4, 0x92, 0xa1, 0x3c, 0x28, 0xf5,
    0xe8, 0xdb, 0x4b, 0x96, 0xd2, 0xce, 0x15, 0xf1,
    0xf2, 0x47, 0x6e, 0x5e, 0x38, 0xba, 0x9e, 0xd3,
    0x8e, 0xf6, 0x9f, 0x7a, 0x46, 0x84, 0x83, 0x8c,
    0x6c, 0x6f, 0xcc, 0x8a, 0xab, 0x89, 0xbc, 0x59,
    0xc9, 0xee, 0xae, 0x0e, 0xe5, 0x0d, 0x82, 0x59,
    0x7a, 0xca, 0xe5, 0x0d, 0x1a, 0x52, 0x3f, 0x81,
    0x93, 0x5b, 0x56, 0x5e, 0x00, 0xa5, 0xe0, 0x74,
    0x6e, 0x5e, 0xca, 0x25, 0xc5, 0xc3, 0xee, 0xd9,
    0x3b, 0x71, 0x65, 0x84, 0xca, 0x75, 0x08, 0xe7,
    0x06, 0x97, 0xed, 0x07, 0x94, 0xe5, 0xcc, 0xcf,
    0x5e, 0x08, 0xc2, 0xd5, 0x0c, 0xcd, 0x3a, 0xb3,
    0x15, 0x61, 0xb5, 0x99, 0x3e, 0x3f, 0x9a, 0xa7,
    0xcc, 0x3e, 0x94, 0x32, 0x61, 0x75, 0x9e, 0x74,
    0x10, 0xa8, 0x10, 0xfe, 0x21, 0x3d, 0xed, 0x5a,
    0x3d, 0xe3, 0x07, 0x75, 0xaa, 0xd4, 0xcb, 0x3b,
    0x5b, 0xc0, 0x36, 0xad, 0xc0, 0xc5, 0x4c, 0xd7,
    0xdb, 0x37, 0x1e, 0xb7, 0x18, 0xc4, 0xd9, 0x10,
    0x77, 0xcf, 0xa4, 0xb8, 0x05, 0x89, 0xd9, 0x13,
    0x46, 0xef, 0x62, 0x65, 0x0c, 0x40, 0x22, 0x9f,
    0xd3, 0x2b, 0xd9, 0x1c, 0x55, 0x7a, 0x83, 0xd0,
    0x44, 0x9b, 0xc4, 0x17, 0x59, 0xdc, 0xc7, 0xcb,
    0xa8, 0xb7, 0xe9, 0x8e, 0x53, 0x5e, 0x1c, 0x25,
    0x41, 0x3a, 0xa7, 0x43, 0x7b, 0xee, 0x62, 0x8e,
    0x8f, 0xd2, 0x19, 0xd7, 0xdb, 0xdd, 0xc0, 0xc8,
    0xe7, 0xf4, 0xc7, 0x9d, 0x20, 0x7e, 0x0d, 0xa4,
    0xe7, 0x74, 0x87, 0xef, 0xba, 0xba, 0xf0, 0xbe,
    0x29, 0x12, 0xc9, 0xe9, 0x62, 0x3b, 0xe3, 0x36,
    0x38, 0xed, 0x35, 0x92, 0xd9, 0xef, 0xbf, 0xf0,
    0x24, 0x4e, 0x09, 0xfa, 0xa0, 0xbd, 0xab, 0x22,
    0xbd, 0x7b, 0xdb, 0x9f, 0x56, 0x26, 0x74, 0x4b,
    0x96, 0x4a, 0x53, 0x57, 0xa2, 0xc6, 0x2e, 0x94,
    0x2c, 0x4a, 0xbb, 0x00, 0x76, 0x1f, 0x67, 0x48,
    0x90, 0xcb, 0x8c, 0xbe, 0xc4, 0x77, 0x2a, 0xdf,
    0x5b, 0xe5, 0x54, 0x07, 0xa9, 0x4c, 0xeb, 0xac,
    0x9c, 0x13, 0x19, 0x14, 0x64, 0x85, 0xd3, 0xef,
    0x47, 0x28, 0xed, 0x5f, 0xfd, 0xcd, 0x6d, 0x8f,
    0x9f, 0x72, 0xaa, 0x02, 0x68, 0xc3, 0x6e, 0xb9,
    0x43, 0x22, 0x91, 0x09, 0x0f, 0x29, 0x1e, 0xab,
    0x0d, 0x0d, 0xae, 0x43, 0x0b, 0x23, 0xa7, 0x63,
    0x48, 0xf2, 0xe1, 0x58, 0x21, 0xf0, 0x6f, 0xc0,
    0x8c, 0xbf, 0xba, 0xe0, 0x8d, 0xdf, 0xfd, 0x91,
    0xe1, 0xc2, 0x93, 0x77, 0x42, 0xc7, 0x94, 0xb4,
    0x0d, 0xa6, 0xf5, 0x26, 0xb1, 0xf2, 0x52, 0x23,
    0xf4, 0x27, 0xd4, 0x56, 0x6f, 0xa6, 0xf6, 0x28,
    0x89, 0x89, 0xe1, 0xe6, 0x32, 0xa4, 0x51, 0x67,
    0xfe, 0x30, 0xb8, 0x25, 0x24, 0x67, 0x93, 0xac,
    0x1e, 0x1b, 0x21, 0xb2, 0x92, 0x78, 0x66, 0xf4,
    0xc2, 0xee, 0xe1, 0xd0, 0x21, 0x83, 0xe5, 0x8e,
    0x4e, 0x36, 0x6c, 0xfa, 0xd5, 0xc7, 0x65, 0x96,
    0x21, 0x42, 0xa1, 0x7f, 0xeb, 0xd2, 0xbb, 0x48,
    0x19, 0x38, 0x3e, 0xb0, 0x9f, 0xb6, 0x4f, 0x21,
    0x25, 0xcf, 0xe6, 0xc7, 0x50, 0x9d, 0x5c, 0xcb,
    0xa2, 0x50, 0x81, 0x32, 0x85, 0x90, 0x03, 0x01,
    0x25, 0xa0, 0x06, 0x39, 0x5a, 0x17, 0xf3, 0x0d,
    0x35, 0x19, 0x32, 0xc7, 0x24, 0x25, 0xf6, 0x58,
    0x8c, 0x1a, 0xe9, 0xf7, 0xfc, 0xcc, 0xf2, 0x9b,
    0x80, 0x21, 0x4e, 0x11, 0x5b, 0x4b, 0xf0, 0xab,
    0x07, 0xdd, 0xc9, 0xb2, 0x90, 0xa6, 0x3e, 0x3d,
    0x14, 0x0c, 0x64, 0x6b, 0x4f, 0x57, 0xab, 0x5f,
    0xc8, 0x5a, 0x68, 0xb1, 0xa7, 0x3f, 0x2b, 0xe9,
    0x5f, 0x23, 0x35, 0x57, 0xbf, 0xc3, 0xfd, 0xb1,
    0x14, 0xfa, 0x6b, 0x50, 0x06, 0x7f, 0x61, 0x4d,
    0x3e, 0xd6, 0x00, 0x7c, 0x79, 0xb1, 0xba, 0xa0,
    0x9c, 0xda, 0x17, 0xfc, 0x78, 0x8c, 0x57, 0xb9,
    0x10, 0x22, 0xd2, 0x0a, 0xfe, 0x2d, 0x87, 0xf8,
    0x9e, 0x5c, 0x8c, 0x19, 0x77, 0x98, 0x66, 0x5d,
    0x8f, 0x6a, 0xb7, 0xe2, 0xe6, 0x91, 0xdb, 0x35,
    0x83, 0xc2, 0x53, 0x9f, 0x88, 0x9a, 0x5e, 0x61,
    0xbb, 0x21, 0xbe, 0x18, 0xd3, 0xdc, 0x3a, 0xef,
    0x99, 0x1f, 0xb6, 0x08, 0xcd, 0xef, 0x46, 0xa8,
    0x7c, 0xa2, 0x01, 0x60, 0x1c, 0xf1, 0x52, 0x7a,
    0x50, 0x61, 0x56, 0xfa, 0x40, 0x28, 0xd7, 0x58,
    0xc6, 0x49, 0xff, 0xd7, 0x78, 0xdb, 0xf6, 0x7f,
    0xbd, 0x40, 0x36, 0xa4, 0x98, 0x0f, 0xdf, 0xc3,
    0xf1, 0x41, 0x33, 0x1c, 0x6d, 0x04, 0xd7, 0x84,
    0xfd, 0x96, 0xed, 0x8b, 0x47, 0xa4, 0x5d, 0xc1,
    0x7e, 0x17, 0x74, 0x69, 0xa0, 0x88, 0xe1, 0x43,
    0x85, 0xa7, 0x35, 0x25, 0x86, 0x7f, 0x26, 0xb7,
    0xab, 0xe9, 0xcf, 0x74, 0x78, 0x03, 0x83, 0x3c,
    0x81, 0xbe, 0x14, 0xb9, 0x3b, 0x3b, 0x33, 0xf4,
    0xcd, 0x23, 0x83, 0xd3, 0x72, 0xeb, 0xaf, 0x4a,
    0xfe, 0xa7, 0xc5, 0x82, 0xee, 0x95, 0x0d, 0x12,
    0x01, 0x6c, 0x17, 0x55, 0x66, 0xdb, 0x88, 0x3d,
    0x95, 0x34, 0x09, 0x7a, 0x50, 0xc9, 0x9c, 0xce,
    0x0b, 0x0e, 0x8f, 0x1b, 0x11, 0xdb, 0xfb, 0x4d,
    0x30, 0x2b, 0xad, 0x9c, 0x2c, 0x9f, 0xa5, 0x72,
    0x27, 0xc8, 0x82, 0x5c, 0x75, 0x4a, 0xda, 0x7d,
    0xb3, 0xf3, 0x2d, 0x22, 0xf5, 0x1c, 0xda, 0x9f,
    0xc5, 0x80, 0x59, 0x4f, 0x34, 0x2d, 0x1d, 0x6a,
    0x49, 0xf1, 0x5b, 0xc1, 0x33, 0x71, 0x8b, 0x54,
    0xda, 0xc0, 0xa6, 0xdd, 0x05, 0x17, 0x6e, 0x7d,
    0xf0, 0xd9, 0x1f, 0x6d, 0xa8, 0x5a, 0x63, 0x81,
    0x29, 0x09, 0x0a, 0x96, 0xd9, 0xcf, 0x29, 0x87,
    0x78, 0x4d, 0x20, 0x64, 0x16, 0x3d, 0xcc, 0x71,
    0x07, 0x0f, 0x64, 0x41, 0xe2, 0x3f, 0x40, 0xa9,
    0x77, 0xfe, 0x64, 0x43, 0x02, 0xbc, 0x31, 0x7f,
    0xd6, 0x24, 0x79, 0xd0, 0x69, 0xab, 0x11, 0x8b,
    0x0f, 0xd3, 0x78, 0xc3, 0xe5, 0xcf, 0xbd, 0xb8,
    0x28, 0xe7, 0xd3, 0xa1, 0xf0, 0x69, 0xc0, 0x37,
    0x4c, 0x1f, 0xc6, 0x2a, 0xe6, 0xd8, 0x31, 0x24,
    0x11, 0xbc, 0x70, 0x67, 0x95, 0x75, 0x72, 0xb1,
    0x57, 0x45, 0xa9, 0x43, 0xec, 0xfd, 0xf9, 0x50,
    0x50, 0xef, 0x70, 0x16, 0xcd, 0xb4, 0x73, 0x8c,
    0x10, 0xc5, 0x32, 0xed, 0xef, 0xf1, 0x2e, 0x4b,
    0xc3, 0x84, 0xf2, 0x2f, 0xab, 0xf5, 0xe6, 0x1a,
    0x48, 0xe8, 0x05, 0x88, 0x00, 0x00, 0x00, 0x00
};



/* 32 bytes */
static const unsigned char expect_ctx_out[] =
{
    /*  computed mac out */
    0x4f, 0x4a, 0xf3, 0x3f, 0x10, 0x29, 0xfc, 0xb2,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,

    /* mic out */
    0x46, 0x39, 0xdc, 0x6c, 0x8e, 0x9c, 0xf4, 0xb9,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
};


static unsigned char cipher[976 + 32];


int testccmp(int fd)
{

    CCMP_REQ CcmpReq;
    int status;
    
    
    memset(&CcmpReq, 0, sizeof(CcmpReq));

    CcmpReq.opId           = DPD_802_11_CCMP_OUTBOUND;

    /* inputs */
    CcmpReq.ctxBytes       = 56;
    CcmpReq.context        = (unsigned char *)context;

    CcmpReq.keyBytes       = 16;
    CcmpReq.keyData        = (unsigned char *)key;

    CcmpReq.AADBytes       = 32;
    CcmpReq.AADData        = (unsigned char *)AAheader;

    CcmpReq.FrameDataBytes = 972;
    CcmpReq.FrameData      = (unsigned char *)plaintext;

    /* outputs */
    CcmpReq.cryptDataBytes = 972;
    CcmpReq.cryptDataOut   = cipher;

    CcmpReq.MICBytes       = 32;
    CcmpReq.MICData        = &cipher[CcmpReq.cryptDataBytes];


    status = putKmem(fd, (void *)context, (void **)&CcmpReq.context, CcmpReq.ctxBytes);
    if (status)
        return status;
    
    status = putKmem(fd, (void *)key, (void **)&CcmpReq.keyData,  CcmpReq.keyBytes);
    if (status) {
        freeKmem(fd, (void **)&CcmpReq.context);
        return status;
    }

    status = putKmem(fd, (void *)plaintext, (void **)&CcmpReq.FrameData,  CcmpReq.FrameDataBytes);
    if (status) {
        freeKmem(fd, (void **)&CcmpReq.context);
        freeKmem(fd, (void **)&CcmpReq.keyData);
        return status;
    }

    status = putKmem(fd, (void *)AAheader, (void **)&CcmpReq.AADData,  CcmpReq.AADBytes);
    if (status) {
        freeKmem(fd, (void **)&CcmpReq.context);
        freeKmem(fd, (void **)&CcmpReq.keyData);
        freeKmem(fd, (void **)&CcmpReq.FrameData);
        return status;
    }

    status = putKmem(fd, NULL, (void **)&CcmpReq.MICData,  CcmpReq.MICBytes);
    if (status) {
        freeKmem(fd, (void **)&CcmpReq.context);
        freeKmem(fd, (void **)&CcmpReq.keyData);
        freeKmem(fd, (void **)&CcmpReq.FrameData);
        freeKmem(fd, (void **)&CcmpReq.AADData);
        return status;
    }

    status = putKmem(fd, NULL, (void **)&CcmpReq.cryptDataOut,  CcmpReq.cryptDataBytes);
    if (status) {
        freeKmem(fd, (void **)&CcmpReq.context);
        freeKmem(fd, (void **)&CcmpReq.keyData);
        freeKmem(fd, (void **)&CcmpReq.FrameData);
        freeKmem(fd, (void **)&CcmpReq.AADData);
        freeKmem(fd, (void **)&CcmpReq.MICData);
        return status;
    }

    armCompletion(&CcmpReq);
    status = ioctl(fd, IOCTL_PROC_REQ, (int)&CcmpReq);
    if ((status = waitCompletion("testccmp(): CCMP packet test", status, &CcmpReq)))
    {
        freeKmem(fd, (void **)&CcmpReq.context);
        freeKmem(fd, (void **)&CcmpReq.keyData);
        freeKmem(fd, (void **)&CcmpReq.FrameData);
        freeKmem(fd, (void **)&CcmpReq.AADData);
        freeKmem(fd, (void **)&CcmpReq.MICData);
        freeKmem(fd, (void **)&CcmpReq.cryptDataOut);
        return -1;
    }

    getKmem(fd, cipher, (void **)&CcmpReq.cryptDataOut, CcmpReq.cryptDataBytes);
    getKmem(fd, &cipher[CcmpReq.cryptDataBytes], (void **)&CcmpReq.MICData, CcmpReq.MICBytes);
    
    freeKmem(fd, (void **)&CcmpReq.context);
    freeKmem(fd, (void **)&CcmpReq.keyData);
    freeKmem(fd, (void **)&CcmpReq.FrameData);
    freeKmem(fd, (void **)&CcmpReq.AADData);
    freeKmem(fd, (void **)&CcmpReq.MICData);
    freeKmem(fd, (void **)&CcmpReq.cryptDataOut);

    if (memcmp((unsigned char *)expected_out, cipher, CcmpReq.cryptDataBytes) != 0)
    {
        printf("PAYLOAD OUT DOES NOT MATCH:\n");
        dumpm(cipher, CcmpReq.cryptDataBytes);
        return -1;
    }

    printf("Payload Out Matches Expected\n");

    if (memcmp((unsigned char *)expect_ctx_out, &cipher[CcmpReq.cryptDataBytes], 16) != 0)
    {
        printf("Computed MAC OUT DOES NOT MATCH:\n");
        dumpm(&cipher[CcmpReq.cryptDataBytes], 16);
        return -1;
    }

    printf("Computed MAC Matches Expected\n");

    if (memcmp((unsigned char *) &expect_ctx_out[16], &cipher[CcmpReq.cryptDataBytes + 16], 16) != 0)
    {           
        printf("MIC OUT DOES NOT MATCH:\n");
        dumpm(&cipher[CcmpReq.cryptDataBytes + 16], 16);
        return -1;
    }

    printf ("MIC OUT Matches Expected\n");
    
    return 0;
}

