/*
 * (C) Copyright Nissin Systems 2008
 *
 * See file CREDITS for list of people who contributed to this
 * project.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 * MA 02111-1307 USA
 */

#include <linux/config.h>
#include <linux/module.h>
#include <linux/moduleparam.h>
#include <linux/device.h>
#include <linux/devfs_fs_kernel.h>
#include <linux/cdev.h>
#include <linux/ctype.h>
#include <asm/uaccess.h>
#include <asm/io.h>
#ifdef CONFIG_PROC_FS
#include <linux/proc_fs.h>
#endif
#include <linux/mifaxadp-pldio.h>

MODULE_AUTHOR("Nissin Systems Co..Ltd.");
MODULE_DESCRIPTION("MITSUBISHI FAXADP PLDIO driver.");
MODULE_LICENSE("GPL");

#define PLDIO_MODULE_NAME "pldio"
#define PLDIO_VERSION "1.00"

#define PLDIO_PHYS_BASE 0x80000000
#define PLDIO_SIZE 0x20

#define PLDIO_REGS_TB_OFF 0x00
#define PLDIO_REGS_MODE_OFF 0x01
#define PLDIO_REGS_LEDDBG_OFF 0x02
#define PLDIO_REGS_LEDPWR_OFF 0x03
#define PLDIO_REGS_LEDAIR_OFF 0x04
#define PLDIO_REGS_LEDFTX_OFF 0x05
#define PLDIO_REGS_LEDFRX_OFF 0x06
#define PLDIO_REGS_GPIO_OFF 0x07
#define PLDIO_REGS_UFM0_OFF 0x08
#define PLDIO_REGS_UFM1_OFF 0x09
#define PLDIO_REGS_UFM2_OFF 0x0a
#define PLDIO_REGS_UFM3_OFF 0x0b
#define PLDIO_REGS_INTS_OFF 0x10
#define PLDIO_REGS_INTM_OFF 0x11
#define PLDIO_REGS_WDTCLR_OFF 0x14
#define PLDIO_REGS_VER_OFF 0x1f

#define PLDIO_IS_WO(off) (PLDIO_REGS_WDTCLR_OFF == off)

#define PLDIO_REGS_TB_WRITE_MASK 0x9f
#define PLDIO_REGS_MODE_WRITE_MASK 0x1f
#define PLDIO_REGS_LEDDBG_WRITE_MASK 0x02
#define PLDIO_REGS_LEDPWR_WRITE_MASK 0x3f
#define PLDIO_REGS_LEDAIR_WRITE_MASK 0x3f
#define PLDIO_REGS_LEDFTX_WRITE_MASK 0x3f
#define PLDIO_REGS_LEDFRX_WRITE_MASK 0x3f
#define PLDIO_REGS_GPIO_WRITE_MASK 0x00
#define PLDIO_REGS_UFM0_WRITE_MASK 0x00
#define PLDIO_REGS_UFM1_WRITE_MASK 0x00
#define PLDIO_REGS_UFM2_WRITE_MASK 0x00
#define PLDIO_REGS_UFM3_WRITE_MASK 0x00
#define PLDIO_REGS_INTS_WRITE_MASK 0x00
#define PLDIO_REGS_INTM_WRITE_MASK 0x00
#define PLDIO_REGS_WDTCLR_WRITE_MASK 0xff

typedef struct mifaxadp_pldio_dev_t {
	struct cdev char_dev;
	struct class *class;
	unsigned int virt_base;
#ifdef CONFIG_PROC_FS
	struct proc_dir_entry *p_tb;
	struct proc_dir_entry *p_mode;
	struct proc_dir_entry *p_leddbg;
	struct proc_dir_entry *p_ledpwr;
	struct proc_dir_entry *p_ledair;
	struct proc_dir_entry *p_ledftx;
	struct proc_dir_entry *p_ledfrx;
	struct proc_dir_entry *p_gpio;
	struct proc_dir_entry *p_ufm0;
	struct proc_dir_entry *p_ufm1;
	struct proc_dir_entry *p_ufm2;
	struct proc_dir_entry *p_ufm3;
	struct proc_dir_entry *p_ints;
	struct proc_dir_entry *p_intm;
	struct proc_dir_entry *p_wdtclr;
	struct proc_dir_entry *p_ver;
#endif
} MIFAXADP_PLDIO_DEV;

/* Initialize default major number */
static int major_num = 254; 
module_param(major_num, int, S_IRUGO);
MODULE_PARM_DESC(major_num, "Number of major");

static MIFAXADP_PLDIO_DEV *pldio_dev;
static spinlock_t pldio_lock = SPIN_LOCK_UNLOCKED;

static int pldio_open(struct inode *inode, struct file *file)
{
	return 0;
}

static int pldio_close(struct inode *inode, struct file *file)
{
	return 0;
}

static ssize_t pldio_read(struct file *file, char *buf, size_t count, loff_t *ppos)
{
	return -ENOSYS;
}

static ssize_t pldio_write(struct file *file, const char __user *buf, size_t count, 
		loff_t *ppos)
{
	return -ENOSYS;
}

static int pldio_get_reg(unsigned int offset)
{
	unsigned char reg_val;

	switch (offset) {
	case PLDIO_REGS_TB_OFF:
	case PLDIO_REGS_MODE_OFF:
	case PLDIO_REGS_LEDDBG_OFF:
	case PLDIO_REGS_LEDPWR_OFF:
	case PLDIO_REGS_LEDAIR_OFF:
	case PLDIO_REGS_LEDFTX_OFF:
	case PLDIO_REGS_LEDFRX_OFF:
	case PLDIO_REGS_GPIO_OFF:
	case PLDIO_REGS_UFM0_OFF:
	case PLDIO_REGS_UFM1_OFF:
	case PLDIO_REGS_UFM2_OFF:
	case PLDIO_REGS_UFM3_OFF:
	case PLDIO_REGS_INTS_OFF:
	case PLDIO_REGS_INTM_OFF:
	case PLDIO_REGS_VER_OFF:
		break;

	default:
		return -1;
	}
	
	spin_lock(&pldio_lock);
	reg_val = __raw_readb((void *)(pldio_dev->virt_base + offset));
	spin_unlock(&pldio_lock);

	return reg_val;
}

static int pldio_set_reg(unsigned int offset, unsigned char value)
{
	switch (offset) {
	case PLDIO_REGS_TB_OFF:
		value |= PLDIO_REGS_TB_WRITE_MASK;
		break;

	case PLDIO_REGS_MODE_OFF:
		value |= PLDIO_REGS_MODE_WRITE_MASK;
		break;

	case PLDIO_REGS_LEDDBG_OFF:
		value |= PLDIO_REGS_LEDDBG_WRITE_MASK;
		break;

	case PLDIO_REGS_LEDPWR_OFF:
		value |= PLDIO_REGS_LEDPWR_WRITE_MASK;
		break;

	case PLDIO_REGS_LEDAIR_OFF:
		value |= PLDIO_REGS_LEDAIR_WRITE_MASK;
		break;

	case PLDIO_REGS_LEDFTX_OFF:
		value |= PLDIO_REGS_LEDFTX_WRITE_MASK;
		break;

	case PLDIO_REGS_LEDFRX_OFF:
		value |= PLDIO_REGS_LEDFRX_WRITE_MASK;
		break;

	case PLDIO_REGS_GPIO_OFF:
		value |= PLDIO_REGS_GPIO_WRITE_MASK;
		break;

	case PLDIO_REGS_UFM0_OFF:
		value |= PLDIO_REGS_UFM0_WRITE_MASK;
		break;

	case PLDIO_REGS_UFM1_OFF:
		value |= PLDIO_REGS_UFM1_WRITE_MASK;
		break;

	case PLDIO_REGS_UFM2_OFF:
		value |= PLDIO_REGS_UFM2_WRITE_MASK;
		break;

	case PLDIO_REGS_UFM3_OFF:
		value |= PLDIO_REGS_UFM3_WRITE_MASK;
		break;

	case PLDIO_REGS_INTS_OFF:
		value |= PLDIO_REGS_INTS_WRITE_MASK;
		break;

	case PLDIO_REGS_INTM_OFF:
		value |= PLDIO_REGS_INTM_WRITE_MASK;
		break;

	case PLDIO_REGS_WDTCLR_OFF:
		value |= PLDIO_REGS_WDTCLR_WRITE_MASK;
		break;

	default:
		return -1;
	}

	spin_lock(&pldio_lock);
	__raw_writeb(value, (void *)(pldio_dev->virt_base + offset));
	spin_unlock(&pldio_lock);

	return 0;
}

static int pldio_ioctl(struct inode *inode, struct file *file, unsigned int cmd,
		unsigned long arg)
{
	PLDIO_REGS usr_val;
	int ret;

	ret = copy_from_user(&usr_val, (void *)arg, sizeof(usr_val));
	if (ret) {
		return -EFAULT;
	}

	switch (cmd) {
	case PLDIO_REG_GET:
		ret = pldio_get_reg(usr_val.offset);
		if (ret == -1) {
			return -EINVAL;
		}
		usr_val.value = (unsigned char)ret;

		ret = copy_to_user((void *)arg, &usr_val, sizeof(usr_val));
		if (ret) {
			return -EFAULT;
		}
		break;

	case PLDIO_REG_SET:
		ret = pldio_set_reg(usr_val.offset, usr_val.value);
		if (ret) {
			return -EINVAL;
		}
		break;

	default:
		return -ENOIOCTLCMD;
	}

	return 0;
}

static struct file_operations mixfax_pldio_fops = {
	open: pldio_open,
	release: pldio_close,
	read: pldio_read,
	write: pldio_write,
	ioctl: pldio_ioctl
};

static int binary_to_str(char *buf, unsigned char num)
{
	char *bp = buf;
	int n;
	int cnt = 7;

	do {
		n = num >> cnt & 0x1;
		*bp++ = n + '0';
	} while (cnt--);

	*bp++ = '\n';
	*bp = '\0';

	return bp - buf;
}

static int pldio_proc_read(char *page, char **start, off_t off, int count, int *eof,
		void *data)
{
	unsigned char reg_val;
	unsigned int offset = (unsigned int)data;
	int ret;
	int len;
	

	if (count < 9) {  /* length of output string */
		return -EFAULT;
	}
	ret = pldio_get_reg(offset);
	if (ret == -1) {
		return -EINVAL;
	}
	reg_val = (unsigned char)ret;
	len = binary_to_str(page, reg_val);

	return len;
}

static int str_to_binary(unsigned char *mask, const char *buf, int len)
{
	const char *bp = buf;
	int n;
	int num = 0;
	int cnt = 7;

	*mask = 0;
	while (len--) {
		if (!isdigit(*bp)) {
			if (*bp == '\n' || *bp == '\0') {
				break;
			}
			return -1;
		}
		n = *bp - '0';
		if (n != 1 && n != 0) {
			return -1;
		}
		bp++;
		*mask |= 1 << cnt;
		num |= n << cnt--;
	}

	return num;
}

static int pldio_proc_write(struct file *file, const char *buffer, unsigned long count,
		void *data)
{
	int ret;
	unsigned int offset = (unsigned int)data;
	unsigned char usr_val;
	unsigned char reg_val;
	unsigned char usr_mask;

	if (count > 9) {
		return -EFAULT;
	}
	ret = str_to_binary(&usr_mask, buffer, count);
	if (ret == -1) {
		return -EINVAL;
	}
	usr_val = (unsigned char)ret;
	
	if (!PLDIO_IS_WO(offset)) {
		ret = pldio_get_reg(offset);
		if (ret == -1) {
			return -EINVAL;
		}
		reg_val = (unsigned char)ret;
	} else {
		reg_val = 0;
	}
	usr_val |= (reg_val & ~usr_mask);

	ret = pldio_set_reg(offset, usr_val);
	if (ret) {
		return -EINVAL;
	}

	return count;
}

#ifdef CONFIG_PROC_FS
#define SET_PROC(a, b) \
{ \
	pldio_dev->a = create_proc_entry(#a, 0, NULL); \
	pldio_dev->a->read_proc = pldio_proc_read; \
	pldio_dev->a->write_proc = pldio_proc_write; \
	pldio_dev->a->data = (void*)b; \
}
static void setup_proc_fs(void)
{
	SET_PROC(p_tb, PLDIO_REGS_TB_OFF);
	SET_PROC(p_mode, PLDIO_REGS_MODE_OFF);
	SET_PROC(p_leddbg, PLDIO_REGS_LEDDBG_OFF);
	SET_PROC(p_ledpwr, PLDIO_REGS_LEDPWR_OFF);
	SET_PROC(p_ledair, PLDIO_REGS_LEDAIR_OFF);
	SET_PROC(p_ledftx, PLDIO_REGS_LEDFTX_OFF);
	SET_PROC(p_ledfrx, PLDIO_REGS_LEDFRX_OFF);
	SET_PROC(p_gpio, PLDIO_REGS_GPIO_OFF);
	SET_PROC(p_ufm0, PLDIO_REGS_UFM0_OFF);
	SET_PROC(p_ufm1, PLDIO_REGS_UFM1_OFF);
	SET_PROC(p_ufm2, PLDIO_REGS_UFM2_OFF);
	SET_PROC(p_ufm3, PLDIO_REGS_UFM3_OFF);
	SET_PROC(p_ints, PLDIO_REGS_INTS_OFF);
	SET_PROC(p_intm, PLDIO_REGS_INTM_OFF);
	SET_PROC(p_wdtclr, PLDIO_REGS_WDTCLR_OFF);
	SET_PROC(p_ver, PLDIO_REGS_VER_OFF);
}

#define STOP_PROC(a) \
{ \
	if (pldio_dev->a) { \
		remove_proc_entry(#a, NULL); \
	} \
}
static void stop_proc_fs(void)
{
	STOP_PROC(p_tb);
	STOP_PROC(p_mode);
	STOP_PROC(p_leddbg);
	STOP_PROC(p_ledpwr);
	STOP_PROC(p_ledair);
	STOP_PROC(p_ledftx);
	STOP_PROC(p_ledfrx);
	STOP_PROC(p_gpio);
	STOP_PROC(p_ufm0);
	STOP_PROC(p_ufm1);
	STOP_PROC(p_ufm2);
	STOP_PROC(p_ufm3);
	STOP_PROC(p_ints);
	STOP_PROC(p_intm);
	STOP_PROC(p_wdtclr);
	STOP_PROC(p_ver);
}
#endif

static int __init pldio_init(void)
{
	dev_t dev;
	int fs_mode = S_IFCHR | S_IRUGO | S_IWUGO;
	int ret = -ENODEV;
	int rc;

	pldio_dev = (MIFAXADP_PLDIO_DEV *)kzalloc(sizeof(MIFAXADP_PLDIO_DEV), GFP_KERNEL);
	if (!pldio_dev) {
		ret = -ENOMEM;
		goto error;
	}

	pldio_dev->virt_base = (unsigned int)ioremap_nocache(PLDIO_PHYS_BASE, PLDIO_SIZE);
	if (!pldio_dev->virt_base) {
		goto error;
	}

	if (major_num) {
		dev = MKDEV(major_num, 0);
		rc = register_chrdev_region(dev, 1, PLDIO_MODULE_NAME);
		major_num = rc;
	} else {
		rc = alloc_chrdev_region(&dev, 0, 1, PLDIO_MODULE_NAME);
		major_num = MAJOR(dev);
	}
	if (rc < 0) {
		printk(KERN_WARNING "unable to get major %d for %s device\n",
				major_num, PLDIO_MODULE_NAME);
		ret = rc;
		goto error;
	}

	cdev_init(&pldio_dev->char_dev, &mixfax_pldio_fops);
	pldio_dev->char_dev.owner = THIS_MODULE;
	rc = cdev_add(&pldio_dev->char_dev, dev, 1);
	if (rc) {
		printk(KERN_NOTICE "Error %d adding pldio char device", rc);
	}

	pldio_dev->class = class_create(THIS_MODULE, PLDIO_MODULE_NAME);
	if (IS_ERR(pldio_dev->class)) {
		printk(KERN_ERR "Error creating pldio class.\n");
		goto error1;
	}
	class_device_create(pldio_dev->class, dev, NULL, PLDIO_MODULE_NAME);
	devfs_mk_cdev(dev, fs_mode, PLDIO_MODULE_NAME);

#ifdef CONFIG_PROC_FS
	setup_proc_fs();
#endif

	printk("MIFAXADP PLDIO Driver Initialized (ver:%s major:%d)\n",
			PLDIO_VERSION, major_num);
	return ret;

error1:
	cdev_del(&pldio_dev->char_dev);
	unregister_chrdev_region(dev, 1);
error:
	kfree(pldio_dev);
	return ret;
}

static void __exit pldio_exit(void)
{
#ifdef CONFIG_PROC_FS
	stop_proc_fs();
#endif
	if (major_num > 0) {
		devfs_remove(PLDIO_MODULE_NAME);
		class_device_destroy(pldio_dev->class, MKDEV(major_num, 0));
		class_destroy(pldio_dev->class);
		cdev_del(&pldio_dev->char_dev);
		unregister_chrdev_region(MKDEV(major_num, 0), 1);
		major_num = 0;
	}

	if (pldio_dev) {
		kfree(pldio_dev);
		pldio_dev = NULL;
	}
}

module_init(pldio_init);
module_exit(pldio_exit);

void pldio_wdt_reset(void)
{
	pldio_set_reg(PLDIO_REGS_MODE_OFF, 0x9f);
}
EXPORT_SYMBOL(pldio_wdt_reset);
