/*
 * (C) Copyright Nissin Systems 2008
 *
 * See file CREDITS for list of people who contributed to this
 * project.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 * MA 02111-1307 USA
 */

#include <linux/config.h>
#include <linux/module.h>
#include <linux/moduleparam.h>
#include <linux/device.h>
#include <linux/devfs_fs_kernel.h>
#include <linux/cdev.h>
#include <asm/uaccess.h>
#include <asm/io.h>
#include <linux/mifaxadp-dsphpi.h>


MODULE_AUTHOR("Nissin Systems Co..Ltd.");
MODULE_DESCRIPTION("MITSUBISHI FAXADP DSP-HPI driver.");
MODULE_LICENSE("GPL");

#define DSPHPI_MODULE_NAME "dsphpi"
#define DSPHPI_VERSION "1.00"

#define DSPHPI_PHYS_BASE 0x90000000
#define DSPHPI_SIZE 0x10000

#define DSP_PROGRAM_START 0x100
#define DSP_PROGRAM_SIZE 0xff00

#define REG_FIX_USER_START 0x0
#define REG_FIX_START 0x160
#define REG_FIX_END 0x019E

#define REG_CNG_USER_START 0x20
#define REG_CNG_START 0x120
#define REG_CNG_END 0x14F

#define DSP_STARTOFF_USER 0x00ff
#define DSP_STARTOFF 0x00fe

#define USER_TO_REG(off, user_start, reg_start) (((off - user_start) * 2) + reg_start)
#define USER_TO_REG_FIX(off) USER_TO_REG(off, REG_FIX_USER_START, REG_FIX_START)
#define USER_TO_REG_CNG(off) USER_TO_REG(off, REG_CNG_USER_START, REG_CNG_START)

#define IS_REG(off, start, end) (start <= off && off <= end)
#define IS_REG_FIX(off) IS_REG(USER_TO_REG_FIX(off), REG_FIX_START, REG_FIX_END)
#define IS_REG_CNG(off) IS_REG(USER_TO_REG_CNG(off), REG_CNG_START, REG_CNG_END)

#define IS_WO(off) (off == 0x120 || off == 0x126 || off == 0x14c)
#define IS_RO(off) (IS_REG_CNG(off) && !IS_WO(off))

typedef struct mifaxadp_dsphpi_dev_t {
	struct cdev char_dev;
	struct class *class;
	unsigned int virt_base;
} MIFAXADP_DSPHPI_DEV;

/* Initialize default major number */
static int major_num = 253; 
module_param(major_num, int, S_IRUGO);
MODULE_PARM_DESC(major_num, "Number of major");

enum dsphpi_cng_minor {
	dspst0 = 1,
	dspst1,
	dspst2,
	dspst3,
	dspst4,
	dspver,
	dspst22,
	dspst23,
	MINOR_NUM
};

#define DSPST0_OFFSET 0x120
#define DSPST1_OFFSET 0x122
#define DSPST2_OFFSET 0x124
#define DSPST3_OFFSET 0x126
#define DSPST4_OFFSET 0x128
#define DSPVER_OFFSET 0x13c
#define DSPST22_OFFSET 0x14c
#define DSPST23_OFFSET 0x14e
#define DSPST4_STRLEN 20
#define DSPVER_STRLEN 16

static MIFAXADP_DSPHPI_DEV *dsphpi_dev, *dsphpi_cng_dev[];
static spinlock_t dsphpi_lock = SPIN_LOCK_UNLOCKED;

static int dsphpi_open(struct inode *inode, struct file *file)
{
	file->private_data = (void *)iminor(inode);
	return 0;
}

static int dsphpi_close(struct inode *inode, struct file *file)
{
	return 0;
}

static unsigned short dsphpi_read_reg(unsigned int offset)
{
	unsigned short val;

	spin_lock(&dsphpi_lock);
	val = __raw_readw((void *)(offset + dsphpi_dev->virt_base));
	spin_unlock(&dsphpi_lock);

	return val;
}

static void dsphpi_write_reg(unsigned int offset, unsigned short val)
{
	spin_lock(&dsphpi_lock);
	__raw_writew(val, (void *)(offset + dsphpi_dev->virt_base));
	spin_unlock(&dsphpi_lock);
}

static void dsphpi_read_str(char *buf, unsigned int offset, ssize_t count)
{
	int i;

	spin_lock(&dsphpi_lock);
	for (i = 0; i < count; i++) {
		buf[i] = __raw_readb((void *)(offset + dsphpi_dev->virt_base + i));
	}
	spin_unlock(&dsphpi_lock);
}

static ssize_t dsphpi_read(struct file *file, char *buf, size_t count, loff_t *ppos)
{
	int ret;
	int minor;
	unsigned int offset = 0;
	char buffer[32];
	ssize_t len = sizeof(buffer);

	minor = (int)file->private_data;
	switch (minor) {
	case dspst1:
		offset = DSPST1_OFFSET;
		break;
	case dspst2:
		offset = DSPST2_OFFSET;
		break;
	case dspst23:
		offset = DSPST23_OFFSET;
		break;
	case dspst4:
		len = DSPST4_STRLEN;
		dsphpi_read_str(buffer, DSPST4_OFFSET, len);
		break;
	case dspver:
		len = DSPVER_STRLEN;
		dsphpi_read_str(buffer, DSPVER_OFFSET, len);
		break;
	default:
		return -ENOSYS;
	}
	if (offset) {
		len = sizeof(short);
		*(unsigned short *)buffer = dsphpi_read_reg(offset);
	}

	if (len > count) len = count;
	ret = copy_to_user(buf, buffer, len);
	if (ret) {
		return -EFAULT;
	}

	return len;
}

static ssize_t dsphpi_write(struct file *file, const char __user *buf, size_t count, 
		loff_t *ppos)
{
	int ret;
	int minor;
	int len;
	unsigned int offset;
	unsigned short val;

	if (count > sizeof(short)) {
		len = sizeof(short);
	} else {
		len = count;
	}
	ret = copy_from_user(&val, buf, len);
	if (ret) {
		return -EFAULT;
	}

	minor = (int)file->private_data;
	switch (minor) {
	case dspst0:
		offset = DSPST0_OFFSET;
		break;

	case dspst3:
		offset = DSPST3_OFFSET;
		break;

	case dspst22:
		offset = DSPST22_OFFSET;
		break;

	default:
		return -ENOSYS;
	}

	dsphpi_write_reg(offset, val);

	return len;
}

static int dsphpi_start(unsigned long arg)
{
	DSPHPI_START_IMAGE usr;
	int ret;
	unsigned short *bp;
	unsigned short *virt;

	ret = copy_from_user(&usr, (void *)arg, sizeof(usr));
	if (ret) {
		return -EFAULT;
	}
	if (usr.size > DSP_PROGRAM_SIZE) {
		return -EINVAL;
	}
	
	bp = (unsigned short *)usr.buffer;
	virt = (unsigned short *)(dsphpi_dev->virt_base + DSP_PROGRAM_START);

	spin_lock(&dsphpi_lock);
	while (usr.size >= 2) {
		__raw_writew(*bp, virt);
		bp++;
		virt++;
		usr.size -= 2;
	}
	if (usr.size) {
		__raw_writeb(*(unsigned char *)bp, (unsigned char *)virt);
	}
	spin_unlock(&dsphpi_lock);

	return 0;
}

static int dsphpi_reg_get(unsigned long arg)
{
	DSPHPI_REGS usr;
	int ret;
	unsigned int offset;

	ret = copy_from_user(&usr, (void *)arg, sizeof(usr));
	if (ret) {
		return -EFAULT;
	}

	if (IS_REG_FIX(usr.offset)) {
		offset = USER_TO_REG_FIX(usr.offset);
	} else if (IS_REG_CNG(usr.offset)) {
		offset = USER_TO_REG_CNG(usr.offset);
	} else {
		return -EINVAL;
	}

	if (IS_WO(offset)) {
		return -EINVAL;
	}

	usr.value = dsphpi_read_reg(offset);

	ret = copy_to_user((void *)arg, &usr, sizeof(usr));
	if (ret) {
		return -EFAULT;
	}

	return 0;
}

int dsphpi_reg_set(unsigned long arg)
{
	DSPHPI_REGS usr;
	unsigned int offset;
	int ret;

	ret = copy_from_user(&usr, (void *)arg, sizeof(usr));
	if (ret) {
		return -EFAULT;
	}

	if (IS_REG_FIX(usr.offset)) {
		offset = USER_TO_REG_FIX(usr.offset);
	} else if (IS_REG_CNG(usr.offset)) {
		offset = USER_TO_REG_CNG(usr.offset);
	} else if (usr.offset == DSP_STARTOFF_USER){
		offset = DSP_STARTOFF;
	} else {
		return -EINVAL;
	}

	if (IS_RO(offset)) {
		return -EINVAL;
	}

	dsphpi_write_reg(offset, usr.value);

	return 0;
}

static int dsphpi_ioctl(struct inode *inode, struct file *file, unsigned int cmd,
		unsigned long arg)
{
	int minor;

	minor = (int)file->private_data;
	if (minor) {
		return -ENOSYS;
	}

	switch (cmd) {
	case DSPHPI_START:
		return dsphpi_start(arg);

	case DSPHPI_REG_GET:
		return dsphpi_reg_get(arg);

	case DSPHPI_REG_SET:
		return dsphpi_reg_set(arg);

	default:
		break;
	}
	return -ENOIOCTLCMD;
}

static struct file_operations mixfax_dsphpi_fops = {
	open: dsphpi_open,
	release: dsphpi_close,
	read: dsphpi_read,
	write: dsphpi_write,
	ioctl: dsphpi_ioctl
};

static int __init dsphpi_init(void)
{
	dev_t dev;
	int fs_mode = S_IFCHR | S_IRUGO | S_IWUGO;
	int ret = -ENODEV;
	int rc;

	dsphpi_dev = (MIFAXADP_DSPHPI_DEV *)kzalloc(sizeof(MIFAXADP_DSPHPI_DEV), GFP_KERNEL);
	if (!dsphpi_dev) {
		ret = -ENOMEM;
		goto error;
	}

	dsphpi_dev->virt_base = (unsigned int)ioremap_nocache(DSPHPI_PHYS_BASE, DSPHPI_SIZE);
	if (!dsphpi_dev->virt_base) {
		goto error;
	}

	if (major_num) {
		dev = MKDEV(major_num, 0);
		rc = register_chrdev_region(dev, MINOR_NUM, DSPHPI_MODULE_NAME);
		major_num = rc;
	} else {
		rc = alloc_chrdev_region(&dev, 0, MINOR_NUM, DSPHPI_MODULE_NAME);
		major_num = MAJOR(dev);
	}
	if (rc < 0) {
		printk(KERN_WARNING "unable to get major %d for %s device\n",
				major_num, DSPHPI_MODULE_NAME);
		ret = rc;
		goto error;
	}

	cdev_init(&dsphpi_dev->char_dev, &mixfax_dsphpi_fops);
	dsphpi_dev->char_dev.owner = THIS_MODULE;
	rc = cdev_add(&dsphpi_dev->char_dev, dev, MINOR_NUM);
	if (rc) {
		printk(KERN_NOTICE "Error %d adding dsphpi char device", rc);
	}

	dsphpi_dev->class = class_create(THIS_MODULE, DSPHPI_MODULE_NAME);
	if (IS_ERR(dsphpi_dev->class)) {
		printk(KERN_ERR "Error creating dsphpi class.\n");
		goto error1;
	}
	class_device_create(dsphpi_dev->class, dev, NULL, DSPHPI_MODULE_NAME);
	devfs_mk_cdev(dev, fs_mode, DSPHPI_MODULE_NAME);

	printk("MIFAXADP dsphpi Driver Initialized (ver:%s major:%d)\n",
			DSPHPI_VERSION, major_num);

	return ret;

error1:
	cdev_del(&dsphpi_dev->char_dev);
	unregister_chrdev_region(dev, 1);
error:
	kfree(dsphpi_dev);
	return ret;
}

static void __exit dsphpi_exit(void)
{
	if (major_num > 0) {
		devfs_remove(DSPHPI_MODULE_NAME);
		class_device_destroy(dsphpi_dev->class, MKDEV(major_num, 0));
		class_destroy(dsphpi_dev->class);
		cdev_del(&dsphpi_dev->char_dev);
		unregister_chrdev_region(MKDEV(major_num, 0), 1);
		major_num = 0;
	}

	if (dsphpi_dev) {
		kfree(dsphpi_dev);
		dsphpi_dev = NULL;
	}
}

module_init(dsphpi_init);
module_exit(dsphpi_exit);
