/*
 * (C) Copyright Nissin Systems 2008
 *
 * See file CREDITS for list of people who contributed to this
 * project.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 * MA 02111-1307 USA
 */

#include <linux/config.h>
#include <linux/stddef.h>
#include <linux/interrupt.h>
#include <linux/irq.h>
#include <linux/init.h>
#include <linux/module.h>
#include <linux/mc146818rtc.h>

#include <asm/mpc8260.h>
#include <asm/machdep.h>
#include <asm/io.h>
#include <asm/todc.h>
#include <asm/immap_cpm2.h>
#include <asm/delay.h>
#include "mifaxadp_rtc.h"

#define RTC_DELAY_TIME 10
#define RTC_SLAVE_ADDR_WRITE 0xD0
#define RTC_SLAVE_ADDR_READ 0xD1
#define RTC_DT 0x00010000
#define RTC_CK 0x00020000

static int getbit(volatile unsigned int *addr, unsigned int bit)
{
	int ret = 0;

	if (*addr & bit) {
		ret = 1;
	}

	return ret;
}

static void setbit(volatile unsigned int *addr, unsigned int bit)
{
	*addr |= bit;
	udelay(RTC_DELAY_TIME);
}

static void clrbit(volatile unsigned int *addr, unsigned int bit)
{
	*addr &= ~bit;
	udelay(RTC_DELAY_TIME);
}

static void rtc_start(void)
{
	volatile unsigned int *dat = (volatile unsigned int *)&cpm2_immr->im_ioport.iop_pdatd;
	volatile unsigned int *dir = (volatile unsigned int *)&cpm2_immr->im_ioport.iop_pdird;

	/* dir is set to the output */
	setbit(dir, RTC_DT | RTC_CK);

	/* start condition */
	setbit(dat, RTC_CK);
	setbit(dat, RTC_DT);
	clrbit(dat, RTC_DT);
	clrbit(dat, RTC_CK);
}

static void rtc_stop(void)
{
	volatile unsigned int *dat = (volatile unsigned int *)&cpm2_immr->im_ioport.iop_pdatd;
	volatile unsigned int *dir = (volatile unsigned int *)&cpm2_immr->im_ioport.iop_pdird;

	/* dir is set to the output */
	setbit(dir, RTC_DT | RTC_CK);

	/* stop condition */
	clrbit(dat, RTC_CK);
	clrbit(dat, RTC_DT);
	setbit(dat, RTC_CK);
	setbit(dat, RTC_DT);
}

static void rtc_write(unsigned char bit)
{
	volatile unsigned int *dat = (volatile unsigned int *)&cpm2_immr->im_ioport.iop_pdatd;
	volatile unsigned int *dir = (volatile unsigned int *)&cpm2_immr->im_ioport.iop_pdird;
	int i;
	unsigned char sendbits;

	sendbits = bit;

	/* dir is set to the output */
	setbit(dir, RTC_DT | RTC_CK);

	/* send bit to rtc */
	for (i = 0; i < 8; i++) {
		clrbit(dat, RTC_CK);
		if (sendbits & 0x80) {
			setbit(dat, RTC_DT);
		} else {
			clrbit(dat, RTC_DT);
		}
		setbit(dat, RTC_CK);

		sendbits <<= 1;
	}
	clrbit(dat, RTC_CK);
	setbit(dat, RTC_CK);
	clrbit(dat, RTC_CK);
}

static unsigned char rtc_read(void)
{
	volatile unsigned int *dat = (volatile unsigned int *)&cpm2_immr->im_ioport.iop_pdatd;
	volatile unsigned int *dir = (volatile unsigned int *)&cpm2_immr->im_ioport.iop_pdird;
	int i;
	unsigned char recvbits;

	/* RTC_CK dir is set to the output */
	setbit(dir, RTC_CK);

	/* RTC_DT dir is set to the input */
	clrbit(dir, RTC_DT);

	/* recv bit from rtc */
	recvbits = 0x00;
	for (i = 0; i < 8; i++) {
		recvbits <<= 1;

		clrbit(dat, RTC_CK);
		if (getbit(dat, RTC_DT)) {
			recvbits |= 0x01;
		}
		setbit(dat, RTC_CK);
	}
	clrbit(dat, RTC_CK);
	setbit(dat, RTC_CK);
	clrbit(dat, RTC_CK);

	return recvbits;
}

static void mifaxadp_rtc_init(void) {
	static int done = 0;
	volatile unsigned int *arc = (volatile unsigned int *)&cpm2_immr->im_ioport.iop_ppard;
	volatile unsigned int *sor = (volatile unsigned int *)&cpm2_immr->im_ioport.iop_psord;
	volatile unsigned int *dir = (volatile unsigned int *)&cpm2_immr->im_ioport.iop_pdird;
	volatile unsigned int *odr = (volatile unsigned int *)&cpm2_immr->im_ioport.iop_podrd;
	volatile unsigned int *dat = (volatile unsigned int *)&cpm2_immr->im_ioport.iop_pdatd;

	if (!done) {
		/* initialize ioport */
		clrbit(arc, RTC_DT | RTC_CK);
		clrbit(sor, RTC_DT | RTC_CK);
		clrbit(odr, RTC_DT | RTC_CK);
		clrbit(dir, RTC_DT | RTC_CK);
		clrbit(dat, RTC_DT | RTC_CK);

		/* initialize rtc control register */
		rtc_start();
		rtc_write(RTC_SLAVE_ADDR_WRITE);
		rtc_write(RTC_CONTROLREG);
		rtc_write(0x00);
		rtc_stop();
		done++;
	}
}

static unsigned int rtc_dummy_reg(unsigned int addr)
{
	unsigned int ret;

	switch (addr) {
	case RTC_CONTROL:
		ret = RTC_24H;
		break;
	case RTC_FREQ_SELECT:
		ret = RTC_UIP;
		break;
	case RTC_VALID:
		ret = RTC_VRT;
		break;
	default:
		ret = 0;
		break;
	}
	return ret;
}

void mifaxadp_rtc_write(unsigned int in_dat, unsigned int addr)
{
	unsigned char dat;

	mifaxadp_rtc_init();

	dat = (unsigned char)(in_dat & 0xff);
	switch (addr) {
	case RTC_SECONDS:
	case RTC_MINUTES:
	case RTC_HOURS:
	case RTC_DAY_OF_WEEK:
	case RTC_DAY_OF_MONTH:
	case RTC_MONTH:
	case RTC_YEAR:
		rtc_start();
		rtc_write(RTC_SLAVE_ADDR_WRITE);
		rtc_write(addr);
		rtc_write(dat);
		rtc_stop();
		break;
	default:
		/* nothing to do */
		break;
	}
}

unsigned int mifaxadp_rtc_read(unsigned int addr)
{
	unsigned char dat = 0;

	mifaxadp_rtc_init();

	switch (addr) {
	case RTC_SECONDS:
	case RTC_MINUTES:
	case RTC_HOURS:
	case RTC_DAY_OF_WEEK:
	case RTC_DAY_OF_MONTH:
	case RTC_MONTH:
	case RTC_YEAR:
		rtc_start();
		rtc_write(RTC_SLAVE_ADDR_WRITE);
		rtc_write(addr);
		rtc_start();
		rtc_write(RTC_SLAVE_ADDR_READ);
		dat = rtc_read();
		rtc_stop();
		break;
	default:
		dat = rtc_dummy_reg(addr);
		break;
	}

	return dat;
}

EXPORT_SYMBOL(mifaxadp_rtc_write);
EXPORT_SYMBOL(mifaxadp_rtc_read);
EXPORT_SYMBOL(mifaxadp_rtc_init);
