#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <errno.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <limits.h>

#include "mifaxadp-dsphpi.h"

#define HEX_PREFIX "0x"
#define IS_SHORT(val) (SHRT_MIN <= val && val <= USHRT_MAX)

int dsphpi_write_register(int fd_dsphpi, unsigned int offset, unsigned short value)
{
	DSPHPI_REGS reg;

	reg.offset = offset;
	reg.value = value;

	return ioctl(fd_dsphpi, DSPHPI_REG_SET, (unsigned int)&reg);
}

unsigned short str_to_ushort(char *str, int *err)
{
	int base;
	long value;
	char *ep;

	if (!str || !err) {
		goto error;
	}

	if (strncmp(str, HEX_PREFIX, strlen(HEX_PREFIX)) == 0) {
		base = 16;
	} else {
		base = 10;
	}

	value = strtol(str, &ep, base);
	if (strncmp(ep, "", 1)) {
		goto error;
	}

	if (!IS_SHORT(value)) {
		goto error;
	}

	*err = 0;
	return (unsigned short)value;

error:
	*err = -1;
	return 0;
}

int main(int argc, char *argv[])
{
	int ret = -1;
	int fd_dsphpi = -1;
	char *dsphpi_device = NULL;
	unsigned int offset;
	unsigned short value;

	if (argc != 4) {
		printf("usage : %s [dsphpi device file] [register offset] [write value]\n", argv[0]);
		goto err;
	}

	dsphpi_device = argv[1];
	offset = str_to_ushort(argv[2], &ret);
	if (ret) {
		printf("[register offset] is an illegal value.\n");
		goto err;
	}
	value = str_to_ushort(argv[3], &ret);
	if (ret) {
		printf("[write value] is an illegal value.\n");
		goto err;
	}

	fd_dsphpi = open(dsphpi_device, O_RDWR);
	if (fd_dsphpi == -1) {
		perror("open");
		goto err;
	}
	
	ret = dsphpi_write_register(fd_dsphpi, offset, value);
	if (ret) {
		printf("Failed in writing in dsp.\n");
		printf(" register offset = 0x%x, write value = 0x%x\n", offset, value);
		goto err;
	}

err:
	if (fd_dsphpi >= 0) {
		close(fd_dsphpi);
	}

	return ret;
}
