#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <errno.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

#include "mifaxadp-dsphpi.h"

#define DSPIMAGE_MAX_SIZE 0xff00
#define MTD_IS_EMPTY(byte) (byte == 0xff)

static int get_dspimage_size(char *image_buf, int buf_size)
{
	unsigned char *cp;
	int size = buf_size;
	cp = image_buf + buf_size - 1;

	do {
		if (!MTD_IS_EMPTY(*cp)) {
			break;
		}
		cp--;
	} while (size--);

	return size;
}

static int read_dspimage(int fd, char *image, int size)
{
	int rc = 0;
	ssize_t ret;
	ssize_t rd_size = 0;
	
	while (rd_size != size) {
		ret = read(fd, image, size);
		if (!ret) {
			rc = -1;
			break;
		}
		rd_size += ret;
	}

	return rc;
}

static int write_dspimage(int fd, char *image, int size)
{
	DSPHPI_START_IMAGE dsp;
	
	dsp.buffer = image;
	dsp.size = size;

	return ioctl(fd, DSPHPI_START, (unsigned int)&dsp);
}

int init_dspimage(int fd_dsphpi, int fd_mtd, char *dspimage, int dspimage_max_size)
{
	int ret;

	ret = read_dspimage(fd_mtd, dspimage, dspimage_max_size);
	if (ret) {
		printf("dspimage in cannot be read\n");
		return -1;
	}
	ret = write_dspimage(fd_dsphpi, dspimage, get_dspimage_size(dspimage, dspimage_max_size));
	if (ret) {
		printf("dspimage cannot be written\n");
		return -1;
	}
	return 0;
}

int main(int argc, char *argv[])
{
	int ret = -1;
	int fd_dsphpi = -1;
	int fd_mtd = -1;
	char *dspimage = NULL;
	char *dsphpi_device = NULL;
	char *mtd_device = NULL;

	if (argc != 3) {
		printf("usage : %s [dsphpi device file] [mtd device file with dspimage]\n", argv[0]);
		goto err;
	}

	dsphpi_device = argv[1];
	mtd_device = argv[2];

	fd_dsphpi = open(dsphpi_device, O_RDWR);
	if (fd_dsphpi == -1) {
		perror("open");
		goto err;
	}
	fd_mtd = open(mtd_device, O_RDONLY);
	if (fd_mtd == -1) {
		perror("open");
		goto err;
	}
	dspimage = (char *)malloc(DSPIMAGE_MAX_SIZE);
	if (!dspimage) {
		perror("malloc");
		goto err;
	}
	
	ret = init_dspimage(fd_dsphpi, fd_mtd, dspimage, DSPIMAGE_MAX_SIZE);
	if (ret) {
		printf("Failed in the initialize of dsp\n");
		goto err;
	}
	printf("Succeeded in the initialize of dsp\n");

err:
	if (dspimage) {
		free(dspimage);
	}
	if (fd_mtd >= 0) {
		close(fd_mtd);
	}
	if (fd_dsphpi >= 0) {
		close(fd_dsphpi);
	}

	return ret;
}
